﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// The original file is
//   Programs/Eris/Sources/Libraries/settings/detail/settings_SystemSettingsDatabase.cpp
// It should be checked if the orignal would be changed.

#pragma once

#define SAME_TO_ORIGINAL

#include <algorithm>
#include <cmath>
#include <cstddef>
#include <cstring>
#include <functional>
#include <mutex>
#include <new>
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Keyboard.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/system/settings_Backlight.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_DataDeletion.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Fatal.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/settings/system/settings_Mii.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/settings/system/settings_News.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_SixAxisSensor.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Telemetry.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/system/settings_Web.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/time/time_LocationName.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_Uuid.h>

#ifndef SAME_TO_ORIGINAL

#include "settings_CalibrationBinaryDatabase.h"
#include "settings_FirmwareDebugSettingsDatabase.h"
#include "settings_LockableMutexType.h"
#include "settings_Platform.h"
#include "settings_PrivateSettingsDatabase.h"
#include "settings_Spl.h"
#include "settings_SystemSettingsDatabase.h"
#include "settings_Uuid.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemSaveData-os.win.h"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2017) && defined(NN_SDK_BUILD_DEBUG)
#undef offsetof
#define offsetof(type, member) \
    (reinterpret_cast<size_t>(&(reinterpret_cast<type*>(0)->member)))
#endif

//!< システム設定の警告ログを出力します。
#define NN_SETTINGS_SYSTEM_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[system settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< システムセーブデータの識別子
const ::nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000050;

//!< システムセーブデータの合計サイズ
const int64_t SystemSaveDataTotalSize = (512 + 32) << 10;

//!< システムセーブデータのジャーナルサイズ
const int64_t SystemSaveDataJournalSize = (512 + 32) << 10;

//!< マウント名
const char MountName[] = "SystemSettings";

//!< 利用可能な言語コード
const LanguageCode AvailableLanguageCodes[] =
{
    { "ja" },
    { "en-US" },
    { "fr" },
    { "de" },
    { "it" },
    { "es" },
    { "zh-CN" },
    { "ko" },
    { "nl" },
    { "pt" },
    { "ru" },
    { "zh-TW" },
    { "en-GB" },
    { "fr-CA" },
    { "es-419" },
    { "zh-Hans" },
    { "zh-Hant" },
};

//!< 言語コードのデフォルト値
const LanguageCode DefaultLanguageCode = AvailableLanguageCodes[1];

//!< ネットワーク設定のデフォルト値
const ::nn::settings::system::NetworkSettings DefaultNetworkSettings[] =
{
    {
        "default-ethernet",
        {
            0xbb, 0x6c, 0x70, 0x7e, 0x00, 0xab, 0x11, 0xe6,
            0x94, 0x44, 0xd8, 0xcb, 0x8a, 0xc3, 0x46, 0xa9,
        },
        { { 0x00000003 } },
        static_cast<int32_t>(::nn::settings::system::NicType_Ethernet),
        {},
        {},
        {
            { { 0x00000003 } },
            {},
            {},
            {},
            {},
            {},
        },
        {
            { { 0x00000000 } },
            {},
            80,
            {},
            {},
        },
        1400,
    },
};

//!< リージョンコードのデフォルト値
const int32_t DefaultRegionCode =
    static_cast<int32_t>(::nn::settings::system::RegionCode_Usa);

//!< 携帯モード時誤操作防止画面の有無のデフォルト値
const bool DefaultLockScreenFlag = true;

//!< カラーセット ID のデフォルト値
const int32_t DefaultColorSetId = 0;

//!< バッテリー残量のパーセンテージ表示を行うか否か表す値のデフォルト値
const bool DefaultBatteryPercentageFlag = false;

//!< アプレット起動の制御フラグのデフォルト値
const ::nn::settings::system::AppletLaunchFlagSet DefaultAppletLaunchFlags =
{
    {}
};

//!< Quest メニューを起動するか否かを表す値のデフォルト値
const bool DefaultQuestFlag = false;

//!< 画面の明るさの設定のデフォルト値
const ::nn::settings::system::BacklightSettingsEx DefaultBacklightSettings =
{
    { { 0x00000001 } },
    0.624f,
    0.843f,
    { { { 0.0f, 1.0f, 0.0f } } },
    { { { 0.0f, 1.0f, 0.0f } } },
    1.0f,
    2.0f,
};

//!< UserSystemClock のコンテキストのデフォルト値
const ::nn::settings::system::SystemClockContext DefaultUserSystemClockContext =
{
    0LL,
    {
        0LL,
        ::nn::util::InvalidUuid,
    }
};

//!< NetworkSystemClock のコンテキストのデフォルト値
const ::nn::settings::system::SystemClockContext
    DefaultNetworkSystemClockContext =
{
    0LL,
    {
        0LL,
        ::nn::util::InvalidUuid,
    }
};

//!< UserSystemClock の自動補正フラグのデフォルト値
const bool DefaultUserSystemClockAutomaticCorrectionFlag = true;

//!< アカウント設定のデフォルト値
const ::nn::settings::system::AccountSettings DefaultAccountSettings =
{
    {},
};

//!< スピーカーの本体音量設定のデフォルト値
const ::nn::settings::system::AudioVolume DefaultSpeakerAudioVolume =
{
    {},
    10
};

//!< ヘッドフォンの本体音量設定のデフォルト値
const ::nn::settings::system::AudioVolume DefaultHeadphoneAudioVolume =
{
    { { 0x00000002 } },
    7
};

//!< USB オーディオの本体音量設定のデフォルト値
const ::nn::settings::system::AudioVolume DefaultUsbAudioVolume =
{
    {},
    8
};

//!< HDMI の本体オーディオ出力設定のデフォルト値
const int32_t DefaultHdmiAudioOutputMode = static_cast<int32_t>(
    ::nn::settings::system::AudioOutputMode_2ch);

//!< スピーカーの本体オーディオ出力設定のデフォルト値
const int32_t DefaultSpeakerAudioOutputMode = static_cast<int32_t>(
    ::nn::settings::system::AudioOutputMode_2ch);

//!< ヘッドフォンの本体オーディオ出力設定のデフォルト値
const int32_t DefaultHeadphoneAudioOutputMode = static_cast<int32_t>(
    ::nn::settings::system::AudioOutputMode_2ch);

//!< USB オーディオの本体オーディオ出力設定のデフォルト値
const int32_t DefaultUsbAudioOutputMode = static_cast<int32_t>(
    ::nn::settings::system::AudioOutputMode_2ch);

//!< ヘッドフォン抜け時の強制ミュート設定のデフォルト値
const bool DefaultForceMuteOnHeadphoneRemovedFlag = true;

//!< ヘッドフォン音量の警告表示回数のデフォルト値
const int32_t DefaultHeadphoneVolumeWarningCount = 0;

//!< ヘッドフォンの本体音量が更新済みか否かを表す値のデフォルト値
const bool DefaultHeadphoneVolumeUpdateFlag = true;

//!< ヘッドフォンの本体音量の変換テーブル
const int8_t HeadphoneVolumeRemappingTable[] = {
    0, 1, 1, 1, 2, 3, 3, 4, 5, 5, 6, 6, 7, 7, 8, 9,
};

//!< 本体情報アップロードが有効か否かを表す値のデフォルト値
const bool DefaultConsoleInformationUploadFlag = false;

//!< アプリケーション自動配信が有効か否かを表す値のデフォルト値
const bool DefaultAutomaticApplicationDownloadFlag = false;

//!< 通知設定のデフォルト値
const ::nn::settings::system::NotificationSettings DefaultNotificationSettings =
{
    { { 0x00000303 } },
    static_cast<int32_t>(::nn::settings::system::NotificationVolume_High),
    {  9, 00 },
    { 21, 00 },
};

//!< Pro Controller 有線 USB 通信機能が有効か否かを表す値のデフォルト値
const bool DefaultUsbFullKeyEnableFlag = false;

//!< 振動のマスターボリュームのデフォルト値
const float DefaultVibrationMasterVolume = 1.0f;

//!< TV 設定のデフォルト値
const ::nn::settings::system::TvSettings DefaultTvSettings =
{
    { { 0x0000000C } },
    static_cast<int32_t>(::nn::settings::system::TvResolution_Auto),
    static_cast<int32_t>(::nn::settings::system::HdmiContentType_Game),
    static_cast<int32_t>(::nn::settings::system::RgbRange_Auto),
    static_cast<int32_t>(::nn::settings::system::CmuMode_None),
    0,
    1.0f,
    0.5f,
};

//!< おすすめ削除設定のデフォルト値
const ::nn::settings::system::DataDeletionSettings DefaultDataDeletionSettings =
{
    {},
    0,
};

//!< 初回ブートするシステムアプレットの ProgramId のデフォルト値
const ::nn::ncm::ProgramId DefaultInitialSystemAppletProgramId =
    ::nn::ncm::ProgramId::GetInvalidId();

//!< 初回ブートするオーバーレイ表示の ProgramId のデフォルト値
const ::nn::ncm::ProgramId DefaultOverlayDispProgramId =
    ::nn::ncm::ProgramId::GetInvalidId();

//!< デバイスの地域名（タイムゾーン名）のデフォルト値
const ::nn::time::LocationName DefaultDeviceTimeZoneLocationName = { "UTC" };

//!< 撮影した画面写真を保存するストレージのデフォルト値
const int DefaultPrimaryAlbumStorage =
    static_cast<int>(::nn::settings::system::PrimaryAlbumStorage_SdCard);

//!< USB 3.0 が有効か否かを表す値のデフォルト値
const bool DefaultUsb30EnableFlag = false;

//!< NFC が有効か否かを表す値のデフォルト値
const bool DefaultNfcEnableFlag = true;

//!< スリープ設定のデフォルト値
const ::nn::settings::system::SleepSettings DefaultSleepSettings =
{
    { { 0x00000001 } },
    static_cast<int32_t>(::nn::settings::system::HandheldSleepPlan_10Min),
    static_cast<int32_t>(::nn::settings::system::ConsoleSleepPlan_1Hour),
};

//!< 無線 LAN が有効か否かを表す値のデフォルト値
const bool DefaultWirelessLanEnableFlag = true;

//!< Bluetooth が有効か否かを表す値のデフォルト値
const bool DefaultBluetoothEnableFlag = true;

//!< Bluetooth の AFH が有効か否かを表す値のデフォルト値
const bool DefaultBluetoothAfhEnableFlag = true;

//!< Bluetooth の Boost が有効か否かを表す値のデフォルト値
const bool DefaultBluetoothBoostEnableFlag = false;

//!< 本体の識別名のデフォルト値
const ::nn::settings::system::DeviceNickName DefaultDeviceNickName = {};

//!< ローカル通信のチャンネルのデフォルト値
const int32_t DefaultLdnChannel = -1;

//!< パッチの自動受信が有効か否かを表す値のデフォルト値
const bool DefaultAutoUpdateEnableFlag = true;

//!< 外部 RTC のリセットが発生したか否かを表す値のデフォルト値
const bool DefaultExternalRtcResetFlag = false;

//!< 修理中か否かを表す値のデフォルト値
const bool DefaultInRepairProcessFlag = false;

//!< プッシュ通知のスリープ状態における動作モードのデフォルト値
const int32_t DefaultPushNotificationActivityModeOnSleep = 0;

//!< エラーレポートを他社に提供してよいかの許可状態のデフォルト値
const int32_t DefaultErrorReportSharePermission =
    static_cast<int32_t>(
        ::nn::settings::system::ErrorReportSharePermission_NotConfirmed);

//!< 本体 6 軸センサーの加速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias
    DefaultConsoleSixAxisSensorAccelerationBias =
{
    0.f, 0.f, 0.f
};

//!< 本体 6 軸センサーの角速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias
    DefaultConsoleSixAxisSensorAngularVelocityBias =
{
    0.f, 0.f, 0.f
};

//!< 本体 6 軸センサーの加速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain
    DefaultConsoleSixAxisSensorAccelerationGain =
{
    1.f, 0.f, 0.f,
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
};

//!< 本体 6 軸センサーの角速度バイアス値のデフォルト値
const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain
    DefaultConsoleSixAxisSensorAngularVelocityGain =
{
    1.f, 0.f, 0.f,
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
};

//!< キーボード配列の種別のデフォルト値
const int32_t DefaultKeyboardLayout =
    static_cast<int32_t>(KeyboardLayout_EnglishUs);

//!< Web インスペクタを使用するか否かを表す値のデフォルト値
const bool DefaultWebInspectorFlag = false;

//!< システムセーブデータ操作排他用ミューテックス
LockableMutexType g_SystemSaveDataMutex =
{
    NN_OS_MUTEX_INITIALIZER(false)
};

//!< FATAL 表示向けダーティーフラグ
::nn::settings::system::FatalDirtyFlagSet g_FatalDirtyFlags = {
    { 0xffffffffffffffffULL, 0xffffffffffffffffULL }
};

//!< 障害情報収集向けダーティーフラグ
::nn::settings::system::TelemetryDirtyFlagSet g_TelemetryDirtyFlags = {
    { 0xffffffffffffffffULL, 0xffffffffffffffffULL }
};

//!< 値を指定された範囲に収めます。
template<typename T>
T Clamp(T value, T minValue, T maxValue) NN_NOEXCEPT
{
    return ::std::min<T>(::std::max<T>(minValue, value), maxValue);
}

//!< 画面の明るさの設定のデフォルト値を返します。
const ::nn::settings::system::BacklightSettingsEx& GetDefaultBacklightSettings(
    ) NN_NOEXCEPT;

//!< TV 設定のデフォルト値を返します。
const ::nn::settings::system::TvSettings& GetDefaultTvSettings() NN_NOEXCEPT;

//!< システムセーブデータから値を取得します。
template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータから値を取得します。
template<typename T, typename U>
::nn::Result GetSystemSaveDataValue(
    U* pOutCount, T outValues[], U count,
    ::std::function<::nn::Result(U*, T*, U, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータの値を変更します。
template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT;

//!< システムセーブデータの値を変更します。
template<typename T, typename U>
::nn::Result SetSystemSaveDataValue(
    const T values[], U count,
    ::std::function<::nn::Result(SystemSaveData*, const T*, U) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT;

//!< システム設定から値を取得します。
template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< システム設定から値を取得します。
template<typename T, typename U>
::nn::Result GetSystemSettingsValue(
    U* pOutCount, T outValues[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(U*, T*, U) NN_NOEXCEPT> getter,
    ::std::function<void(U*, T*, U) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< システム設定の値を変更します。
template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT;

//!< システム設定の値を変更します。
template<typename T, typename U>
::nn::Result SetSystemSettingsValue(
    const T values[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(const T*, U) NN_NOEXCEPT> setter) NN_NOEXCEPT;

//!< システム設定の値をリセットします。
::nn::Result ResetSystemSettingsValue(
    const char* const functionName,
    ::std::function<::nn::Result() NN_NOEXCEPT> resetter) NN_NOEXCEPT;

//!< FATAL 表示向けダーティーフラグの変更通知用イベントを取得します。
::nn::os::SystemEvent& GetFatalDirtyFlagEvent() NN_NOEXCEPT;

//!< 障害情報収集向けダーティーフラグの変更通知用イベントを取得します。
::nn::os::SystemEvent& GetTelemetryDirtyFlagEvent() NN_NOEXCEPT;

//!< システムセーブデータから言語コードを取得します。
::nn::Result GetSystemSaveDataLanguageCode(
    LanguageCode* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの言語コードを変更します。
::nn::Result SetSystemSaveDataLanguageCode(
    SystemSaveData* pSystemSaveData,
    const LanguageCode& value) NN_NOEXCEPT;

//!< システムセーブデータからネットワーク設定を取得します。
::nn::Result GetSystemSaveDataNetworkSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NetworkSettings outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのネットワーク設定を変更します。
::nn::Result SetSystemSaveDataNetworkSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NetworkSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータからリージョンコードを取得します。
::nn::Result GetSystemSaveDataRegionCode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのリージョンコードを変更します。
::nn::Result SetSystemSaveDataRegionCode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータから Bluetooth デバイス設定を取得します。
::nn::Result GetSystemSaveDataBluetoothDevicesSettings(
    int32_t* pOutCount,
    ::nn::settings::system::BluetoothDevicesSettings outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Bluetooth デバイス設定を変更します。
::nn::Result SetSystemSaveDataBluetoothDevicesSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::BluetoothDevicesSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータから Bluetooth が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataBluetoothEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Bluetooth が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataBluetoothEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから Bluetooth の AFH  が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataBluetoothAfhEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Bluetooth の AFH  が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataBluetoothAfhEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから Bluetooth の Boost が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataBluetoothBoostEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Bluetooth の Boost  が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataBluetoothBoostEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから携帯モード時誤操作防止画面の有無を取得します。
::nn::Result GetSystemSaveDataLockScreenFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの携帯モード時誤操作防止画面の有無を変更します。
::nn::Result SetSystemSaveDataLockScreenFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから本体 EULA の同意バージョンを取得します。
::nn::Result GetSystemSaveDataEulaVersions(
    int32_t* pOutCount,
    ::nn::settings::system::EulaVersion outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体 EULA の同意バージョンを変更します。
::nn::Result SetSystemSaveDataEulaVersions(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::EulaVersion values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータからエラーレポートを他社に提供してよいかの許可状態を取得します。
::nn::Result GetSystemSaveDataErrorReportSharePermission(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのエラーレポートを他社に提供してよいかの許可状態を変更します。
::nn::Result SetSystemSaveDataErrorReportSharePermission(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからカラーセット ID を取得します。
::nn::Result GetSystemSaveDataColorSetId(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのカラーセット ID を変更します。
::nn::Result SetSystemSaveDataColorSetId(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからバッテリー残量のパーセンテージ表示を行うか否か表す値を取得します。
::nn::Result GetSystemSaveDataBatteryPercentageFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのバッテリー残量のパーセンテージ表示を行うか否か表す値を変更します。
::nn::Result SetSystemSaveDataBatteryPercentageFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから初回起動設定の制御フラグを取得します。
::nn::Result GetSystemSaveDataAppletLaunchFlags(
    ::nn::settings::system::AppletLaunchFlagSet* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの初回起動設定の制御フラグを変更します。
::nn::Result SetSystemSaveDataAppletLaunchFlags(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AppletLaunchFlagSet& value) NN_NOEXCEPT;

//!< システムセーブデータから Quest メニューを起動するか否かを表す値を取得します。
::nn::Result GetSystemSaveDataQuestFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Quest メニューを起動するか否かを表す値を変更します。
::nn::Result SetSystemSaveDataQuestFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから画面の明るさの設定を取得します。
template<size_t Version = 1u>
::nn::Result GetSystemSaveDataBacklightSettings(
    ::nn::settings::system::BacklightSettingsEx* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの画面の明るさの設定を変更します。
template<size_t Version = 1u>
::nn::Result SetSystemSaveDataBacklightSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::BacklightSettingsEx& value) NN_NOEXCEPT;

//!< システムセーブデータから UserSystemClock のコンテキストを取得します。
::nn::Result GetSystemSaveDataUserSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの UserSystemClock のコンテキストを変更します。
::nn::Result SetSystemSaveDataUserSystemClockContext(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT;

//!< システムセーブデータから NetworkSystemClock のコンテキストを取得します。
::nn::Result GetSystemSaveDataNetworkSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの NetworkSystemClock のコンテキストを変更します。
::nn::Result SetSystemSaveDataNetworkSystemClockContext(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT;

//!< システムセーブデータから UserSystemClock の自動補正フラグを取得します。
::nn::Result GetSystemSaveDataUserSystemClockAutomaticCorrectionFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの UserSystemClock の自動補正フラグを変更します。
::nn::Result SetSystemSaveDataUserSystemClockAutomaticCorrectionFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータからアカウント設定を取得します。
::nn::Result GetSystemSaveDataAccountSettings(
    ::nn::settings::system::AccountSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのアカウント設定を変更します。
::nn::Result SetSystemSaveDataAccountSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AccountSettings& value) NN_NOEXCEPT;

//!< システムセーブデータからスピーカーの本体音量設定を取得します。
::nn::Result GetSystemSaveDataSpeakerAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのスピーカーの本体音量設定を変更します。
::nn::Result SetSystemSaveDataSpeakerAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT;

//!< システムセーブデータからヘッドフォンの本体音量設定を取得します。
::nn::Result GetSystemSaveDataHeadphoneAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのヘッドフォンの本体音量設定を変更します。
::nn::Result SetSystemSaveDataHeadphoneAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT;

//!< システムセーブデータから USB オーディオの本体音量設定を取得します。
::nn::Result GetSystemSaveDataUsbAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの USB オーディオの本体音量設定を変更します。
::nn::Result SetSystemSaveDataUsbAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT;

//!< システムセーブデータから HDMI の本体オーディオ出力設定を取得します。
::nn::Result GetSystemSaveDataHdmiAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの HDMI の本体オーディオ出力設定を変更します。
::nn::Result SetSystemSaveDataHdmiAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからスピーカーの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSaveDataSpeakerAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのスピーカーの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSaveDataSpeakerAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからヘッドフォンの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSaveDataHeadphoneAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのヘッドフォンの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSaveDataHeadphoneAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータから USB オーディオの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSaveDataUsbAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの USB オーディオの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSaveDataUsbAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからヘッドフォン抜け時の強制ミュート設定を取得します。
::nn::Result GetSystemSaveDataForceMuteOnHeadphoneRemovedFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのヘッドフォン抜け時の強制ミュート設定を変更します。
::nn::Result SetSystemSaveDataForceMuteOnHeadphoneRemovedFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータからヘッドフォン音量の警告表示回数を取得します。
::nn::Result GetSystemSaveDataHeadphoneVolumeWarningCount(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのヘッドフォン音量の警告表示回数を変更します。
::nn::Result SetSystemSaveDataHeadphoneVolumeWarningCount(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからヘッドフォン音量を更新済みか否かを表す値を取得します。
::nn::Result GetSystemSaveDataHeadphoneVolumeUpdateFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのヘッドフォン音量を更新済みか否かを表す値を変更します。
::nn::Result SetSystemSaveDataHeadphoneVolumeUpdateFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから本体情報アップロードが有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataConsoleInformationUploadFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体情報アップロードが有効か否かを表す値を設定します。
::nn::Result SetSystemSaveDataConsoleInformationUploadFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータからアプリケーション自動配信が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataAutomaticApplicationDownloadFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのアプリケーション自動配信が有効か否かを表す値を設定します。
::nn::Result SetSystemSaveDataAutomaticApplicationDownloadFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから通知設定を取得します。
::nn::Result GetSystemSaveDataNotificationSettings(
    ::nn::settings::system::NotificationSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの通知設定を変更します。
::nn::Result SetSystemSaveDataNotificationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NotificationSettings& value) NN_NOEXCEPT;

//!< システムセーブデータからアカウント毎の通知設定を取得します。
::nn::Result GetSystemSaveDataAccountNotificationSettings(
    int32_t* pOutCount,
    ::nn::settings::system::AccountNotificationSettings outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのアカウント毎の通知設定を変更します。
::nn::Result SetSystemSaveDataAccountNotificationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AccountNotificationSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータから Pro Controller 有線 USB 通信機能が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataUsbFullKeyEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Pro Controller 有線 USB 通信機能が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataUsbFullKeyEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから振動のマスターボリュームを取得します。
::nn::Result GetSystemSaveDataVibrationMasterVolume(
    float* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの振動のマスターボリュームを変更します。
::nn::Result SetSystemSaveDataVibrationMasterVolume(
    SystemSaveData* pSystemSaveData,
    const float& value) NN_NOEXCEPT;

//!< システムセーブデータから本体 6 軸センサーの加速度バイアス値を取得します。
::nn::Result GetSystemSaveDataConsoleSixAxisSensorAccelerationBias(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体 6 軸センサーの加速度バイアス値を変更します。
::nn::Result SetSystemSaveDataConsoleSixAxisSensorAccelerationBias(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias& value) NN_NOEXCEPT;

//!< システムセーブデータから本体 6 軸センサーの角速度バイアス値を取得します。
::nn::Result GetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体 6 軸センサーの角速度バイアス値を変更します。
::nn::Result SetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias& value) NN_NOEXCEPT;

//!< システムセーブデータから本体 6 軸センサーの加速度ゲイン値を取得します。
::nn::Result GetSystemSaveDataConsoleSixAxisSensorAccelerationGain(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体 6 軸センサーの加速度ゲイン値を変更します。
::nn::Result SetSystemSaveDataConsoleSixAxisSensorAccelerationGain(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain& value) NN_NOEXCEPT;

//!< システムセーブデータから本体 6 軸センサーの角速度ゲイン値を取得します。
::nn::Result GetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体 6 軸センサーの角速度ゲイン値を変更します。
::nn::Result SetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain& value) NN_NOEXCEPT;

//!< システムセーブデータからコントローラ情報を取得します。
::nn::Result GetSystemSaveDataNxControllerSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NxControllerSettings outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのコントローラ情報を変更します。
::nn::Result SetSystemSaveDataNxControllerSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NxControllerSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータから TV 設定を取得します。
::nn::Result GetSystemSaveDataTvSettings(
    ::nn::settings::system::TvSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの TV 設定を変更します。
::nn::Result SetSystemSaveDataTvSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::TvSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから EDID を取得します。
::nn::Result GetSystemSaveDataEdid(::nn::settings::system::Edid* pOutValue,
                                   SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの EDID を変更します。
::nn::Result SetSystemSaveDataEdid(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::Edid& value) NN_NOEXCEPT;

//!< システムセーブデータからおすすめ削除設定を取得します。
::nn::Result GetSystemSaveDataDataDeletionSettings(
    ::nn::settings::system::DataDeletionSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのおすすめ削除設定を変更します。
::nn::Result SetSystemSaveDataDataDeletionSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::DataDeletionSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから初回ブートするシステムアプレットの ProgramId を取得します。
::nn::Result GetSystemSaveDataInitialSystemAppletProgramId(
    ::nn::ncm::ProgramId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの初回ブートするシステムアプレットの ProgramId を設定します。
::nn::Result SetSystemSaveDataInitialSystemAppletProgramId(
    SystemSaveData* pSystemSaveData,
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT;

//!< システムセーブデータから初回ブートするオーバーレイ表示の ProgramId を取得します。
::nn::Result GetSystemSaveDataOverlayDispProgramId(
    ::nn::ncm::ProgramId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの初回ブートするオーバーレイ表示の ProgramId を設定します。
::nn::Result SetSystemSaveDataOverlayDispProgramId(
    SystemSaveData* pSystemSaveData,
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT;

//!< システムセーブデータからデバイスの地域名（タイムゾーン名）を取得します。
::nn::Result GetSystemSaveDataDeviceTimeZoneLocationName(
    ::nn::time::LocationName* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのデバイスの地域名（タイムゾーン名）を変更します。
::nn::Result SetSystemSaveDataDeviceTimeZoneLocationName(
    SystemSaveData* pSystemSaveData,
    const ::nn::time::LocationName& value) NN_NOEXCEPT;

//!< システムセーブデータから撮影した画面写真を保存するストレージを取得します。
::nn::Result GetSystemSaveDataPrimaryAlbumStorage(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの撮影した画面写真を保存するストレージを変更します。
::nn::Result SetSystemSaveDataPrimaryAlbumStorage(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータから USB 3.0 が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataUsb30EnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの USB 3.0 が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataUsb30EnableFlag(SystemSaveData* pSystemSaveData,
                                              const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから NFC が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataNfcEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの NFC が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataNfcEnableFlag(SystemSaveData* pSystemSaveData,
                                            const bool& value) NN_NOEXCEPT;

//!< システムセーブデータからスリープ設定を取得します。
::nn::Result GetSystemSaveDataSleepSettings(
    ::nn::settings::system::SleepSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのスリープ設定を変更します。
::nn::Result SetSystemSaveDataSleepSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SleepSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから無線 LAN が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataWirelessLanEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの無線 LAN が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataWirelessLanEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから初回起動設定を取得します。
::nn::Result GetSystemSaveDataInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの初回起動設定を変更します。
::nn::Result SetSystemSaveDataInitialLaunchSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから本体の識別名を取得します。
::nn::Result GetSystemSaveDataDeviceNickName(
    ::nn::settings::system::DeviceNickName* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの本体の識別名を変更します。
::nn::Result SetSystemSaveDataDeviceNickName(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::DeviceNickName& value) NN_NOEXCEPT;

//!< システムセーブデータからローカル通信のチャンネルを取得します。
::nn::Result GetSystemSaveDataLdnChannel(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのローカル通信のチャンネルを変更します。
::nn::Result SetSystemSaveDataLdnChannel(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータから Mii 作者 ID を取得します。
::nn::Result GetSystemSaveDataMiiAuthorId(
    ::nn::util::Uuid* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Mii 作者 ID を変更します。
::nn::Result SetSystemSaveDataMiiAuthorId(
    SystemSaveData* pSystemSaveData,
    const ::nn::util::Uuid& value) NN_NOEXCEPT;

//!< システムセーブデータからパッチの自動受信が有効か否かを表す値を取得します。
::nn::Result GetSystemSaveDataAutoUpdateEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのパッチの自動受信が有効か否かを表す値を変更します。
::nn::Result SetSystemSaveDataAutoUpdateEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから外部 RTC のリセットが発生したか否かを表す値を取得します。
::nn::Result GetSystemSaveDataExternalRtcResetFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの外部 RTC のリセットが発生したか否かを表す値を変更します。
::nn::Result SetSystemSaveDataExternalRtcResetFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから修理中か否かを表す値を取得します。
::nn::Result GetSystemSaveDataInRepairProcessEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの修理中か否かを表す値を変更します。
::nn::Result SetSystemSaveDataInRepairProcessEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータからプッシュ通知のスリープ状態における動作モードを取得します。
::nn::Result GetSystemSaveDataPushNotificationActivityModeOnSleep(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのプッシュ通知のスリープ状態における動作モードを変更します。
::nn::Result SetSystemSaveDataPushNotificationActivityModeOnSleep(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータからキーボード配列の種別を取得します。
::nn::Result GetSystemSaveDataKeyboardLayout(
    int32_t* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのキーボード配列の種別を変更します。
::nn::Result SetSystemSaveDataKeyboardLayout(
    SystemSaveData* pSystemSaveData, const int32_t& value) NN_NOEXCEPT;

//!< システムセーブデータから Web インスペクタを使用するか否かを表す値を取得します。
::nn::Result GetSystemSaveDataWebInspectorFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Web インスペクタを使用するか否かを表す値を変更します。
::nn::Result SetSystemSaveDataWebInspectorFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT;

//!< システムセーブデータから SSL チェック無効化ホストを取得します。
::nn::Result GetSystemSaveDataAllowedSslHosts(
    int32_t* pOutCount,
    ::nn::settings::system::AllowedSslHost outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの SSL チェック無効化ホストを変更します。
::nn::Result SetSystemSaveDataAllowedSslHosts(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AllowedSslHost values[],
    int32_t count) NN_NOEXCEPT;

//!< システムセーブデータからホスト FS のマウントポイントを取得します。
::nn::Result GetSystemSaveDataHostFsMountPoint(
    ::nn::settings::system::HostFsMountPoint* pOutValue,
        SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのホスト FS のマウントポイントを変更します。
::nn::Result SetSystemSaveDataHostFsMountPoint(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT;

//!< システムセーブデータ上のシステム設定を読み出します。
::nn::Result ReadSystemSaveDataSystemSettings(
    uint64_t* pOutCount, char outBuffer[], uint64_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータ上のシステム設定を更新します。
::nn::Result UpdateSystemSaveDataSystemSettings() NN_NOEXCEPT;

//!< システムセーブデータ上のシステム設定をリセットします。
::nn::Result ResetSystemSaveDataSystemSettings() NN_NOEXCEPT;

//!< システムセーブデータのインメモリモードを有効にします。（デバッグ用）
::nn::Result EnableSystemSaveDataInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT;

} // namespace

::nn::Result AcquireSystemSettingsFatalDirtyFlagEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::os::SystemEvent& event = GetFatalDirtyFlagEvent();

    // ハンドルの取得直後はシグナル状態であることを保証
    event.Signal();

    *pOutValue = event.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsFatalDirtyFlags(
    ::nn::settings::system::FatalDirtyFlagSet* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullFatalDirtyFlagsBuffer());

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    *pOutValue = g_FatalDirtyFlags;

    g_FatalDirtyFlags.Reset();

    NN_RESULT_SUCCESS;
}

::nn::Result AcquireSystemSettingsTelemetryDirtyFlagEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::os::SystemEvent& event = GetTelemetryDirtyFlagEvent();

    // ハンドルの取得直後はシグナル状態であることを保証
    event.Signal();

    *pOutValue = event.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsTelemetryDirtyFlags(
    ::nn::settings::system::TelemetryDirtyFlagSet* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullTelemetryDirtyFlagsBuffer());

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    *pOutValue = g_TelemetryDirtyFlags;

    g_TelemetryDirtyFlags.Reset();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsLanguageCode(LanguageCode* pOutValue) NN_NOEXCEPT
{
    typedef LanguageCode Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetLanguageCode()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullLanguageCodeBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataLanguageCode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultLanguageCode;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsLanguageCode(LanguageCode value) NN_NOEXCEPT
{
    typedef LanguageCode Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetLanguageCode()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataLanguageCode,
                        true));
                g_FatalDirtyFlags.Set<
                    ::nn::settings::system::FatalDirtyFlag::LanguageCode>();
                GetFatalDirtyFlagEvent().Signal();
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::LanguageCode>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAvailableLanguageCodeCount(
    int32_t* pOutCount) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullLanguageCodeCountBuffer());
    *pOutCount = static_cast<int32_t>(NN_ARRAY_SIZE(AvailableLanguageCodes));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAvailableLanguageCode(
    LanguageCode* pOutValue, int32_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullLanguageCodeBuffer());
    NN_RESULT_THROW_UNLESS(
        index >= 0 &&
        index < static_cast<int32_t>(NN_ARRAY_SIZE(AvailableLanguageCodes)),
        ResultInvalidLanguage());
    *pOutValue = AvailableLanguageCodes[index];
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAvailableLanguageCodes(
    int32_t* pOutCount, LanguageCode outCodes[], int32_t count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullLanguageCodeCountBuffer());

    NN_RESULT_THROW_UNLESS(
        outCodes != nullptr, ResultNullLanguageCodeBuffer());

    const auto codeCount = Clamp<int32_t>(
        count, 0, static_cast<int32_t>(NN_ARRAY_SIZE(AvailableLanguageCodes)));

    for (int i = 0; i < codeCount; ++i)
    {
        outCodes[i] = AvailableLanguageCodes[i];
    }

    *pOutCount = codeCount;

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsNetworkSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NetworkSettings outValues[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NetworkSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, static_cast<CountType>(count),
            "GetNetworkSettings()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr,
                    ResultNullNetworkSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outValues != nullptr,
                    ResultNullNetworkSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataNetworkSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                if (pOutCount != nullptr)
                {
                    if (outValues == nullptr)
                    {
                        *pOutCount = 0;
                    }
                    else
                    {
                        const auto valueCountMax = static_cast<CountType>(
                            NN_ARRAY_SIZE(DefaultNetworkSettings));

                        const auto valueCount =
                            Clamp<CountType>(count, 0, valueCountMax);

                        for (int i = 0; i < valueCount; ++i)
                        {
                            outValues[i] = DefaultNetworkSettings[i];
                        }

                        *pOutCount = valueCount;
                    }
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsNetworkSettings(
    const ::nn::settings::system::NetworkSettings* values,
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NetworkSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, static_cast<int32_t>(count),
            "SetNetworkSettings()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(count == 0 || values != nullptr,
                                       ResultNullNetworkSettingsBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count,
                        SetSystemSaveDataNetworkSettings,
                        false));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::NetworkSettings
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsRegionCode(int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetRegionCode()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullRegionCodeBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataRegionCode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultRegionCode;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsRegionCode(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetRegionCode()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataRegionCode,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::RegionCode>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBluetoothDevicesSettings(
    int32_t* pOutCount,
    ::nn::settings::system::BluetoothDevicesSettings outValues[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::BluetoothDevicesSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, static_cast<CountType>(count),
            "GetBluetoothDevicesSettings()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr,
                    ResultNullBluetoothDevicesSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outValues != nullptr,
                    ResultNullBluetoothDevicesSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataBluetoothDevicesSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBluetoothDevicesSettings(
    const ::nn::settings::system::BluetoothDevicesSettings values[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::BluetoothDevicesSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, static_cast<int32_t>(count),
            "SetBluetoothDevicesSettings()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    count == 0 || values != nullptr,
                    ResultNullBluetoothDevicesSettingsBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count,
                        SetSystemSaveDataBluetoothDevicesSettings,
                        false));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        BluetoothDevicesSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBluetoothEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsBluetoothEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullBluetoothEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataBluetoothEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultBluetoothEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBluetoothEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetBluetoothEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataBluetoothEnableFlag, false));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        BluetoothEnableFlag>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBluetoothAfhEnableFlag(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsBluetoothAfhEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullBluetoothAfhEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataBluetoothAfhEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultBluetoothAfhEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBluetoothAfhEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetBluetoothAfhEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataBluetoothAfhEnableFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBluetoothBoostEnableFlag(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsBluetoothBoostEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullBluetoothBoostEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataBluetoothBoostEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultBluetoothBoostEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBluetoothBoostEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetBluetoothBoostEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataBluetoothBoostEnableFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsLockScreenFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetLockScreenFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullLockScreenFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataLockScreenFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultLockScreenFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsLockScreenFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetLockScreenFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataLockScreenFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::LockScreenFlag
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsEulaVersions(
    int32_t* pOutCount,
    ::nn::settings::system::EulaVersion outValues[],
    size_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::EulaVersion ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, static_cast<CountType>(count),
            "GetEulaVersions()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutCount != nullptr,
                                       ResultNullEulaVersionCountBuffer());
                NN_RESULT_THROW_UNLESS(outValues != nullptr,
                                       ResultNullEulaVersionBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataEulaVersions));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsEulaVersions(
    const ::nn::settings::system::EulaVersion values[],
    size_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::EulaVersion ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, static_cast<int32_t>(count),
            "SetEulaVersions()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(count == 0 || values != nullptr,
                                       ResultNullEulaVersionBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count, SetSystemSaveDataEulaVersions, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::EulaVersions>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsErrorReportSharePermission(
    int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetErrorReportSharePermission()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullErrorReportSharePermissionBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataErrorReportSharePermission));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultErrorReportSharePermission;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsErrorReportSharePermission(
    int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetErrorReportSharePermission()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataErrorReportSharePermission,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ErrorReportSharePermission>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsColorSetId(int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetColorSetId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullColorSetIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataColorSetId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultColorSetId;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsColorSetId(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetColorSetId()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataColorSetId,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::ColorSetId>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBatteryPercentageFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetBatteryPercentageFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullBatteryPercentageFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataBatteryPercentageFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultBatteryPercentageFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBatteryPercentageFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetBatteryPercentageFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataBatteryPercentageFlag,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsQuestFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetQuestFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr, ResultNullQuestFlagBuffer());

                if (!IsSplDevelopment())
                {
                    *pOutValue = IsSplRetailInteractiveDisplayStateEnabled();
                }
                else
                {
                    NN_RESULT_DO(
                        GetSystemSaveDataValue<Type>(
                            pOutValue, GetSystemSaveDataQuestFlag));
                }

                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = !IsSplDevelopment()
                        ? IsSplRetailInteractiveDisplayStateEnabled()
                        : DefaultQuestFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsQuestFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetQuestFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataQuestFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::QuestFlag>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

template<size_t Version>
::nn::Result GetSystemSettingsBacklightSettingsImpl(
    ::nn::settings::system::BacklightSettingsEx* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::BacklightSettingsEx Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetBacklightSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullBacklightSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataBacklightSettings<Version>));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = GetDefaultBacklightSettings();
                }
            }));
    NN_RESULT_SUCCESS;
}

template<size_t Version>
::nn::Result SetSystemSettingsBacklightSettingsImpl(
    const ::nn::settings::system::BacklightSettingsEx& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::BacklightSettingsEx Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetBacklightSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataBacklightSettings<Version>,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        BacklightSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBacklightSettings(
    ::nn::settings::system::BacklightSettings* pOutValue) NN_NOEXCEPT
{
    if (pOutValue == nullptr)
    {
        NN_RESULT_DO(GetSystemSettingsBacklightSettingsImpl<0>(nullptr));
    }
    else
    {
        ::nn::settings::system::BacklightSettingsEx settings = {};
        NN_RESULT_DO(GetSystemSettingsBacklightSettingsImpl<0>(&settings));
        pOutValue->flags = settings.flags;
        pOutValue->brightness = settings.brightness;
        pOutValue->brightnessCoefficients = settings.brightnessCoefficients;
        pOutValue->sensorCoefficients = settings.sensorCoefficients;
        pOutValue->delayDarkToBright = settings.delayDarkToBright;
        pOutValue->delayBrightToDark = settings.delayBrightToDark;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBacklightSettings(
    const ::nn::settings::system::BacklightSettings& value) NN_NOEXCEPT
{
    ::nn::settings::system::BacklightSettingsEx settings = {};
    settings.flags = value.flags;
    settings.brightness = value.brightness;
    settings.brightnessCoefficients = value.brightnessCoefficients;
    settings.sensorCoefficients = value.sensorCoefficients;
    settings.delayDarkToBright = value.delayDarkToBright;
    settings.delayBrightToDark = value.delayBrightToDark;
    NN_RESULT_DO(SetSystemSettingsBacklightSettingsImpl<0>(settings));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsBacklightSettings(
    ::nn::settings::system::BacklightSettingsEx* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(GetSystemSettingsBacklightSettingsImpl<1>(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsBacklightSettings(
    const ::nn::settings::system::BacklightSettingsEx& value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetSystemSettingsBacklightSettingsImpl<1>(value));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsExternalSteadyClockSourceId(
    ::nn::settings::system::ClockSourceId* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(GetPrivateSettingsExternalSteadyClockSourceId(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsExternalSteadyClockSourceId(
    const ::nn::settings::system::ClockSourceId& value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetPrivateSettingsExternalSteadyClockSourceId(value));

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    g_TelemetryDirtyFlags.Set<
        ::nn::settings::system::TelemetryDirtyFlag::
            ExternalSteadyClockSourceId>();

    GetTelemetryDirtyFlagEvent().Signal();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsShutdownRtcValue(int64_t* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(GetPrivateSettingsShutdownRtcValue(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsShutdownRtcValue(int64_t value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetPrivateSettingsShutdownRtcValue(value));

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    g_TelemetryDirtyFlags.Set<
        ::nn::settings::system::TelemetryDirtyFlag::
            ShutdownRtcValue>();

    GetTelemetryDirtyFlagEvent().Signal();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsExternalSteadyClockInternalOffset(
    int64_t* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetPrivateSettingsExternalSteadyClockInternalOffset(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsExternalSteadyClockInternalOffset(
    int64_t value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetPrivateSettingsExternalSteadyClockInternalOffset(value));

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    g_TelemetryDirtyFlags.Set<
        ::nn::settings::system::TelemetryDirtyFlag::
            ExternalSteadyClockInternalOffset>();

    GetTelemetryDirtyFlagEvent().Signal();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsUserSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SystemClockContext Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetUserSystemClockContext()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullUserSystemClockContextBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataUserSystemClockContext));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultUserSystemClockContext;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsUserSystemClockContext(
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SystemClockContext Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetUserSystemClockContext()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataUserSystemClockContext,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        UserSystemClockContext>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsNetworkSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SystemClockContext Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetNetworkSystemClockContext()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullNetworkSystemClockContextBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataNetworkSystemClockContext));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultNetworkSystemClockContext;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsNetworkSystemClockContext(
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SystemClockContext Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetNetworkSystemClockContext()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataNetworkSystemClockContext,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        NetworkSystemClockContext>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsUserSystemClockAutomaticCorrectionFlag(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsUserSystemClockAutomaticCorrectionEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullUserSystemClockAutomaticCorrectionEnabledBuffer()
                    );
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataUserSystemClockAutomaticCorrectionFlag)
                    );
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultUserSystemClockAutomaticCorrectionFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsUserSystemClockAutomaticCorrectionFlag(
    bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetUserSystemClockAutomaticCorrectionEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataUserSystemClockAutomaticCorrectionFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        UserSystemClockAutomaticCorrection>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAccountSettings(
    ::nn::settings::system::AccountSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AccountSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAccountSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullAccountSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataAccountSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultAccountSettings;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAccountSettings(
    const ::nn::settings::system::AccountSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AccountSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAccountSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataAccountSettings,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::AccountSettings
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue, int32_t index) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AudioVolume Type;

    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullAudioVolumeBuffer());

    switch (static_cast<::nn::settings::system::AudioVolumeTarget>(index))
    {
    case ::nn::settings::system::AudioVolumeTarget_Speaker:
        NN_RESULT_DO(
            GetSystemSettingsValue<Type>(
                pOutValue,
                "GetAudioVolume(AudioVolumeTarget_Speaker)",
                [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                    NN_RESULT_DO(
                        GetSystemSaveDataValue<Type>(
                            pOutValue, GetSystemSaveDataSpeakerAudioVolume));
                    NN_RESULT_SUCCESS;
                },
                [] (Type* pOutValue) NN_NOEXCEPT
                {
                    if (pOutValue != nullptr)
                    {
                        *pOutValue = DefaultSpeakerAudioVolume;
                    }
                }));
        break;

    case ::nn::settings::system::AudioVolumeTarget_Headphone:
        NN_RESULT_DO(
            GetSystemSettingsValue<Type>(
                pOutValue,
                "GetAudioVolume(AudioVolumeTarget_Headphone)",
                [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                    NN_RESULT_DO(
                        GetSystemSaveDataValue<Type>(
                            pOutValue, GetSystemSaveDataHeadphoneAudioVolume));
                    NN_RESULT_SUCCESS;
                },
                [] (Type* pOutValue) NN_NOEXCEPT
                {
                    if (pOutValue != nullptr)
                    {
                        *pOutValue = DefaultHeadphoneAudioVolume;
                    }
                }));
        break;

    case ::nn::settings::system::AudioVolumeTarget_Usb:
        NN_RESULT_DO(
            GetSystemSettingsValue<Type>(
                pOutValue,
                "GetAudioVolume(AudioVolumeTarget_Usb)",
                [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                    NN_RESULT_DO(
                        GetSystemSaveDataValue<Type>(
                            pOutValue, GetSystemSaveDataUsbAudioVolume));
                    NN_RESULT_SUCCESS;
                },
                [] (Type* pOutValue) NN_NOEXCEPT
                {
                    if (pOutValue != nullptr)
                    {
                        *pOutValue = DefaultUsbAudioVolume;
                    }
                }));
        break;

    default:
        NN_RESULT_THROW(ResultInvalidAudioVolumeTarget());
    };

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAudioVolume(
    const ::nn::settings::system::AudioVolume& value, int32_t index) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AudioVolume Type;

    switch (static_cast<::nn::settings::system::AudioVolumeTarget>(index))
    {
    case ::nn::settings::system::AudioVolumeTarget_Speaker:
        NN_RESULT_DO(
            SetSystemSettingsValue<Type>(
                value,
                "SetAudioVolume(AudioVolumeTarget_Speaker)",
                [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<Type>(
                            value,
                            SetSystemSaveDataSpeakerAudioVolume,
                            false));
                    g_TelemetryDirtyFlags.Set<
                        ::nn::settings::system::TelemetryDirtyFlag::AudioVolume
                        >();
                    GetTelemetryDirtyFlagEvent().Signal();
                    NN_RESULT_SUCCESS;
                }));
        break;

    case ::nn::settings::system::AudioVolumeTarget_Headphone:
        NN_RESULT_DO(
            SetSystemSettingsValue<Type>(
                value,
                "SetAudioVolume(AudioVolumeTarget_Headphone)",
                [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<Type>(
                            value,
                            SetSystemSaveDataHeadphoneAudioVolume,
                            false));
                    g_TelemetryDirtyFlags.Set<
                        ::nn::settings::system::TelemetryDirtyFlag::AudioVolume
                        >();
                    GetTelemetryDirtyFlagEvent().Signal();
                    NN_RESULT_SUCCESS;
                }));
        break;

    case ::nn::settings::system::AudioVolumeTarget_Usb:
        NN_RESULT_DO(
            SetSystemSettingsValue<Type>(
                value,
                "SetAudioVolume(AudioVolumeTarget_Usb)",
                [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<Type>(
                            value,
                            SetSystemSaveDataUsbAudioVolume,
                            false));
                    g_TelemetryDirtyFlags.Set<
                        ::nn::settings::system::TelemetryDirtyFlag::AudioVolume
                        >();
                    GetTelemetryDirtyFlagEvent().Signal();
                    NN_RESULT_SUCCESS;
                }));
        break;

    default:
        NN_RESULT_THROW(ResultInvalidAudioVolumeTarget());
    };

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsRequiresRunRepairTimeReviser(bool* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetPrivateSettingsRequiresRunRepairTimeReviser(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsRequiresRunRepairTimeReviser(bool value) NN_NOEXCEPT
{
    NN_RESULT_DO(
        SetPrivateSettingsRequiresRunRepairTimeReviser(value));
    NN_RESULT_SUCCESS;
}

namespace {

//!< システム設定から HDMI の本体オーディオ出力設定を取得します。
::nn::Result GetSystemSettingsHdmiAudioOutputMode(
    int32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    typedef int32_t Type;

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAudioOutputMode(AudioOutputModeTarget_Hdmi)",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataHdmiAudioOutputMode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultHdmiAudioOutputMode;
                }
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定からスピーカーの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSettingsSpeakerAudioOutputMode(
    int32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    typedef int32_t Type;

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAudioOutputMode(AudioOutputModeTarget_Speaker)",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataSpeakerAudioOutputMode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultSpeakerAudioOutputMode;
                }
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定からヘッドフォンの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSettingsHeadphoneAudioOutputMode(
    int32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    typedef int32_t Type;

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAudioOutputMode(AudioOutputModeTarget_Headphone)",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataHeadphoneAudioOutputMode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultHeadphoneAudioOutputMode;
                }
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定から USB オーディオの本体オーディオ出力設定を取得します。
::nn::Result GetSystemSettingsUsbAudioOutputMode(int32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    typedef int32_t Type;

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAudioOutputMode(AudioOutputModeTarget_Usb)",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataUsbAudioOutputMode));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultUsbAudioOutputMode;
                }
            }));

    NN_RESULT_SUCCESS;
}

} // namespace

::nn::Result GetSystemSettingsAudioOutputMode(
    int32_t* pOutValue, int32_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAudioOutputModeBuffer());

    switch (static_cast<::nn::settings::system::AudioOutputModeTarget>(index))
    {
    case ::nn::settings::system::AudioOutputModeTarget_Hdmi:
        NN_RESULT_DO(GetSystemSettingsHdmiAudioOutputMode(pOutValue));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Speaker:
        NN_RESULT_DO(GetSystemSettingsSpeakerAudioOutputMode(pOutValue));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Headphone:
        NN_RESULT_DO(GetSystemSettingsHeadphoneAudioOutputMode(pOutValue));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Usb:
        NN_RESULT_DO(GetSystemSettingsUsbAudioOutputMode(pOutValue));
        break;

    default:
        NN_RESULT_THROW(ResultInvalidAudioOutputModeTarget());
    };

    NN_RESULT_SUCCESS;
}

namespace {

//!< システム設定の HDMI の本体オーディオ出力設定を変更します。
::nn::Result SetSystemSettingsHdmiAudioOutputMode(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;

    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAudioOutputMode(AudioOutputModeTarget_Hdmi)",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataHdmiAudioOutputMode,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定のスピーカーの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSettingsSpeakerAudioOutputMode(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;

    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAudioOutputMode(AudioOutputModeTarget_Speaker)",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataSpeakerAudioOutputMode,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定のヘッドフォンの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSettingsHeadphoneAudioOutputMode(
    int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;

    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAudioOutputMode(AudioOutputModeTarget_Headphone)",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataHeadphoneAudioOutputMode,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));

    NN_RESULT_SUCCESS;
}

//!< システム設定の USB オーディオの本体オーディオ出力設定を変更します。
::nn::Result SetSystemSettingsUsbAudioOutputMode(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;

    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAudioOutputMode(AudioOutputModeTarget_Usb)",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataUsbAudioOutputMode,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));

    NN_RESULT_SUCCESS;
}

} // namespace

::nn::Result SetSystemSettingsAudioOutputMode(
    int32_t value, int32_t index) NN_NOEXCEPT
{
    switch (static_cast<::nn::settings::system::AudioOutputModeTarget>(index))
    {
    case ::nn::settings::system::AudioOutputModeTarget_Hdmi:
        NN_RESULT_DO(SetSystemSettingsHdmiAudioOutputMode(value));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Speaker:
        NN_RESULT_DO(SetSystemSettingsSpeakerAudioOutputMode(value));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Headphone:
        NN_RESULT_DO(SetSystemSettingsHeadphoneAudioOutputMode(value));
        break;

    case ::nn::settings::system::AudioOutputModeTarget_Usb:
        NN_RESULT_DO(SetSystemSettingsUsbAudioOutputMode(value));
        break;

    default:
        NN_RESULT_THROW(ResultInvalidAudioOutputModeTarget());
    };

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsForceMuteOnHeadphoneRemovedFlag(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsForceMuteOnHeadphoneRemoved()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullForceMuteOnHeadphoneRemovedBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataForceMuteOnHeadphoneRemovedFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultForceMuteOnHeadphoneRemovedFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsForceMuteOnHeadphoneRemovedFlag(
    bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetForceMuteOnHeadphoneRemoved()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataForceMuteOnHeadphoneRemovedFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::ForceMute>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsHeadphoneVolumeWarningCount(
    int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetHeadphoneVolumeWarningDisplayedEventCount()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullHeadphoneVolumeWarningCountBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataHeadphoneVolumeWarningCount));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultHeadphoneVolumeWarningCount;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsHeadphoneVolumeWarningCount(
    int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetHeadphoneVolumeWarningDisplayedEventCount()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataHeadphoneVolumeWarningCount,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsHeadphoneVolumeUpdateFlag(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetHeadphoneVolumeUpdateFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullHeadphoneVolumeUpdateFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataHeadphoneVolumeUpdateFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultHeadphoneVolumeUpdateFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsHeadphoneVolumeUpdateFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetHeadphoneVolumeUpdateFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataHeadphoneVolumeUpdateFlag,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result NeedsToUpdateSystemSettingsHeadphoneVolume(
    bool* pOutValue,
    bool* pOutIsHighPowerAvailable,
    int8_t* pOutVolume,
    bool isRestricted) NN_NOEXCEPT
{
    struct Type final
    {
        bool* pValue;
        bool* pIsHighPowerAvailable;
        int8_t* pVolume;
        bool isAvailable;
        ::nn::settings::system::AudioVolume audioVolume;
    };

    Type value = {};
    value.pValue = pOutValue;
    value.pIsHighPowerAvailable = pOutIsHighPowerAvailable;
    value.pVolume = pOutVolume;

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            &value,
            "NeedsToUpdateHeadphoneVolume()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_RESULT_THROW_UNLESS(
                    pOutValue->pValue != nullptr,
                    ResultNullHeadphoneVolumeUpdateFlagBuffer());
                NN_RESULT_THROW_UNLESS(
                    pOutValue->pIsHighPowerAvailable != nullptr,
                    ResultNullHighPowerAvailableFlagBuffer());
                NN_RESULT_THROW_UNLESS(
                    pOutValue->pVolume != nullptr,
                    ResultNullAudioVolumeBuffer());
                auto isUpdated = bool();
                NN_RESULT_DO(
                    GetSystemSaveDataValue<bool>(
                        &isUpdated,
                        GetSystemSaveDataHeadphoneVolumeUpdateFlag));
                NN_RESULT_DO(
                    GetSystemSaveDataValue<::nn::settings::system::AudioVolume>(
                        &pOutValue->audioVolume,
                        GetSystemSaveDataHeadphoneAudioVolume));
                pOutValue->isAvailable = true;
                *(pOutValue->pValue) = !isUpdated;
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                if (pOutValue->pValue != nullptr)
                {
                    *(pOutValue->pValue) = false;
                }
            }));

    if (value.isAvailable)
    {
        int8_t& volume = value.audioVolume.volume;

        if (isRestricted || volume < 0 ||
            NN_ARRAY_SIZE(HeadphoneVolumeRemappingTable) <= volume)
        {
            *pOutValue = false;
        }

        if (*pOutValue)
        {
            *pOutVolume = HeadphoneVolumeRemappingTable[volume];
            *pOutIsHighPowerAvailable = true;
            volume = *pOutVolume;
            value.audioVolume.flags.Set<
                ::nn::settings::system::AudioVolumeFlag::HighPower>();
        }

        NN_RESULT_DO(
            SetSystemSettingsValue<Type>(
                value,
                "NeedsToUpdateHeadphoneVolume()",
                [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<
                            ::nn::settings::system::AudioVolume>(
                            value.audioVolume,
                            SetSystemSaveDataHeadphoneAudioVolume, false));
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<bool>(
                            true,
                            SetSystemSaveDataHeadphoneVolumeUpdateFlag, true));
                    NN_RESULT_SUCCESS;
                }));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsConsoleInformationUploadFlag(bool* pOutValue
                                                           ) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetConsoleInformationUploadFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullConsoleInformationUploadFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataConsoleInformationUploadFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultConsoleInformationUploadFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsConsoleInformationUploadFlag(bool value
                                                           ) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetConsoleInformationUploadFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataConsoleInformationUploadFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ConsoleInformationUploadFlag>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAutomaticApplicationDownloadFlag(bool* pOutValue
                                                               ) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAutomaticApplicationDownloadFlag()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullAutomaticApplicationDownloadFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataAutomaticApplicationDownloadFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultAutomaticApplicationDownloadFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAutomaticApplicationDownloadFlag(bool value
                                                               ) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAutomaticApplicationDownloadFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataAutomaticApplicationDownloadFlag,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        AutomaticApplicationDownloadFlag>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsNotificationSettings(
    ::nn::settings::system::NotificationSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NotificationSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetNotificationSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullNotificationSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataNotificationSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultNotificationSettings;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsNotificationSettings(
    const ::nn::settings::system::NotificationSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NotificationSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetNotificationSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataNotificationSettings, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        NotificationSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAccountNotificationSettings(
    int32_t* pOutCount,
    ::nn::settings::system::AccountNotificationSettings outValues[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AccountNotificationSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, count, "GetAccountNotificationSettings()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr,
                    ResultNullAccountNotificationSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outValues != nullptr,
                    ResultNullAccountNotificationSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataAccountNotificationSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAccountNotificationSettings(
    const ::nn::settings::system::AccountNotificationSettings values[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AccountNotificationSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, count, "SetAccountNotificationSettings()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    count == 0 || values != nullptr,
                    ResultNullAccountNotificationSettingsBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count,
                        SetSystemSaveDataAccountNotificationSettings,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        AccountNotificationSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsUsbFullKeyEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsUsbFullKeyEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullUsbFullKeyEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataUsbFullKeyEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultUsbFullKeyEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsUsbFullKeyEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetUsbFullKeyEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataUsbFullKeyEnableFlag, false));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsVibrationMasterVolume(float* pOutValue
                                                    ) NN_NOEXCEPT
{
    typedef float Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetVibrationMasterVolume()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullVibrationMasterVolumeBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataVibrationMasterVolume));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultVibrationMasterVolume;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsVibrationMasterVolume(float value) NN_NOEXCEPT
{
    typedef float Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetVibrationMasterVolume()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataVibrationMasterVolume,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        VibrationMasterVolume>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsConsoleSixAxisSensorAccelerationBias(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetConsoleSixAxisSensorAccelerationBias()",
            [](Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullConsoleSixAxisSensorAccelerationBiasBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataConsoleSixAxisSensorAccelerationBias));
                NN_RESULT_SUCCESS;
            },
            [](Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultConsoleSixAxisSensorAccelerationBias;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsConsoleSixAxisSensorAccelerationBias(
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetConsoleSixAxisSensorAccelerationBias()",
            [](const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataConsoleSixAxisSensorAccelerationBias,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ConsoleSixAxisSensorAccelerationBias>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularVelocityBias(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetConsoleSixAxisSensorAngularVelocityBias()",
            [](Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullConsoleSixAxisSensorAngularVelocityBiasBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias));
                NN_RESULT_SUCCESS;
            },
            [](Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultConsoleSixAxisSensorAngularVelocityBias;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularVelocityBias(
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetConsoleSixAxisSensorAngularVelocityBias()",
            [](const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ConsoleSixAxisSensorAngularVelocityBias>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsConsoleSixAxisSensorAccelerationGain(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetConsoleSixAxisSensorAccelerationGain()",
            [](Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullConsoleSixAxisSensorAccelerationGainBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataConsoleSixAxisSensorAccelerationGain));
                NN_RESULT_SUCCESS;
            },
            [](Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultConsoleSixAxisSensorAccelerationGain;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsConsoleSixAxisSensorAccelerationGain(
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetConsoleSixAxisSensorAccelerationGain()",
            [](const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataConsoleSixAxisSensorAccelerationGain,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ConsoleSixAxisSensorAccelerationGain>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularVelocityGain(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetConsoleSixAxisSensorAngularVelocityGain()",
            [](Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullConsoleSixAxisSensorAngularVelocityGainBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain));
                NN_RESULT_SUCCESS;
            },
            [](Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultConsoleSixAxisSensorAngularVelocityGain;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularVelocityGain(
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetConsoleSixAxisSensorAngularVelocityGain()",
            [](const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        ConsoleSixAxisSensorAngularVelocityGain>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsNxControllerSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NxControllerSettings outValues[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NxControllerSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, static_cast<CountType>(count),
            "GetNxControllerSettings()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr,
                    ResultNullNxControllerSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outValues != nullptr,
                    ResultNullNxControllerSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataNxControllerSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsNxControllerSettings(
    const ::nn::settings::system::NxControllerSettings values[],
    int32_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::NxControllerSettings ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, static_cast<int32_t>(count),
            "SetNxControllerSettings()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    count == 0 || values != nullptr,
                    ResultNullNxControllerSettingsBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count,
                        SetSystemSaveDataNxControllerSettings,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsTvSettings(
    ::nn::settings::system::TvSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::TvSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetTvSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullTvSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataTvSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = GetDefaultTvSettings();
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsTvSettings(
    const ::nn::settings::system::TvSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::TvSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetTvSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataTvSettings, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::TvSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsEdid(::nn::settings::system::Edid* pOutValue
                                   ) NN_NOEXCEPT
{
    typedef ::nn::settings::system::Edid Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetEdid()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullEdidBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataEdid));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    ::std::memset(pOutValue, 0, sizeof(*pOutValue));
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsEdid(const ::nn::settings::system::Edid& value
                                   ) NN_NOEXCEPT
{
    typedef ::nn::settings::system::Edid Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetEdid()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataEdid, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::Edid>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsDataDeletionSettings(
    ::nn::settings::system::DataDeletionSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::DataDeletionSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetDataDeletionSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullDataDeletionSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataDataDeletionSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultDataDeletionSettings;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsDataDeletionSettings(
    const ::nn::settings::system::DataDeletionSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::DataDeletionSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetDataDeletionSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataDataDeletionSettings,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        DataDeletionSettings>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsInitialSystemAppletProgramId(
    ::nn::ncm::ProgramId* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::ncm::ProgramId Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetInitialSystemAppletProgramId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullInitialSystemAppletProgramIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataInitialSystemAppletProgramId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultInitialSystemAppletProgramId;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsInitialSystemAppletProgramId(
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT
{
    typedef ::nn::ncm::ProgramId Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetInitialSystemAppletProgramId()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataInitialSystemAppletProgramId,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsOverlayDispProgramId(
    ::nn::ncm::ProgramId* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::ncm::ProgramId Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetOverlayDispProgramId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullOverlayDispProgramIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataOverlayDispProgramId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultOverlayDispProgramId;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsOverlayDispProgramId(
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT
{
    typedef ::nn::ncm::ProgramId Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetOverlayDispProgramId()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataOverlayDispProgramId,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsDeviceTimeZoneLocationName(
    ::nn::time::LocationName* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::time::LocationName Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetDeviceTimeZoneLocationName()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullDeviceTimeZoneLocationNameBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataDeviceTimeZoneLocationName));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultDeviceTimeZoneLocationName;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsDeviceTimeZoneLocationName(
    const ::nn::time::LocationName& value) NN_NOEXCEPT
{
    typedef ::nn::time::LocationName Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetDeviceTimeZoneLocationName()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataDeviceTimeZoneLocationName,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        DeviceTimeZoneLocationName>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsPrimaryAlbumStorage(
    int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetPrimaryAlbumStorage()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullPrimaryAlbumStorageBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataPrimaryAlbumStorage));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultPrimaryAlbumStorage;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsPrimaryAlbumStorage(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetPrimaryAlbumStorage()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataPrimaryAlbumStorage,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        PrimaryAlbumStorage>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsUsb30EnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsUsb30Enabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullUsb30EnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataUsb30EnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultUsb30EnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsUsb30EnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetUsb30Enabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataUsb30EnableFlag, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::Usb30EnableFlag
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsNfcEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsNfcEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullNfcEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataNfcEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultNfcEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsNfcEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetNfcEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataNfcEnableFlag, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::NfcEnableFlag
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsSleepSettings(
    ::nn::settings::system::SleepSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SleepSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetSleepSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullSleepSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataSleepSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultSleepSettings;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsSleepSettings(
    const ::nn::settings::system::SleepSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::SleepSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetSleepSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    value.handheldSleepPlan >=
                    static_cast<int32_t>(
                        ::nn::settings::system::HandheldSleepPlan_1Min) &&
                    value.handheldSleepPlan <=
                    static_cast<int32_t>(
                        ::nn::settings::system::HandheldSleepPlan_Never),
                    ResultInvalidHandheldSleepPlan());
                NN_RESULT_THROW_UNLESS(
                    value.consoleSleepPlan >=
                    static_cast<int32_t>(
                        ::nn::settings::system::ConsoleSleepPlan_1Hour) &&
                    value.consoleSleepPlan <=
                    static_cast<int32_t>(
                        ::nn::settings::system::ConsoleSleepPlan_Never),
                    ResultInvalidConsoleSleepPlan());
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataSleepSettings, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::SleepSettings
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsWirelessLanEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsWirelessLanEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullWirelessLanEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataWirelessLanEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultWirelessLanEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsWirelessLanEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetWirelessLanEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataWirelessLanEnableFlag, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::
                        WirelessLanEnableFlag>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::InitialLaunchSettings Type;

    const ::nn::settings::system::InitialLaunchFlagSet mask =
        ::nn::settings::system::InitialLaunchFlag::HasTimeStamp::Mask;

    Type defaultValue = {};

    NN_RESULT_DO(GetPrivateSettingsInitialLaunchSettings(&defaultValue));

    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetInitialLaunchSettings()",
            [&] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullInitialLaunchSettingsBuffer());

                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataInitialLaunchSettings));

                if (defaultValue.flags != (pOutValue->flags & mask) ||
                    defaultValue.timeStamp != pOutValue->timeStamp)
                {
                    *pOutValue = defaultValue;
                }

                NN_RESULT_SUCCESS;
            },
            [&] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = defaultValue;
                }
            }));

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsInitialLaunchSettings(
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::InitialLaunchSettings Type;

    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetInitialLaunchSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataInitialLaunchSettings, true));

                NN_RESULT_SUCCESS;
            }));

    Type defaultValue = value;

    defaultValue.flags &=
        ::nn::settings::system::InitialLaunchFlag::HasTimeStamp::Mask;

    NN_RESULT_DO(SetPrivateSettingsInitialLaunchSettings(defaultValue));

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    g_TelemetryDirtyFlags.Set<
        ::nn::settings::system::TelemetryDirtyFlag::InitialLaunchSettings>();

    GetTelemetryDirtyFlagEvent().Signal();

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsDeviceNickName(
    ::nn::settings::system::DeviceNickName* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::DeviceNickName Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetDeviceNickName()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullDeviceNickNameBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataDeviceNickName));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultDeviceNickName;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsDeviceNickName(
    const ::nn::settings::system::DeviceNickName& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::DeviceNickName Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetDeviceNickName()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataDeviceNickName, true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::DeviceNickName
                    >();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAppletLaunchFlags(
    ::nn::settings::system::AppletLaunchFlagSet* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AppletLaunchFlagSet Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetAppletLaunchFlags()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullAppletLaunchFlagsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataAppletLaunchFlags));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultAppletLaunchFlags;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAppletLaunchFlags(
    const ::nn::settings::system::AppletLaunchFlagSet& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AppletLaunchFlagSet Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAppletLaunchFlags()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataAppletLaunchFlags, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsLdnChannel(int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetLdnChannel()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullLdnChannelBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataLdnChannel));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultLdnChannel;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsLdnChannel(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetLdnChannel()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataLdnChannel,
                        true));
                g_TelemetryDirtyFlags.Set<
                    ::nn::settings::system::TelemetryDirtyFlag::LdnChannel>();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsMiiAuthorId(
    ::nn::util::Uuid* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::util::Uuid Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetMiiAuthorId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullMiiAuthorIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataMiiAuthorId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = GenerateUuid();
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAutoUpdateEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsAutoUpdateEnabled()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullAutoUpdateEnableFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataAutoUpdateEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultAutoUpdateEnableFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAutoUpdateEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetAutoUpdateEnable()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataAutoUpdateEnableFlag,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsExternalRtcResetFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsExternalRtcReset()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullExternalRtcResetFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataExternalRtcResetFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultExternalRtcResetFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsExternalRtcResetFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetExternalRtcReset()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataExternalRtcResetFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsInRepairProcessEnableFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsInRepairProcess()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullIsInRepairBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataInRepairProcessEnableFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultInRepairProcessFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsInRepairProcessEnableFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetInRepairProcessEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataInRepairProcessEnableFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsPushNotificationActivityModeOnSleep(
    int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetPushNotificationActivityModeOnSleep()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullPushNotificationActivityModeBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataPushNotificationActivityModeOnSleep));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultPushNotificationActivityModeOnSleep;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsPushNotificationActivityModeOnSleep(
    int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetPushNotificationActivityModeOnSleep()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataPushNotificationActivityModeOnSleep,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsKeyboardLayout(int32_t* pOutValue) NN_NOEXCEPT
{
    typedef int32_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetKeyboardLayout()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullKeyboardLayoutBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataKeyboardLayout));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultKeyboardLayout;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsKeyboardLayout(int32_t value) NN_NOEXCEPT
{
    typedef int32_t Type;

    switch (static_cast<KeyboardLayout>(value))
    {
        case KeyboardLayout_Japanese:
        case KeyboardLayout_EnglishUs:
        case KeyboardLayout_EnglishUsInternational:
        case KeyboardLayout_EnglishUk:
        case KeyboardLayout_French:
        case KeyboardLayout_FrenchCa:
        case KeyboardLayout_Spanish:
        case KeyboardLayout_SpanishLatin:
        case KeyboardLayout_German:
        case KeyboardLayout_Italian:
        case KeyboardLayout_Portuguese:
        case KeyboardLayout_Russian:
        NN_RESULT_DO(
            SetSystemSettingsValue<Type>(
                value,
                "SetKeyboardLayout()",
                [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
                {
                    NN_RESULT_DO(
                        SetSystemSaveDataValue<Type>(
                            value,
                            SetSystemSaveDataKeyboardLayout,
                            true));
                    NN_RESULT_SUCCESS;
                }));
        break;

    default:
        NN_RESULT_THROW(ResultInvalidKeyboardLayout());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsWebInspectorFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "UsesWebInspector()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr, ResultNullWebInspectorFlagBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataWebInspectorFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultWebInspectorFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsWebInspectorFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetWebInspectorFlag()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataWebInspectorFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsAllowedSslHosts(
    int32_t* pOutCount,
    ::nn::settings::system::AllowedSslHost outValues[], size_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AllowedSslHost ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outValues, static_cast<CountType>(count),
            "GetAllowedSslHosts()",
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr,
                    ResultNullAllowedSslHostsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outValues != nullptr,
                    ResultNullAllowedSslHostsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outValues, count,
                        GetSystemSaveDataAllowedSslHosts));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsAllowedSslHosts(
    const ::nn::settings::system::AllowedSslHost values[],
    size_t count) NN_NOEXCEPT
{
    typedef ::nn::settings::system::AllowedSslHost ValueType;
    typedef int32_t CountType;
    NN_RESULT_DO(
        (SetSystemSettingsValue<ValueType, CountType>)(
            values, static_cast<CountType>(count),
            "SetAllowedSslHosts()",
            [] (const ValueType values[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    count == 0 || values != nullptr,
                    ResultNullAllowedSslHostsBuffer());
                NN_RESULT_DO(
                    (SetSystemSaveDataValue<ValueType, CountType>)(
                        values, count,
                        SetSystemSaveDataAllowedSslHosts,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSettingsHostFsMountPoint(
    ::nn::settings::system::HostFsMountPoint* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::HostFsMountPoint Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetHostFsMountPoint()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr, ResultNullHostFsMountPointBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataHostFsMountPoint));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    ::std::memset(pOutValue, 0, sizeof(*pOutValue));
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSettingsHostFsMountPoint(
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::HostFsMountPoint Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetHostFsMountPoint()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataHostFsMountPoint, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemSettings(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    typedef char ValueType;
    typedef uint64_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outBuffer, static_cast<CountType>(count),
            "ReadSettings()",
            [] (CountType* pOutCount, ValueType outBuffer[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr, ResultNullSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outBuffer != nullptr, ResultNullSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outBuffer, count,
                        ReadSystemSaveDataSystemSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSettings() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ResetSystemSettingsValue(
            "ResetSettings()", [] () NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(ResetSystemSaveDataSystemSettings());
                g_FatalDirtyFlags.Set();
                GetFatalDirtyFlagEvent().Signal();
                g_TelemetryDirtyFlags.Set();
                GetTelemetryDirtyFlagEvent().Signal();
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result EnableSystemSettingsInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    NN_RESULT_DO(EnableSystemSaveDataInMemoryModeForDebug(buffer, size));

    NN_RESULT_SUCCESS;
}

namespace {

#else // SAME_TO_ORIGINAL

namespace nn { namespace repair { namespace systemSettings {

#endif

//!< システム設定データベースの現在のフォーマットバージョンです。
const uint32_t CurrentFormatVersion = 0x040001;

//!< 言語設定のデータベースを表す構造体です。
struct LanguageSettingsDatabase final
{
    nn::settings::LanguageCode languageCode;

    ::nn::Bit8 margin[56];
};

NN_STATIC_ASSERT(sizeof(LanguageSettingsDatabase) == 64);

//!< ネットワーク設定のデータベースのエントリを表す構造体です。
struct NetworkSettingsDatabaseEntry final
{
    char name[::nn::settings::system::NetworkSettingsNameLengthMax];
    ::nn::Bit8 id[::nn::settings::system::NetworkSettingsIdLengthMax];
    ::nn::settings::system::ProfileFlagSet flags;
    int32_t nicType;
    ::nn::settings::system::SsidFlagSet ssidFlags;
    ::nn::Bit8 ssid[::nn::settings::system::SsidLengthMax];
    int32_t ssidLength;
    int32_t securityAuthenticationMode;
    int32_t securityEncryptionMode;
    char securityKeyMaterial[::nn::settings::system::KeyMaterialLengthMax];
    int32_t securityKeyMaterialLength;
    ::nn::settings::system::IpFlagSet ipFlags;
    ::nn::Bit8 ipAddress[4];
    ::nn::Bit8 ipSubnetMask[4];
    ::nn::Bit8 ipDefaultGateway[4];
    ::nn::Bit8 ipPreferredDns[4];
    ::nn::Bit8 ipAlternateDns[4];
    ::nn::settings::system::ProxyFlagSet proxyFlags;
    char proxyHostName[::nn::settings::system::ProxyHostNameLengthMax];
    uint16_t proxyPortNumber;
    NN_PADDING2;
    char proxyUserName[::nn::settings::system::ProxyUserNameLengthMax];
    char proxyPassword[::nn::settings::system::ProxyPasswordLengthMax];
    uint16_t mtu;
    NN_PADDING2;

    ::nn::Bit8 margin[592];
};

NN_STATIC_ASSERT(sizeof(NetworkSettingsDatabaseEntry) == 1024);

//!< ネットワーク設定のデータベースを表す構造体です。
struct NetworkSettingsDatabase final
{
    int32_t count;
    int32_t isWirelessLanEnabled;
    NN_PADDING8;
    NetworkSettingsDatabaseEntry entries[
        ::nn::settings::system::NetworkSettingsCountMax];
};

NN_STATIC_ASSERT(sizeof(NetworkSettingsDatabase) == 32784);

//!< Bluetooth デバイス設定のデータベースを表す構造体です。
struct BluetoothDevicesSettingsDatabase final
{
    int32_t count;
    int32_t isBluetoothEnabled;
    int32_t isBluetoothAfhEnabled;
    int32_t isBluetoothBoostEnabled;
    ::nn::settings::system::BluetoothDevicesSettings entries[
        ::nn::settings::system::BluetoothDevicesSettingsCountMax];
};

NN_STATIC_ASSERT(sizeof(BluetoothDevicesSettingsDatabase) == 5136);

//!< ローカル通信のデータベースを表す構造体です。
struct LdnSettingsDatabase final
{
    int32_t ldnChannel;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(LdnSettingsDatabase) == 64);

//!< Mii 設定のデータベースを表す構造体です。
struct MiiSettingsDatabase final
{
    ::nn::util::Uuid miiAuthorId;

    ::nn::Bit8 margin[48];
};

NN_STATIC_ASSERT(sizeof(MiiSettingsDatabase) == 64);

//!< PCV 設定のデータベースを表す構造体です。
struct PcvSettingsDatabase final
{
    int32_t externalRtcResetFlag;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(PcvSettingsDatabase) == 64);

//!< プッシュ通知設定のデータベースを表す構造体です。
struct PushNotificationSettingsDatabase final
{
    int32_t activityModeOnSleep;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(PushNotificationSettingsDatabase) == 64);

//!< エラーレポート設定のデータベースを表す構造体です。
struct ErrorReportSettingsDatabase final
{
    int32_t sharePermission;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(ErrorReportSettingsDatabase) == 64);

//!< Cruiser 用デバッグ設定のデータベースを表す構造体です。
struct WebSettingsDatabase final
{
    int32_t webInspectorFlag;
    int32_t allowedSslHostsCount;
    ::nn::Bit8 _padding1[56];
    ::nn::settings::system::HostFsMountPoint hostFsMountPoint;
    ::nn::settings::system::AllowedSslHost allowedSslHosts[
        ::nn::settings::system::AllowedSslHostCountMax];
    ::nn::Bit8 _padding2[1728];
};

NN_STATIC_ASSERT(sizeof(WebSettingsDatabase) == 4096);


//!< サービスディスカバリ設定のデータベースを表す構造体です。
struct ServiceDiscoverySettingsDatabase final
{
    ::nn::Bit8 margin[125312];
};

NN_STATIC_ASSERT(sizeof(ServiceDiscoverySettingsDatabase) == 125312);

//!< システムアプリケーション設定のデータベースを表す構造体です。
struct SystemApplicationSettingsDatabase final
{
    int32_t lockScreenFlag;
    uint32_t eulaVersion;
    int32_t colorSetId;
    int32_t questFlag;
    int32_t regionCode;
    uint32_t initialLaunchSettingsFlags;
    ::nn::time::SteadyClockTimePoint initialLaunchSettingsTimeStamp;
    int32_t batteryPercentageFlag;
    uint32_t appletLaunchFlags;

    ::nn::Bit8 margin[72];
};

NN_STATIC_ASSERT(sizeof(SystemApplicationSettingsDatabase) == 128);

//!< 画面の明るさの設定のデータベースを表す構造体です。
struct BacklightSettingsDatabase final
{
    ::nn::settings::system::BacklightFlagSet flags;
    float brightness;
    ::nn::util::Float3 brightnessCoefficients;
    ::nn::util::Float3 sensorCoefficients;
    float delayDarkToBright;
    float delayBrightToDark;
    float brightnessVr;

    ::nn::Bit8 margin[84];
};

NN_STATIC_ASSERT(sizeof(BacklightSettingsDatabase) == 128);

//!< クロック設定のデータベースを表す構造体です。
struct ClockSettingsDatabase final
{
    ::nn::Bit8 _padding[16];    // 書き換え禁止
    ::nn::settings::system::SystemClockContext userSystemClockContext;
    ::nn::settings::system::SystemClockContext networkSystemClockContext;
    int32_t userSystemClockAutomaticCorrectionFlag;

    ::nn::Bit8 margin[44];
};

NN_STATIC_ASSERT(sizeof(ClockSettingsDatabase) == 128);

//!< アカウント設定のデータベースを表す構造体です。
struct AccountSettingsDatabase final
{
    ::nn::settings::system::UserSelectorFlagSet userSelectorFlags;

    ::nn::Bit8 margin[252];
};

NN_STATIC_ASSERT(sizeof(AccountSettingsDatabase) == 256);

//!< オーディオ設定のデータベースを表す構造体です。
struct AudioSettingsDatabase final
{
    ::nn::settings::system::AudioVolumeFlagSet speakerAudioVolumeFlags;
    int8_t speakerAudioVolume;
    ::nn::Bit8 _padding1[3];    // 書き換え禁止
    ::nn::settings::system::AudioVolumeFlagSet headphoneAudioVolumeFlags;
    int8_t headphoneAudioVolume;
    ::nn::Bit8 _padding2[3];    // 書き換え禁止
    int32_t hdmiAudioOutputMode;
    int32_t speakerAudioOutputMode;
    int32_t headphoneAudioOutputMode;
    int32_t forceMuteOnHeadphoneRemovedFlag;
    int32_t headphoneVolumeWarningCount;
    int32_t headphoneVolumeUpdateFlag;
    ::nn::settings::system::AudioVolumeFlagSet usbAudioVolumeFlags;
    int8_t usbAudioVolume;
    ::nn::Bit8 _padding3[3];
    int32_t usbAudioOutputMode;

    ::nn::Bit8 margin[76];
};

NN_STATIC_ASSERT(sizeof(AudioSettingsDatabase) == 128);

//!< ネットワークサービス設定のデータベースを表す構造体です。
struct NetworkServiceSettingsDatabase final
{
    int32_t consoleInformationUploadFlag;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(NetworkServiceSettingsDatabase) == 64);

//!< 通知設定のデータベースを表す構造体です。
struct NotificationSettingsDatabase final
{
    int32_t automaticApplicationDownloadFlag;
    ::nn::Bit8 _padding[4]; // 書き換え禁止
    ::nn::settings::system::NotificationFlagSet flags;
    int32_t volume;
    ::nn::settings::system::NotificationTime headTime;
    ::nn::settings::system::NotificationTime tailTime;

    ::nn::Bit8 margin[96];
};

NN_STATIC_ASSERT(sizeof(NotificationSettingsDatabase) == 128);

//!< アカウント毎の通知設定のデータベースのエントリを表す構造体です。
struct AccountNotificationSettingsDatabaseEntry final
{
    ::nn::account::Uid uid;
    ::nn::settings::system::AccountNotificationFlagSet flags;
    ::nn::Bit8 _padding1[4]; // 書き換え禁止
    int8_t friendPresenceOverlayPermission;
    ::nn::Bit8 _padding2[7];

    ::nn::Bit8 _reserved[32];
};

NN_STATIC_ASSERT(sizeof(AccountNotificationSettingsDatabaseEntry) == 64);

//!< アカウント毎の通知設定のデータベースを表す構造体です。
struct AccountNotificationSettingsDatabase final
{
    int32_t count;
    ::nn::Bit8 _padding[12];    // 書き換え禁止
    AccountNotificationSettingsDatabaseEntry entries[
        ::nn::settings::system::AccountNotificationSettingsCountMax];
};

NN_STATIC_ASSERT(sizeof(AccountNotificationSettingsDatabase) == 528);

//!< HID 設定のデータベースを表す構造体です。
struct HidSettingsDatabase final
{
    float vibrationMasterVolume;
    int32_t usbFullKeyEnableFlag;

    ::nn::Bit8 margin[56];
};

NN_STATIC_ASSERT(sizeof(HidSettingsDatabase) == 64);

//!< コントローラ情報のデータベースのエントリを表す構造体です。
struct NxControllerSettingsDatabaseEntry final
{
    ::nn::settings::system::NxControllerSettings value;

    ::nn::Bit8 margin[23];
};

NN_STATIC_ASSERT(sizeof(NxControllerSettingsDatabaseEntry) == 64);

//!< コントローラ情報のデータベースを表す構造体です。
struct NxControllerSettingsDatabase final
{
    int32_t count;

    ::nn::Bit8 margin1[12];

    NxControllerSettingsDatabaseEntry entries[
        ::nn::settings::system::NxControllerSettingsCountMax];

    ::nn::Bit8 margin2[368];
};

NN_STATIC_ASSERT(sizeof(NxControllerSettingsDatabase) == 1024);

//!< TV 設定のデータベースを表す構造体です。
struct TvSettingsDatabase final
{
    int32_t tvResolution;
    int32_t hdmiContentType;
    int32_t rgbRange;
    int32_t cmuMode;
    uint32_t tvUnderscan;
    float tvGamma;
    float contrastRatio;
    ::nn::settings::system::TvFlagSet flags;
    ::nn::Bit8 edidData0[128];
    ::nn::Bit8 edidData1[128];

    ::nn::Bit8 margin[736];
};

NN_STATIC_ASSERT(sizeof(TvSettingsDatabase) == 1024);

//!< おすすめ削除設定のデータベースを表す構造体です。
struct DataDeletionSettingsDatabase final
{
    ::nn::settings::system::DataDeletionFlagSet flags;
    int32_t useCount;

    ::nn::Bit8 margin[56];
};

NN_STATIC_ASSERT(sizeof(DataDeletionSettingsDatabase) == 64);

//!< ブート設定のデータベースを表す構造体です。
struct BootSettingsDatabase final
{
    ::nn::ncm::ProgramId initialSystemAppletProgramId;
    ::nn::ncm::ProgramId overlayDispProgramId;
    int32_t inRepairFlag;

    ::nn::Bit8 margin[108];
};

NN_STATIC_ASSERT(sizeof(BootSettingsDatabase) == 128);

//!< タイムゾーン設定のデータベースを表す構造体です。
struct TimeZoneSettingsDatabase final
{
    ::nn::time::LocationName deviceTimeZoneLocationName;

    ::nn::Bit8 margin[220];
};

NN_STATIC_ASSERT(sizeof(TimeZoneSettingsDatabase) == 256);

//!< キャプチャ設定のデータベースを表す構造体です。
struct CaptureSettingsDatabase final
{
    int32_t primaryAlbumStorage;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(CaptureSettingsDatabase) == 64);

//!< Usb 設定のデータベースを表す構造体です。
struct UsbSettingsDatabase final
{
    int32_t isUsb30Enabled;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(UsbSettingsDatabase) == 64);

//!< Nfc 設定のデータベースを表す構造体です。
struct NfcSettingsDatabase final
{
    int32_t isNfcEnabled;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(NfcSettingsDatabase) == 64);

//!< スリープ設定のデータベースを表す構造体です。
struct SleepSettingsDatabase final
{
    ::nn::settings::system::SleepFlagSet flags;
    int32_t handheldSleepPlan;
    int32_t consoleSleepPlan;

    ::nn::Bit8 margin[52];
};

NN_STATIC_ASSERT(sizeof(SleepSettingsDatabase) == 64);

//!< 本体 EULA の同意バージョンのデータベースのエントリを表す構造体です。
struct EulaVersionsDatabaseEntry final
{
    uint32_t version;
    int32_t regionCode;
    int32_t clockType;
    NN_PADDING4;
    ::nn::time::PosixTime networkSystemClock;
    ::nn::time::SteadyClockTimePoint steadyClock;

    ::nn::Bit8 margin[16];
};

NN_STATIC_ASSERT(sizeof(EulaVersionsDatabaseEntry) == 64);

//!< 本体 EULA の同意バージョンのデータベースを表す構造体です。
struct EulaVersionsDatabase final
{
    int32_t count;
    NN_PADDING4;
    NN_PADDING8;
    EulaVersionsDatabaseEntry entries[
        ::nn::settings::system::EulaVersionCountMax];
};

NN_STATIC_ASSERT(sizeof(EulaVersionsDatabase) == 2064);

//!< 本体の識別名のデータベースを表す構造体です。
struct DeviceNickNameDatabase final
{
    ::nn::settings::system::DeviceNickName deviceNickName;

    ::nn::Bit8 margin[128];
};

NN_STATIC_ASSERT(sizeof(DeviceNickNameDatabase) == 256);

//!< DRM 設定のデータベースを表す構造体です。
struct DrmSettingsDatabase final
{
    int32_t isAutoUpdateEnabled;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(DrmSettingsDatabase) == 64);

//!< SixAxisSensor のデータベースを表す構造体です。
struct SixAxisSensorDatabase final
{
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias consoleSixAxisSensorAccelerationBias;
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias consoleSixAxisSensorAngularVelocityBias;

    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain consoleSixAxisSensorAccelerationGain;
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain consoleSixAxisSensorAngularVelocityGain;

    ::nn::Bit8 margin[160];
};

NN_STATIC_ASSERT(sizeof(SixAxisSensorDatabase) == 256);

//!< キーボード設定のデータベースを表す構造体です。
struct KeyboardSettingsDatabase final
{
    int32_t keyboardLayout;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(KeyboardSettingsDatabase) == 64);

//!< システム設定データベースのヘッダを表す構造体です。
struct SystemSettingsDatabaseHeader final
{
    uint32_t version;
    ::nn::Bit8 margin[12];
};

NN_STATIC_ASSERT(sizeof(SystemSettingsDatabaseHeader) == 16);

//!< システム設定データベースを表す構造体です。
struct SystemSettingsDatabase final
{
    SystemSettingsDatabaseHeader header;
    LanguageSettingsDatabase languageSettingsDatabase;
    NetworkSettingsDatabase networkSettingsDatabase;
    BluetoothDevicesSettingsDatabase bluetoothDevicesSettingsDatabase;
    LdnSettingsDatabase ldnSettingsDatabase;
    MiiSettingsDatabase miiSettingsDatabase;
    NxControllerSettingsDatabase nxControllerSettingsDatabase;
    PcvSettingsDatabase pcvSettingsDatabase;
    PushNotificationSettingsDatabase pushNotificationSettingsDatabase;
    ErrorReportSettingsDatabase errorReportSettingsDatabase;
    KeyboardSettingsDatabase keyboardSettingsDatabase;
    WebSettingsDatabase webSettingsDatabase;
    ServiceDiscoverySettingsDatabase serviceDiscoverySettingsDatabase;
    SixAxisSensorDatabase sixAxisSensorDatabase;
    SystemApplicationSettingsDatabase systemApplicationSettingsDatabase;
    BacklightSettingsDatabase backlightSettingsDatabase;
    ClockSettingsDatabase clockSettingsDatabase;
    AccountSettingsDatabase accountSettingsDatabase;
    AudioSettingsDatabase audioSettingsDatabase;
    NetworkServiceSettingsDatabase networkServiceSettingsDatabase;
    NotificationSettingsDatabase notificationSettingsDatabase;
    AccountNotificationSettingsDatabase accountNotificationSettingsDatabase;
    HidSettingsDatabase hidSettingsDatabase;
    TvSettingsDatabase tvSettingsDatabase;
    DataDeletionSettingsDatabase dataDeletionSettingsDatabase;
    BootSettingsDatabase bootSettingsDatabase;
    TimeZoneSettingsDatabase timeZoneSettingsDatabase;
    CaptureSettingsDatabase captureSettingsDatabase;
    UsbSettingsDatabase usbSettingsDatabase;
    NfcSettingsDatabase nfcSettingsDatabase;
    SleepSettingsDatabase sleepSettingsDatabase;
    EulaVersionsDatabase eulaVersionsDatabase;
    DeviceNickNameDatabase deviceNickNameDatabase;
    DrmSettingsDatabase drmSettingsDatabase;

    ::nn::Bit8 margin[349552];
};

NN_STATIC_ASSERT(sizeof(SystemSettingsDatabase) == 524288);

#ifndef SAME_TO_ORIGINAL

const ::nn::settings::system::BacklightSettingsEx& GetDefaultBacklightSettings(
    ) NN_NOEXCEPT
{
    static ::nn::settings::system::BacklightSettingsEx s_BacklightSettings =
        DefaultBacklightSettings;

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        ::nn::util::Float3 brightnessCoefficients = {};

        if (GetCalibBinBacklightBrightnessCoefficientsForInternal(
                &brightnessCoefficients).IsSuccess())
        {
            s_BacklightSettings.brightnessCoefficients = brightnessCoefficients;
        }

        s_IsInitialized = true;
    }

    return s_BacklightSettings;
}

const ::nn::settings::system::TvSettings& GetDefaultTvSettings() NN_NOEXCEPT
{
    static ::nn::settings::system::TvSettings s_TvSettings = DefaultTvSettings;

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        auto isHdmiCecSuppressed = bool();

        if(GetFwdbgHdmiCecSuppressionFlagForInternal(
               &isHdmiCecSuppressed).IsSuccess() && isHdmiCecSuppressed)
        {
            s_TvSettings.flags.Reset<
                ::nn::settings::system::TvFlag::AllowsCec>();
        }

        s_IsInitialized = true;
    }

    return s_TvSettings;
}

//!< システムセーブデータを返します。
::nn::Result GetSystemSaveData(SystemSaveData** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemSaveData,
                             sizeof(SystemSaveData),
                         NN_ALIGNOF(SystemSaveData)> s_Storage;

    SystemSaveData& systemSaveData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemSaveData) SystemSaveData();
        systemSaveData.SetSystemSaveDataId(SystemSaveDataId);
        systemSaveData.SetTotalSize(SystemSaveDataTotalSize);
        systemSaveData.SetJournalSize(SystemSaveDataJournalSize);
        systemSaveData.SetMountName(MountName);

        NN_RESULT_DO(systemSaveData.Mount(true));

        s_IsInitialized = true;
    }

    *ppOutValue = &systemSaveData;

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result = getter(pOutValue, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result GetSystemSaveDataValue(
    U* pOutValue,  T outValues[], U count,
    ::std::function<::nn::Result(U*, T*, U, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result =
        getter(pOutValue, outValues, count, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(setter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    const ::nn::Result result = setter(pSystemSaveData, value);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(synchronizes));
    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result SetSystemSaveDataValue(
    const T values[], U count,
    ::std::function<::nn::Result(SystemSaveData*, const T*, U) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(setter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    const ::nn::Result result = setter(pSystemSaveData, values, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(synchronizes));
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = UpdateSystemSaveDataSystemSettings();

    if (result.IsSuccess())
    {
        result = getter(pOutValue);
    }
    else if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataSystemSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutValue);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_SYSTEM_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutValue);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result GetSystemSettingsValue(
    U* pOutCount, T outValues[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(U*, T*, U) NN_NOEXCEPT> getter,
    ::std::function<void(U*, T*, U) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = UpdateSystemSaveDataSystemSettings();

    if (result.IsSuccess())
    {
        result = getter(pOutCount, outValues, count);
    }
    else if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataSystemSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutCount, outValues, count);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_SYSTEM_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutCount, outValues, count);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(setter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = UpdateSystemSaveDataSystemSettings();

    if (result.IsSuccess())
    {
        result = setter(value);
    }
    else if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataSystemSettings();

        if (result.IsSuccess())
        {
            result = setter(value);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_SYSTEM_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result SetSystemSettingsValue(
    const T values[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(const T*, U) NN_NOEXCEPT> setter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(setter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = UpdateSystemSaveDataSystemSettings();

    if (result.IsSuccess())
    {
        result = setter(values, count);
    }
    else if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataSystemSettings();

        if (result.IsSuccess())
        {
            result = setter(values, count);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_SYSTEM_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSettingsValue(
    const char* const functionName,
    ::std::function<::nn::Result() NN_NOEXCEPT> resetter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(resetter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = UpdateSystemSaveDataSystemSettings();

    if (result.IsSuccess())
    {
        result = resetter();
    }
    else if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataSystemSettings();

        if (result.IsSuccess())
        {
            result = resetter();
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_SYSTEM_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

::nn::os::SystemEvent& GetFatalDirtyFlagEvent() NN_NOEXCEPT
{
    static ::nn::util::TypedStorage<::nn::os::SystemEvent,
                             sizeof(::nn::os::SystemEvent),
                         NN_ALIGNOF(::nn::os::SystemEvent)> s_Storage;

    ::nn::os::SystemEvent& event = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&event) ::nn::os::SystemEvent(::nn::os::EventClearMode_ManualClear,
                                          true);

        s_IsInitialized = true;
    }

    return event;
}

::nn::os::SystemEvent& GetTelemetryDirtyFlagEvent() NN_NOEXCEPT
{
    static ::nn::util::TypedStorage<::nn::os::SystemEvent,
                             sizeof(::nn::os::SystemEvent),
                         NN_ALIGNOF(::nn::os::SystemEvent)> s_Storage;

    ::nn::os::SystemEvent& event = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&event) ::nn::os::SystemEvent(::nn::os::EventClearMode_ManualClear,
                                          true);

        s_IsInitialized = true;
    }

    return event;
}

//!< システムセーブデータからバージョンを取得します。
::nn::Result GetSystemSaveDataVersion(
    uint32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, header) +
            offsetof(SystemSettingsDatabaseHeader, version)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータのバージョンを変更します。
::nn::Result SetSystemSaveDataVersion(
    SystemSaveData* pSystemSaveData,
    const uint32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, header) +
            offsetof(SystemSettingsDatabaseHeader, version)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataLanguageCode(
    LanguageCode* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, languageSettingsDatabase) +
            offsetof(LanguageSettingsDatabase, languageCode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataLanguageCode(
    SystemSaveData* pSystemSaveData,
    const LanguageCode& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, languageSettingsDatabase) +
            offsetof(LanguageSettingsDatabase, languageCode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータから SSID 設定を取得します。
::nn::Result GetSystemSaveDataSsidSettings(
    ::nn::settings::system::SsidSettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssidFlags)),
            &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssid)),
            pOutValue->ssid, sizeof(pOutValue->ssid)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssidLength)),
            &pOutValue->ssidLength, sizeof(pOutValue->ssidLength)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータからセキュリティ設定を取得します。
::nn::Result GetSystemSaveDataSecuritySettings(
    ::nn::settings::system::SecuritySettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityAuthenticationMode)),
            &pOutValue->authenticationMode,
            sizeof(pOutValue->authenticationMode)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityEncryptionMode)),
            &pOutValue->encryptionMode, sizeof(pOutValue->encryptionMode)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityKeyMaterial)),
            pOutValue->keyMaterial, sizeof(pOutValue->keyMaterial)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityKeyMaterialLength)),
            &pOutValue->keyMaterialLength,
            sizeof(pOutValue->keyMaterialLength)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータから IP 設定を取得します。
::nn::Result GetSystemSaveDataIpSettings(
    ::nn::settings::system::IpSettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipFlags)),
            &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipAddress)),
            pOutValue->ipAddress, sizeof(pOutValue->ipAddress)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipSubnetMask)),
            pOutValue->subnetMask, sizeof(pOutValue->subnetMask)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipDefaultGateway)),
            pOutValue->defaultGateway, sizeof(pOutValue->defaultGateway)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipPreferredDns)),
            pOutValue->preferredDns, sizeof(pOutValue->preferredDns)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipAlternateDns)),
            pOutValue->alternateDns, sizeof(pOutValue->alternateDns)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータから Proxy 設定を取得します。
::nn::Result GetSystemSaveDataProxySettings(
    ::nn::settings::system::ProxySettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyFlags)),
            &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyHostName)),
            pOutValue->hostName, sizeof(pOutValue->hostName)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyPortNumber)),
            &pOutValue->portNumber, sizeof(pOutValue->portNumber)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyUserName)),
            pOutValue->userName, sizeof(pOutValue->userName)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyPassword)),
            pOutValue->password, sizeof(pOutValue->password)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータからネットワーク設定を取得します。
::nn::Result GetSystemSaveDataNetworkSettings(
    ::nn::settings::system::NetworkSettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   name)),
            pOutValue->name, sizeof(pOutValue->name)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   id)),
            pOutValue->id, sizeof(pOutValue->id)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   flags)),
            &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   nicType)),
            &pOutValue->nicType, sizeof(pOutValue->nicType)));
    NN_RESULT_DO(
        GetSystemSaveDataSsidSettings(
            &pOutValue->ssidSettings, pSystemSaveData, offset));
    NN_RESULT_DO(
        GetSystemSaveDataSecuritySettings(
            &pOutValue->securitySettings, pSystemSaveData, offset));
    NN_RESULT_DO(
        GetSystemSaveDataIpSettings(
            &pOutValue->ipSettings, pSystemSaveData, offset));
    NN_RESULT_DO(
        GetSystemSaveDataProxySettings(
            &pOutValue->proxySettings, pSystemSaveData, offset));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   mtu)),
            &pOutValue->mtu, sizeof(pOutValue->mtu)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataNetworkSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NetworkSettings outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
                offsetof(NetworkSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));
    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(::nn::settings::system::NetworkSettingsCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            GetSystemSaveDataNetworkSettings(
                &outValues[i],
                pSystemSaveData,
                offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
                offsetof(NetworkSettingsDatabase, entries) +
                sizeof(NetworkSettingsDatabaseEntry) * static_cast<size_t>(i)));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータの SSID 設定を変更します。
::nn::Result SetSystemSaveDataSsidSettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::SsidSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssidFlags)),
            &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssid)),
            value.ssid, sizeof(value.ssid)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(NetworkSettingsDatabaseEntry, ssidLength)),
            &value.ssidLength, sizeof(value.ssidLength)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータのセキュリティ設定を変更します。
::nn::Result SetSystemSaveDataSecuritySettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::SecuritySettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityAuthenticationMode)),
            &value.authenticationMode, sizeof(value.authenticationMode)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityEncryptionMode)),
            &value.encryptionMode, sizeof(value.encryptionMode)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityKeyMaterial)),
            value.keyMaterial, sizeof(value.keyMaterial)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   securityKeyMaterialLength)),
            &value.keyMaterialLength, sizeof(value.keyMaterialLength)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータの IP 設定を変更します。
::nn::Result SetSystemSaveDataIpSettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::IpSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipFlags)),
            &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipAddress)),
            value.ipAddress, sizeof(value.ipAddress)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipSubnetMask)),
            value.subnetMask, sizeof(value.subnetMask)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipDefaultGateway)),
            value.defaultGateway, sizeof(value.defaultGateway)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipPreferredDns)),
            value.preferredDns, sizeof(value.preferredDns)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   ipAlternateDns)),
            value.alternateDns, sizeof(value.alternateDns)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータの Proxy 設定を変更します。
::nn::Result SetSystemSaveDataProxySettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::ProxySettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyFlags)),
            &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyHostName)),
            value.hostName, sizeof(value.hostName)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyPortNumber)),
            &value.portNumber, sizeof(value.portNumber)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyUserName)),
            value.userName, sizeof(value.userName)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   proxyPassword)),
            value.password, sizeof(value.password)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータのネットワーク設定を変更します。
::nn::Result SetSystemSaveDataNetworkSettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::NetworkSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    // 書き込みキャッシュを予約
    static const NetworkSettingsDatabaseEntry entry = {};
    NN_RESULT_DO(pSystemSaveData->Write(offset, &entry, sizeof(entry)));

    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   name)),
            value.name, sizeof(value.name)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   id)),
            value.id, sizeof(value.id)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   flags)),
            &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   nicType)),
            &value.nicType, sizeof(value.nicType)));
    NN_RESULT_DO(
        SetSystemSaveDataSsidSettings(
            pSystemSaveData, offset, value.ssidSettings));
    NN_RESULT_DO(
        SetSystemSaveDataSecuritySettings(
            pSystemSaveData, offset, value.securitySettings));
    NN_RESULT_DO(
        SetSystemSaveDataIpSettings(
            pSystemSaveData, offset, value.ipSettings));
    NN_RESULT_DO(
        SetSystemSaveDataProxySettings(
            pSystemSaveData, offset, value.proxySettings));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(NetworkSettingsDatabaseEntry,
                                                   mtu)),
            &value.mtu, sizeof(value.mtu)));

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataNetworkSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NetworkSettings values[],
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(::nn::settings::system::NetworkSettingsCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            SetSystemSaveDataNetworkSettings(
                pSystemSaveData,
                offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
                offsetof(NetworkSettingsDatabase, entries) +
                sizeof(NetworkSettingsDatabaseEntry) * static_cast<size_t>(i),
                values[i]));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
                offsetof(NetworkSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataRegionCode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, regionCode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataRegionCode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, regionCode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataBluetoothDevicesSettings(
    int32_t* pOutCount,
    ::nn::settings::system::BluetoothDevicesSettings* outValues,
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         bluetoothDevicesSettingsDatabase) +
                offsetof(BluetoothDevicesSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::BluetoothDevicesSettingsCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Read(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             bluetoothDevicesSettingsDatabase) +
                    offsetof(BluetoothDevicesSettingsDatabase, entries) +
                    sizeof(::nn::settings::system::BluetoothDevicesSettings) *
                    static_cast<size_t>(i)),
                &outValues[i], sizeof(outValues[i])));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataBluetoothDevicesSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::BluetoothDevicesSettings* values,
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::BluetoothDevicesSettingsCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Write(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             bluetoothDevicesSettingsDatabase) +
                    offsetof(BluetoothDevicesSettingsDatabase, entries) +
                    sizeof(::nn::settings::system::BluetoothDevicesSettings) *
                    static_cast<size_t>(i)),
                &values[i], sizeof(values[i])));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         bluetoothDevicesSettingsDatabase) +
                offsetof(BluetoothDevicesSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataBluetoothEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isBluetoothEnabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(BluetoothDevicesSettingsDatabase, isBluetoothEnabled)),
        &isBluetoothEnabled, sizeof(isBluetoothEnabled)));
    *pOutValue = (isBluetoothEnabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataBluetoothEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isBluetoothEnabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(BluetoothDevicesSettingsDatabase, isBluetoothEnabled)),
        &isBluetoothEnabled, sizeof(isBluetoothEnabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataBluetoothAfhEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isBluetoothAfhEnabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(BluetoothDevicesSettingsDatabase, isBluetoothAfhEnabled)),
        &isBluetoothAfhEnabled, sizeof(isBluetoothAfhEnabled)));
    *pOutValue = (isBluetoothAfhEnabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataBluetoothAfhEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isBluetoothAfhEnabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(BluetoothDevicesSettingsDatabase, isBluetoothAfhEnabled)),
        &isBluetoothAfhEnabled, sizeof(isBluetoothAfhEnabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataBluetoothBoostEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isBluetoothBoostEnabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(
                BluetoothDevicesSettingsDatabase, isBluetoothBoostEnabled)),
        &isBluetoothBoostEnabled, sizeof(isBluetoothBoostEnabled)));
    *pOutValue = (isBluetoothBoostEnabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataBluetoothBoostEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isBluetoothBoostEnabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bluetoothDevicesSettingsDatabase) +
            offsetof(
                BluetoothDevicesSettingsDatabase, isBluetoothBoostEnabled)),
        &isBluetoothBoostEnabled, sizeof(isBluetoothBoostEnabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataLockScreenFlag(bool* pOutValue,
                                             SystemSaveData* pSystemSaveData
                                             ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto lockScreenFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, lockScreenFlag)),
        &lockScreenFlag, sizeof(lockScreenFlag)));
    *pOutValue = (lockScreenFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataLockScreenFlag(SystemSaveData* pSystemSaveData,
                                             const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto lockScreenFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, lockScreenFlag)),
        &lockScreenFlag, sizeof(lockScreenFlag)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータから本体 EULA の同意バージョンを取得します。
::nn::Result GetSystemSaveDataEulaVersions(
    ::nn::settings::system::EulaVersion* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   version)),
            &pOutValue->version, sizeof(pOutValue->version)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   regionCode)),
            &pOutValue->regionCode, sizeof(pOutValue->regionCode)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   clockType)),
            &pOutValue->clockType, sizeof(pOutValue->clockType)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   networkSystemClock)),
            &pOutValue->networkSystemClock,
            sizeof(pOutValue->networkSystemClock)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   steadyClock)),
            &pOutValue->steadyClock, sizeof(pOutValue->steadyClock)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataEulaVersions(
    int32_t* pOutCount,
    ::nn::settings::system::EulaVersion outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, eulaVersionsDatabase) +
                offsetof(EulaVersionsDatabase, count)),
            &valueCount, sizeof(valueCount)));
    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(::nn::settings::system::EulaVersionCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            GetSystemSaveDataEulaVersions(
                &outValues[i],
                pSystemSaveData,
                offsetof(SystemSettingsDatabase, eulaVersionsDatabase) +
                offsetof(EulaVersionsDatabase, entries) +
                sizeof(EulaVersionsDatabaseEntry) * static_cast<size_t>(i)));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータの本体 EULA の同意バージョンを変更します。
::nn::Result SetSystemSaveDataEulaVersions(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::EulaVersion& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    // 書き込みキャッシュを予約
    const EulaVersionsDatabaseEntry entry = {};
    NN_RESULT_DO(pSystemSaveData->Write(offset, &entry, sizeof(entry)));

    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   version)),
            &value.version, sizeof(value.version)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   regionCode)),
            &value.regionCode, sizeof(value.regionCode)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   clockType)),
            &value.clockType, sizeof(value.clockType)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   networkSystemClock)),
            &value.networkSystemClock, sizeof(value.networkSystemClock)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(offsetof(EulaVersionsDatabaseEntry,
                                                   steadyClock)),
            &value.steadyClock, sizeof(value.steadyClock)));

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataEulaVersions(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::EulaVersion values[],
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(::nn::settings::system::EulaVersionCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            SetSystemSaveDataEulaVersions(
                pSystemSaveData,
                offsetof(SystemSettingsDatabase, eulaVersionsDatabase) +
                offsetof(EulaVersionsDatabase, entries) +
                sizeof(EulaVersionsDatabaseEntry) * static_cast<size_t>(i),
                values[i]));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, eulaVersionsDatabase) +
                offsetof(EulaVersionsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataErrorReportSharePermission(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     errorReportSettingsDatabase) +
            offsetof(ErrorReportSettingsDatabase, sharePermission)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataErrorReportSharePermission(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     errorReportSettingsDatabase) +
            offsetof(ErrorReportSettingsDatabase, sharePermission)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataColorSetId(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, colorSetId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataColorSetId(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, colorSetId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataBatteryPercentageFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto batteryPercentageFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, batteryPercentageFlag)),
        &batteryPercentageFlag, sizeof(batteryPercentageFlag)));
    *pOutValue = (batteryPercentageFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataBatteryPercentageFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto batteryPercentageFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, batteryPercentageFlag)),
        &batteryPercentageFlag, sizeof(batteryPercentageFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAppletLaunchFlags(
    ::nn::settings::system::AppletLaunchFlagSet* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, appletLaunchFlags)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAppletLaunchFlags(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AppletLaunchFlagSet& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, appletLaunchFlags)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataQuestFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto questFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, questFlag)),
        &questFlag, sizeof(questFlag)));
    *pOutValue = (questFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataQuestFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto questFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase, questFlag)),
        &questFlag, sizeof(questFlag)));
    NN_RESULT_SUCCESS;
}

template<size_t Version>
::nn::Result GetSystemSaveDataBacklightSettings(
    ::nn::settings::system::BacklightSettingsEx* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, flags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, brightness)),
        &pOutValue->brightness, sizeof(pOutValue->brightness)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, brightnessVr)),
        &pOutValue->brightnessVr, sizeof(pOutValue->brightnessVr)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, brightnessCoefficients)),
        &pOutValue->brightnessCoefficients,
        sizeof(pOutValue->brightnessCoefficients)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, sensorCoefficients)),
        &pOutValue->sensorCoefficients, sizeof(pOutValue->sensorCoefficients)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, delayDarkToBright)),
        &pOutValue->delayDarkToBright, sizeof(pOutValue->delayDarkToBright)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, delayBrightToDark)),
        &pOutValue->delayBrightToDark, sizeof(pOutValue->delayBrightToDark)));
    NN_RESULT_SUCCESS;
}

template<size_t Version>
::nn::Result SetSystemSaveDataBacklightSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::BacklightSettingsEx& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, flags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, brightness)),
        &value.brightness, sizeof(value.brightness)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, brightnessCoefficients)),
        &value.brightnessCoefficients, sizeof(value.brightnessCoefficients)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, sensorCoefficients)),
        &value.sensorCoefficients, sizeof(value.sensorCoefficients)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, delayDarkToBright)),
        &value.delayDarkToBright, sizeof(value.delayDarkToBright)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
            offsetof(BacklightSettingsDatabase, delayBrightToDark)),
        &value.delayBrightToDark, sizeof(value.delayBrightToDark)));

    if (NN_STATIC_CONDITION(0 < Version))
    {
        NN_RESULT_DO(pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, backlightSettingsDatabase) +
                offsetof(BacklightSettingsDatabase, brightnessVr)),
            &value.brightnessVr, sizeof(value.brightnessVr)));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUserSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, userSystemClockContext)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUserSystemClockContext(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, userSystemClockContext)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataNetworkSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, networkSystemClockContext)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataNetworkSystemClockContext(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, networkSystemClockContext)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUserSystemClockAutomaticCorrectionFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto userSystemClockAutomaticCorrectionFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase,
                     userSystemClockAutomaticCorrectionFlag)),
        &userSystemClockAutomaticCorrectionFlag,
        sizeof(userSystemClockAutomaticCorrectionFlag)));
    *pOutValue = (userSystemClockAutomaticCorrectionFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUserSystemClockAutomaticCorrectionFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto userSystemClockAutomaticCorrectionFlag =
        static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase,
                     userSystemClockAutomaticCorrectionFlag)),
        &userSystemClockAutomaticCorrectionFlag,
        sizeof(userSystemClockAutomaticCorrectionFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAccountSettings(
    ::nn::settings::system::AccountSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, accountSettingsDatabase) +
            offsetof(AccountSettingsDatabase, userSelectorFlags)),
        &pOutValue->userSelectorSettings.flags,
        sizeof(pOutValue->userSelectorSettings.flags)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAccountSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AccountSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, accountSettingsDatabase) +
            offsetof(AccountSettingsDatabase, userSelectorFlags)),
        &value.userSelectorSettings.flags,
        sizeof(value.userSelectorSettings.flags)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataSpeakerAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioVolumeFlags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioVolume)),
        &pOutValue->volume, sizeof(pOutValue->volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataSpeakerAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioVolumeFlags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioVolume)),
        &value.volume, sizeof(value.volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHeadphoneAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioVolumeFlags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioVolume)),
        &pOutValue->volume, sizeof(pOutValue->volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHeadphoneAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioVolumeFlags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioVolume)),
        &value.volume, sizeof(value.volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUsbAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioVolumeFlags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioVolume)),
        &pOutValue->volume, sizeof(pOutValue->volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUsbAudioVolume(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AudioVolume& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioVolumeFlags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioVolume)),
        &value.volume, sizeof(value.volume)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHdmiAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, hdmiAudioOutputMode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHdmiAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, hdmiAudioOutputMode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataSpeakerAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioOutputMode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataSpeakerAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, speakerAudioOutputMode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHeadphoneAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioOutputMode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHeadphoneAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneAudioOutputMode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUsbAudioOutputMode(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioOutputMode)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUsbAudioOutputMode(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, usbAudioOutputMode)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataForceMuteOnHeadphoneRemovedFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto forceMuteOnHeadphoneRemovedFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, forceMuteOnHeadphoneRemovedFlag)),
        &forceMuteOnHeadphoneRemovedFlag,
        sizeof(forceMuteOnHeadphoneRemovedFlag)));
    *pOutValue = (forceMuteOnHeadphoneRemovedFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataForceMuteOnHeadphoneRemovedFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto forceMuteOnHeadphoneRemovedFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, forceMuteOnHeadphoneRemovedFlag)),
        &forceMuteOnHeadphoneRemovedFlag,
        sizeof(forceMuteOnHeadphoneRemovedFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHeadphoneVolumeWarningCount(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneVolumeWarningCount)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHeadphoneVolumeWarningCount(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneVolumeWarningCount)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHeadphoneVolumeUpdateFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto headphoneVolumeUpdateFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneVolumeUpdateFlag)),
        &headphoneVolumeUpdateFlag, sizeof(headphoneVolumeUpdateFlag)));
    *pOutValue = (headphoneVolumeUpdateFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHeadphoneVolumeUpdateFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto headphoneVolumeUpdateFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, audioSettingsDatabase) +
            offsetof(AudioSettingsDatabase, headphoneVolumeUpdateFlag)),
        &headphoneVolumeUpdateFlag, sizeof(headphoneVolumeUpdateFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataConsoleInformationUploadFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto consoleInformationUploadFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, networkServiceSettingsDatabase) +
            offsetof(NetworkServiceSettingsDatabase,
                     consoleInformationUploadFlag)),
        &consoleInformationUploadFlag,
        sizeof(consoleInformationUploadFlag)));
    *pOutValue = (consoleInformationUploadFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataConsoleInformationUploadFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto consoleInformationUploadFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, networkServiceSettingsDatabase) +
            offsetof(NetworkServiceSettingsDatabase,
                     consoleInformationUploadFlag)),
        &consoleInformationUploadFlag,
        sizeof(consoleInformationUploadFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAutomaticApplicationDownloadFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto automaticApplicationDownloadFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase,
                     automaticApplicationDownloadFlag)),
        &automaticApplicationDownloadFlag,
        sizeof(automaticApplicationDownloadFlag)));
    *pOutValue = (automaticApplicationDownloadFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAutomaticApplicationDownloadFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto automaticApplicationDownloadFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase,
                     automaticApplicationDownloadFlag)),
        &automaticApplicationDownloadFlag,
        sizeof(automaticApplicationDownloadFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataNotificationSettings(
    ::nn::settings::system::NotificationSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, flags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, volume)),
        &pOutValue->volume, sizeof(pOutValue->volume)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, headTime)),
        &pOutValue->headTime, sizeof(pOutValue->headTime)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, tailTime)),
        &pOutValue->tailTime, sizeof(pOutValue->tailTime)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataNotificationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NotificationSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, flags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, volume)),
        &value.volume, sizeof(value.volume)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, headTime)),
        &value.headTime, sizeof(value.headTime)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, notificationSettingsDatabase) +
            offsetof(NotificationSettingsDatabase, tailTime)),
        &value.tailTime, sizeof(value.tailTime)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータからアカウント毎の通知設定を取得します。
::nn::Result GetSystemSaveDataAccountNotificationSettings(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    SystemSaveData* pSystemSaveData,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, uid)),
            &pOutValue->uid, sizeof(pOutValue->uid)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, flags)),
            &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(
        pSystemSaveData->Read(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, friendPresenceOverlayPermission)),
            &pOutValue->friendPresenceOverlayPermission, sizeof(pOutValue->friendPresenceOverlayPermission)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAccountNotificationSettings(
    int32_t* pOutCount,
    ::nn::settings::system::AccountNotificationSettings* outValues,
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         accountNotificationSettingsDatabase) +
                offsetof(AccountNotificationSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));
    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::AccountNotificationSettingsCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            GetSystemSaveDataAccountNotificationSettings(
                &outValues[i],
                pSystemSaveData,
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             accountNotificationSettingsDatabase) +
                    offsetof(AccountNotificationSettingsDatabase, entries) +
                    sizeof(AccountNotificationSettingsDatabaseEntry) *
                    static_cast<size_t>(i))));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータのアカウント毎の通知設定を変更します。
::nn::Result SetSystemSaveDataAccountNotificationSettings(
    SystemSaveData* pSystemSaveData,
    int64_t offset,
    const ::nn::settings::system::AccountNotificationSettings& value
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    // 書き込みキャッシュを予約
    const AccountNotificationSettingsDatabaseEntry entry = {};
    NN_RESULT_DO(pSystemSaveData->Write(offset, &entry, sizeof(entry)));

    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, uid)),
            &value.uid, sizeof(value.uid)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, flags)),
            &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(
        pSystemSaveData->Write(
            offset + static_cast<int64_t>(
                offsetof(AccountNotificationSettingsDatabaseEntry, friendPresenceOverlayPermission)),
            &value.friendPresenceOverlayPermission, sizeof(value.friendPresenceOverlayPermission)));

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAccountNotificationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AccountNotificationSettings* values,
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::AccountNotificationSettingsCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            SetSystemSaveDataAccountNotificationSettings(
                pSystemSaveData,
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             accountNotificationSettingsDatabase) +
                    offsetof(AccountNotificationSettingsDatabase, entries) +
                    sizeof(AccountNotificationSettingsDatabaseEntry) *
                    static_cast<size_t>(i)),
                values[i]));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         accountNotificationSettingsDatabase) +
                offsetof(AccountNotificationSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUsbFullKeyEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto usbFullKeyEnableFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, hidSettingsDatabase) +
            offsetof(HidSettingsDatabase, usbFullKeyEnableFlag)),
        &usbFullKeyEnableFlag, sizeof(usbFullKeyEnableFlag)));
    *pOutValue = (usbFullKeyEnableFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUsbFullKeyEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto usbFullKeyEnableFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, hidSettingsDatabase) +
            offsetof(HidSettingsDatabase, usbFullKeyEnableFlag)),
        &usbFullKeyEnableFlag, sizeof(usbFullKeyEnableFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataVibrationMasterVolume(
    float* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, hidSettingsDatabase) +
            offsetof(HidSettingsDatabase, vibrationMasterVolume)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataVibrationMasterVolume(
    SystemSaveData* pSystemSaveData,
    const float& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, hidSettingsDatabase) +
            offsetof(HidSettingsDatabase, vibrationMasterVolume)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataConsoleSixAxisSensorAccelerationBias(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAccelerationBias)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataConsoleSixAxisSensorAccelerationBias(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAccelerationBias)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAngularVelocityBias)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAngularVelocityBias)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataConsoleSixAxisSensorAccelerationGain(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAccelerationGain)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataConsoleSixAxisSensorAccelerationGain(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAccelerationGain)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAngularVelocityGain)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sixAxisSensorDatabase) +
            offsetof(SixAxisSensorDatabase, consoleSixAxisSensorAngularVelocityGain)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataNxControllerSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NxControllerSettings* outValues,
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         nxControllerSettingsDatabase) +
                offsetof(NxControllerSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::NxControllerSettingsCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Read(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             nxControllerSettingsDatabase) +
                    offsetof(NxControllerSettingsDatabase, entries) +
                    sizeof(NxControllerSettingsDatabaseEntry) *
                    static_cast<size_t>(i) +
                    offsetof(NxControllerSettingsDatabaseEntry, value)),
                &outValues[i], sizeof(outValues[i])));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataNxControllerSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::NxControllerSettings* values,
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(
            ::nn::settings::system::NxControllerSettingsCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Write(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase,
                             nxControllerSettingsDatabase) +
                    offsetof(NxControllerSettingsDatabase, entries) +
                    sizeof(NxControllerSettingsDatabaseEntry) *
                    static_cast<size_t>(i) +
                    offsetof(NxControllerSettingsDatabaseEntry, value)),
                &values[i], sizeof(values[i])));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase,
                         nxControllerSettingsDatabase) +
                offsetof(NxControllerSettingsDatabase, count)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataTvSettings(
    ::nn::settings::system::TvSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, flags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvResolution)),
        &pOutValue->tvResolution, sizeof(pOutValue->tvResolution)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, hdmiContentType)),
        &pOutValue->hdmiContentType, sizeof(pOutValue->hdmiContentType)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, rgbRange)),
        &pOutValue->rgbRange, sizeof(pOutValue->rgbRange)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, cmuMode)),
        &pOutValue->cmuMode, sizeof(pOutValue->cmuMode)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvUnderscan)),
        &pOutValue->tvUnderscan, sizeof(pOutValue->tvUnderscan)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvGamma)),
        &pOutValue->tvGamma, sizeof(pOutValue->tvGamma)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, contrastRatio)),
        &pOutValue->contrastRatio, sizeof(pOutValue->contrastRatio)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataTvSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::TvSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, flags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvResolution)),
        &value.tvResolution, sizeof(value.tvResolution)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, hdmiContentType)),
        &value.hdmiContentType, sizeof(value.hdmiContentType)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, rgbRange)),
        &value.rgbRange, sizeof(value.rgbRange)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, cmuMode)),
        &value.cmuMode, sizeof(value.cmuMode)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvUnderscan)),
        &value.tvUnderscan, sizeof(value.tvUnderscan)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, tvGamma)),
        &value.tvGamma, sizeof(value.tvGamma)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, contrastRatio)),
        &value.contrastRatio, sizeof(value.contrastRatio)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataEdid(::nn::settings::system::Edid* pOutValue,
                                   SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, edidData0)),
        pOutValue->data0, sizeof(pOutValue->data0)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, edidData1)),
        pOutValue->data1, sizeof(pOutValue->data1)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataEdid(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::Edid& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, edidData0)),
        value.data0, sizeof(value.data0)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, tvSettingsDatabase) +
            offsetof(TvSettingsDatabase, edidData1)),
        value.data1, sizeof(value.data1)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataDataDeletionSettings(
    ::nn::settings::system::DataDeletionSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, dataDeletionSettingsDatabase) +
            offsetof(DataDeletionSettingsDatabase, flags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, dataDeletionSettingsDatabase) +
            offsetof(DataDeletionSettingsDatabase, useCount)),
        &pOutValue->useCount, sizeof(pOutValue->useCount)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataDataDeletionSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::DataDeletionSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, dataDeletionSettingsDatabase) +
            offsetof(DataDeletionSettingsDatabase, flags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, dataDeletionSettingsDatabase) +
            offsetof(DataDeletionSettingsDatabase, useCount)),
        &value.useCount, sizeof(value.useCount)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataInitialSystemAppletProgramId(
    ::nn::ncm::ProgramId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, initialSystemAppletProgramId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataInitialSystemAppletProgramId(
    SystemSaveData* pSystemSaveData,
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, initialSystemAppletProgramId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataOverlayDispProgramId(
    ::nn::ncm::ProgramId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, overlayDispProgramId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataOverlayDispProgramId(
    SystemSaveData* pSystemSaveData,
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, overlayDispProgramId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataDeviceTimeZoneLocationName(
    ::nn::time::LocationName* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, timeZoneSettingsDatabase) +
            offsetof(TimeZoneSettingsDatabase, deviceTimeZoneLocationName)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataDeviceTimeZoneLocationName(
    SystemSaveData* pSystemSaveData,
    const ::nn::time::LocationName& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, timeZoneSettingsDatabase) +
            offsetof(TimeZoneSettingsDatabase, deviceTimeZoneLocationName)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataPrimaryAlbumStorage(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, captureSettingsDatabase) +
            offsetof(CaptureSettingsDatabase, primaryAlbumStorage)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataPrimaryAlbumStorage(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, captureSettingsDatabase) +
            offsetof(CaptureSettingsDatabase, primaryAlbumStorage)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataUsb30EnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isUsb30Enabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, usbSettingsDatabase) +
            offsetof(UsbSettingsDatabase, isUsb30Enabled)),
        &isUsb30Enabled, sizeof(isUsb30Enabled)));
    *pOutValue = (isUsb30Enabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataUsb30EnableFlag(SystemSaveData* pSystemSaveData,
                                              const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isUsb30Enabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, usbSettingsDatabase) +
            offsetof(UsbSettingsDatabase, isUsb30Enabled)),
        &isUsb30Enabled, sizeof(isUsb30Enabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataNfcEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isNfcEnabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, nfcSettingsDatabase) +
            offsetof(NfcSettingsDatabase, isNfcEnabled)),
        &isNfcEnabled, sizeof(isNfcEnabled)));
    *pOutValue = (isNfcEnabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataNfcEnableFlag(SystemSaveData* pSystemSaveData,
                                            const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isNfcEnabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, nfcSettingsDatabase) +
            offsetof(NfcSettingsDatabase, isNfcEnabled)),
        &isNfcEnabled, sizeof(isNfcEnabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataSleepSettings(
    ::nn::settings::system::SleepSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, flags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, handheldSleepPlan)),
        &pOutValue->handheldSleepPlan, sizeof(pOutValue->handheldSleepPlan)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, consoleSleepPlan)),
        &pOutValue->consoleSleepPlan, sizeof(pOutValue->consoleSleepPlan)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataSleepSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::SleepSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, flags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, handheldSleepPlan)),
        &value.handheldSleepPlan, sizeof(value.handheldSleepPlan)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, sleepSettingsDatabase) +
            offsetof(SleepSettingsDatabase, consoleSleepPlan)),
        &value.consoleSleepPlan, sizeof(value.consoleSleepPlan)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataWirelessLanEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto isWirelessLanEnabled = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
            offsetof(NetworkSettingsDatabase, isWirelessLanEnabled)),
        &isWirelessLanEnabled, sizeof(isWirelessLanEnabled)));
    *pOutValue = (isWirelessLanEnabled == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataWirelessLanEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto isWirelessLanEnabled = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, networkSettingsDatabase) +
            offsetof(NetworkSettingsDatabase, isWirelessLanEnabled)),
        &isWirelessLanEnabled, sizeof(isWirelessLanEnabled)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsFlags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsTimeStamp)),
        &pOutValue->timeStamp, sizeof(pOutValue->timeStamp)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataInitialLaunchSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsFlags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsTimeStamp)),
        &value.timeStamp, sizeof(value.timeStamp)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataDeviceNickName(
    ::nn::settings::system::DeviceNickName* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, deviceNickNameDatabase) +
            offsetof(DeviceNickNameDatabase, deviceNickName)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataDeviceNickName(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::DeviceNickName& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, deviceNickNameDatabase) +
            offsetof(DeviceNickNameDatabase, deviceNickName)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataLdnChannel(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, ldnSettingsDatabase) +
            offsetof(LdnSettingsDatabase, ldnChannel)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataLdnChannel(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, ldnSettingsDatabase) +
            offsetof(LdnSettingsDatabase, ldnChannel)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataMiiAuthorId(
    ::nn::util::Uuid* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, miiSettingsDatabase) +
            offsetof(MiiSettingsDatabase, miiAuthorId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataMiiAuthorId(
    SystemSaveData* pSystemSaveData,
    const ::nn::util::Uuid& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, miiSettingsDatabase) +
            offsetof(MiiSettingsDatabase, miiAuthorId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAutoUpdateEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto enableFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, drmSettingsDatabase) +
            offsetof(DrmSettingsDatabase, isAutoUpdateEnabled)),
        &enableFlag, sizeof(enableFlag)));
    *pOutValue = (enableFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAutoUpdateEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto enableFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, drmSettingsDatabase) +
            offsetof(DrmSettingsDatabase, isAutoUpdateEnabled)),
        &enableFlag, sizeof(enableFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataExternalRtcResetFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto resetFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, pcvSettingsDatabase) +
            offsetof(PcvSettingsDatabase, externalRtcResetFlag)),
        &resetFlag, sizeof(resetFlag)));
    *pOutValue = (resetFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataExternalRtcResetFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto resetFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, pcvSettingsDatabase) +
            offsetof(PcvSettingsDatabase, externalRtcResetFlag)),
        &resetFlag, sizeof(resetFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataInRepairProcessEnableFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto inRepairFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, inRepairFlag)),
        &inRepairFlag, sizeof(inRepairFlag)));
    *pOutValue = (inRepairFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataInRepairProcessEnableFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto inRepairFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, bootSettingsDatabase) +
            offsetof(BootSettingsDatabase, inRepairFlag)),
        &inRepairFlag, sizeof(inRepairFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataPushNotificationActivityModeOnSleep(
    int32_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, pushNotificationSettingsDatabase) +
            offsetof(PushNotificationSettingsDatabase, activityModeOnSleep)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataPushNotificationActivityModeOnSleep(
    SystemSaveData* pSystemSaveData,
    const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, pushNotificationSettingsDatabase) +
            offsetof(PushNotificationSettingsDatabase, activityModeOnSleep)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataKeyboardLayout(
    int32_t* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, keyboardSettingsDatabase) +
            offsetof(KeyboardSettingsDatabase, keyboardLayout)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataKeyboardLayout(
    SystemSaveData* pSystemSaveData, const int32_t& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, keyboardSettingsDatabase) +
            offsetof(KeyboardSettingsDatabase, keyboardLayout)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataWebInspectorFlag(
    bool* pOutValue, SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto webInspectorFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, webSettingsDatabase) +
            offsetof(WebSettingsDatabase, webInspectorFlag)),
        &webInspectorFlag, sizeof(webInspectorFlag)));
    *pOutValue = (webInspectorFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataWebInspectorFlag(
    SystemSaveData* pSystemSaveData, const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto webInspectorFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, webSettingsDatabase) +
            offsetof(WebSettingsDatabase, webInspectorFlag)),
        &webInspectorFlag, sizeof(webInspectorFlag)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataAllowedSslHosts(
    int32_t* pOutCount,
    ::nn::settings::system::AllowedSslHost outValues[],
    int32_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto valueCount = int32_t();
    NN_RESULT_DO(
        pSystemSaveData->Read(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, webSettingsDatabase) +
                offsetof(WebSettingsDatabase, allowedSslHostsCount)),
            &valueCount, sizeof(valueCount)));
    valueCount = Clamp<int32_t>(
        valueCount,
        0,
        static_cast<int32_t>(::nn::settings::system::AllowedSslHostCountMax));

    valueCount = Clamp<int32_t>(count, 0, valueCount);

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Read(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase, webSettingsDatabase) +
                    offsetof(WebSettingsDatabase, allowedSslHosts) +
                    sizeof(outValues[i]) * static_cast<size_t>(i)),
                &outValues[i], sizeof(outValues[i])));
    }

    *pOutCount = valueCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataAllowedSslHosts(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AllowedSslHost values[],
    int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(values != nullptr || count == 0);

    const auto valueCount = Clamp<int32_t>(
        count,
        0,
        static_cast<int32_t>(::nn::settings::system::AllowedSslHostCountMax));

    for (int i = 0; i < valueCount; ++i)
    {
        NN_RESULT_DO(
            pSystemSaveData->Write(
                static_cast<int64_t>(
                    offsetof(SystemSettingsDatabase, webSettingsDatabase) +
                    offsetof(WebSettingsDatabase, allowedSslHosts) +
                    sizeof(values[i]) * static_cast<size_t>(i)),
                &values[i], sizeof(values[i])));
    }

    NN_RESULT_DO(
        pSystemSaveData->Write(
            static_cast<int64_t>(
                offsetof(SystemSettingsDatabase, webSettingsDatabase) +
                offsetof(WebSettingsDatabase, allowedSslHostsCount)),
            &valueCount, sizeof(valueCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataHostFsMountPoint(
    ::nn::settings::system::HostFsMountPoint* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, webSettingsDatabase) +
            offsetof(WebSettingsDatabase, hostFsMountPoint)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataHostFsMountPoint(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(SystemSettingsDatabase, webSettingsDatabase) +
            offsetof(WebSettingsDatabase, hostFsMountPoint)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemSaveDataSystemSettings(
    uint64_t* pOutCount, char outBuffer[], uint64_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto size = ::std::min(
        static_cast<size_t>(count), sizeof(SystemSettingsDatabase));
    NN_RESULT_DO(pSystemSaveData->Read(0, outBuffer, size));
    *pOutCount = size;
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のシステム設定をゼロ初期化します。
::nn::Result ClearSystemSaveDataSystemSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    const auto fileSize = static_cast<int32_t>(sizeof(SystemSettingsDatabase));
    const ::nn::Bit8 zeros[512] = {};
    for (int32_t restSize = fileSize; 0 < restSize; )
    {
        const auto size = ::std::min(
            static_cast<int32_t>(NN_ARRAY_SIZE(zeros)), restSize);
        NN_RESULT_DO(pSystemSaveData->Write(fileSize - restSize, zeros, size));
        restSize -= size;
    }

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のネットワーク設定をリセットします。
::nn::Result ResetSystemSaveDataNetworkSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_STATIC_ASSERT(sizeof(DefaultNetworkSettings) /
                     sizeof(DefaultNetworkSettings[0]) == 1);

    const ::nn::util::Uuid uuid = GenerateUuid();

    ::nn::settings::system::NetworkSettings value = DefaultNetworkSettings[0];

    ::std::copy(::std::begin(uuid.data), ::std::end(uuid.data), value.id);

    NN_RESULT_DO(SetSystemSaveDataNetworkSettings(pSystemSaveData, &value, 1));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上の Mii 設定をリセットします。
::nn::Result ResetSystemSaveDataMiiSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SetSystemSaveDataMiiAuthorId(pSystemSaveData, GenerateUuid()));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のオーディオ設定をリセットします。
::nn::Result ResetSystemSaveDataAudioSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SetSystemSaveDataSpeakerAudioVolume(
                     pSystemSaveData, DefaultSpeakerAudioVolume));
    NN_RESULT_DO(SetSystemSaveDataHeadphoneAudioVolume(
                     pSystemSaveData, DefaultHeadphoneAudioVolume));
    NN_RESULT_DO(SetSystemSaveDataUsbAudioVolume(
                     pSystemSaveData, DefaultUsbAudioVolume));
    NN_RESULT_DO(SetSystemSaveDataHdmiAudioOutputMode(
                     pSystemSaveData, DefaultHdmiAudioOutputMode));
    NN_RESULT_DO(SetSystemSaveDataSpeakerAudioOutputMode(
                     pSystemSaveData, DefaultSpeakerAudioOutputMode));
    NN_RESULT_DO(SetSystemSaveDataHeadphoneAudioOutputMode(
                     pSystemSaveData, DefaultHeadphoneAudioOutputMode));
    NN_RESULT_DO(SetSystemSaveDataUsbAudioOutputMode(
                     pSystemSaveData, DefaultUsbAudioOutputMode));
    NN_RESULT_DO(SetSystemSaveDataForceMuteOnHeadphoneRemovedFlag(
                     pSystemSaveData, DefaultForceMuteOnHeadphoneRemovedFlag));
    NN_RESULT_DO(SetSystemSaveDataHeadphoneVolumeWarningCount(
                     pSystemSaveData, DefaultHeadphoneVolumeWarningCount));
    NN_RESULT_DO(SetSystemSaveDataHeadphoneVolumeUpdateFlag(
                     pSystemSaveData, DefaultHeadphoneVolumeUpdateFlag));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のシステムアプリケーション設定をリセットします。
::nn::Result ResetSystemSaveDataSystemApplicationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& initialLaunchSettings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SetSystemSaveDataLockScreenFlag(
                     pSystemSaveData, DefaultLockScreenFlag));
    NN_RESULT_DO(SetSystemSaveDataColorSetId(
                     pSystemSaveData, DefaultColorSetId));
    NN_RESULT_DO(SetSystemSaveDataInitialLaunchSettings(
                     pSystemSaveData, initialLaunchSettings));
    NN_RESULT_DO(SetSystemSaveDataQuestFlag(pSystemSaveData, DefaultQuestFlag));
    NN_RESULT_DO(SetSystemSaveDataDeviceNickName(
                     pSystemSaveData, DefaultDeviceNickName));
    NN_RESULT_DO(SetSystemSaveDataAppletLaunchFlags(
                     pSystemSaveData, DefaultAppletLaunchFlags));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のクロック設定をリセットします。
::nn::Result ResetSystemSaveDataClockSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SetSystemSaveDataUserSystemClockContext(
                     pSystemSaveData, DefaultUserSystemClockContext));
    NN_RESULT_DO(SetSystemSaveDataNetworkSystemClockContext(
                     pSystemSaveData, DefaultNetworkSystemClockContext));
    NN_RESULT_DO(SetSystemSaveDataUserSystemClockAutomaticCorrectionFlag(
                     pSystemSaveData,
                     DefaultUserSystemClockAutomaticCorrectionFlag));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上の本体 6 軸センサーのキャリブレーション値をリセットします。
::nn::Result ResetSystemSaveDataConsoleSixAxisSensorCalibrationSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SetSystemSaveDataConsoleSixAxisSensorAccelerationBias(
        pSystemSaveData, DefaultConsoleSixAxisSensorAccelerationBias));
    NN_RESULT_DO(SetSystemSaveDataConsoleSixAxisSensorAngularVelocityBias(
        pSystemSaveData, DefaultConsoleSixAxisSensorAngularVelocityBias));
    NN_RESULT_DO(SetSystemSaveDataConsoleSixAxisSensorAccelerationGain(
        pSystemSaveData, DefaultConsoleSixAxisSensorAccelerationGain));
    NN_RESULT_DO(SetSystemSaveDataConsoleSixAxisSensorAngularVelocityGain(
        pSystemSaveData, DefaultConsoleSixAxisSensorAngularVelocityGain));

    NN_RESULT_SUCCESS;
}

//!< 更新された画面の明るさの輝度設定値を返します。
float GetUpdatedBacklightBrightness(
    const ::nn::settings::system::BacklightSettingsEx& backlightSettings
    ) NN_NOEXCEPT
{
    if (backlightSettings.flags.Test<
            ::nn::settings::system::BacklightFlag::Automatic>())
    {
        return backlightSettings.brightness;
    }
    else
    {
        const double brightness = backlightSettings.brightness;

        return static_cast<float>(
            (::std::sqrt(brightness * brightness + 0.07) - 0.063) / 0.937);
    }
}

//!< システムセーブデータ上のシステム設定を更新します。
::nn::Result UpdateSystemSaveDataSystemSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::AccountNotificationSettings
        accountNotificationSettings[],
    int32_t accountNotificationSettingsCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(
        accountNotificationSettings != nullptr ||
        accountNotificationSettingsCount == 0);

    ::nn::settings::system::AccountNotificationSettings
        lastAccountNotificationSettings[
            ::nn::settings::system::AccountNotificationSettingsCountMax] = {};

    for (int32_t i = 0; i < accountNotificationSettingsCount; ++i)
    {
        ::nn::settings::system::AccountNotificationSettings&
            settings = lastAccountNotificationSettings[i];
        settings = accountNotificationSettings[i];
        settings.flags.Set<
            ::nn::settings::system::AccountNotificationFlag::CoralInvitation>();
    }

    NN_RESULT_DO(SetSystemSaveDataAccountNotificationSettings(
                     pSystemSaveData,
                     lastAccountNotificationSettings,
                     accountNotificationSettingsCount));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のシステム設定を更新します。
::nn::Result UpdateSystemSaveDataSystemSettings(
    SystemSaveData* pSystemSaveData,
    uint32_t version,
    const ::nn::settings::system::TvSettings& tvSettings,
    const ::nn::settings::system::BacklightSettingsEx& backlightSettings,
    const ::nn::settings::system::AccountNotificationSettings
        accountNotificationSettings[],
    int32_t accountNotificationSettingsCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES(
        accountNotificationSettings != nullptr ||
        accountNotificationSettingsCount == 0);

    ::nn::settings::system::BacklightSettingsEx localBacklightSettings =
        backlightSettings;

    if (version < 0x020000) // 2.0.0 NUP
    {
        NN_RESULT_DO(SetSystemSaveDataNxControllerSettings(
                         pSystemSaveData, nullptr, 0));
        NN_RESULT_DO(SetSystemSaveDataAutoUpdateEnableFlag(
                         pSystemSaveData, DefaultAutoUpdateEnableFlag));
        NN_RESULT_DO(SetSystemSaveDataBatteryPercentageFlag(
                         pSystemSaveData, DefaultBatteryPercentageFlag));
        NN_RESULT_DO(SetSystemSaveDataExternalRtcResetFlag(
                         pSystemSaveData, DefaultExternalRtcResetFlag));
        NN_RESULT_DO(SetSystemSaveDataUsb30EnableFlag(
                         pSystemSaveData, DefaultUsb30EnableFlag));
        ::nn::settings::system::TvSettings localTvSettings = tvSettings;
        localTvSettings.flags.Set<::nn::settings::system::TvFlag::AllowsCec>(
            GetDefaultTvSettings().flags.Test<
                ::nn::settings::system::TvFlag::AllowsCec>());
        localTvSettings.hdmiContentType =
            GetDefaultTvSettings().hdmiContentType;
        NN_RESULT_DO(SetSystemSaveDataTvSettings(
                         pSystemSaveData, localTvSettings));
    }

    if (version < 0x030000) // 3.0.0 NUP
    {
        NN_RESULT_DO(SetSystemSaveDataBluetoothAfhEnableFlag(
                         pSystemSaveData, DefaultBluetoothAfhEnableFlag));
        NN_RESULT_DO(SetSystemSaveDataBluetoothBoostEnableFlag(
                         pSystemSaveData, DefaultBluetoothBoostEnableFlag));
    }

    if (version < 0x040000) // 4.0.0 NUP
    {
        localBacklightSettings.brightness =
            GetUpdatedBacklightBrightness(backlightSettings);
        localBacklightSettings.brightnessVr =
            GetDefaultBacklightSettings().brightnessVr;
        NN_RESULT_DO(SetSystemSaveDataUsbAudioVolume(
                         pSystemSaveData, DefaultUsbAudioVolume));
        NN_RESULT_DO(SetSystemSaveDataUsbAudioOutputMode(
                         pSystemSaveData, DefaultUsbAudioOutputMode));
        NN_RESULT_DO(SetSystemSaveDataKeyboardLayout(
                         pSystemSaveData, DefaultKeyboardLayout));
        NN_RESULT_DO(ResetSystemSaveDataConsoleSixAxisSensorCalibrationSettings(
                         pSystemSaveData));
        NN_RESULT_DO(UpdateSystemSaveDataSystemSettings(
                         pSystemSaveData,
                         accountNotificationSettings,
                         accountNotificationSettingsCount));
    }

    if (version < 0x040001) // 4.0.0 NUP (NXBTS-16243)
    {
        localBacklightSettings.brightnessCoefficients =
            GetDefaultBacklightSettings().brightnessCoefficients;
        NN_RESULT_DO(SetSystemSaveDataBacklightSettings(
                         pSystemSaveData, localBacklightSettings));
    }

    NN_RESULT_DO(SetSystemSaveDataVersion(
                     pSystemSaveData, CurrentFormatVersion));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のシステム設定を更新します。
::nn::Result UpdateSystemSaveDataSystemSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    auto version = uint32_t();
    ::nn::settings::system::TvSettings tvSettings = {};
    ::nn::settings::system::BacklightSettingsEx backlightSettings = {};
    auto accountNotificationSettingsCount = int32_t();
    ::nn::settings::system::AccountNotificationSettings
        accountNotificationSettings[
            ::nn::settings::system::AccountNotificationSettingsCountMax] = {};

    {
        NN_RESULT_DO(pSystemSaveData->OpenToRead());

        NN_UTIL_SCOPE_EXIT
        {
            pSystemSaveData->Close();
        };

        NN_RESULT_DO(GetSystemSaveDataVersion(&version, pSystemSaveData));
        NN_RESULT_DO(GetSystemSaveDataTvSettings(&tvSettings, pSystemSaveData));
        NN_RESULT_DO(GetSystemSaveDataBacklightSettings(
                         &backlightSettings, pSystemSaveData));
        NN_RESULT_DO(GetSystemSaveDataAccountNotificationSettings(
                         &accountNotificationSettingsCount,
                         accountNotificationSettings,
                         ::nn::settings::system::
                             AccountNotificationSettingsCountMax,
                         pSystemSaveData));
    }

    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    ::nn::Result result = UpdateSystemSaveDataSystemSettings(
        pSystemSaveData,
        version, tvSettings, backlightSettings,
        accountNotificationSettings, accountNotificationSettingsCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(true));

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のシステム設定をリセットします。
::nn::Result ResetSystemSaveDataSystemSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& initialLaunchSettings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(ClearSystemSaveDataSystemSettings(pSystemSaveData));
    NN_RESULT_DO(SetSystemSaveDataLanguageCode(
                     pSystemSaveData, DefaultLanguageCode));
    NN_RESULT_DO(SetSystemSaveDataRegionCode(
                     pSystemSaveData, DefaultRegionCode));
    NN_RESULT_DO(SetSystemSaveDataBluetoothDevicesSettings(
                     pSystemSaveData, nullptr, 0));
    NN_RESULT_DO(SetSystemSaveDataBluetoothEnableFlag(
                     pSystemSaveData, DefaultBluetoothEnableFlag));
    NN_RESULT_DO(SetSystemSaveDataLdnChannel(
                     pSystemSaveData, DefaultLdnChannel));
    NN_RESULT_DO(SetSystemSaveDataAccountSettings(
                     pSystemSaveData, DefaultAccountSettings));
    NN_RESULT_DO(SetSystemSaveDataConsoleInformationUploadFlag(
                     pSystemSaveData, DefaultConsoleInformationUploadFlag));
    NN_RESULT_DO(SetSystemSaveDataAutomaticApplicationDownloadFlag(
                     pSystemSaveData, DefaultAutomaticApplicationDownloadFlag));
    NN_RESULT_DO(SetSystemSaveDataNotificationSettings(
                     pSystemSaveData, DefaultNotificationSettings));
    NN_RESULT_DO(SetSystemSaveDataAccountNotificationSettings(
                     pSystemSaveData, nullptr, 0));
    NN_RESULT_DO(SetSystemSaveDataVibrationMasterVolume(
                     pSystemSaveData, DefaultVibrationMasterVolume));
    NN_RESULT_DO(SetSystemSaveDataDataDeletionSettings(
                     pSystemSaveData, DefaultDataDeletionSettings));
    NN_RESULT_DO(SetSystemSaveDataInitialSystemAppletProgramId(
                     pSystemSaveData, DefaultInitialSystemAppletProgramId));
    NN_RESULT_DO(SetSystemSaveDataOverlayDispProgramId(
                     pSystemSaveData, DefaultOverlayDispProgramId));
    NN_RESULT_DO(SetSystemSaveDataDeviceTimeZoneLocationName(
                     pSystemSaveData, DefaultDeviceTimeZoneLocationName));
    NN_RESULT_DO(SetSystemSaveDataPrimaryAlbumStorage(
                     pSystemSaveData, DefaultPrimaryAlbumStorage));
    NN_RESULT_DO(SetSystemSaveDataNfcEnableFlag(
                     pSystemSaveData, DefaultNfcEnableFlag));
    NN_RESULT_DO(SetSystemSaveDataSleepSettings(
                     pSystemSaveData, DefaultSleepSettings));
    NN_RESULT_DO(SetSystemSaveDataEulaVersions(pSystemSaveData, nullptr, 0));
    NN_RESULT_DO(SetSystemSaveDataErrorReportSharePermission(
                     pSystemSaveData, DefaultErrorReportSharePermission));
    NN_RESULT_DO(SetSystemSaveDataWirelessLanEnableFlag(
                     pSystemSaveData, DefaultWirelessLanEnableFlag));
    NN_RESULT_DO(SetSystemSaveDataPushNotificationActivityModeOnSleep(
                     pSystemSaveData,
                     DefaultPushNotificationActivityModeOnSleep));
    NN_RESULT_DO(SetSystemSaveDataKeyboardLayout(
                     pSystemSaveData, DefaultKeyboardLayout));
    NN_RESULT_DO(ResetSystemSaveDataNetworkSettings(pSystemSaveData));
    NN_RESULT_DO(ResetSystemSaveDataMiiSettings(pSystemSaveData));
    NN_RESULT_DO(ResetSystemSaveDataAudioSettings(pSystemSaveData));
    NN_RESULT_DO(ResetSystemSaveDataSystemApplicationSettings(
                     pSystemSaveData, initialLaunchSettings));
    NN_RESULT_DO(ResetSystemSaveDataClockSettings(pSystemSaveData));
    NN_RESULT_DO(ResetSystemSaveDataConsoleSixAxisSensorCalibrationSettings(
                     pSystemSaveData));
    NN_RESULT_DO(UpdateSystemSaveDataSystemSettings(
                     pSystemSaveData, 0,
                     GetDefaultTvSettings(),
                     GetDefaultBacklightSettings(), nullptr, 0));

    NN_RESULT_SUCCESS;
}

::nn::Result UpdateSystemSaveDataSystemSettings() NN_NOEXCEPT
{
    static bool s_IsUpdated = false;

    if (!s_IsUpdated)
    {
        s_IsUpdated = true;

        SystemSaveData* pSystemSaveData = nullptr;
        NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
        NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);

        NN_RESULT_DO(UpdateSystemSaveDataSystemSettings(pSystemSaveData));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSaveDataSystemSettings() NN_NOEXCEPT
{
    ::nn::settings::system::InitialLaunchSettings initialLaunchSettings = {};
    NN_RESULT_DO(
        GetPrivateSettingsInitialLaunchSettings(&initialLaunchSettings));

    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    ::nn::Result result = pSystemSaveData->Create(
        static_cast<int64_t>(sizeof(SystemSettingsDatabase)));
    NN_RESULT_THROW_UNLESS(
        result.IsSuccess() ||
        ::nn::fs::ResultPathAlreadyExists::Includes(result),
        result);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    result = ResetSystemSaveDataSystemSettings(
        pSystemSaveData, initialLaunchSettings);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(true));
    NN_RESULT_SUCCESS;
}

::nn::Result EnableSystemSaveDataInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->EnableInMemoryModeForDebug(buffer, size));
    NN_RESULT_DO(UpdateSystemSaveDataSystemSettings(pSystemSaveData));
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail

#else

}}} // namespace nn::repair::systemSettings

#endif
