﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/result/result_HandlingUtility.h>
#include "../repair_utility.h"
#include "repair_settings_SystemSettingsDatabase.h"
#include "repair_settings_PrivateSettingsDatabase.h"
#include "repair_settings_ApplicationSettingsDatabase.h"
#include "repair_settingsFlag.h"

namespace nn { namespace repair { namespace detail {

    namespace {

        const std::string s_MountName = "save";
        const std::string s_RootPath  = s_MountName + ":/";

        //!< スペース ID
        const nn::fs::SaveDataSpaceId s_SpaceId = nn::fs::SaveDataSpaceId::ProperSystem;

        //!< SystemSavedata ID
        const nn::fs::SaveDataId SystemSettingsSavedataId      = 0x8000000000000050ull;
        const nn::fs::SaveDataId PrivateSettingsSavedataId     = 0x8000000000000052ull; // 本体初期化で消えないsettingsのセーブデータ
        const nn::fs::SaveDataId ApplicationSettingsSavedataId = 0x8000000000000054ull; // ユーザデータ残し初期化で消えないsettingsのセーブデータ

        nn::Result IsInMigration(bool* pOut, std::string mountName, std::string rootPath)
        {
            nn::settings::system::AppletLaunchFlagSet appletLaunchFlags;
            std::string settingsFileName = "file";

            nn::fs::FileHandle handle;
            NN_RESULT_DO(nn::fs::OpenFile(&handle, (rootPath + settingsFileName).c_str(), nn::fs::OpenMode_Read));
            NN_RESULT_DO(nn::fs::ReadFile(handle,
                                          static_cast<int64_t>(offsetof(nn::repair::systemSettings::SystemSettingsDatabase, systemApplicationSettingsDatabase) +
                                                               offsetof(nn::repair::systemSettings::SystemApplicationSettingsDatabase, appletLaunchFlags)), &appletLaunchFlags, sizeof(appletLaunchFlags)));
            nn::fs::CloseFile(handle);

            *pOut = appletLaunchFlags.Test<nn::settings::system::AppletLaunchFlag::Migration>();

            NN_RESULT_SUCCESS;
        }

        nn::Result SetRequiresRunRepairTimeReviserFlag(std::string mountName, std::string rootPath, bool flag)
        {
            const auto requiresRunRepairTimeReviser = static_cast<uint32_t>(flag);
            // Programs\Eris\Sources\Libraries\settings\detail\settings_SystemSaveData-os.horizon.cpp の SystemSaveDataFileName がファイル名です。
            std::string settingsFileName = "file";

            nn::fs::FileHandle handle;
            NN_RESULT_DO(nn::fs::OpenFile(&handle, (rootPath + settingsFileName).c_str(), nn::fs::OpenMode_Write));
            NN_RESULT_DO(nn::fs::WriteFile(handle,
                                           static_cast<int64_t>(offsetof(nn::repair::privateSettings::PrivateSettingsDatabase, repairSettingsDatabase) +
                                                                offsetof(nn::repair::privateSettings::RepairSettingsDatabase, requiresRunRepairTimeReviser)), &requiresRunRepairTimeReviser, sizeof(requiresRunRepairTimeReviser),
                                           nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            nn::fs::CloseFile(handle);
            NN_RESULT_DO(nn::fs::CommitSaveData(mountName.c_str()));

            NN_RESULT_SUCCESS;
        }

        nn::Result SetInRepairFlag(std::string mountName, std::string rootPath, bool flag)
        {
            const auto inRepairFlag = static_cast<int32_t>(flag);
            std::string settingsFileName = "file";

            nn::fs::FileHandle handle;
            NN_RESULT_DO(nn::fs::OpenFile(&handle, (rootPath + settingsFileName).c_str(), nn::fs::OpenMode_Write));
            NN_RESULT_DO(nn::fs::WriteFile(handle,
                                           static_cast<int64_t>(offsetof(nn::repair::applicationSettings::ApplicationSettingsDatabase, repairSettingsDatabase) +
                                                                offsetof(nn::repair::applicationSettings::RepairSettingsDatabase, inRepairProcessFlag)), &inRepairFlag, sizeof(inRepairFlag),
                                           nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            nn::fs::CloseFile(handle);
            NN_RESULT_DO(nn::fs::CommitSaveData(mountName.c_str()));

            NN_RESULT_SUCCESS;
        }

    } // local namespace

    nn::Result SetInRepairFlag(bool flag)
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        NN_REPAIR_RESULT_DO( nn::fs::MountSystemSaveData( s_MountName.c_str(), s_SpaceId, ApplicationSettingsSavedataId) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(s_MountName.c_str());
        };

        NN_REPAIR_RESULT_DO( SetInRepairFlag( s_MountName, s_RootPath, flag ) );
        NN_RESULT_SUCCESS;
    }

    nn::Result IsInMigration(bool* pOut)
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        NN_REPAIR_RESULT_DO( nn::fs::MountSystemSaveData( s_MountName.c_str(), s_SpaceId, SystemSettingsSavedataId) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(s_MountName.c_str());
        };

        return IsInMigration(pOut, s_MountName, s_RootPath);
    }

    nn::Result SetRequiresRunRepairTimeReviserFlag(bool flag)
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        NN_REPAIR_RESULT_DO( nn::fs::MountSystemSaveData( s_MountName.c_str(), s_SpaceId, PrivateSettingsSavedataId) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(s_MountName.c_str());
        };

        NN_REPAIR_RESULT_DO( SetRequiresRunRepairTimeReviserFlag( s_MountName, s_RootPath, flag ) );
        NN_RESULT_SUCCESS;
    }

}}} // namespace nn::repair::detail
