﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <forward_list>
#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <sstream>
#include <string>
#include <iostream>
#include <iomanip>

#include "../repair_Utility.h"
#include "repair_UnsafeEntry.h"
#include "repair_Record.h"

namespace nn { namespace repair { namespace detail {

    const std::string UnsafeEntry::MountName = "save";

nn::Result UnsafeEntry::WriteTree() const NN_NOEXCEPT
{
    // abort 禁止
    nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

    // ID 記録ファイルの準備
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, BackupRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    NN_REPAIR_RESULT_DO( nn::fs::CreateDirectory(m_outRootPath.c_str()) );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, m_SpaceId));
    std::ostringstream oss;

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        nn::Result result;
        // SaveData saveData(info);
        // nn::Result result = saveData.Mount(MountName);
        SendMessage("SaveData %llX, %llX, %llX, %llX, %llX",
                    info.saveDataId, info.systemSaveDataId, info.applicationId.value, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

        // 出力フォルダ名
        oss.str("");

        switch (info.saveDataType)
        {
            case nn::fs::SaveDataType::Account:
            {
                result = nn::fs::MountSaveData(MountName.c_str(), info.applicationId, info.saveDataUserId);
                oss << "a";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.applicationId.value;
                oss << "_";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.saveDataUserId._data[0];
                oss << "_";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.saveDataUserId._data[1];
                oss << "/";
            }
            break;
            case nn::fs::SaveDataType::Bcat:
            {
                result = nn::fs::MountBcatSaveData(MountName.c_str(), info.applicationId);
                oss << "b";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.applicationId.value;
                oss << "/";
            }
            break;
            case nn::fs::SaveDataType::Device:
            {
                result = nn::fs::MountDeviceSaveData(MountName.c_str(), info.applicationId);
                oss << "d";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.applicationId.value;
                oss << "/";
            }
            break;
            case nn::fs::SaveDataType::System:
            {
                result = nn::fs::MountSystemSaveData( MountName.c_str(), m_SpaceId, info.systemSaveDataId, info.saveDataUserId );

                oss << "s";
                oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.systemSaveDataId;

                if( (info.saveDataUserId._data[0] != 0x0ull) || (info.saveDataUserId._data[1] != 0x0ull))
                {
                    oss << "_";
                    oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.saveDataUserId._data[0];
                    oss << "_";
                    oss << std::setw(16) << std::setfill('0') << std::hex << std::uppercase << info.saveDataUserId._data[1];
                }
                oss << "/";
            }
            break;
            default:
            {
                SendMessage(
                    "Unexpected SaveDataType %d\n",
                    static_cast<int>(info.saveDataType));
            }
        }

        SendRecordLog(handle, "SaveData %llX, %llX, %llX, %llX, %08x\n",
                      info.saveDataId, info.applicationId.value, info.saveDataUserId._data[0], info.saveDataUserId._data[1], result.GetInnerValueForDebug() );

        if( result.IsFailure() )
        {
            SendMessage(" FAIL! (%08x)\n", result.GetInnerValueForDebug() );
            continue;
        }
        else
        {
            SendMessage(" \n");
        }

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName.c_str());
        };

        NN_REPAIR_RESULT_DO(this->WriteHeader(info, oss.str()));
        NN_REPAIR_RESULT_DO(this->WriteBody(oss.str()));
    }

    NN_RESULT_SUCCESS;
}

nn::Result UnsafeEntry::WriteHeader(const nn::fs::SaveDataInfo& info, const std::string& folder) const NN_NOEXCEPT
{
    {
        // セーブデータ出力フォルダの作成
        std::string outpath = m_outRootPath + folder;
        nn::Result result = nn::fs::CreateDirectory(outpath.c_str());

        if( result.IsFailure() )
        {
            SendMessage("Create folder Fail! %s (%08x) \n",outpath.c_str(), result.GetInnerValueForDebug() );
        }
    }

    if (m_OutputMetaInfo)
    {
        nn::Bit64 ownerId;
        int64_t   availableSize;
        int64_t   journalSize;
        uint32_t  flags;

        nn::fs::GetSaveDataOwnerId(&ownerId, m_SpaceId, info.saveDataId);
        nn::fs::GetSaveDataAvailableSize(&availableSize, m_SpaceId, info.saveDataId);
        nn::fs::GetSaveDataJournalSize(&journalSize, m_SpaceId, info.saveDataId);
        nn::fs::GetSaveDataFlags(&flags, m_SpaceId, info.saveDataId);

        // (TODO)
        // ファイル数、ディレクトリ数のカウント
        int64_t count = 0;

        Header header = {count, info, ownerId, availableSize, journalSize, flags};

        nn::fs::FileHandle outhandle;
        std::string outpath = m_outRootPath + folder + "header.bin";
        NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&outhandle, outpath.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(outhandle);
        };

        NN_REPAIR_RESULT_DO( nn::fs::WriteFile(outhandle, 0, &header, sizeof(Header), nn::fs::WriteOption()));
        NN_REPAIR_RESULT_DO( nn::fs::FlushFile(outhandle));
    }

    NN_RESULT_SUCCESS;

}

nn::Result UnsafeEntry::WriteBody(const std::string& folder) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(this->RegisterFiles("", folder, true));
    NN_REPAIR_RESULT_DO(this->RegisterFiles("", folder, false));

    NN_RESULT_SUCCESS;
}

nn::Result UnsafeEntry::RegisterFiles(const std::string& parentDir, const std::string& folder , bool isFile) const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;

    std::string inpath = m_inRootPath + parentDir;

    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(
        &handle,
        inpath.c_str(),
        isFile ? nn::fs::OpenDirectoryMode_File : nn::fs::OpenDirectoryMode_Directory));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;

        if (isFile)
        {
            // エントリの登録作業をする
            const std::string& FilePath = parentDir + Name;
            NN_REPAIR_RESULT_DO(this->WriteFileData(FilePath, folder));
        }
        else
        {
            const std::string& childDir = parentDir + Name + "/";

            const std::string outpath = m_outRootPath + folder + childDir;
            NN_REPAIR_RESULT_DO( nn::fs::CreateDirectory(outpath.c_str()) );

            NN_REPAIR_RESULT_DO(this->RegisterFiles(childDir, folder, true));
            NN_REPAIR_RESULT_DO(this->RegisterFiles(childDir, folder, false));
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result UnsafeEntry::WriteFileData(const std::string& apath, const std::string& folder) const NN_NOEXCEPT
{
    // in
    std::string inpath = m_inRootPath + apath;

    nn::fs::FileHandle inhandle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&inhandle, inpath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(inhandle);
    };

    // out
    std::string outpath = m_outRootPath + folder + apath;
    NN_REPAIR_RESULT_DO(nn::fs::CreateFile(outpath.c_str(), 0));

    nn::fs::FileHandle outhandle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&outhandle, outpath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(outhandle);
    };

    int64_t srcSize;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&srcSize, inhandle));
    SendMessage("%s (%lld bytes) :", outpath.c_str(), srcSize);

    const int BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char []> buffer(new char[BufferSize]);

    for (int64_t cursor = 0; cursor < srcSize; cursor += BufferSize)
    {
        // 進捗表示
        (cursor / BufferSize) % 30 != 0 ? SendMessage("|") : SendMessage("\n(%lld/%lld) |", cursor, srcSize);

        // 書き込むサイズ
        int64_t size = ((srcSize - cursor) < BufferSize) ? (srcSize - cursor) : BufferSize;

        NN_REPAIR_RESULT_DO( nn::fs::ReadFile(inhandle, cursor, buffer.get(), static_cast<size_t>(size)));
        NN_REPAIR_RESULT_DO( nn::fs::WriteFile(outhandle, cursor, buffer.get(), size, nn::fs::WriteOption()));
    }

    NN_REPAIR_RESULT_DO( nn::fs::FlushFile(outhandle));

    SendMessage("\n");

    NN_RESULT_SUCCESS;
}


nn::Result UnsafeEntry::SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    char str[512];
    va_list args;
    va_start(args, pFormat);
    vsnprintf(str, sizeof(str), pFormat, args);
    va_end(args);

    SendRecordLog(handle, str);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

