﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <string>
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_Record.h"

namespace nn { namespace repair { namespace detail {

//!< データ移行を扱うクラスです。
class Transporter final
{
private:
    //!< アーカイブファイル名
    static const std::string ArchiveName;

    //!< アーカイブファイルのパス
    std::string m_ArchivePath;

    //!< セキュアモードフラグ
    bool m_IsSecurely;

public:
    explicit Transporter(const std::string& path) NN_NOEXCEPT
        : m_ArchivePath(path + ArchiveName)
        , m_IsSecurely(true)
    {
        SetRecordDirectoryPath(path.c_str());
    }

    Transporter(const std::string& path, bool isSecurely) NN_NOEXCEPT
        : m_ArchivePath(path + ArchiveName)
        , m_IsSecurely(isSecurely)
    {
        SetRecordDirectoryPath(path.c_str());
    }

    //!< データをエクスポートします。
    nn::Result Export(const BlackListStruct* pList) const NN_NOEXCEPT;

    //!< データをインポートします。
    nn::Result Import() const NN_NOEXCEPT;

    //!< エクスポートした ID か調べます。
    nn::Result ExportedId(bool* pOut, nn::Bit64 id) const NN_NOEXCEPT;

    //!< インポートした ID か調べます。
    nn::Result ImportedId(bool* pOut, nn::Bit64 id) const NN_NOEXCEPT;

    //!< 記録されたデータを表示します。
    nn::Result ShowRecordedData(
            bool* pOutHasCorruption, const char* name,
            ReportFunctionPointer const pFunction, void* const pParameter) const NN_NOEXCEPT;

private:

    //!< アーカイブファイルを作成します。
    nn::Result CreateArchiveFile(std::shared_ptr<CachedArchiveFile>* pOutFile) const NN_NOEXCEPT;

    //!< アーカイブファイルを開きます。
    nn::Result OpenArchiveFile(std::shared_ptr<CachedArchiveFile>* pOutFile) const NN_NOEXCEPT;

    //!< システムセーブデータをエクスポートします。
    nn::Result ExportSystemData(
            std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT;

    //!< 指定したシステムセーブデータをエクスポートします。
    nn::Result ExportSystemData(
            std::shared_ptr<CachedArchiveFile> pFile,
            nn::Bit64 id,
            const BlackList* pBlackList) const NN_NOEXCEPT;

    //!< ユーザセーブデータをエクスポートします。
    nn::Result ExportUserData(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< サムネイルデータをエクスポートします。
    nn::Result ExportThumbnailData(
            std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT;

    //!< アルバムデータをエクスポートします。
    nn::Result ExportAlbumData(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< システムセーブデータをインポートします。
    nn::Result ImportSystemData(
            int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< ユーザセーブデータをインポートします。
    nn::Result ImportUserData(
            int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< サムネイルデータをインポートします。
    nn::Result ImportThumbnailData(
            int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< アルバムデータをインポートします。
    nn::Result ImportAlbumData(
            int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< アカウントデータが正常か調べます。
    bool IsValidAccount() const NN_NOEXCEPT;

    //!< システムセーブが正常か調べます。
    bool IsValidSystem(const BlackList* pBlackList) const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

