﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// #include <nn/mem.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/sf/sf_MemUtility.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/kvdb/kvdb_FlatMapKeyValueStore.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_MaxCount.h>

#include <nn/repair.h>
#include "../repair_utility.h"
#include "repair_SystemUpdateVersion.h"

namespace nn { namespace repair {

    nn::Result GetSystemUpdateVersion(uint32_t* pOutVersion) NN_NOEXCEPT
    {
        return detail::GetSystemUpdateVersion(pOutVersion);
    }

    namespace detail {

    namespace {

        const std::string s_MountName = "save";
        const std::string s_RootPath  = s_MountName + ":/";

        //!< スペース ID
        const nn::fs::SaveDataSpaceId s_SpaceId = nn::fs::SaveDataSpaceId::ProperSystem;

        //!< SystemSavedata ID
        const nn::fs::SaveDataId NcmSavedataId      = 0x8000000000000120ull;

        //!< 確保するHeapSize
        // 参照 : Chris/Sources/Libraries/ncm/ncm_ContentManagerImpl.cpp
        //        NN_ALIGNAS(4096) char g_SystemContentMetaDatabaseHeap[512 * 1024];
        const size_t MemoryHeapSize = 512 * 1024; // ncm のものとあわせる

    } // local namespace

    nn::Result GetSystemUpdateVersion(uint32_t* pOutVersion) NN_NOEXCEPT
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        std::unique_ptr<char []>buffer(new char[MemoryHeapSize + 4096]); // 4096 は 内部 alignment 用
        nn::mem::StandardAllocator allocator;
        allocator.Initialize((void *)buffer.get(), MemoryHeapSize );
        nn::sf::StandardAllocatorMemoryResource * memoryResource = new nn::sf::StandardAllocatorMemoryResource(&allocator);
        NN_UTIL_SCOPE_EXIT
        {
            delete memoryResource;
            allocator.Finalize();
        };

        NN_REPAIR_RESULT_DO( nn::fs::MountSystemSaveData( s_MountName.c_str(), s_SpaceId, NcmSavedataId) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(s_MountName.c_str());
        };

        kvdb::FlatMapKeyValueStore<nn::ncm::ContentMetaKey> kvs;

        const std::string metaFilePath  = s_RootPath + "meta";
        NN_REPAIR_RESULT_DO( kvs.Initialize(metaFilePath.c_str(), nn::ncm::SystemMaxContentMetaCount, memoryResource ));
        NN_REPAIR_RESULT_DO( kvs.Load() );

        auto iter = kvs.GetBeginIterator();
        for (; ! iter.IsEnd(); iter.Next())
        {
            auto iterated = iter.Get();
            if (iterated.type == ncm::ContentMetaType::SystemUpdate)
            {
                *pOutVersion = iterated.version;
                break;
            }
        }

        NN_RESULT_SUCCESS;
    }


    }
}} // namespace nn::repair::detail
