﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/time/time_PosixTime.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <string>
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_SaveData.h"
#include "repair_Surveyor.h"

namespace nn { namespace repair { namespace detail {

//!< セーブデータエントリを扱うクラスです。
class SaveDataEntry final
{
private:

    //!< ヘッダ用の構造体です。(ver2)
    struct LegacyHeader final
    {
        //!< ファイル数
        int64_t count;

        //!< セーブデータ情報
        nn::fs::SaveDataInfo info;

        //!< 所有者の識別子
        nn::Bit64 ownerId;

        //!< データ保存領域のサイズ
        int64_t availableSize;

        //!< ジャーナル領域のサイズ
        int64_t journalSize;

        //!< セーブデータ設定フラグ
        uint32_t flags;
    };

    //!< ヘッダ用の構造体です。(ver3)
    struct Header final
    {
        //!< 旧ヘッダ
        LegacyHeader legacy;

        //!< TimeStamp
        nn::time::PosixTime timeStamp;

        //!< CommitId
        nn::fs::SaveDataCommitId commitId;
    };

    //!< corruption 情報を末尾側につけます (ver2)
    struct Footer final
    {
        int64_t corruptedCount;
    };

    //!< メタ用の構造体です。
    struct Meta final
    {
        //!< サイズ
        int64_t size;

        //!< パス
        char path[nn::fs::EntryNameLengthMax + 1];
    };

    //!< マウント名
    static const std::string MountName;

    //!< ルートパス
    static const std::string RootPath;

    //!< スペース ID
    nn::fs::SaveDataSpaceId m_SpaceId;

    //!< Format Version
    const uint32_t m_Version;

public:
    SaveDataEntry(const nn::fs::SaveDataSpaceId& spaceId, uint32_t version) NN_NOEXCEPT
        : m_SpaceId(spaceId), m_Version(version)
    {
    }

    //!< エントリを数えます。
    nn::Result CountEntries(
            int64_t *pOutCount, const BlackList* pBlackList = nullptr) const NN_NOEXCEPT;

    //!< アーカイブファイルに書き込みます。
    nn::Result WriteArchive(
            std::shared_ptr<CachedArchiveFile> pFile,
            const BlackList* pBlackList = nullptr,
            nn::Bit64 id = 0) const NN_NOEXCEPT;

    //!< エントリを書き込みます。
    nn::Result WriteEntries(
            int64_t* pOutSize,
            std::shared_ptr<CachedArchiveFile> pFile,
            int64_t entriesCount,
            int64_t offset) const NN_NOEXCEPT;

    //!< セーブデータをベリファイします。
    bool VerifyData(nn::Bit64 id) const NN_NOEXCEPT;

    //!< 全てのセーブデータをベリファイします。
    bool VerifyDataAll(const BlackList* pBlackList) const NN_NOEXCEPT;

private:
    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT;

    //!< ボディを書き込みます。
    nn::Result WriteBody(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< フッタを書き込みます。
    nn::Result WriteFooter(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< ファイルを数えます。
    nn::Result CountFiles(int64_t *pOutCount, const std::string& parent, bool isFile) const NN_NOEXCEPT;

    //!< ファイルを登録します。
    nn::Result RegisterFiles(
        int64_t *pOutCount, std::shared_ptr<CachedArchiveFile> pFile, const std::string& parent, bool isFile) const NN_NOEXCEPT;

    //!< ファイルメタとファイルデータを書き込みます。
    nn::Result WriteFileMetaAndData(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT;

    //!< ディレクトリメタを書き込みます。
    nn::Result WriteDirectoryMeta(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT;

    //!< セーブデータ情報を取得します。
    nn::Result GetHeader(
        Header *pOutInfo, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< セーブデータ破損情報を取得します。
    nn::Result GetFooter(
        Footer *pFooter, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルパスとファイルサイズを取得します。
    nn::Result GetFilePathAndSize(
            std::string* pOutPath, int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< Journalバッファが少なくなったら Commit します
    bool CommitIfJournalFull(const SaveData& saveData, int64_t journalSize, int64_t comsumptionBlock) const NN_NOEXCEPT;

    //!< すべてのファイルを書き込みます。
    nn::Result WriteFiles(
            int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile,
            const SaveData& saveData, int64_t journalSize, int64_t fileCount, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルを書き込みます。
    nn::Result WriteFile(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path,
            const SaveData& saveData, int64_t journalSize, int64_t offset, int64_t fileSize) const NN_NOEXCEPT;

    //!< 指定のセーブデータをベリファイします。
    nn::Result VerifyDataImpl(nn::Bit64 id) const NN_NOEXCEPT;

    //!< ファイルをベリファイします。
    nn::Result VerifyFiles(const SaveData& saveData) const NN_NOEXCEPT;

    //!< ファイルをベリファイします。
    nn::Result VerifyFile( const std::string& path ) const NN_NOEXCEPT;

    //!< ファイルを バックアップ します。
    nn::Result ExportFiles(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT;

    //!< 破損情報を記録します。
    nn::Result SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT;

    //!< 既存のセーブデータを削除します。
    nn::Result RemoveExistingData() const NN_NOEXCEPT;

    //!< Migration のセーブデータを削除します。
    nn::Result RemoveMigrationData() const NN_NOEXCEPT;

    //!< 転送速度（経過)を表示します
    void ShowLapRate(uint64_t srcSize) const NN_NOEXCEPT;

    //!< 転送速度（最終計測値）を表示します
    void ShowFinalRate() const NN_NOEXCEPT;

};

}}} // namespace nn::repair::detail

