﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <forward_list>
#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <sstream>
#include <string>

#include "../repair_Utility.h"
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_Directory.h"
#include "repair_MessageReporter.h"
#include "repair_Record.h"
#include "repair_SaveData.h"
#include "repair_SaveDataEntry.h"
#include "repair_settingsFlag.h"
#include "repair_Surveyor.h"

namespace nn { namespace repair { namespace detail {

const std::string SaveDataEntry::MountName = "save";
const std::string SaveDataEntry::RootPath = MountName + ":/";
const nn::fs::SaveDataId MigrationSystemSavedataIds[] = { 0x8000000000000130, 0x8000000000000131 };

namespace {
    int64_t  s_CorruptedFileCount;
    Surveyor s_Surveyor;
    int64_t  s_JournalHistorySize = 0;
    const int64_t JOURNAL_BLOCK_SIZE = 0x4000;
    const int64_t IMPORT_BUFFER_SIZE = 0x100000;
}

nn::Result SaveDataEntry::CountEntries(
        int64_t* pOutCount, const BlackList* pBlackList) const NN_NOEXCEPT
{
    // カウントを増やす前に Fail したときのため
    *pOutCount = 0;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    auto result = nn::fs::OpenSaveDataIterator(&iter, m_SpaceId);
    if (result.IsFailure())
    {
        NN_RESULT_SUCCESS;
    }

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        result = iter->ReadSaveDataInfo(&count, &info, 1);
        if (result.IsFailure())
        {
            NN_RESULT_SUCCESS;
        }

        if (count == 0)
        {
            break;
        }

        SaveData saveData(info);
        if (saveData.IsNecessary(pBlackList))
        {
            if (this->VerifyFiles(saveData).IsFailure())
            {
                continue;
            }
            *pOutCount += 1;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::WriteArchive(
        std::shared_ptr<CachedArchiveFile> pFile,
        const BlackList* pBlackList,
        nn::Bit64 id) const NN_NOEXCEPT
{
    // ID 記録ファイルの準備
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, BackupRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    s_Surveyor.StartTick();

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, m_SpaceId));

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        SaveData saveData(info);

        // バックアップの必要があるデータか確認
        // id の指定がある場合それだけをバックアップ
        bool isNecessaryData = saveData.IsNecessary(pBlackList);
        if (isNecessaryData && (id != 0))
        {
            isNecessaryData = (saveData.GetInfo().saveDataId == id) ? true : false;
        }

        if (!isNecessaryData)
        {
            continue;
        }

        if( !pFile ) // this->VerifyDataAll (Transporter::IsValidSystem)
        {
            auto result = this->VerifyFiles(saveData);
            if (result.IsFailure())
            {
                SendMessage("Corrupted!! SaveData %llX, %llX, %llX,\n\t %llX, %llX\n",
                            info.saveDataId, info.systemSaveDataId, info.applicationId.value,
                            info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
                return result;
            }
        }
        else
        {
            // User Data の verify はやめる
            // system save data については Transporter::IsValidSystem (pFile=null) で事前走査している

            SendMessage("SaveData %llX, %llX, %llX,\n\t %llX, %llX \n",
                        info.saveDataId, info.systemSaveDataId, info.applicationId.value,
                        info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

            s_CorruptedFileCount = 0;
            NN_REPAIR_RESULT_DO(this->ExportFiles(pFile, saveData));

            // User Data / Thumbnail / Album は ExportFiles 中で corruption が出る可能性がある
            if (s_CorruptedFileCount > 0)
            {
                NN_REPAIR_RESULT_DO(this->SendCorruptionLog("save data : %llX\n", info.applicationId.value));
                continue;
            }

            // 正しくバックアップできたデータを記録
            SendRecordLog(handle, "SaveData %llX, %llX, %llX, %llX, %llX\n",
                          info.saveDataId, info.systemSaveDataId, info.applicationId.value, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

            SendMessage("\n");
        }
    }

    // 計測結果表示
    s_Surveyor.ShowResult();

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT
{
    nn::Bit64 ownerId;
    int64_t   availableSize;
    int64_t   journalSize;
    uint32_t  flags;
    NN_REPAIR_RESULT_DO(saveData.GetOwnerId(&ownerId, m_SpaceId));
    NN_REPAIR_RESULT_DO(saveData.GetAvailableSize(&availableSize, m_SpaceId));
    NN_REPAIR_RESULT_DO(saveData.GetJournalSize(&journalSize, m_SpaceId));
    NN_REPAIR_RESULT_DO(saveData.GetFlags(&flags, m_SpaceId));

    int64_t count = 0;
    NN_REPAIR_RESULT_DO(this->CountFiles(&count, RootPath, true));
    NN_REPAIR_RESULT_DO(this->CountFiles(&count, RootPath, false));

    // 有効な pFile が渡されたときのみ書き込みを実行
    if (pFile)
    {
        s_Surveyor.ShowFileCount(count);

        nn::time::PosixTime timestamp;
        nn::fs::SaveDataCommitId commitId;
        NN_REPAIR_RESULT_DO(
            nn::fs::GetSaveDataTimeStamp(&timestamp, m_SpaceId, saveData.GetInfo().saveDataId));
        NN_REPAIR_RESULT_DO(
            nn::fs::GetSaveDataCommitId(&commitId, m_SpaceId, saveData.GetInfo().saveDataId));

        LegacyHeader legacy = {count, saveData.GetInfo(), ownerId, availableSize, journalSize, flags};
        Header header = {legacy, timestamp, commitId};

        // for debug
        // SendMessage("SaveDataId %llx , TimeStamp %llX , CommitId %llX\n", saveData.GetInfo().saveDataId, timestamp.value, commitId);

        auto offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &header, sizeof(Header), false));
    }

    NN_RESULT_SUCCESS;
}

// エントリ末尾にキャンセル情報を入れます
nn::Result SaveDataEntry::WriteFooter(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    // 有効な pFile が渡されたときのみ書き込みを実行
    if (pFile)
    {
        Footer footer = {s_CorruptedFileCount};

        auto offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &footer, sizeof(Footer), false));
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::CountFiles(int64_t* pOutCount, const std::string& parent, bool isFile) const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(
                &handle,
                parent.c_str(),
                isFile ? nn::fs::OpenDirectoryMode_File : nn::fs::OpenDirectoryMode_Directory));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;

        if (isFile)
        {
            *pOutCount += 1;
        }
        else
        {
            const std::string& ChildPath = parent + Name + "/";

            const auto CurrentCount = *pOutCount;
            NN_REPAIR_RESULT_DO(this->CountFiles(pOutCount, ChildPath, true));
            NN_REPAIR_RESULT_DO(this->CountFiles(pOutCount, ChildPath, false));

            // 中に何も入っていないディレクトリもカウントする
            if (CurrentCount == *pOutCount)
            {
                *pOutCount += 1;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::WriteBody(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    int64_t count = 0;
    NN_REPAIR_RESULT_DO(this->RegisterFiles(&count, pFile, RootPath, true));
    NN_REPAIR_RESULT_DO(this->RegisterFiles(&count, pFile, RootPath, false));

    NN_RESULT_SUCCESS;
}

// caution!
//   RegisterFiles 内に Delete の処理を含むので Unmount 前に二回目の走査をするとおそらくハマります。

nn::Result SaveDataEntry::RegisterFiles(
    int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, const std::string& parent, bool isFile) const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(
                &handle,
                parent.c_str(),
                isFile ? nn::fs::OpenDirectoryMode_File : nn::fs::OpenDirectoryMode_Directory));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;

        if (isFile)
        {
            // エントリの登録作業をする
            const std::string& FilePath = parent + Name;
            NN_REPAIR_RESULT_DO(this->WriteFileMetaAndData(pFile, FilePath));

            *pOutCount += 1;
        }
        else
        {
            const std::string& ChildPath = parent + Name + "/";
            int64_t count = 0;
            NN_REPAIR_RESULT_DO(this->RegisterFiles(&count, pFile, ChildPath, true));
            NN_REPAIR_RESULT_DO(this->RegisterFiles(&count, pFile, ChildPath, false));

            if (count == 0)
            {
                const std::string& DirectoryPath = parent + Name + "/";
                NN_REPAIR_RESULT_DO(this->WriteDirectoryMeta(pFile, DirectoryPath));
                *pOutCount += 1;
            }
            else
            {
                *pOutCount += count;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::WriteDirectoryMeta(
        std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT
{
    if (path.back() != '/')
    {
        NN_RESULT_THROW(ResultInvalidName());
    }

    if (pFile)
    {

        Meta meta;

        // データサイズは 0 にしておく
        meta.size = 0;

        // データパスをコピー
        strncpy(meta.path, path.c_str(), sizeof(meta.path));

        // 有効な pFile が渡されたときのみ書き込みを実行
        auto offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), false));
    }

    NN_RESULT_SUCCESS;
}

// 読むだけ
nn::Result SaveDataEntry::VerifyFile( const std::string& path ) const NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    //
    // データサイズを取得
    int64_t srcSize;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&srcSize, handle));

    const int BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char []> buffer(new char[BufferSize]);

    for (int64_t cursor = 0; cursor < srcSize; cursor += BufferSize)
    {
        int64_t size = ((srcSize - cursor) < BufferSize) ? (srcSize - cursor) : BufferSize;
        NN_REPAIR_RESULT_DO(nn::fs::ReadFile(handle, cursor, buffer.get(), static_cast<size_t>(size)));
    }

    NN_RESULT_SUCCESS;
}

// open/close を一回でまとめる
// アーカイブは巻き戻って書き直せないので先に書いたヘッダとの整合性を取るために
// エラーが出てもとりあえず書く

nn::Result SaveDataEntry::WriteFileMetaAndData(
    std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT
{
    Meta meta;

    if (path.back() == '/')
    {
        NN_RESULT_THROW(ResultInvalidName());
    }

    if(pFile == nullptr)
    {
        return this->VerifyFile(path);
    }

    // corrupted 状態を正確に把握する場合はDEBUG_COUNT_ALL_CORRUPTEDを定義する
#ifndef DEBUG_GET_ALL_CORRUPTED
    if(s_CorruptedFileCount > 0)
    {
        // corrupted が出たらそのエントリは捨てるので以降は
        // ヘッダのエントリ数分だけ空ファイルデータを書く (delete は不要)
        meta.size = 0;
        strncpy(meta.path, path.c_str(), sizeof(meta.path));

        int64_t offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), false));

        s_Surveyor.Snap(0);

        NN_RESULT_SUCCESS;
    }
#endif

    nn::fs::FileHandle handle;
    nn::Result result;
    result = nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
        // 処理が終わったら消してしまえ (FAT読み取り速度遅延対策)
        nn::fs::DeleteFile( path.c_str() );
    };

    int64_t srcSize = 0;

    if(result.IsSuccess())
    {
        // データサイズを取得
        result = nn::fs::GetFileSize(&srcSize, handle);
    }

    if(result.IsFailure())
    {
        srcSize = 0;
        s_CorruptedFileCount ++;
        SendMessage("\n=== Corrupted File : %s (%08x) ===\n", path.c_str(), result.GetInnerValueForDebug());

        // go through
        // エラーでも meta をファイルサイズ０で書く（先に書いたヘッダとの整合性維持の為）
    }
    meta.size = srcSize;

    // データパスをコピー
    strncpy(meta.path, path.c_str(), sizeof(meta.path));

    int64_t offset = pFile->GetEndOffset();
    NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), false));

    const int BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char []> buffer(new char[BufferSize]);

    if(srcSize > BufferSize * 30)
    {
        // 大きいファイルの場合だけ表示
        SendMessage("file size %lld bytes :", srcSize);
    }

    offset = pFile->GetEndOffset();

    bool isSkip = false;
    for (int64_t cursor = 0; cursor < srcSize; cursor += BufferSize)
    {
        // 進捗表示
        s_Surveyor.ShowTick(cursor / BufferSize, srcSize / BufferSize);

        // 書き込むサイズ
        int64_t size = ((srcSize - cursor) < BufferSize) ? (srcSize - cursor) : BufferSize;

        if(!isSkip)
        {
            result = nn::fs::ReadFile(handle, cursor, buffer.get(), static_cast<size_t>(size));

            if(result.IsFailure())
            {
                isSkip = true;
                s_CorruptedFileCount ++;
                SendMessage("=== Corrupted File : %s (%08x) ===\n", path.c_str(), result.GetInnerValueForDebug());

                // go through
                // エラーでもとりあえず同サイズ分書く
            }
        }

        NN_REPAIR_RESULT_DO(pFile->Write(offset + cursor, buffer.get(), static_cast<size_t>(size), false));
    }

    s_Surveyor.Snap(srcSize);

    NN_RESULT_SUCCESS;
}

//
// for restore
//

nn::Result SaveDataEntry::WriteEntries(
        int64_t* pOutSize,
        std::shared_ptr<CachedArchiveFile> pFile,
        int64_t entriesCount,
        int64_t offset) const NN_NOEXCEPT
{

    if(pFile == nullptr)
    {
        return nn::repair::ResultUnexpected();
    }

    // ID 記録ファイルの準備
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, RestoreRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    s_Surveyor.StartTick();

    // ターゲット内にあるユーザセーブデータをすべて削除
    // 症状確認時にユーザの遊んでないゲームで確認が行われることがあるため
    if (m_SpaceId == nn::fs::SaveDataSpaceId::User)
    {
        NN_REPAIR_RESULT_DO(this->RemoveExistingData());
    }

    int64_t cursor = 0;
    for (int64_t count = 0; count < entriesCount; count++)
    {
        Header header;
        Footer footer = {0};
        NN_REPAIR_RESULT_DO(this->GetHeader(&header, pFile, offset + cursor));

        LegacyHeader* pLegacy = &header.legacy;
        SendMessage("SaveData %llX, %llX, %llX,\n\t %llX, %llX, %llX\n",
                    pLegacy->info.saveDataId, pLegacy->info.systemSaveDataId, pLegacy->info.applicationId.value,
                    header.timeStamp.value, pLegacy->info.saveDataUserId._data[0], pLegacy->info.saveDataUserId._data[1]);

        SaveData saveData(pLegacy->info);

        // リストア前にターゲット内の当該セーブデータを削除する
        NN_REPAIR_RESULT_DO(saveData.Delete(m_SpaceId));

        NN_REPAIR_RESULT_DO(
            saveData.Create(pLegacy->ownerId, pLegacy->availableSize, pLegacy->journalSize, pLegacy->flags)
        );

        NN_REPAIR_RESULT_DO(saveData.Mount(MountName));
        NN_UTIL_SCOPE_EXIT
        {
            saveData.Unmount(MountName);
            if( footer.corruptedCount > 0 )
            {
                // 書いたセーブデータを削除
                saveData.Delete(m_SpaceId);
                SendMessage("=== Delete SaveData %d, %llX, %llX, %llX,\n\t %llX, %llX ===\n", (int)footer.corruptedCount,
                            pLegacy->info.saveDataId, pLegacy->info.systemSaveDataId, pLegacy->info.applicationId.value,
                            pLegacy->info.saveDataUserId._data[0], pLegacy->info.saveDataUserId._data[1]);
            }
        };

        s_Surveyor.ShowFileCount(pLegacy->count);
        // 旧バージョンは Header Size が異なる
        if( m_Version >= 3 )
        {
            cursor += sizeof(Header);
        }
        else
        {
            cursor += sizeof(LegacyHeader);
        }

        s_JournalHistorySize = 0;
        int64_t entrySize = 0;
        NN_REPAIR_RESULT_DO(this->WriteFiles(&entrySize, pFile, saveData, pLegacy->journalSize, pLegacy->count, offset + cursor));
        cursor += entrySize;

        // 旧バージョンには Footer 無し
        if( m_Version >= 2 )
        {
            NN_REPAIR_RESULT_DO(this->GetFooter(&footer, pFile, offset + cursor));
            cursor += sizeof(Footer);

            if( footer.corruptedCount > 0 )
            {
                continue;
            }
        }
        else
        {
            footer.corruptedCount = 0;
        }

        // やり残しが無い様に
        NN_REPAIR_RESULT_DO(saveData.Commit(MountName));

        // savedataid の再取得(create 後でないと savedataid はわからない)
        nn::fs::SaveDataId myId;
        NN_REPAIR_RESULT_DO(SaveData::GetSaveDataId(&myId, m_SpaceId, pLegacy->info));

        // commit 後に付加属性情報の引っ越し
        if( m_Version >= 3 )
        {
            // タイムスタンプの修正
            NN_REPAIR_RESULT_DO( SetSaveDataTimeStamp(m_SpaceId, myId, header.timeStamp) );

            // commit id の修正
            NN_REPAIR_RESULT_DO( SetSaveDataCommitId(m_SpaceId, myId, header.commitId) );
        }

        // 正しくリストアできたデータを記録
        {
            nn::time::PosixTime timestamp;
            nn::fs::SaveDataCommitId commitId;

            NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataTimeStamp(&timestamp, m_SpaceId, myId));
            NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataCommitId(&commitId, m_SpaceId, myId));

            // for debug
            // SendMessage("SaveDataId %llx , TimeStamp0 %llX , CommitId0 %llX\n", pLegacy->info.saveDataId, timestamp.value, commitId);

            SendRecordLog(handle, "SaveData %llX, %llX, %llX, %llX, %llX, %llX, %llX\n",
                          myId, pLegacy->info.systemSaveDataId, pLegacy->info.applicationId.value,
                          pLegacy->info.saveDataUserId._data[0], pLegacy->info.saveDataUserId._data[1],
                          timestamp.value, commitId);
        }

        SendMessage("\n");
    }

    // 計測結果表示
    s_Surveyor.ShowResult();

    // 修理工程では必ずユーザ移行は解除される
    // migration のセーブデータはシステムプロセスにマウントされてるのでリストア時に削除する
    if (m_SpaceId == nn::fs::SaveDataSpaceId::ProperSystem)
    {
        NN_REPAIR_RESULT_DO(this->RemoveMigrationData());
    }

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::GetFilePathAndSize(
    std::string* pOutPath, int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Meta meta;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &meta, sizeof(Meta)));

    // meta 情報壊れチェック
    NN_REPAIR_ABORT_UNLESS( meta.path[0] != 0 );
    NN_REPAIR_ABORT_UNLESS( strnlen( meta.path, nn::fs::EntryNameLengthMax + 1) <= nn::fs::EntryNameLengthMax );
    NN_REPAIR_ABORT_UNLESS( meta.size <= pFile->GetEndOffset() );

    *pOutPath = meta.path;
    *pOutSize = meta.size;

    NN_RESULT_SUCCESS;
}

bool SaveDataEntry::CommitIfJournalFull(const SaveData& saveData, int64_t journalSize, int64_t comsumptionBlock) const NN_NOEXCEPT
{
    // Journal ブロック消費表
    // 管理ブロック 2
    // ファイル削除 3
    // ファイル作成 4
    // ファイル個数管理 1000 ファイルなら 8 位
    // ディレクトリ作成 多くて16 位
    s_JournalHistorySize += comsumptionBlock * JOURNAL_BLOCK_SIZE;
    bool isFull = s_JournalHistorySize > journalSize - IMPORT_BUFFER_SIZE - JOURNAL_BLOCK_SIZE * 0x20; // 余裕をもたせる

    if ( isFull )
    {
        saveData.Commit(MountName);
        s_JournalHistorySize = 0;
    }

    return isFull;
}

nn::Result SaveDataEntry::WriteFiles(
        int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile,
        const SaveData& saveData, int64_t journalSize, int64_t fileCount, int64_t offset) const NN_NOEXCEPT
{
    // fileCount = 0 も許容する
    int64_t cursor = 0;
    for (int64_t count = 0; count < fileCount; count++)
    {
        std::string fullpath;
        int64_t fileSize;
        NN_REPAIR_RESULT_DO(this->GetFilePathAndSize(&fullpath, &fileSize, pFile, offset + cursor));
        std::stringstream ss(fullpath);

        const auto isFilePath = (fullpath.back() != '/');

        std::string buffer;
        std::string path;

        auto IsLastLine = [](const std::stringstream& ss)
        {
            return ss.eof();
        };

        for ( ; ; )
        {
            std::getline(ss, buffer, '/');

            if (IsLastLine(ss) && isFilePath)
            {
                path += buffer;

                NN_REPAIR_RESULT_DO(
                    this->WriteFile(pFile, path, saveData, journalSize, offset + cursor, fileSize));
                cursor += sizeof(Meta) + fileSize;
                break;
            }
            else
            {
                if (IsLastLine(ss))
                {
                    cursor += sizeof(Meta);
                    break;
                }

                path += buffer + "/";

                if (buffer == MountName + ":")
                {
                    continue;
                }

                // パス中のディレクトリがなければ作る
                Directory directory(path);
                if (!directory.IsFound())
                {
                    NN_REPAIR_RESULT_DO(directory.Create());
                    CommitIfJournalFull(saveData , journalSize, 2); // ディレクトリを新規に作成した場合は最大 2 ブロック 消費
                }
            }
        }
    }

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::WriteFile(
    std::shared_ptr<CachedArchiveFile> pFile, const std::string& path,
    const SaveData& saveData, int64_t journalSize, int64_t offset, int64_t fileSize) const NN_NOEXCEPT
{
    // 既にあるファイルを削除
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        NN_REPAIR_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
        CommitIfJournalFull(saveData , journalSize, 3); // ファイルを削除した場合 最大 3 ブロック 消費
    }

    // ファイル作成(ファイルサイズを指定する)
    NN_REPAIR_RESULT_DO(nn::fs::CreateFile(path.c_str(), fileSize));
    CommitIfJournalFull(saveData , journalSize, 4); // ファイルを新規作成した場合 最大 4 ブロック 消費

    // ジャーナル領域のうち 2 Block (32KByte) は FS が使用する可能性がある
    const int64_t BufferSize = std::min<int64_t>( journalSize - 2 * JOURNAL_BLOCK_SIZE , IMPORT_BUFFER_SIZE );
    std::unique_ptr<char []> buffer(new char[static_cast<int>(BufferSize)]);

    if(fileSize > BufferSize * 30)
    {
        // 大きいファイルの場合だけ表示
        SendMessage("file size %llx bytes(%llx) :", fileSize, journalSize);
    }

    for (int64_t cursor = 0; cursor < fileSize; cursor += BufferSize)
    {
        // 進捗表示
        s_Surveyor.ShowTick(cursor / BufferSize, fileSize / BufferSize);

        // 最大 BufferSize 分読む
        int64_t size = ((fileSize - cursor) < BufferSize) ? (fileSize - cursor) : BufferSize;

        // アーカイブファイルからデータを読み込む
        NN_REPAIR_RESULT_DO(pFile->Read(offset + sizeof(Meta) + cursor, buffer.get(), static_cast<size_t>(size)));

        // ファイルを開く
        nn::fs::FileHandle handle;
        NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Write));

        // 読み込んだデータを本体に書き込む(Append は遅くなるので使わない)
        NN_REPAIR_RESULT_DO(
                nn::fs::WriteFile(handle, cursor, buffer.get(), static_cast<size_t>(size), nn::fs::WriteOption()));
        NN_REPAIR_RESULT_DO(nn::fs::FlushFile(handle));

        // コミット前にファイルを閉じる
        nn::fs::CloseFile(handle);

        // ジャーナル履歴が充分溜まったらコミット
        CommitIfJournalFull(saveData , journalSize, (size / JOURNAL_BLOCK_SIZE) + 1);
    }

    s_Surveyor.Snap(fileSize);

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::GetHeader(
    Header *pOutInfo, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Header header;

    if( m_Version >= 3 )
    {
        NN_REPAIR_RESULT_DO(pFile->Read(offset, &header, sizeof(Header)));
    }
    else
    {
        // 旧ヘッダ
        NN_REPAIR_RESULT_DO(pFile->Read(offset, &header.legacy, sizeof(LegacyHeader)));
        header.timeStamp.value = 0; // 1970.1.1
    }

    *pOutInfo = header;

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::GetFooter(
    Footer *pFooter, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Footer footer;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &footer, sizeof(Footer)));

    *pFooter = footer;

    NN_RESULT_SUCCESS;
}

bool SaveDataEntry::VerifyData(nn::Bit64 id) const NN_NOEXCEPT
{
    SendMessage("verify save data (0x%llX)\n", id);
    return this->VerifyDataImpl(id).IsSuccess() ? true : false;
}

nn::Result SaveDataEntry::VerifyDataImpl(nn::Bit64 id) const NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, m_SpaceId));

    bool isFound = false;
    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        SaveData saveData(info);
        if (id == ((m_SpaceId == nn::fs::SaveDataSpaceId::ProperSystem) ?
                saveData.GetInfo().saveDataId : saveData.GetInfo().applicationId.value))
        {
            isFound = true;

            NN_REPAIR_RESULT_DO(saveData.Mount(MountName));
            NN_UTIL_SCOPE_EXIT
            {
                saveData.Unmount(MountName);
            };

            // ヌルを渡したときは対象の Read のみ実行
            NN_REPAIR_RESULT_DO(this->WriteHeader(nullptr, saveData));
            NN_REPAIR_RESULT_DO(this->WriteBody(nullptr));

            break;
        }
    }

    // 見つからなかったら壊れていると判断
    if (!isFound)
    {
        SendMessage("0x%llx is not found\n", id);
        NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
    }

    NN_RESULT_SUCCESS;
}

bool SaveDataEntry::VerifyDataAll(const BlackList* pBlackList) const NN_NOEXCEPT
{
    SendMessage("verify data all\n");
    return this->WriteArchive(nullptr, pBlackList).IsSuccess() ? true : false;
}

nn::Result SaveDataEntry::VerifyFiles(const SaveData& saveData) const NN_NOEXCEPT
{
    // System save data 以外は verify しない
    if( m_SpaceId != nn::fs::SaveDataSpaceId::ProperSystem )
    {
        return ::nn::ResultSuccess();
    }

    NN_REPAIR_RESULT_DO(saveData.Mount(MountName));
    NN_UTIL_SCOPE_EXIT
    {
        saveData.Unmount(MountName);
    };

    NN_REPAIR_RESULT_DO(this->WriteHeader(nullptr, saveData));
    NN_REPAIR_RESULT_DO(this->WriteBody(nullptr));

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::ExportFiles(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(saveData.Mount(MountName));
    NN_UTIL_SCOPE_EXIT
    {
        saveData.Unmount(MountName);
    };

    NN_REPAIR_RESULT_DO(this->WriteHeader(pFile, saveData));
    NN_REPAIR_RESULT_DO(this->WriteBody(pFile));
    NN_REPAIR_RESULT_DO(this->WriteFooter(pFile));
    NN_REPAIR_RESULT_DO(pFile->Flush());

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    char str[512];
    va_list args;
    va_start(args, pFormat);
    vsnprintf(str, sizeof(str), pFormat, args);
    va_end(args);

    SendRecordLog(handle, str);

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::RemoveMigrationData() const NN_NOEXCEPT
{
    const nn::fs::SaveDataSpaceId MigrationSpaceId
        = nn::fs::SaveDataSpaceId::ProperSystem;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(
            nn::fs::OpenSaveDataIterator(&iter, MigrationSpaceId));

    std::forward_list<nn::fs::SaveDataId> ids;
    ids.clear();

    // 本体内にあるすべてのシステムセーブデータ ID をチェックし、
    // Migration システムセーブデータの ID と一致するものを見つけたら
    // そのセーブデータ ID を削除用の ID リストに登録していく
    for ( ; ; )
    {
        auto count = int64_t();
        auto info = nn::fs::SaveDataInfo();
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        for (const auto& id : MigrationSystemSavedataIds)
        {
            if (info.systemSaveDataId == id)
            {
                ids.push_front(info.saveDataId);
            }
        }
    }

    // リストに登録されたセーブデータ ID を持つセーブデータを削除
    for (const auto& id : ids)
    {
        SendMessage("Remove migration data 0x%llX\n", id);
        NN_REPAIR_RESULT_DO(
                nn::fs::DeleteSaveData(MigrationSpaceId, id));
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataEntry::RemoveExistingData() const NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, m_SpaceId));

    std::forward_list<nn::fs::SaveDataId> ids;
    ids.clear();

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        ids.push_front(info.saveDataId);
    }

    for (const nn::fs::SaveDataId& id : ids)
    {
        SendMessage("Remove save data %llX\n", id);
        NN_REPAIR_RESULT_DO(nn::fs::DeleteSaveData(m_SpaceId, id));
    }

    NN_RESULT_SUCCESS;
}


}}} // namespace nn::repair::detail

