﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SaveDataTypes.h>
#include <string>
#include "repair_BlackList.h"

namespace nn { namespace repair { namespace detail {

//!< セーブデータを扱うクラスです。
class SaveData final
{
private:
    //!< セーブデータ情報
    nn::fs::SaveDataInfo m_Info;

public:
    explicit SaveData(const nn::fs::SaveDataInfo& info) NN_NOEXCEPT
        : m_Info(info)
    {
    }

    //!< 必要なデータか確認します。
    bool IsNecessary(const BlackList* pList) const NN_NOEXCEPT;

    //!< サムネイルを持つデータか確認します。
    bool HasThumbnail() const NN_NOEXCEPT;

    //!< マウントします。
    nn::Result Mount(const std::string& name) const NN_NOEXCEPT;

    //!< アンマウントします。
    void Unmount(const std::string& name) const NN_NOEXCEPT;

    //!< 作成します。
    nn::Result Create(
            nn::Bit64 ownerId, int64_t availableSize, int64_t journalSize, uint32_t flags) const NN_NOEXCEPT;

    //!< 削除します。
    nn::Result Delete(const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT;

     //!< セーブデータの更新をコミットします
    nn::Result Commit(const std::string& name) const NN_NOEXCEPT;

    //!< セーブデータ情報を返します。
    nn::fs::SaveDataInfo GetInfo() const NN_NOEXCEPT;

    //!< 所有者 ID を返します。
    nn::Result GetOwnerId(nn::Bit64* pOutOwnerId, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT;

    //!< 保存領域のサイズを返します。
    nn::Result GetAvailableSize(int64_t* pOutAvailableSize, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT;

    //!< ジャーナルサイズを返します。
    nn::Result GetJournalSize(int64_t* pOutJournalSize, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT;

    //!< フラグを返します。
    nn::Result GetFlags(uint32_t* pOutFlags, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT;

    static nn::Result GetSaveDataId(nn::fs::SaveDataId* pOutId, const nn::fs::SaveDataSpaceId& spaceId, const nn::fs::SaveDataInfo& ref) NN_NOEXCEPT;

private:
    //!< 偽のセーブデータか確認します。
    bool IsPseudo() const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

