﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <forward_list>
#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <sstream>
#include <string>
#include <iostream>
#include <iomanip>
#include <nn/settings/system/settings_FirmwareVersion.h>

#include "../repair_Utility.h"
#include "repair_ExtractMigration.h"
#include "repair_settingsFlag.h"

namespace nn { namespace repair { namespace detail {

    const std::string ExtractMigration::MountName = "save";

    namespace {

        struct MigrationStoredState
        {
            uint32_t fwVersion;
            nn::migration::user::MigrationState state;
        };

        typedef struct
        {
            uint32_t magic;
            nn::migration::user::MigrationRole   role;
            nn::migration::user::MigrationState  state;

        }MigrationInfoFile;

        const std::string s_targetFilename            = "migration.bin";
        const nn::fs::SaveDataId MigrationStateSavedataId = 0x8000000000000130ull;

        nn::migration::user::MigrationRole                 s_migrationRole;  // origin or destination
        nn::migration::user::MigrationState                s_migrationState; // nn::migration::user::MigrationState
    }

    bool ExtractMigration::GetMigrationIsActive() NN_NOEXCEPT
    {
        bool isActive = false;
        nn::Result result = IsInMigration(&isActive);

        return result.IsSuccess() ? isActive : false;
    }

    //
    // 想定される State か
    //
    bool ExtractMigration::IsResumeableMigrationState(nn::migration::user::MigrationState state) const NN_NOEXCEPT
    {
        return
          ((state == nn::migration::user::MigrationState_InInitialization)
           || (state == nn::migration::user::MigrationState_InTransfer)
           || (state == nn::migration::user::MigrationState_InFinalization)
           || (state == nn::migration::user::MigrationState_Finalized));
    }

    //
    // 適切な範囲内にあるか
    //
    bool ExtractMigration::CheckFirmwareVersion(uint32_t ver) const NN_NOEXCEPT
    {
        nn::settings::system::FirmwareVersion fwVer4;
        fwVer4.major = 4;
        fwVer4.minor = 0;
        fwVer4.micro = 0;

        nn::settings::system::FirmwareVersion fwVerNow;
        nn::settings::system::GetFirmwareVersion(&fwVerNow);

        return (ver >= fwVer4.GetComparableVersion()) && (ver <= fwVerNow.GetComparableVersion());
    }

    bool ExtractMigration::ReadMigrationStateFile(void * pData, size_t dataSize, const std::string & filename) NN_NOEXCEPT
    {
        bool isValid = false;
        nn::fs::FileHandle handle;
        size_t read_size;

        const int BufferSize = 1 * 1024;
        std::unique_ptr<char []> buffer(new char[BufferSize]);

        MigrationStoredState *mig = (MigrationStoredState *)pData;

        nn::Result result = nn::fs::OpenFile(&handle, (m_inRootPath + filename).c_str(), nn::fs::OpenMode_Read);

        if( result.IsSuccess() )
        {
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            result = nn::fs::ReadFile(&read_size, handle, 0, buffer.get(), BufferSize);

            // format check
            if( result.IsSuccess() && read_size == dataSize )
            {
                std::memcpy(mig, buffer.get(), read_size);
                isValid = CheckFirmwareVersion(mig->fwVersion);
            }
            else
            {
                SendMessage("[GetMigrationState] read migration state FAIL! (%08x)\n", result.GetInnerValueForDebug() );
            }
        }
        else if( !nn::fs::ResultPathNotFound().Includes(result) )
        {
            SendMessage("[GetMigrationState] open migration state FAIL! (%08x)\n", result.GetInnerValueForDebug() );
        }

        return isValid;
    }

    bool ExtractMigration::GetMigrationState() NN_NOEXCEPT
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        MigrationStoredState migOrig = {0};
        MigrationStoredState migDest = {0};

        nn::Result result = nn::fs::MountSystemSaveData( MountName.c_str(), m_SpaceId, MigrationStateSavedataId );
        if( !result.IsSuccess() )
        {
            SendMessage("[GetMigrationState] mount fail (%08x).\n", result.GetInnerValueForDebug());
            return false;
        }

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName.c_str());
        };

        bool isValidOrig = ReadMigrationStateFile(&migOrig, sizeof(MigrationStoredState), "server.bin");
        bool isValidDest = ReadMigrationStateFile(&migDest, sizeof(MigrationStoredState), "client.bin");

        if( isValidOrig && isValidDest )
        {
            if( IsResumeableMigrationState(migOrig.state) && IsResumeableMigrationState(migDest.state) )
            {
                SendMessage("[GetMigrationState] can't get role. wrong internal state.\n" );
                return false;
            }
            else if( IsResumeableMigrationState(migOrig.state) )
            {
                s_migrationRole  = nn::migration::user::MigrationRole_Server; // origin
                s_migrationState = migOrig.state;
                SendMessage("[GetMigrationState] the role is origin. \n");
                return true;
            }
            else if (IsResumeableMigrationState(migDest.state) )
            {
                s_migrationRole  = nn::migration::user::MigrationRole_Client; // dest
                s_migrationState = migDest.state;
                SendMessage("[GetMigrationState] the role is destination. \n");
                return true;
            }
            else
            {
                SendMessage("[GetMigrationState] can't get state.(maybe already finished) \n" );
                return false;
            }
        }
        else
        {
            if( isValidOrig )
            {
                s_migrationRole  = nn::migration::user::MigrationRole_Server; // origin
                s_migrationState = migOrig.state;
                SendMessage("[GetMigrationState] the role is origin. \n");
                return true;
            }
            else if ( isValidDest )
            {
                s_migrationRole  = nn::migration::user::MigrationRole_Client; // dest
                s_migrationState = migDest.state;
                SendMessage("[GetMigrationState] the role is destination. \n");
                return true;
            }
            else
            {
                SendMessage("[GetMigrationState] can't get state.(maybe already finished)\n");
                return false;
            }
        }
        return false;
    }

    nn::Result ExtractMigration::Export() NN_NOEXCEPT
    {
        if( this->GetMigrationState() )
        {
            // 取得できたのでsdに保存
            NN_REPAIR_RESULT_DO(this->WriteFileData(s_targetFilename));
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result ExtractMigration::WriteFileData(const std::string& apath) NN_NOEXCEPT
    {
        // out
        std::string outpath = m_outRootPath + apath;
        NN_REPAIR_RESULT_DO(nn::fs::CreateFile(outpath.c_str(), 0));

        nn::fs::FileHandle outhandle;
        NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&outhandle, outpath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(outhandle);
        };

        MigrationInfoFile plainData;
        plainData.magic = 'MIGS';
        plainData.role  = s_migrationRole;
        plainData.state = s_migrationState;

        NN_REPAIR_RESULT_DO( nn::fs::WriteFile(outhandle, 0, (char *)&plainData, sizeof(MigrationInfoFile), nn::fs::WriteOption()));
        NN_REPAIR_RESULT_DO( nn::fs::FlushFile(outhandle));

        NN_RESULT_SUCCESS;
    }


}}} // namespace nn::repair::detail

