﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/repair/repair_ProtectedFile.h>
#include <nn/os.h>

namespace nn { namespace repair { namespace detail {

    //!< ProtectedFile へのアクセスをキャッシュするラッパークラスです
    //!< ４エリアから成る単純なループバッファを使用します  | Z | A | B | C |
    //!< Read/Write の呼び出しに以下の制限があります。
    //!<
    //!< 1. offset の巻き戻りが発生してはいけません
    //!< 2. offset + size 位置が 前回終端から エリアサイズ（1M byte）を超えてはいけません
    //!<    (offset が 前回終端位置である場合は size が エリアサイズ内であればOK）


    class CachedArchiveFile
    {
        NN_DISALLOW_COPY(CachedArchiveFile);

    public:
        static const int STACK_SIZE = 0x4000;
        struct CommandParam
        {
            nn::fs::OpenMode mode;
            int64_t offset;
            const void* buffer;
            void* mirror;
            size_t size;
            nn::Result result;
        };

        CachedArchiveFile(std::shared_ptr<ProtectedFile> pProtectedFile, nn::fs::OpenMode mode);

        virtual ~CachedArchiveFile();

        nn::Result Write(int64_t offset, const void * buffer, size_t size, bool isFlush);
        nn::Result Read(int64_t offset, void* buffer, size_t size);
        nn::Result Flush();
        nn::Result Close();

        int64_t    GetEndOffset();

        static nn::Result CreateArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath);
        static nn::Result CreateArchive(std::shared_ptr<CachedArchiveFile>* pOut, const char* archivePath, const Key128 &key, const AuthenticationArchiveContent &authContent);
        static nn::Result OpenArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath);
        static nn::Result OpenArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath, const Key128 &key);
        static nn::Result CloseArchive(std::shared_ptr<CachedArchiveFile>* pOut);

        static void ReadWriteProcess(void* arg) NN_NOEXCEPT;

    private:
        void Reset();
        nn::Result ReadFirst();
        nn::Result WriteLast();
        nn::Result StartUp();

        nn::Result DoWriteAsync(int64_t offset, const void* buffer, size_t size);
        nn::Result DoReadAsync(int64_t offset, void* buffer, void* mirror, size_t size);

        //!< ラップする ProtectedFile
        std::shared_ptr<ProtectedFile> m_pCore;

        //!< Read or Write (インスタンス生成時に決定)
        nn::fs::OpenMode m_Mode;

        //!< ループバッファ制御
        uint8_t* m_Cache;        //!< キャッシュ用メモリ (4M Byte)
        int64_t m_BlockSize;      //!< エリアのサイズ（= ProtectedFile に渡すのに最適なサイズ）
        int64_t m_DataSize;       //!< Archive Data サイズ
        int64_t m_NextFP;         //!< Read/Write 終端オフセット(次回の開始点)
        int64_t m_StartFP;        //!< 有効エリア先頭のファイルオフセット ＆ 書き込み完了位置
        int     m_StartArea;      //!< 有効エリア先頭(cache 内オフセット)

        //!< スレッド処理用
        nn::os::ThreadType m_Thread;
        bool m_isThreadAlive;
        nn::os::EventType  m_RequestEvent;
        nn::os::EventType  m_DoneEvent;
        CommandParam       m_Param;
        uint8_t m_ThreadStack[STACK_SIZE + nn::os::ThreadStackAlignment];
    };

}}} // namespace nn::repair::detail
