﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "repair_cachedArchiveFile.h"
#include "repair_MessageReporter.h"
#include "../repair_utility.h"

#define  ENABLE_THREAD_FOR_SD

namespace nn { namespace repair { namespace detail {

    namespace {

        const int SD_ACCESS_PRIORITY = nn::os::DefaultThreadPriority - 1; // 優先度高

    }

    // 固定鍵、修理サーバ認証スキップ（デバッグ専用）
    nn::Result CachedArchiveFile::CreateArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath)
    {
        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());

        // アーカイブファイルの作成
        std::shared_ptr<FileSystem> fileSystem = nn::repair::FileSystem::Create("nnfs");
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "FixedKey"));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            CreateProtectedFile(&file, fileSystem, archivePath,
                                encryptor, Key128::MakeZero(), AuthenticationArchiveContent::MakeZero()));

        *pOut = std::shared_ptr<CachedArchiveFile>(new CachedArchiveFile(file,nn::fs::OpenMode_Write));

        NN_RESULT_DO((*pOut) -> StartUp());

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::CreateArchive(std::shared_ptr<CachedArchiveFile>* pOut, const char* archivePath, const Key128 &key, const AuthenticationArchiveContent &authContent)
    {
        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());

        // アーカイブファイルの作成
        std::shared_ptr<FileSystem> fileSystem = nn::repair::FileSystem::Create("nnfs");
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            CreateProtectedFile(&file, fileSystem, archivePath, encryptor, key, authContent));

        *pOut = std::shared_ptr<CachedArchiveFile>(new CachedArchiveFile(file,nn::fs::OpenMode_Write));

        NN_RESULT_DO((*pOut) -> StartUp());

        NN_RESULT_SUCCESS;
    }

    // 固定鍵（デバッグ専用）
    nn::Result CachedArchiveFile::OpenArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath)
    {
        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());

        // アーカイブファイルを開く
        std::shared_ptr<FileSystem> fileSystem = nn::repair::FileSystem::Create("nnfs");
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "FixedKey"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            OpenProtectedFile(&file, fileSystem, archivePath, encryptor, Key128::MakeZero()));

        *pOut = std::shared_ptr<CachedArchiveFile>(new CachedArchiveFile(file,nn::fs::OpenMode_Read));

        NN_RESULT_DO((*pOut) -> StartUp());
        NN_RESULT_DO((*pOut) -> ReadFirst());

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::OpenArchive(std::shared_ptr<CachedArchiveFile>* pOut , const char* archivePath, const Key128 &key)
    {
        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());

        // アーカイブファイルを開く
        std::shared_ptr<FileSystem> fileSystem = nn::repair::FileSystem::Create("nnfs");
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            OpenProtectedFile(&file, fileSystem, archivePath, encryptor, key));

        *pOut = std::shared_ptr<CachedArchiveFile>(new CachedArchiveFile(file,nn::fs::OpenMode_Read));

        NN_RESULT_DO((*pOut) -> StartUp());
        NN_RESULT_DO((*pOut) -> ReadFirst());

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::CloseArchive(std::shared_ptr<CachedArchiveFile>* pOut)
    {
        NN_RESULT_SUCCESS;
    }

    CachedArchiveFile::CachedArchiveFile(std::shared_ptr<ProtectedFile> pProtectedFile, nn::fs::OpenMode mode)
      : m_pCore(pProtectedFile),m_Mode(mode)
    {
        m_BlockSize = ProtectedFile::GetBlockSize();
        m_Cache = new uint8_t[m_BlockSize * 4];

        Reset();
    }

    nn::Result CachedArchiveFile::StartUp()
    {
        uint64_t alm = (uint64_t)(nn::os::ThreadStackAlignment - 1);
        uint64_t stackAddress = (((uint64_t)m_ThreadStack + alm) & ~alm);
        size_t stackSize = sizeof(m_ThreadStack) - nn::os::ThreadStackAlignment;

        m_isThreadAlive = true;
        nn::Result result = nn::os::CreateThread(
            &m_Thread, ReadWriteProcess, (void *)this, (void *)stackAddress,
            stackSize, SD_ACCESS_PRIORITY);

        nn::os::InitializeEvent(&m_RequestEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&m_DoneEvent, true, nn::os::EventClearMode_ManualClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::StartThread(&m_Thread);

        NN_RESULT_SUCCESS;
    }

    CachedArchiveFile::~CachedArchiveFile()
    {
        delete m_Cache;
        m_pCore = nullptr;
    }

    void CachedArchiveFile::Reset()
    {
        m_NextFP = 0;
        m_DataSize  = 0;
        m_StartArea = 1;
        m_StartFP   = 0;
        std::memset(&m_Cache[0] ,0, m_BlockSize * 4);
        m_Param.offset = 0;
        m_Param.size   = 0;
        m_Param.result = nn::ResultSuccess();

    }

    nn::Result CachedArchiveFile::Close()
    {
        if (m_Mode == nn::fs::OpenMode_Write)
        {
            NN_RESULT_DO(
                this->WriteLast());
        }

        m_pCore->Close();

        m_isThreadAlive = false;
        nn::os::SignalEvent( &m_RequestEvent );
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::ReadFirst()
    {
        m_DataSize = m_pCore->GetDataSize();

        // ３エリア分Cache に溜める
        size_t targetSize = std::min<size_t>( m_DataSize , m_BlockSize * 3);
#ifdef ENABLE_THREAD_FOR_SD
        NN_RESULT_DO(
            DoReadAsync(0, &m_Cache[m_BlockSize], nullptr, targetSize));
        nn::os::WaitEvent(&m_DoneEvent);
#else
        size_t unused;
        NN_RESULT_DO(
            m_pCore->Read(&unused, 0, &m_Cache[m_BlockSize], targetSize));
#endif
        m_Param.offset = targetSize;

        int remainSize = targetSize - m_BlockSize * 2;

        if(remainSize>0)
        {
            std::memcpy(&m_Cache[0] ,&m_Cache[m_BlockSize * 3], remainSize);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::WriteLast()
    {
        // 残りの Cache を払い出す
        size_t size = m_DataSize - m_StartFP;
        if (size > 0)
        {
            int64_t stylus = m_BlockSize * m_StartArea;
            NN_REPAIR_ABORT_UNLESS( stylus >= 0 );
            NN_REPAIR_ABORT_UNLESS( stylus + size  <=  m_BlockSize * 4 );

#ifdef ENABLE_THREAD_FOR_SD
            NN_RESULT_DO(DoWriteAsync(m_StartFP ,&m_Cache[stylus], size));
            nn::os::WaitEvent(&m_DoneEvent);
#else
            NN_RESULT_DO(m_pCore->Write(m_StartFP ,&m_Cache[stylus], size, true));
#endif

        }

        NN_RESULT_SUCCESS;
    }

    void CachedArchiveFile::ReadWriteProcess(void* arg) NN_NOEXCEPT
    {
        CachedArchiveFile *me = reinterpret_cast<CachedArchiveFile *>(arg);

        do
        {
            nn::os::WaitEvent(&me->m_RequestEvent);

            if(!me->m_isThreadAlive)
            {
                break;
            }

            if(me->m_Param.mode == nn::fs::OpenMode_Write)
            {
                me->m_Param.result = me->m_pCore->Write(me->m_Param.offset, me->m_Param.buffer, me->m_Param.size, true);
            }
            else if(me->m_Param.mode == nn::fs::OpenMode_Read)
            {
                size_t unused;
                me->m_Param.result = me->m_pCore->Read(&unused, me->m_Param.offset, const_cast<void *>(me->m_Param.buffer), me->m_Param.size);
                if(me->m_Param.mirror != nullptr)
                {
                    std::memcpy(me->m_Param.mirror, me->m_Param.buffer, me->m_Param.size);
                }
            }

            nn::os::SignalEvent( &me->m_DoneEvent );
        }
        while(me->m_isThreadAlive);
    }

    // Result は前回の結果
    nn::Result CachedArchiveFile::DoWriteAsync(int64_t offset, const void* buffer, size_t size)
    {
        nn::Result result;

        nn::os::WaitEvent(&m_DoneEvent);
        nn::os::ClearEvent(&m_DoneEvent);

        result = m_Param.result; // 前回の結果

        m_Param.mode = nn::fs::OpenMode_Write;
        m_Param.offset = offset;
        m_Param.buffer = buffer;
        m_Param.size   = size;

        nn::os::SignalEvent( &m_RequestEvent );

        return result;
    }

    nn::Result CachedArchiveFile::DoReadAsync(int64_t offset, void* buffer, void* mirror, size_t size)
    {
        nn::Result result;

        nn::os::WaitEvent(&m_DoneEvent);
        nn::os::ClearEvent(&m_DoneEvent);

        result = m_Param.result; // 前回の結果

        m_Param.mode = nn::fs::OpenMode_Read;
        m_Param.offset = offset;
        m_Param.buffer = buffer;
        m_Param.mirror = mirror;
        m_Param.size   = size;

        nn::os::SignalEvent( &m_RequestEvent );

        return result;
    }

    // FP : FilePosition
    // stylus : Cache Memory上の走査位置
    nn::Result CachedArchiveFile::Write(int64_t offset, const void * buffer, size_t size, bool isFlush)
    {
        NN_UNUSED(isFlush);
        NN_REPAIR_ABORT_UNLESS( m_Mode == nn::fs::OpenMode_Write ); //
        NN_REPAIR_ABORT_UNLESS( m_NextFP <= offset ); // 巻き戻りが無いことの確認
        NN_REPAIR_ABORT_UNLESS( m_StartArea >= 0 && m_StartArea < 3 ); // area のチェック

        int64_t stylus = offset - m_StartFP + m_BlockSize * m_StartArea ;

#ifdef ENABLE_THREAD_FOR_SD
        NN_REPAIR_ABORT_UNLESS( m_Param.offset + m_BlockSize * 3 >= offset + size ); // 一周して追いついた
#endif
        // for debug
        // SendMessage("CW FPos:%08x Size:%08x stylus:%08x\n", (uint32_t)offset, size, (uint32_t)stylus + size);

        NN_REPAIR_ABORT_UNLESS( stylus >= 0 ); // stylus の範囲チェック
        NN_REPAIR_ABORT_UNLESS( stylus + size <=  m_BlockSize * 4 ); // size の範囲チェック

        // デバイス側のFP
        int64_t alignedFP = nn::util::align_down(offset + size, m_BlockSize);
        NN_REPAIR_ABORT_UNLESS( alignedFP - m_StartFP <= m_BlockSize ); // stylus が一気に2M byte 進むことは想定していない

        // Cache に書き込む
        memcpy( &m_Cache[stylus] , buffer , size );

        // 針をすすめる
        m_NextFP = offset + size;

        // GetEndOffset で取れる値
        m_DataSize = std::max<int64_t>(m_DataSize, m_NextFP);

        // Cache が溜まった
        if (alignedFP - m_StartFP > 0)
        {
            // 非同期でやる場合は
            // 前回の Write 実行完了を待つ

            // Area | Z | A | B | C |
            // Area A に stylus が到達 : Area Z が確定
            // Area B に stylus が到達 : Area A が確定
            // Area C に stylus が到達 : Area B が確定, AreaC の内容を AreaZ にコピー, stylus を Area Z に移動
#ifdef ENABLE_THREAD_FOR_SD
            DoWriteAsync(m_StartFP, &m_Cache[m_BlockSize * m_StartArea], m_BlockSize);
#else
            NN_RESULT_DO(m_pCore->Write(m_StartFP, &m_Cache[m_BlockSize * m_StartArea], m_BlockSize, true));
#endif

            m_StartArea = (m_StartArea==2) ? 0 : m_StartArea + 1;
            m_StartFP += m_BlockSize;

            if (m_StartArea == 0) // Area C に stylus が到達
            {
                int remainSize = m_DataSize - m_StartFP;
                NN_REPAIR_ABORT_UNLESS( remainSize >= 0 );            // 範囲チェック
                NN_REPAIR_ABORT_UNLESS( remainSize <=  m_BlockSize ); // 範囲チェック

                std::memset(&m_Cache[0] ,0, m_BlockSize);
                std::memcpy(&m_Cache[0] ,&m_Cache[m_BlockSize * 3], remainSize);
            }
        }
        NN_RESULT_SUCCESS;
    }

    //
    // ファイルオープン時に m_BlockSize * 3 の先読みを行う
    // 以降 1 エリア読むごとに３エリア先を Read する
    //
    nn::Result CachedArchiveFile::Read(int64_t offset, void* pOutBuffer, size_t size)
    {
        NN_REPAIR_ABORT_UNLESS( m_Mode == nn::fs::OpenMode_Read );
        NN_REPAIR_ABORT_UNLESS( m_NextFP <= offset ); // 巻き戻りが無いことの確認

        // size_t unused;
        int64_t endFP = std::min<int64_t>(m_DataSize, m_StartFP + m_BlockSize * 2);

        int64_t  stylus = offset - m_StartFP + m_BlockSize * m_StartArea;

#ifdef ENABLE_THREAD_FOR_SD
        NN_REPAIR_ABORT_UNLESS( m_Param.offset >= offset + size ); // 先読み領域に追いついてしまった
#endif
        // for debug
        // SendMessage("CR FPos:%08x Size:%08x stylus:%08x\n", (uint32_t)offset, size, (uint32_t)stylus + size);

        NN_REPAIR_ABORT_UNLESS( m_StartFP <= offset ); // 巻き戻りが無いことの確認
        NN_REPAIR_ABORT_UNLESS( offset + size  <= endFP ); // size の 範囲チェック
        NN_REPAIR_ABORT_UNLESS( m_StartArea >= 0 && m_StartArea < 3 ); // area のチェック

        // cache から読み込む
        memcpy(pOutBuffer , &m_Cache[stylus], size );

        // 針を進める
        m_NextFP = offset + size;

        // Area | Z | A | B | C |
        if (m_NextFP - m_StartFP >= m_BlockSize)
        {
            // 非同期で行う場合は
            // 前回の Read 実行完了を待つ
            int64_t targetFP    = m_StartFP + 3 * m_BlockSize;
            int64_t targetSize  = std::min<int64_t>( m_DataSize - targetFP , m_BlockSize);

            // Area A に stylus が到達 : Area Z に先読み, Area C にもコピー
            // Area B に stylus が到達 : Area A に先読み
            // Area C に stylus が到達 : Area B に先読み, stylus を Z に移動

            if(targetSize > 0)
            {
                // for debug
                // SendMessage("FP:%08x To:%08x size:%lld\n", targetFP, m_BlockSize*m_StartArea, targetSize);
#ifdef ENABLE_THREAD_FOR_SD
                NN_RESULT_DO(DoReadAsync(targetFP, &m_Cache[m_BlockSize*m_StartArea],
                                         (m_StartArea == 0) ? &m_Cache[m_BlockSize * 3] : nullptr,
                                         (size_t)targetSize));
#else
                size_t unused;
                NN_RESULT_DO(m_pCore->Read(&unused, targetFP, &m_Cache[m_BlockSize*m_StartArea], (size_t)targetSize));
                if (m_StartArea == 0) // AreaZ => A
                {
                    std::memcpy(&m_Cache[m_BlockSize * 3], &m_Cache[0], targetSize);
                }
#endif
            }
            else
            {
#ifdef ENABLE_THREAD_FOR_SD
                nn::os::WaitEvent(&m_DoneEvent);
                m_Param.offset = m_Param.offset + m_Param.size;
#endif
            }

            m_StartArea = (m_StartArea==2) ? 0 : m_StartArea + 1;
            m_StartFP += m_BlockSize;

        }
        NN_RESULT_SUCCESS;
    }

    nn::Result CachedArchiveFile::Flush()
    {
        // nothing to be done
        NN_RESULT_SUCCESS;
    }

    int64_t CachedArchiveFile::GetEndOffset()
    {
        return m_DataSize;
    }

}}}
