﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include "repair_CachedArchiveFile.h"
#include <string>

namespace nn { namespace repair { namespace detail {

//!< アルバムデータエントリを扱うクラスです。
class AlbumDataEntry final
{
private:
    //!< ヘッダ用の構造体です。
    struct Header final
    {
        //!< ファイル数
        int64_t count;
    };

    //!< ファイルメタ用の構造体です。
    struct Meta final
    {
        //!< ファイルサイズ
        int64_t size;

        //!< ファイルパス
        char path[nn::fs::EntryNameLengthMax + 1];
    };

    //!< マウント名
    static const std::string MountName;

    //!< ルートパス
    static const std::string RootPath;
public:
    AlbumDataEntry()
    {
    }

    //!< エントリを数えます。
    nn::Result CountEntries(int64_t* pOutCount) const NN_NOEXCEPT;

    //!< アーカイブファイルに書き込みます。
    nn::Result WriteArchive(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< エントリを書き込みます。
    nn::Result WriteEntries(
            int64_t* pOutSize,
            std::shared_ptr<CachedArchiveFile> pFile,
            int64_t entriesCount,
            int64_t offset) const NN_NOEXCEPT;

private:
    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, int64_t count) const NN_NOEXCEPT;

    //!< ボディを書き込みます。
    nn::Result WriteBody(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT;

    //!< ファイルを数えます。
    nn::Result CountFiles(int64_t *pOutCount, const std::string& parent, bool isFile) const NN_NOEXCEPT;

    //!< ファイルを登録します。
    nn::Result RegisterFiles(
            int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, const std::string& parent, bool isFile) const NN_NOEXCEPT;

    //!< ファイルメタ&データを書き込みます。
    nn::Result WriteFileMetaAndData(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT;

    //!< ディレクトリメタを書き込みます。
    nn::Result WriteDirectoryMeta(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT;

    //!< ファイル数を取得します。
    nn::Result GetFileCount(
            int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルパスとファイルサイズを取得します。
    nn::Result GetFilePathAndSize(
            std::string* pOutPath, int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< すべてのファイルを書き込みます。
    nn::Result WriteFiles(
            int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t fileCount, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルを書き込みます。
    nn::Result WriteFile(
            std::shared_ptr<CachedArchiveFile> pFile, const std::string& path, int64_t offset, int64_t fileSize) const NN_NOEXCEPT;

    //!< ファイルをベリファイします。
    // nn::Result VerifyFile(const std::string& path) const NN_NOEXCEPT;

    //!< 破損情報を記録します。
    nn::Result SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT;

    //!< 既存のデータをすべて削除します
    nn::Result RemoveExistingData() const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

