﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/nn_Result.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <sstream>
#include <string>

#include "../repair_Utility.h"
#include "repair_AlbumDataEntry.h"
#include "repair_CachedArchiveFile.h"
#include "repair_Directory.h"
#include "repair_MessageReporter.h"
#include "repair_Record.h"
#include "repair_Surveyor.h"

namespace nn { namespace repair { namespace detail {

const std::string AlbumDataEntry::MountName = "save";
const std::string AlbumDataEntry::RootPath = MountName + ":/";

namespace {
    int64_t  s_CorruptedFileCount;
    Surveyor s_Surveyor;
}

nn::Result AlbumDataEntry::CountEntries(int64_t* pOutCount) const NN_NOEXCEPT
{
    // アルバムのエントリは 1 で固定
    *pOutCount = 1;
    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteArchive(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    s_Surveyor.StartTick();
    s_CorruptedFileCount = 0;

    auto result = nn::fs::MountImageDirectory(MountName.c_str(), nn::fs::ImageDirectoryId::Nand);
    if (result.IsFailure())
    {
        NN_REPAIR_RESULT_DO(this->SendCorruptionLog("all album data\n"));

        // マウントすら出来ない場合はファイル数 0 とする
        NN_REPAIR_RESULT_DO(this->WriteHeader(pFile, 0));
    }
    else
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName.c_str());
        };

        NN_REPAIR_RESULT_DO(this->WriteHeader(pFile));
        NN_REPAIR_RESULT_DO(this->WriteBody(pFile));
        if(pFile)
        {
            NN_REPAIR_RESULT_DO(pFile->Flush());
        }
    }

    s_Surveyor.ShowResult();

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteHeader(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    int64_t count = 0;

    auto result = this->CountFiles(&count, RootPath, true);
    if (result.IsFailure())
    {
        SendMessage("%s files are broken\n", RootPath.c_str());
    }

    result = this->CountFiles(&count, RootPath, false);
    if (result.IsFailure())
    {
        SendMessage("%s directories are broken\n", RootPath.c_str());
    }

    NN_REPAIR_RESULT_DO(this->WriteHeader(pFile, count));

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, int64_t count) const NN_NOEXCEPT
{
    s_Surveyor.ShowFileCount(count);

    Header header = {count};

    auto offset = pFile->GetEndOffset();
    NN_REPAIR_RESULT_DO(pFile->Write(offset, &header, sizeof(Header), false));

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::CountFiles(int64_t* pOutCount, const std::string& parent, bool isFile) const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(
                &handle,
                parent.c_str(),
                isFile ? nn::fs::OpenDirectoryMode_File : nn::fs::OpenDirectoryMode_Directory));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;

        if (isFile)
        {
            *pOutCount += 1;
        }
        else
        {
            const std::string& ParentPath = parent + Name + "/";

            const auto CurrentCount = *pOutCount;
            auto result = this->CountFiles(pOutCount, ParentPath, true);
            if (result.IsFailure())
            {
                SendMessage("%s files are broken\n", Name.c_str());
            }

            result = this->CountFiles(pOutCount, ParentPath, false);
            if (result.IsFailure())
            {
                SendMessage("%s directories are broken\n", Name.c_str());
            }

            // 中に何も入っていないディレクトリもカウント
            if (CurrentCount == *pOutCount)
            {
                *pOutCount += 1;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteBody(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    int64_t count = 0;
    auto result = this->RegisterFiles(&count, pFile, RootPath, true);
    if (result.IsFailure())
    {
        SendMessage("%s files are broken\n", RootPath.c_str());
    }

    result = this->RegisterFiles(&count, pFile, RootPath, false);
    if (result.IsFailure())
    {
        SendMessage("%s directories are broken\n", RootPath.c_str());
    }

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::RegisterFiles(
    int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, const std::string& parent, bool isFile) const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(
                &handle,
                parent.c_str(),
                isFile ? nn::fs::OpenDirectoryMode_File : nn::fs::OpenDirectoryMode_Directory));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;

        if (isFile)
        {
            // エントリの登録作業をする
            const std::string& FilePath = parent + Name;

            NN_REPAIR_RESULT_DO(this->WriteFileMetaAndData(pFile, FilePath));
            *pOutCount += 1;
        }
        else
        {
            const std::string& ParentPath = parent + Name + "/";
            int64_t count = 0;
            auto result = this->RegisterFiles(&count, pFile, ParentPath, true);
            if (result.IsFailure())
            {
                SendMessage("%s files are broken\n", Name.c_str());
            }

            result = this->RegisterFiles(&count, pFile, ParentPath, false);
            if (result.IsFailure())
            {
                SendMessage("%s directories are broken\n", Name.c_str());
            }

            if (count == 0)
            {
                const std::string& DirectoryPath = parent + Name + "/";
                NN_REPAIR_RESULT_DO(this->WriteDirectoryMeta(pFile, DirectoryPath));
                *pOutCount += 1;
            }
            else
            {
                *pOutCount += count;;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteDirectoryMeta(
        std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT
{
    if (path.back() != '/')
    {
        NN_RESULT_THROW(ResultInvalidName());
    }

    if (pFile)
    {
        Meta meta;

        // データサイズは 0 にしておく
        meta.size = 0;

        // データパスをコピー
        strncpy(meta.path, path.c_str(), sizeof(meta.path));

        // 有効な pFile が渡されたときのみ書き込みを実行
        auto offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), true));
    }

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteFileMetaAndData(
        std::shared_ptr<CachedArchiveFile> pFile, const std::string& path) const NN_NOEXCEPT
{
    if (path.back() == '/')
    {
        NN_RESULT_THROW(ResultInvalidName());
    }

    if(pFile == nullptr)
    {
        return nn::repair::ResultUnexpected();
    }

    nn::fs::FileHandle handle;
    nn::Result result;
    result = nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);

        // 高速化の為の仮 delete 技は使えない(ほんとに消える)
    };

    // meta 情報書き込み
    Meta meta;
    int64_t srcSize;

    // データサイズを取得
    if(result.IsSuccess())
    {
        // データサイズを取得
        result = nn::fs::GetFileSize(&srcSize, handle);
    }

    if(result.IsFailure())
    {
        // エラーでも meta をファイルサイズ０で書く（先に書いたヘッダとの整合性維持の為）
        srcSize = 0;

        // go through
    }

    const int BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char []> buffer(new char[BufferSize]);

    // 先に verify read する
    for (int64_t cursor = 0; cursor < srcSize; cursor += BufferSize)
    {
        // サイズ
        int64_t size = ((srcSize - cursor) < BufferSize) ? (srcSize - cursor) : BufferSize;

        result = nn::fs::ReadFile(handle, cursor, buffer.get(), static_cast<size_t>(size));

        if(result.IsFailure())
        {
            // エラーでも meta をファイルサイズ０で書く（先に書いたヘッダとの整合性維持の為）
            srcSize = 0;
            break;
        }
    }

    if(result.IsFailure())
    {
        s_CorruptedFileCount ++;
        SendMessage("\n === Corrupted File : %s (%08x) ===\n", path.c_str(), result.GetInnerValueForDebug());

        NN_REPAIR_RESULT_DO(SendCorruptionLog("album data : %s\n", path.c_str()));
        // go through
    }

    // verify fail の場合は size 0 で書く
    meta.size = srcSize;

    // データパスをコピー
    strncpy(meta.path, path.c_str(), sizeof(meta.path));

    int64_t offset = pFile->GetEndOffset();
    NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), false));

    if(srcSize > BufferSize * 30)
    {
        SendMessage("file size %lld bytes :", srcSize);
    }

    // file data 書き込み
    offset = pFile->GetEndOffset();

    for (int64_t cursor = 0; cursor < srcSize; cursor += BufferSize)
    {
        // 進捗表示
        s_Surveyor.ShowTick(cursor / BufferSize, srcSize / BufferSize);

        // 書き込むサイズ
        int64_t size = ((srcSize - cursor) < BufferSize) ? (srcSize - cursor) : BufferSize;
        NN_REPAIR_RESULT_DO(nn::fs::ReadFile(handle, cursor, buffer.get(), static_cast<size_t>(size)));
        NN_REPAIR_RESULT_DO(pFile->Write(offset + cursor, buffer.get(), static_cast<size_t>(size), false));
    }

    s_Surveyor.Snap(srcSize);

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::GetFileCount(
        int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Header header;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &header, sizeof(Header)));

    *pOutCount = header.count;

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteEntries(
        int64_t* pOutSize,
        std::shared_ptr<CachedArchiveFile> pFile,
        int64_t entriesCount,
        int64_t offset) const NN_NOEXCEPT
{

    s_Surveyor.StartTick();

    NN_REPAIR_RESULT_DO(nn::fs::MountImageDirectory(MountName.c_str(), nn::fs::ImageDirectoryId::Nand));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    // ターゲット内にあるアルバムデータをすべて削除
    // 症状確認時に新規に作成されることがあるため
    NN_REPAIR_RESULT_DO(this->RemoveExistingData());

    int64_t cursor = 0;
    for (int64_t count = 0; count < entriesCount; count++)
    {
        int64_t fileCount;
        NN_REPAIR_RESULT_DO(this->GetFileCount(&fileCount, pFile, offset + cursor));
        s_Surveyor.ShowFileCount(fileCount);

        int64_t entrySize;
        NN_REPAIR_RESULT_DO(this->WriteFiles(&entrySize, pFile, fileCount, offset + cursor + sizeof(Header)));
        cursor += static_cast<int64_t>(sizeof(Header)) + entrySize;
    }

    s_Surveyor.ShowResult();

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteFiles(
        int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t fileCount, int64_t offset) const NN_NOEXCEPT
{
    int64_t cursor = 0;
    for (int64_t count = 0; count < fileCount; count++)
    {
        std::string fullpath;
        int64_t fileSize;
        NN_REPAIR_RESULT_DO(this->GetFilePathAndSize(&fullpath, &fileSize, pFile, offset + cursor));
        std::stringstream ss(fullpath);

        const auto isFilePath = (fullpath.back() != '/');

        std::string buffer;
        std::string path;

        auto IsLastLine = [](const std::stringstream& ss)
        {
            return ss.eof();
        };

        for ( ; ; )
        {
            std::getline(ss, buffer, '/');

            if (IsLastLine(ss) && isFilePath)
            {
                path += buffer;

                NN_REPAIR_RESULT_DO(
                    this->WriteFile(pFile, path, offset + cursor, fileSize));
                cursor += sizeof(Meta) + fileSize;
                break;
            }
            else
            {
                if (IsLastLine(ss))
                {
                    cursor += sizeof(Meta);
                    break;
                }

                path += buffer + "/";

                if (buffer == MountName + ":")
                {
                    continue;
                }

                // パス中のディレクトリがなければ作る
                Directory directory(path);
                if (!directory.IsFound())
                {
                    NN_REPAIR_RESULT_DO(directory.Create());
                }
            }
        }
    }

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::GetFilePathAndSize(
    std::string* pOutPath, int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Meta meta;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &meta, sizeof(Meta)));

    *pOutPath = meta.path;
    *pOutSize = meta.size;

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::WriteFile(
    std::shared_ptr<CachedArchiveFile> pFile, const std::string& path, int64_t offset, int64_t fileSize) const NN_NOEXCEPT
{
    // 既にあるファイルを削除
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        NN_REPAIR_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
    }

    const int64_t BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char []> buffer(new char[BufferSize]);

    if(fileSize > 0)
    {
        // ファイル作成
        NN_REPAIR_RESULT_DO(nn::fs::CreateFile(path.c_str(), fileSize));
    }

    for (int64_t cursor = 0; cursor < fileSize; cursor += BufferSize)
    {
        // 進捗表示
        s_Surveyor.ShowTick(cursor / BufferSize, fileSize / BufferSize);

        // 最大 BufferSize 分読む
        int64_t size = ((fileSize - cursor) < BufferSize) ? (fileSize - cursor) : BufferSize;

        // アーカイブファイルからデータを読み込む
        NN_REPAIR_RESULT_DO(pFile->Read(offset + sizeof(Meta) + cursor, buffer.get(), static_cast<size_t>(size)));

        // 読み込んだデータを本体に書き込む
        nn::fs::FileHandle handle;
        NN_REPAIR_RESULT_DO(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_REPAIR_RESULT_DO(
                nn::fs::WriteFile(handle, cursor, buffer.get(), static_cast<size_t>(size), nn::fs::WriteOption()));
        NN_REPAIR_RESULT_DO(nn::fs::FlushFile(handle));
    }

    s_Surveyor.Snap(fileSize);

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    char str[512];
    va_list args;
    va_start(args, pFormat);
    vsnprintf(str, sizeof(str), pFormat, args);
    va_end(args);

    SendRecordLog(handle, str);

    NN_RESULT_SUCCESS;
}

nn::Result AlbumDataEntry::RemoveExistingData() const NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    NN_REPAIR_RESULT_DO(nn::fs::OpenDirectory(&handle, RootPath.c_str(), nn::fs::OpenDirectoryMode_Directory));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int64_t entriesCount;
    NN_REPAIR_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entriesCount, handle));

    int64_t readCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> entries(new nn::fs::DirectoryEntry[static_cast<int>(entriesCount)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.get(), handle, entriesCount));

    for (int64_t entry = 0; entry < readCount; entry++)
    {
        const std::string& Name = entries[static_cast<size_t>(entry)].name;
        const std::string& Path = RootPath + Name +  "/";
        NN_REPAIR_RESULT_DO(nn::fs::DeleteDirectoryRecursively(Path.c_str()));
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

