﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/ut/g3d_Perf.h>
#include <limits>
#include <numeric>
#include <algorithm>

#if NW_G3D_IS_HOST_WIN
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#elif NW_G3D_IS_HOST_CAFE
#include <cafe/os/OSTime.h>
#endif

namespace nw { namespace g3d { namespace ut {

PerfManager PerfManager::s_Instance;
s64 PerfManager::s_Freq = 0;

#if NW_G3D_IS_HOST_WIN
CRITICAL_SECTION s_CriticalSection;
#elif NW_G3D_IS_HOST_CAFE
OSMutex s_Mutex;
#endif

void PerfManager::Init()
{
    if (s_Freq == 0)
    {
#if NW_G3D_IS_HOST_WIN
        BOOL success = FALSE;
        success = QueryPerformanceFrequency(reinterpret_cast<LARGE_INTEGER*>(&s_Freq));
        NW_G3D_ASSERT(success);
        InitializeCriticalSection(&s_CriticalSection);
#elif NW_G3D_IS_HOST_CAFE
        s_Freq = OS_TIMER_CLOCK;
        OSInitMutex(&s_Mutex);
#endif
    }

    PerfNodeCPU node;
    for (int i = 0; i < 100; ++i)
    {
        {
            PerfNodeCPU::StopWatch stopwatch(node);
        }
        node.NextFrame();
    }
    DebugPrint("[NW G3D PERF]\n");
    node.Print();

    s_Instance.Reset();
}

void PerfManager::Add(PerfNodeCPU* pNode)
{
    NW_G3D_ASSERT(s_Freq != 0); // Init が呼ばれていなければならない。

#if NW_G3D_IS_HOST_WIN
    EnterCriticalSection(&s_CriticalSection);
#elif NW_G3D_IS_HOST_CAFE
    OSLockMutex(&s_Mutex);
#endif
    if (m_pLastNodeCPU)
    {
        m_pLastNodeCPU->m_pNext = pNode;
        m_pLastNodeCPU = pNode;
    }
    else
    {
        m_pFirstNodeCPU = m_pLastNodeCPU = pNode;
    }
#if NW_G3D_IS_HOST_WIN
    LeaveCriticalSection(&s_CriticalSection);
#elif NW_G3D_IS_HOST_CAFE
    OSUnlockMutex(&s_Mutex);
#endif
}

void PerfManager::Reset()
{
    m_InitialFrame = true;
}

void PerfManager::NextFrame()
{
    PerfNodeCPU* pNode = m_pFirstNodeCPU;
    if (m_InitialFrame)
    {
        m_InitialFrame = false;
        while (pNode)
        {
            pNode->Reset();
            pNode = pNode->m_pNext;
        }
    }
    else
    {
        while (pNode)
        {
            pNode->NextFrame();
            pNode = pNode->m_pNext;
        }
    }
}

void PerfManager::Print()
{
    DebugPrint("----------------------------------------------------------------\n");
    DebugPrint("[NW G3D PERF]\n\n");
    PerfNodeCPU* pNode = m_pFirstNodeCPU;
    while (pNode)
    {
        pNode->Print();
        pNode = pNode->m_pNext;
    }
    DebugPrint("----------------------------------------------------------------\n");
}

void PerfManager::PrintTable()
{
    DebugPrint("----------------------------------------------------------------\n");
    DebugPrint("[NW G3D PERF]\n");
    DebugPrint("|total (%%)|min (%%)|max (%%)|single call (%%)|count|name|\n");
    PerfNodeCPU* pNode = m_pFirstNodeCPU;
    while (pNode)
    {
        pNode->PrintTable();
        pNode = pNode->m_pNext;
    }
    DebugPrint("----------------------------------------------------------------\n");
}

s64 PerfManager::GetTicks()
{
#if NW_G3D_IS_HOST_WIN
    s64 ticks = 0;
    QueryPerformanceCounter(reinterpret_cast<LARGE_INTEGER*>(&ticks));
    return ticks;
#elif NW_G3D_IS_HOST_CAFE
    return OSGetSystemTime();
#else
    return 0;
#endif
}

//--------------------------------------------------------------------------------------------------

void PerfNodeCPU::Reset()
{
    std::fill_n(m_CurElapsed, sizeof(m_CurElapsed) / sizeof(*m_CurElapsed), 0);
    m_MinElapsed = std::numeric_limits<s64>::max();
    m_MaxElapsed = 0;
    m_TotalElapsed = 0;

    std::fill_n(m_CurCount, sizeof(m_CurCount) / sizeof(*m_CurCount), 0);
    m_MinCount = std::numeric_limits<int>::max();
    m_MaxCount = 0;
    m_TotalCount = 0;
    m_TotalFrame = 0;
}

void PerfNodeCPU::NextFrame()
{
    ++m_TotalFrame;

    s64 curElapsed = std::accumulate(m_CurElapsed,
        m_CurElapsed + sizeof(m_CurElapsed) / sizeof(*m_CurElapsed), 0ll);
    if (curElapsed < m_MinElapsed && curElapsed > 0)
    {
        m_MinElapsed = curElapsed;
    }
    if (curElapsed > m_MaxElapsed)
    {
        m_MaxElapsed = curElapsed;
    }
    m_TotalElapsed += curElapsed;
    std::fill_n(m_CurElapsed, sizeof(m_CurElapsed) / sizeof(*m_CurElapsed), 0ll);

    int curCount = std::accumulate(m_CurCount,
        m_CurCount + sizeof(m_CurCount) / sizeof(*m_CurCount), 0);
    if (curCount < m_MinCount)
    {
        m_MinCount = curCount;
    }
    if (curCount > m_MaxCount)
    {
        m_MaxCount = curCount;
    }
    m_TotalCount += curCount;
    std::fill_n(m_CurCount, sizeof(m_CurCount) / sizeof(*m_CurCount), 0);
}

void PerfNodeCPU::Print()
{
    DebugPrint("%s\n", m_Name);

    DebugPrint("  Total Frame: %d\n", m_TotalFrame);

    DebugPrint("  Avg Count: %g\n",
        static_cast<float>(m_TotalCount) / m_TotalFrame);
    DebugPrint("  Min Count: %d\n", m_MinCount);
    DebugPrint("  Max Count: %d\n", m_MaxCount);

    DebugPrint("  Avg Time (Single Call): %g us (%g %%)\n",
        PerfManager::ToMicroSecs(m_TotalElapsed) / m_TotalCount,
        PerfManager::ToPercents(m_TotalElapsed) / m_TotalCount);
    DebugPrint("  Avg Time: %g us (%g %%)\n",
        PerfManager::ToMicroSecs(m_TotalElapsed) / m_TotalFrame,
        PerfManager::ToPercents(m_TotalElapsed) / m_TotalFrame);
    DebugPrint("  Min Time: %g us (%g %%)\n",
        PerfManager::ToMicroSecs(m_MinElapsed), PerfManager::ToPercents(m_MinElapsed));
    DebugPrint("  Max Time: %g us (%g %%)\n",
        PerfManager::ToMicroSecs(m_MaxElapsed), PerfManager::ToPercents(m_MaxElapsed));

    DebugPrint("\n");
}

void PerfNodeCPU::PrintTable()
{
    // |total (%)|min (%)|max (%)|single call (%)|count|name|
    DebugPrint("|%.3g|%.3g|%.3g|%.3g|%.3g|%s|\n",
        PerfManager::ToPercents(m_TotalElapsed) / m_TotalFrame,
        PerfManager::ToPercents(m_MinElapsed),
        PerfManager::ToPercents(m_MaxElapsed),
        PerfManager::ToPercents(m_TotalElapsed) / m_TotalCount,
        static_cast<float>(m_TotalCount) / m_TotalFrame,
        m_Name);
}

}}} // namespace nw::g3d::ut
