﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResVisibilityAnim.h>
#include <algorithm>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

BindResult ResVisibilityAnim::PreBind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    ref().ofsBindModel.set_ptr(pModel->ptr());
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();
    const BinString* pNameArray = ref().ofsNameArray.to_ptr<BinString>();
    const ResDicType* pDic = GetVisibilityType() == BONE_VISIBILITY ?
        NW_G3D_RES_DIC(pModel->GetSkeleton()->ref().ofsBoneDic) :
        NW_G3D_RES_DIC(pModel->ref().ofsMaterialDic);

    BindResult result;
    for (int idxAnim = 0, numAnim = GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResName* pName = pNameArray[idxAnim].GetResName();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            pBindIndexArray[idxAnim] = static_cast<u16>(idxTarget);
            result |= BindResult::Bound();
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResVisibilityAnim::BindCheck(const ResModel* pModel) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    const BinString* pNameArray = ref().ofsNameArray.to_ptr<BinString>();
    const ResDicType* pDic = GetVisibilityType() == BONE_VISIBILITY ?
        NW_G3D_RES_DIC(pModel->GetSkeleton()->ref().ofsBoneDic) :
        NW_G3D_RES_DIC(pModel->ref().ofsMaterialDic);

    BindResult result;
    for (int idxAnim = 0, numAnim = GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResName* pName = pNameArray[idxAnim].GetResName();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

bool ResVisibilityAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalcBakedIntSize();
        curve->BakeInt(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResVisibilityAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
            if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_BOOL)
            {
                pBuffer = curve->ref().ofsKeyArray.to_ptr();
                foundCurve = true;
            }
            curve->Reset();
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResVisibilityAnim::Reset()
{
    ref().ofsBindModel.set_ptr(NULL);
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();
    for (int idxAnim = 0, numAnim = GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
    }

    ResetCurve();
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
