﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/res/g3d_ResUtility.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/fnd/g3d_GfxShader.h>
#include <nw/g3d/ut/g3d_Flag.h>

#include <stdio.h>

#if NW_G3D_IS_HOST_WIN
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#elif defined(ANDROID) || defined(__APPLE__)
#include <zlib.h>
#endif // NW_G3D_IS_HOST_WIN

namespace nw { namespace g3d { namespace res {

namespace {

#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )

class ZLib
{
#if !defined(ANDROID) && !defined(__APPLE__)
    typedef int (WINAPI *PFNUNCOMPRESS) (unsigned char* dest, unsigned long *destLen, const unsigned char *source, unsigned long sourceLen);
    typedef const char* (WINAPI *PFNZERROR) (int);
#endif

public:
#if !defined(ANDROID) && !defined(__APPLE__)
    ZLib() : hDLL(NULL) {}
#else
    ZLib() {}
#endif
    ~ZLib()
    {
        if (IsValid())
        {
#if !defined(ANDROID) && !defined(__APPLE__)
            BOOL result = FreeLibrary(hDLL);
            NW_G3D_UNUSED(result);
            NW_G3D_WARNING(result == TRUE, "failed to free library : %hs\n", GetLibraryPath());
            hDLL = NULL;
#endif
        }
    }

    void Construct()
    {
#if !defined(ANDROID) && !defined(__APPLE__)
        hDLL = LoadLibraryA(GetLibraryPath());
        NW_G3D_ASSERTMSG(hDLL, "failed to load library : %hs\n", GetLibraryPath());

        uncompress = reinterpret_cast<PFNUNCOMPRESS>(GetProcAddress(hDLL, "uncompress"));
        NW_G3D_ASSERTMSG(uncompress, "failed to load proc : uncompress\n");

        zError = reinterpret_cast<PFNZERROR>(GetProcAddress(hDLL, "zError"));
        NW_G3D_ASSERTMSG(zError, "failed to load proc : zError\n");
#endif
    }
#if !defined(ANDROID) && !defined(__APPLE__)
    bool IsValid() const { return hDLL != NULL; }
#else
    bool IsValid() const { return true; }
#endif

    void Uncompress(void* dst, size_t* written, size_t dstSize, const void* src, size_t srcSize)
    {
        unsigned long destLen = static_cast<unsigned long>(dstSize);
        int error = uncompress(
            static_cast<unsigned char*>(dst),
            &destLen,
            static_cast<const unsigned char*>(src),
            static_cast<unsigned long>(srcSize));
        NW_G3D_UNUSED(error);
        NW_G3D_ASSERTMSG(!error, "failed to uncompress : %hs\n", zError(error));
        if (written)
        {
            *written = static_cast<size_t>(destLen);
        }
    }

    static const char* GetLibraryPath() { return "zlibwapi.dll"; }

#if !defined(ANDROID) && !defined(__APPLE__)
private:
    HINSTANCE hDLL;
    PFNUNCOMPRESS uncompress;
    PFNZERROR zError;
#endif
} s_ZLib;

void UpdateProgramGL(ResShadingModel* pShader, int programIndex)
{
    ResShaderProgramData& program = pShader->GetShaderProgram(programIndex)->ref();
    if (program.handle)
    {
        DestroyShaderProgram(program.handle);
    }

    // シンボルテーブルの取り出し
    ResShadingModel::GLShaderInfo* pGLInfo =
        pShader->ref().ofsGLShaderInfo.to_ptr<ResShadingModel::GLShaderInfo>();
    static const int MAX_STREAMOUT = GX2_MAX_STREAMOUT_BUFFERS;
    NW_G3D_ASSERTMSG(pGLInfo->numStreamout <= MAX_STREAMOUT, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));
    const char* varyings[MAX_STREAMOUT];
    const void* pVarying = pGLInfo->ofsStreamout.to_ptr();
    for (int idxVarying = 0; idxVarying < static_cast<int>(pGLInfo->numStreamout); ++idxVarying)
    {
        const ResNameData* pName = static_cast<const ResNameData*>(pVarying);
        varyings[idxVarying] = pName->str;
        pVarying = AddOffset(pVarying, ( pName->len & ~(sizeof(ResName::LengthType) - 1) )
            + sizeof(ResNameData));
    }

    // コンパイル
    GfxShader shader;
    ResShaderArchive* pArchive = pShader->ref().ofsShaderArchive.to_ptr<ResShaderArchive>();
    ResShaderProgram::GLShaderProgram* pVS =
        program.ofsVertexShader.to_ptr<ResShaderProgram::GLShaderProgram>();
    ResShaderProgram::GLShaderProgram* pGS =
        program.ofsGeometryShader.to_ptr<ResShaderProgram::GLShaderProgram>();
    ResShaderProgram::GLShaderProgram* pFS =
        program.ofsFragmentShader.to_ptr<ResShaderProgram::GLShaderProgram>();
    ResShaderProgram::GLShaderProgram* pCS =
        program.ofsComputeShader.to_ptr<ResShaderProgram::GLShaderProgram>();
    NW_G3D_ASSERTMSG((pVS && !pCS) || (!pVS && pCS), "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));

    class Buffer
    {
    public:
        Buffer() : ptr(NULL) {}
        Buffer(size_t size) { Alloc(size); }
        ~Buffer() { Free(); }
        bool Alloc(size_t size)
        {
            Free();
            return ( ptr = malloc(size) ) != NULL;
        }
        void Free()
        {
            if (ptr)
            {
                free(ptr);
                ptr = NULL;
            }
        }
        void* ptr;
    };

    GfxShader::SetupArg arg;
    Buffer binary;
    Buffer vsSource;
    Buffer gsSource;
    Buffer fsSource;
    Buffer csSource;
    const char*sources[NUM_STAGE][2];
    if (pArchive->IsGLBinaryAvailable())
    {
        ResShaderProgram::GLShaderProgram* pGLProgram = pVS ? pVS : pCS;
        arg.pBinary = pGLProgram->ofsShaderCode.to_ptr();
        arg.binarySize = pGLProgram->shaderCodeSize;
        arg.binaryFormat = pGLProgram->binaryFormat;
        if (pArchive->IsShaderCompressed())
        {
            binary.Alloc(pGLProgram->shaderCodeSize);
            s_ZLib.Uncompress(binary.ptr, NULL, pGLProgram->shaderCodeSize,
                pGLProgram->ofsShaderCode.to_ptr(), pGLProgram->compressedShaderCodeSize);
            arg.pBinary = binary.ptr;
        }
    }
    else
    {
        arg.useMultiSource = true;
        if(pVS)
        {
            arg.ppVertexShader = sources[STAGE_VERTEX];
            arg.ppVertexShader[0] = pVS->ofsShaderSource.to_ptr<const char>();
            arg.ppVertexShader[1] = pGLInfo->sharedSources[STAGE_VERTEX].pSharedSource.to_ptr<const char>();
            arg.numVertexShader = arg.ppVertexShader[1] ? 2 : 1;
        }
        if(pGS)
        {
            arg.ppGeometryShader = sources[STAGE_GEOMETRY];
            arg.ppGeometryShader[0] = pGS->ofsShaderSource.to_ptr<const char>();
            arg.ppGeometryShader[1] = pGLInfo->sharedSources[STAGE_GEOMETRY].pSharedSource.to_ptr<const char>();
            arg.numGeometryShader = arg.ppGeometryShader[1] ? 2 : 1;
        }
        if(pFS)
        {
            arg.ppFragmentShader = sources[STAGE_FRAGMENT];
            arg.ppFragmentShader[0] = pFS->ofsShaderSource.to_ptr<const char>();
            arg.ppFragmentShader[1] = pGLInfo->sharedSources[STAGE_FRAGMENT].pSharedSource.to_ptr<const char>();
            arg.numFragmentShader = arg.ppFragmentShader[1] ? 2 : 1;
        }
        if(pCS)
        {
            arg.ppComputeShader = sources[STAGE_COMPUTE];
            arg.ppComputeShader[0] = pCS->ofsShaderSource.to_ptr<const char>();
            arg.ppComputeShader[1] = pGLInfo->sharedSources[STAGE_COMPUTE].pSharedSource.to_ptr<const char>();
            arg.numComputeShader = arg.ppComputeShader[1] ? 2 : 1;
        }
        arg.numVaryings = pGLInfo->numStreamout;
        arg.ppTransformFeedbackVaryings = varyings;

        if (pArchive->IsShaderCompressed())
        {
            if (pVS)
            {
                vsSource.Alloc(pVS->shaderSourceSize);
                s_ZLib.Uncompress(vsSource.ptr, NULL, pVS->shaderSourceSize,
                    pVS->ofsShaderSource.to_ptr(), pVS->compressedShaderSourceSize);
                arg.ppVertexShader[0] = static_cast<char*>(vsSource.ptr);
            }
            if (pGS)
            {
                gsSource.Alloc(pGS->shaderSourceSize);
                s_ZLib.Uncompress(gsSource.ptr, NULL, pGS->shaderSourceSize,
                    pGS->ofsShaderSource.to_ptr(), pGS->compressedShaderSourceSize);
                arg.ppGeometryShader[0] = static_cast<char*>(gsSource.ptr);
            }
            if (pFS)
            {
                fsSource.Alloc(pFS->shaderSourceSize);
                s_ZLib.Uncompress(fsSource.ptr, NULL, pFS->shaderSourceSize,
                    pFS->ofsShaderSource.to_ptr(), pFS->compressedShaderSourceSize);
                arg.ppFragmentShader[0] = static_cast<char*>(fsSource.ptr);
            }
            if (pCS)
            {
                csSource.Alloc(pCS->shaderSourceSize);
                s_ZLib.Uncompress(csSource.ptr, NULL, pCS->shaderSourceSize,
                    pCS->ofsShaderSource.to_ptr(), pCS->compressedShaderSourceSize);
                arg.ppComputeShader[0] = static_cast<char*>(csSource.ptr);
            }
        }
    }
    shader.Setup(arg);
    program.handle = shader.handle;

    // サンプラーロケーションテーブルの書き換え
    s8* pSamplerLocation = program.ofsSamplerTable.to_ptr<s8>();
    const void* pSamplerSymbol = pGLInfo->ofsSamplerTable.to_ptr();
    for (int idxSampler = 0, numSampler = pShader->GetSamplerCount();
        idxSampler < numSampler; ++idxSampler)
    {
        for (int stage = 0; stage < ResShaderProgram::NUM_STAGE; ++stage)
        {
            s8 location = -1;
            const ResNameData* pName = static_cast<const ResNameData*>(pSamplerSymbol);
            if (pName->len > 0)
            {
                location = shader.GetVertexSamplerLocation(pName->str);
                pSamplerSymbol = AddOffset(pSamplerSymbol, pName->len & ~(sizeof(ResNameData::LengthType) - 1));
            }
            pSamplerSymbol = AddOffset(pSamplerSymbol, sizeof(ResNameData));

            pSamplerLocation[idxSampler * ResShaderProgram::NUM_STAGE + stage] = location;
        }
    }

    // ブロックロケーションテーブルの書き換え
    s8* pBlockLocation = program.ofsUniformBlockTable.to_ptr<s8>();
    const void* pBlockSymbol = pGLInfo->ofsBlockTable.to_ptr();
    for (int idxBlock = 0, numBlock = pShader->GetUniformBlockCount();
        idxBlock < numBlock; ++idxBlock)
    {
        for (int stage = 0; stage < ResShaderProgram::NUM_STAGE; ++stage)
        {
            s8 location = -1;
            const ResNameData* pName = reinterpret_cast<const ResNameData*>(pBlockSymbol);
            if (pName->len > 0)
            {
                location = shader.GetVertexBlockLocation(pName->str);
                pBlockSymbol = AddOffset(pBlockSymbol, pName->len & ~(sizeof(ResNameData::LengthType) - 1));
            }
            pBlockSymbol = AddOffset(pBlockSymbol, sizeof(ResNameData));

            pBlockLocation[idxBlock * ResShaderProgram::NUM_STAGE + stage] = location;
        }
    }

    program.flag &= ~ResShaderProgram::UPDATE_REQURIED;
}

#endif // NW_G3D_IS_GL && !defined( NW_STRIP_GL )

#if !NW_G3D_IS_RELEASE // 未使用関数に対する警告対策

inline
const char* GetArchiveName(const void* ptr)
{
    const ResShaderArchiveData* pData = static_cast<const ResShaderArchiveData*>(ptr);
    if (pData->fileHeader.byteOrder == BinaryFileHeader::BYTE_ORDER_MARK)
    {
        return pData->ofsName.to_ptr();
    }
    else
    {
        const u32* ofsName = reinterpret_cast<const u32*>(&pData->ofsName);
        return reinterpret_cast<const char*>(ofsName) + LoadRevU32(ofsName);
    }
}

#endif // !NW_G3D_IS_RELEASE

#if NW_G3D_IS_GX2 && NW_G3D_OFFLINE_DL_ENABLE
const int s_ShaderPtrOffsets[] =
{
    0,
    reinterpret_cast<int>(&reinterpret_cast<GX2FetchShader*>(NULL)->shaderPtr),
    reinterpret_cast<int>(&reinterpret_cast<GX2VertexShader*>(NULL)->shaderPtr),
    reinterpret_cast<int>(&reinterpret_cast<GX2GeometryShader*>(NULL)->copyShaderPtr),
    reinterpret_cast<int>(&reinterpret_cast<GX2GeometryShader*>(NULL)->shaderPtr),
    reinterpret_cast<int>(&reinterpret_cast<GX2PixelShader*>(NULL)->shaderPtr),
#if NW_G3D_COMPUTE_SHADER_ENABLE
    reinterpret_cast<int>(&reinterpret_cast<GX2ComputeShader*>(NULL)->shaderPtr)
#else
    reinterpret_cast<int>(&reinterpret_cast<fnd::detail::GX2ComputeShaderData*>(NULL)->shaderPtr)
#endif
};
#endif

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

class ResOfflineDLShader : private ResOfflineDLShaderData
{
    NW_G3D_RES_COMMON(ResOfflineDLShader);

public:
    void Setup()
    {
#if NW_G3D_IS_GX2
#if NW_G3D_OFFLINE_DL_ENABLE
        void* shaderPtr = ref().ofsShader.to_ptr();
        NW_G3D_ASSERT_INDEX_BOUNDS(ref().gx2PatchType,
            sizeof(s_ShaderPtrOffsets) / sizeof(*s_ShaderPtrOffsets));
        // GX2 シェーダ構造体として渡す
        GX2PatchDisplayList(ref().ofsOfflineDL.to_ptr(),
            static_cast<GX2PatchType>(ref().gx2PatchType), ref().byteOffset,
            AddOffset(&shaderPtr, -s_ShaderPtrOffsets[ref().gx2PatchType]));
#else
        NW_G3D_ASSERTMSG(false,
            "This shader was converted as offline display list that is the functionality of cafe_sdk 2.12.1.\n"
            "Please pass the disable offline display list option to NW4F_g3dshdrcvtr when converting.");
#endif
        DCFlushRange(ref().ofsOfflineDL.to_ptr(), ref().offlineDLSize);
        DCFlushRange(ref().ofsShader.to_ptr(), ref().shaderSize);
#endif
    }

    void CopyDL() const
    {
#if NW_G3D_IS_GX2
        GX2CopyDisplayList(ref().ofsOfflineDL.to_ptr(), ref().offlineDLSize);
#endif
    }

    void CallDL() const
    {
#if NW_G3D_IS_GX2
        GX2CallDisplayList(ref().ofsOfflineDL.to_ptr(), ref().offlineDLSize);
#endif
    }
};

//--------------------------------------------------------------------------------------------------

void ResShaderProgram::Setup()
{
#if NW_G3D_IS_GX2
    if (!CheckFlag(flag, INITIALIZED))
    {
        if (flag & OFFLINE_DL)
        {
            if (flag & OWN_VERTEX_SHADER)
            {
                ResOfflineDLShader* pShader = ref().ofsVertexShader.to_ptr<ResOfflineDLShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                pShader->Setup();
            }

            if (flag & OWN_GEOMETRY_SHADER)
            {
                ResOfflineDLShader* pShader = ref().ofsGeometryShader.to_ptr<ResOfflineDLShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                pShader->Setup();
                ResOfflineDLShader* pCopyShader = ref().ofsGeometryCopyShader.to_ptr<ResOfflineDLShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pCopyShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                pCopyShader->Setup();
            }

            if (flag & OWN_FRAGMENT_SHADER)
            {
                ResOfflineDLShader* pShader = ref().ofsFragmentShader.to_ptr<ResOfflineDLShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                pShader->Setup();
            }

            if (flag & OWN_COMPUTE_SHADER)
            {
                ResOfflineDLShader* pShader = ref().ofsComputeShader.to_ptr<ResOfflineDLShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                pShader->Setup();
            }
        }
        else
        {
            if (flag & OWN_VERTEX_SHADER)
            {
                GX2VertexShader* pShader = ref().ofsVertexShader.to_ptr<GX2VertexShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                pShader->shaderPtr = ofsShaderPtr.to_ptr();
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                pShader->_loopVars = ofsLoopVars.to_ptr();
                DCFlushRange(pShader->shaderPtr, pShader->shaderSize);
            }

            if (flag & OWN_GEOMETRY_SHADER)
            {
                GX2GeometryShader* pShader = ref().ofsGeometryShader.to_ptr<GX2GeometryShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                pShader->shaderPtr = ofsShaderPtr.to_ptr();
                Offset& ofsCopyShaderPtr = reinterpret_cast<Offset&>(pShader->copyShaderPtr);
                pShader->copyShaderPtr = ofsCopyShaderPtr.to_ptr();
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                pShader->_loopVars = ofsLoopVars.to_ptr();
                DCFlushRange(pShader->shaderPtr, pShader->shaderSize);
                DCFlushRange(pShader->copyShaderPtr, pShader->copyShaderSize);
            }

            if (flag & OWN_FRAGMENT_SHADER)
            {
                GX2PixelShader* pShader = ref().ofsFragmentShader.to_ptr<GX2PixelShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                pShader->shaderPtr = ofsShaderPtr.to_ptr();
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                pShader->_loopVars = ofsLoopVars.to_ptr();
                DCFlushRange(pShader->shaderPtr, pShader->shaderSize);
            }
#if NW_G3D_COMPUTE_SHADER_ENABLE
            if (flag & OWN_COMPUTE_SHADER)
            {
                GX2ComputeShader* pShader = ref().ofsComputeShader.to_ptr<GX2ComputeShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                pShader->shaderPtr = ofsShaderPtr.to_ptr();
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                pShader->_loopVars = ofsLoopVars.to_ptr();
                DCFlushRange(pShader->shaderPtr, pShader->shaderSize);
            }
#endif
        }
        flag |= INITIALIZED;
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (ref().handle == 0)
    {
        // 遅延コンパイルのために更新フラグを立てる。
        ref().flag |= UPDATE_REQURIED;
    }
#endif
}

void ResShaderProgram::Cleanup()
{
#if NW_G3D_IS_GX2
    if (CheckFlag(flag, INITIALIZED))
    {
        if (!(flag & OFFLINE_DL))
        {
            if (flag & OWN_VERTEX_SHADER)
            {
                GX2VertexShader* pShader = ref().ofsVertexShader.to_ptr<GX2VertexShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                ofsShaderPtr.set_ptr(pShader->shaderPtr);
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                ofsLoopVars.set_ptr(pShader->_loopVars);
            }

            if (flag & OWN_GEOMETRY_SHADER)
            {
                GX2GeometryShader* pShader = ref().ofsGeometryShader.to_ptr<GX2GeometryShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                ofsShaderPtr.set_ptr(pShader->shaderPtr);
                Offset& ofsCopyShaderPtr = reinterpret_cast<Offset&>(pShader->copyShaderPtr);
                ofsCopyShaderPtr.set_ptr(pShader->copyShaderPtr);
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                ofsLoopVars.set_ptr(pShader->_loopVars);
            }

            if (flag & OWN_FRAGMENT_SHADER)
            {
                GX2PixelShader* pShader = ref().ofsFragmentShader.to_ptr<GX2PixelShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                ofsShaderPtr.set_ptr(pShader->shaderPtr);
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                ofsLoopVars.set_ptr(pShader->_loopVars);
            }
#if NW_G3D_COMPUTE_SHADER_ENABLE
            if (flag & OWN_COMPUTE_SHADER)
            {
                GX2ComputeShader* pShader = ref().ofsComputeShader.to_ptr<GX2ComputeShader>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pShader, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
                Offset& ofsShaderPtr = reinterpret_cast<Offset&>(pShader->shaderPtr);
                ofsShaderPtr.set_ptr(pShader->shaderPtr);
                Offset& ofsLoopVars = reinterpret_cast<Offset&>(pShader->_loopVars);
                ofsLoopVars.set_ptr(pShader->_loopVars);
            }
#endif
        }
        flag &= ~INITIALIZED;
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    u32 hProgram = ref().handle;
    if (hProgram)
    {
        DestroyShaderProgram(hProgram);
        ref().handle = 0;
        ref().flag &= ~UPDATE_REQURIED;
    }
#endif
}

void ResShaderProgram::Update()
{
#if ( NW_G3D_IS_GL && !defined( NW_STRIP_GL ) ) || !NW_G3D_IS_RELEASE
    // 更新があったプログラムを遅延コンパイルする。
    if (ref().flag & UPDATE_REQURIED)
    {
        ResShadingModel* pShadingModel = ref().ofsShadingModel.to_ptr<ResShadingModel>();
        int idxProgram = static_cast<int>(std::distance(
            pShadingModel->ref().ofsShaderProgramArray.to_ptr<ResShaderProgramData>(), ptr()));
        pShadingModel->UpdateProgram(idxProgram);
    }
#endif
}

void ResShaderProgram::Load() const
{
#if NW_G3D_IS_GX2
    if (flag & OFFLINE_DL)
    {
        if (const ResOfflineDLShader* pShader = ref().ofsVertexShader.to_ptr<ResOfflineDLShader>())
        {
            pShader->CopyDL();
        }

        if (const ResOfflineDLShader* pShader = ref().ofsGeometryShader.to_ptr<ResOfflineDLShader>())
        {
            pShader->CopyDL();
        }

        if (const ResOfflineDLShader* pShader = ref().ofsFragmentShader.to_ptr<ResOfflineDLShader>())
        {
            pShader->CopyDL();
        }

        if (const ResOfflineDLShader* pShader = ref().ofsComputeShader.to_ptr<ResOfflineDLShader>())
        {
            pShader->CopyDL();
        }
    }
    else
    {
        if (const GX2VertexShader* pShader = ref().ofsVertexShader.to_ptr<GX2VertexShader>())
        {
            GX2SetVertexShader(pShader);
        }

        if (const GX2GeometryShader* pShader = ref().ofsGeometryShader.to_ptr<GX2GeometryShader>())
        {
            GX2SetGeometryShader(pShader);
        }

        if (const GX2PixelShader* pShader = ref().ofsFragmentShader.to_ptr<GX2PixelShader>())
        {
            GX2SetPixelShader(pShader);
        }
#if NW_G3D_COMPUTE_SHADER_ENABLE
        if (const GX2ComputeShader* pShader = ref().ofsComputeShader.to_ptr<GX2ComputeShader>())
        {
            GX2SetComputeShader(pShader);
        }
#endif
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERTMSG(ref().handle, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
    glUseProgram(ref().handle);
    NW_G3D_GL_ASSERT_DETAIL("%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
#endif
}

void ResShaderProgram::LoadDL() const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERTMSG(ref().flag & OFFLINE_DL, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
    if (const ResOfflineDLShader* pShader = ref().ofsVertexShader.to_ptr<ResOfflineDLShader>())
    {
        pShader->CallDL();
    }
    if (const ResOfflineDLShader* pShader = ref().ofsGeometryShader.to_ptr<ResOfflineDLShader>())
    {
        pShader->CallDL();
    }
    if (const ResOfflineDLShader* pShader = ref().ofsFragmentShader.to_ptr<ResOfflineDLShader>())
    {
        pShader->CallDL();
    }
    if (const ResOfflineDLShader* pShader = ref().ofsComputeShader.to_ptr<ResOfflineDLShader>())
    {
        pShader->CallDL();
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERTMSG(ref().handle, "%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
    glUseProgram(ref().handle);
    NW_G3D_GL_ASSERT_DETAIL("%s\n", NW_G3D_RES_GET_NAME(this->ref().ofsShadingModel.to_ptr<ResShadingModel>(), GetName()));
#endif
}

GX2ShaderMode ResShaderProgram::GetShaderMode() const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (ofsComputeShader.to_ptr())
    {
        return GX2_SHADER_MODE_COMPUTE_SHADER;
    }
    else
#endif
    {
        if (flag & OFFLINE_DL)
        {
            const ResOfflineDLShaderData* pShader = ref().ofsVertexShader.to_ptr<ResOfflineDLShaderData>();
            return static_cast<GX2ShaderMode>(pShader->gx2ShaderMode);
        }
        else
        {
            const nw::g3d::fnd::detail::GX2VertexShaderData* pShader =
                ref().ofsVertexShader.to_ptr<nw::g3d::fnd::detail::GX2VertexShaderData>();
            return pShader->shaderMode;
        }
    }
#elif NW_G3D_IS_GL && !defined(NW_STRIP_GL)
    return GX2_SHADER_MODE_UNIFORM_REGISTER;
#else
    return GX2_SHADER_MODE_UNIFORM_REGISTER;
#endif
}

//--------------------------------------------------------------------------------------------------

void ResShaderOption::WriteStaticKey(bit32* pStaticKey, int choiceIndex) const
{
    NW_G3D_ASSERTMSG(IsStatic(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pStaticKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(choiceIndex, GetChoiceCount(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    bit32& key32 = pStaticKey[ref().bit32Index];
    key32 = (key32 & ~ref().bit32Mask) | static_cast<bit32>(choiceIndex << ref().bit32Shift);
}

int ResShaderOption::ReadStaticKey(const bit32* pStaticKey) const
{
    NW_G3D_ASSERTMSG(IsStatic(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pStaticKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    bit32 key32 = pStaticKey[ref().bit32Index];
    return static_cast<int>((key32 & ref().bit32Mask) >> ref().bit32Shift);
}

void ResShaderOption::WriteDynamicKey(bit32* pDynamicKey, int choiceIndex) const
{
    NW_G3D_ASSERTMSG(IsDynamic(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pDynamicKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(choiceIndex, GetChoiceCount(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    bit32& key32 = pDynamicKey[ref().bit32Index - ref().keyOffset];
    key32 = (key32 & ~ref().bit32Mask) | static_cast<bit32>(choiceIndex << ref().bit32Shift);
}

int ResShaderOption::ReadDynamicKey(const bit32* pDynamicKey) const
{
    NW_G3D_ASSERTMSG(IsDynamic(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pDynamicKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    bit32 key32 = pDynamicKey[ref().bit32Index - ref().keyOffset];
    return static_cast<int>((key32 & ref().bit32Mask) >> ref().bit32Shift);
}

//--------------------------------------------------------------------------------------------------

void ResShadingModel::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    ResShaderArchive* pResShaderArchive = ref().ofsShaderArchive.to_ptr<ResShaderArchive>();
    if((pResShaderArchive->ref().flag & ResShaderArchive::GL_SOURCE)
        && !pResShaderArchive->IsGLBinaryAvailable())
    {
        GLShaderInfo* pGLInfo = ofsGLShaderInfo.to_ptr<GLShaderInfo>();
        for(int stage = 0; stage < NUM_STAGE; ++stage)
        {
            GLShaderInfo::SharedSource* pSharedSource = pGLInfo->sharedSources + stage;
            if(pSharedSource->sharedSourceSize > 0)
            {
                if(pSharedSource->pSharedSource.to_ptr() == NULL)
                {
                    pSharedSource->pSharedSource.set_ptr(malloc(pSharedSource->sharedSourceSize));
                    s_ZLib.Uncompress(pSharedSource->pSharedSource.to_ptr(), NULL, pSharedSource->sharedSourceSize,
                        pSharedSource->ofsCompressedSharedSource.to_ptr(), pSharedSource->compressedSharedSourceSize);
                }
            }
        }
    }
#endif

    for (int idxProgram = 0, numProgram = GetShaderProgramCount();
       idxProgram < numProgram; ++idxProgram)
    {
        ResShaderProgram* pProgram = GetShaderProgram(idxProgram);
        pProgram->Setup();
    }
}

void ResShadingModel::Cleanup()
{
    for (int idxProgram = 0, numProgram = GetShaderProgramCount();
        idxProgram < numProgram; ++idxProgram)
    {
        ResShaderProgram* pProgram = GetShaderProgram(idxProgram);
        pProgram->Cleanup();
    }

#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    ResShaderArchive* pResShaderArchive = ref().ofsShaderArchive.to_ptr<ResShaderArchive>();
    if(pResShaderArchive->ref().flag & ResShaderArchive::GL_SOURCE)
    {
        GLShaderInfo* pGLInfo = ofsGLShaderInfo.to_ptr<GLShaderInfo>();
        for(int stage = 0; stage < NUM_STAGE; ++stage)
        {
            GLShaderInfo::SharedSource* pSharedSource = pGLInfo->sharedSources + stage;
            if(void* ptr = pSharedSource->pSharedSource.to_ptr())
            {
                free(ptr);
                pSharedSource->pSharedSource.set_ptr(NULL);
            }
        }
    }
#endif
}

// ShaderKey 二分探索用のイテレータと比較関数
namespace {
class KeyIter : public std::iterator<std::random_access_iterator_tag, const bit32*>
{
public:
#if defined(ANDROID)
    KeyIter() : pKey(NULL), length(0), stride(0) {}
#endif
    KeyIter(const bit32* pKey, size_t length, size_t stride)
        : pKey(pKey), length(length), stride(stride)
    {
    }

    reference operator*() { return pKey; }
    KeyIter& operator++() { pKey += stride; return *this; }
    KeyIter& operator+=(difference_type diff) { pKey += stride * diff; return *this; }
    KeyIter operator+(difference_type diff) const { return KeyIter(*this) += diff; }
    difference_type operator-(const KeyIter& rhs) const { return (pKey - rhs.pKey) / stride; }

    bool operator==(const KeyIter& rhs) const { return pKey == rhs.pKey; }
#if defined( _MSC_VER ) // VC 実装では使われる。
    bool operator!=(const KeyIter& rhs) const { return pKey != rhs.pKey; }
    bool operator<(const KeyIter& rhs) const { return pKey < rhs.pKey; }
#endif

#if 0 // std::lower_bound() では使われない演算子。
    KeyIter& operator--() { pKey += stride; return *this; }
    KeyIter operator++(int) { KeyIter tmp(*this); ++(*this); return tmp; }
    KeyIter operator--(int) { KeyIter tmp(*this); --(*this); return tmp; }
    KeyIter& operator-=(difference_type diff) { pKey -= stride * diff; return *this; }
    KeyIter operator-(difference_type diff) const { return KeyIter(*this) -= diff; }
    bool operator>(const KeyIter& rhs) const { return pKey > rhs.pKey; }
    bool operator<=(const KeyIter& rhs) const { return pKey <= rhs.pKey; }
    bool operator>=(const KeyIter& rhs) const { return pKey >= rhs.pKey; }
#endif

private:
    const bit32* pKey;
    size_t length;
    size_t stride;
};

struct KeyCmp
{
    KeyCmp(size_t length) : length(length) {}

    bool operator()(const bit32* lhs, const bit32* rhs) const
    {
        for (size_t word = 0; word < length; ++word, ++lhs, ++rhs)
        {
            if (*lhs != *rhs)
            {
                return *lhs < *rhs;
            }
        }
        return false;
    }

    size_t length;
};

} // anonymous namespace

void ResShadingModel::WriteDefaultStaticKey(bit32* pStaticKey) const
{
    int idxDefaultProgram = GetDefaultProgramIndex();
    if (idxDefaultProgram == SHADER_PROGRAM_NONE)
    {
        // 全てのオプションがデフォルトのシェーダが存在しない場合はオプションからキーを作ります。
        std::fill_n(pStaticKey, GetStaticKeyLength(), 0);
        for (int idxOption = 0, numOption = GetStaticOptionCount();
            idxOption < numOption; ++idxOption)
        {
            const ResShaderOption* pOption = GetStaticOption(idxOption);
            pOption->WriteStaticKey(pStaticKey, pOption->GetDefaultIndex());
        }
    }
    else
    {
        // 全てのオプションがデフォルトのシェーダが存在する場合はキーをコピーします。
        const bit32* pDefaultKey = GetStaticKey(idxDefaultProgram);
        std::copy(pDefaultKey, pDefaultKey + GetStaticKeyLength(), pStaticKey);
    }
}

void ResShadingModel::WriteDefaultDynamicKey(bit32* pDynamicKey) const
{
    int idxDefaultProgram = GetDefaultProgramIndex();
    if (idxDefaultProgram == SHADER_PROGRAM_NONE)
    {
        // 全てのオプションがデフォルトのシェーダが存在しない場合はオプションからキーを作ります。
        std::fill_n(pDynamicKey, GetDynamicKeyLength(), 0);
        for (int idxOption = 0, numOption = GetDynamicOptionCount();
            idxOption < numOption; ++idxOption)
        {
            const ResShaderOption* pOption = GetDynamicOption(idxOption);
            pOption->WriteDynamicKey(pDynamicKey, pOption->GetDefaultIndex());
        }
    }
    else
    {
        // 全てのオプションがデフォルトのシェーダが存在する場合はキーをコピーします。
        const bit32* pDefaultKey = GetDynamicKey(idxDefaultProgram);
        std::copy(pDefaultKey, pDefaultKey + GetDynamicKeyLength(), pDynamicKey);
    }
}

void ResShadingModel::WriteInvalidDynamicKey(bit32* pDynamicKey) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pDynamicKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    // システムで予約されているビットがあるため全て1のキーは存在しません。
    std::fill_n(pDynamicKey, GetDynamicKeyLength(), bit32(~0));
}

int ResShadingModel::FindProgramIndex(const bit32* pKey) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pKey, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    int keyLength = GetKeyLength();
    KeyIter iterBegin(ref().ofsKeyTable.to_ptr<bit32>(), keyLength, keyLength);
    KeyIter iterEnd(iterBegin + GetShaderProgramCount());
    KeyIter iterFound = std::lower_bound(iterBegin, iterEnd, pKey, KeyCmp(keyLength));
    if (iterFound == iterEnd || KeyCmp(keyLength)(pKey, *iterFound))
    {
        return -1;
    }
    else
    {
        return static_cast<int>(std::distance(iterBegin, iterFound));
    }
}

int ResShadingModel::FindProgramIndex(const ShaderRange& range, const bit32* pDynamicKey) const
{
    NW_G3D_ASSERTMSG(pDynamicKey != NULL || GetDynamicKeyLength() == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    int keyStride = GetKeyLength();
    int keyOffset = GetStaticKeyLength();
    int keyLength = GetDynamicKeyLength();
    KeyIter iterBegin(range.pBegin, keyLength, keyStride);
    KeyIter iterEnd(range.pEnd, keyLength, keyStride);
    KeyIter iterFound = std::lower_bound(iterBegin, iterEnd, pDynamicKey, KeyCmp(keyLength));
    if (iterFound == iterEnd || KeyCmp(keyLength)(pDynamicKey, *iterFound))
    {
        return -1;
    }
    else
    {
        KeyIter iterBase(ref().ofsKeyTable.to_ptr<bit32>() + keyOffset, keyLength, keyStride);
        return static_cast<int>(std::distance(iterBase, iterFound));
    }
}

bool ResShadingModel::FindProgramRange(ShaderRange* pRange, const bit32* pStaticKey) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pRange, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERTMSG(pStaticKey != NULL || GetStaticKeyLength() == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    int keyStride = GetKeyLength();
    int keyLength = GetStaticKeyLength();
    KeyIter iterBegin(ref().ofsKeyTable.to_ptr<bit32>(), keyLength, keyStride);
    KeyIter iterEnd(iterBegin + GetShaderProgramCount());
    std::pair<KeyIter, KeyIter> rangeFound =
        std::equal_range(iterBegin, iterEnd, pStaticKey, KeyCmp(keyLength));
    pRange->pBegin = *rangeFound.first + keyLength; // dynamic オプションのキーとして記録します。
    pRange->pEnd = *rangeFound.second + keyLength;
    return pRange->pBegin != pRange->pEnd;
}

void ResShadingModel::UpdateProgram(int programIndex)
{
#if ( NW_G3D_IS_GL && !defined( NW_STRIP_GL ) ) || !NW_G3D_IS_RELEASE
    const ResShaderArchive* pArchive = ref().ofsShaderArchive.to_ptr<ResShaderArchive>();
    const UpdateProgramCallback pCallback = pArchive->GetUpdateProgramCallback();
    if (pCallback)
    {
        pCallback(this, programIndex);
    }
#else
    NW_G3D_UNUSED(programIndex);
#endif
}

const bit32* ResShadingModel::GetKey(int programIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(programIndex, GetShaderProgramCount(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    const bit32* pKeyTable = ref().ofsKeyTable.to_ptr<bit32>();
    int keyStride = sizeof(bit32) * GetKeyLength();
    const bit32* pKey = AddOffset<bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

const bit32* ResShadingModel::GetStaticKey(int programIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(programIndex, GetShaderProgramCount(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    const bit32* pKeyTable = ref().ofsKeyTable.to_ptr<bit32>();
    int keyStride = sizeof(bit32) * GetKeyLength();
    const bit32* pKey = AddOffset<bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

const bit32* ResShadingModel::GetDynamicKey(int programIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(programIndex, GetShaderProgramCount(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    const bit32* pKeyTable = ref().ofsKeyTable.to_ptr<bit32>() + GetStaticKeyLength();
    int keyStride = sizeof(bit32) * GetKeyLength();
    const bit32* pKey = AddOffset<bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

int ResShadingModel::PrintKeyTo(char* pStr, int strLength, const bit32* pKey, int keyLength)
{
    NW_G3D_ASSERT(pStr != NULL || strLength == 0);
    NW_G3D_ASSERT(pKey != NULL || keyLength == 0);

    const int lenPerBit32 = 9; // 8桁 + 区切り文字
    int strLengthToWrite = 0;
    if (keyLength > 0)
    {
        strLengthToWrite = keyLength * lenPerBit32 - 1; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return strLengthToWrite;
    }

    NW_G3D_ASSERT(strLength > 0); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    int idxKey = 0;
    do
    {
        sprintf(pStr, "%08X_", pKey[idxKey]); // 書き込み先サイズはチェック済みです。
        pStr += lenPerBit32;
    } while (++idxKey < keyLength);

    *(--pStr) = '\0'; // 最後の '_' を終端文字に置き換えます。
    return strLengthToWrite;
}

int ResShadingModel::PrintStaticOptionTo(char* pStr, int strLength, const bit32* pKey) const
{
    NW_G3D_ASSERTMSG(pStr != NULL || strLength == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERTMSG(pKey != NULL || GetStaticKeyLength() == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    // デバッグ機能なので簡潔さのためにループを2周します。
    int strLengthToWrite = 0;
    for (int idxOption = 0, numOption = GetStaticOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        const ResName* pOptionName = pOption->ref().ofsName.GetResName();
        strLengthToWrite += static_cast<int>(pOptionName->GetLength());

        int choice = pOption->ReadStaticKey(pKey);
        const ResDicType* pDic = NW_G3D_RES_DIC(pOption->ref().ofsChoiceDic);
        const ResName* pChoiceName = pDic->GetResName(choice);
        strLengthToWrite += static_cast<int>(pChoiceName->GetLength());

        strLengthToWrite += 2; // 区切り文字分です。(Option:Choice\t)
    }
    if (strLengthToWrite > 0)
    {
        --strLengthToWrite; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return strLengthToWrite;
    }

    NW_G3D_ASSERTMSG(strLength > 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName())); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    for (int idxOption = 0, numOption = GetStaticOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        const ResName* pOptionName = pOption->ref().ofsName.GetResName();
        memcpy(pStr, pOptionName->GetName(), pOptionName->GetLength());
        pStr += pOptionName->GetLength();
        *pStr++ = ':';

        int choice = pOption->ReadStaticKey(pKey);
        const ResDicType* pDic = NW_G3D_RES_DIC(pOption->ref().ofsChoiceDic);
        const ResName* pChoiceName = pDic->GetResName(choice);
        memcpy(pStr, pChoiceName->GetName(), pChoiceName->GetLength());
        pStr += pChoiceName->GetLength();
        *pStr++ = '\t';
    }

    *(--pStr) = '\0'; // 最後の '\t' を終端文字に置き換えます。
    return strLengthToWrite;
}

int ResShadingModel::PrintDynamicOptionTo(char* pStr, int strLength, const bit32* pKey) const
{
    NW_G3D_ASSERTMSG(pStr != NULL || strLength == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERTMSG(pKey != NULL || GetDynamicKeyLength() == 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    // デバッグ機能なので簡潔さのためにループを2周します。
    int strLengthToWrite = 0;
    for (int idxOption = 0, numOption = GetDynamicOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        const ResName* pOptionName = pOption->ref().ofsName.GetResName();
        strLengthToWrite += static_cast<int>(pOptionName->GetLength());

        int choice = pOption->ReadDynamicKey(pKey);
        const ResDicType* pDic = NW_G3D_RES_DIC(pOption->ref().ofsChoiceDic);
        const ResName* pChoiceName = pDic->GetResName(choice);
        strLengthToWrite += static_cast<int>(pChoiceName->GetLength());

        strLengthToWrite += 2; // 区切り文字分です。(Option:Choice\t)
    }
    if (strLengthToWrite > 0)
    {
        --strLengthToWrite; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return strLengthToWrite;
    }

    NW_G3D_ASSERTMSG(strLength > 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName())); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    for (int idxOption = 0, numOption = GetDynamicOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        const ResName* pOptionName = pOption->ref().ofsName.GetResName();
        memcpy(pStr, pOptionName->GetName(), pOptionName->GetLength());
        pStr += pOptionName->GetLength();
        *pStr++ = ':';

        int choice = pOption->ReadDynamicKey(pKey);
        const ResDicType* pDic = NW_G3D_RES_DIC(pOption->ref().ofsChoiceDic);
        const ResName* pChoiceName = pDic->GetResName(choice);
        memcpy(pStr, pChoiceName->GetName(), pChoiceName->GetLength());
        pStr += pChoiceName->GetLength();
        *pStr++ = '\t';
    }

    *(--pStr) = '\0'; // 最後の '\t' を終端文字に置き換えます。
    return strLengthToWrite;
}

//--------------------------------------------------------------------------------------------------

ResShaderArchive* ResShaderArchive::ResCast(void* ptr)
{
    NW_G3D_ASSERT_NOT_NULL(ptr);
    ResShaderArchiveData* pData = static_cast<ResShaderArchiveData*>(ptr);
    ResShaderArchive* pArchive = static_cast<ResShaderArchive*>(pData);
#if !!NW_G3D_IS_HOST_CAFE
    // ptr が NULL の場合もあるので Valid かどうかチェックする。
    if ((pData != NULL) && pData->fileHeader.byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        // エンディアンを変換する。
        Endian<true>::Swap(pData);
    }
#endif

#if !defined( NW_STRIP_GL ) // GL と一緒にはがす
    NW_G3D_ASSERT_ADDR_ALIGNMENT(ptr, pData->alignment);
#endif
    return pArchive;
}

void ResShaderArchive::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERTMSG(HasGLSource() || HasGLBinary(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    SetUpdateProgramCallback(UpdateProgramGL);

    if (IsShaderCompressed() && !s_ZLib.IsValid())
    {
        s_ZLib.Construct();
    }

    // バイナリ版が使用可能かどうかを調べる
    if (HasGLBinary())
    {
        class Buffer
        {
        public:
            Buffer() : ptr(NULL) {}
            Buffer(size_t size) { Alloc(size); }
            ~Buffer() { Free(); }
            bool Alloc(size_t size)
            {
                Free();
                return ( ptr = malloc(size) ) != NULL;
            }
            void Free()
            {
                if (ptr)
                {
                    free(ptr);
                    ptr = NULL;
                }
            }
            void* ptr;
        } binary;

        GfxShader shader;
        GfxShader::SetupArg arg;
        ResShaderProgram::GLShaderProgram* pProgram = GetShadingModel(0)->GetShaderProgram(0)->
            ref().ofsVertexShader.to_ptr<ResShaderProgram::GLShaderProgram>();
        if (pProgram == NULL)
        {
            pProgram = GetShadingModel(0)->GetShaderProgram(0)->ref().
                ofsComputeShader.to_ptr<ResShaderProgram::GLShaderProgram>();
        }
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pProgram, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        arg.pBinary = pProgram->ofsShaderCode.to_ptr();
        arg.binarySize = pProgram->shaderCodeSize;
        arg.binaryFormat = pProgram->binaryFormat;
        if (IsShaderCompressed())
        {
            binary.Alloc(pProgram->shaderCodeSize);
            s_ZLib.Uncompress(binary.ptr, NULL, pProgram->shaderCodeSize,
                pProgram->ofsShaderCode.to_ptr(), pProgram->compressedShaderCodeSize);
            arg.pBinary = binary.ptr;
        }
        if (shader.IsGLBinaryAvailable(arg))
        {
            ref().flag |= GL_BINARY_AVAILABLE;
        }
        else
        {
            NW_G3D_WARNING(0, "GL binary unavailable (shader archive name : %s)", GetName());
            NW_G3D_ASSERTMSG(HasGLSource(), "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        }
    }
#endif
    for (int idxShadingModel = 0, numShader = GetShadingModelCount(); idxShadingModel < numShader; ++idxShadingModel)
    {
        ResShadingModel* pShadingModel = GetShadingModel(idxShadingModel);
        pShadingModel->Setup();
    }
}

void ResShaderArchive::Cleanup()
{
    for (int idxShadingModel = 0, numShader = GetShadingModelCount(); idxShadingModel < numShader; ++idxShadingModel)
    {
        ResShadingModel* pShadingModel = GetShadingModel(idxShadingModel);
        pShadingModel->Cleanup();
    }
}

bool ResShaderArchive::IsValid(const void* ptr)
{
    NW_G3D_ASSERT_NOT_NULL(ptr);
    const BinaryFileHeader* pHeader = static_cast<const BinaryFileHeader*>(ptr);

    if (pHeader->sigWord != SIGNATURE)
    {
        NW_G3D_WARNING(false, "Signature check failed ('%c%c%c%c' must be '%c%c%c%c').\n",
            pHeader->signature[0],
            pHeader->signature[1],
            pHeader->signature[2],
            pHeader->signature[3],
            NW_G3D_GET_SIGNATURE0(SIGNATURE),
            NW_G3D_GET_SIGNATURE1(SIGNATURE),
            NW_G3D_GET_SIGNATURE2(SIGNATURE),
            NW_G3D_GET_SIGNATURE3(SIGNATURE)
        );
        return false;
    }

    if (pHeader->verWord != NW_G3D_SHADER_VERSION)
    {
        NW_G3D_WARNING(false, "Version check failed (bin:'%d.%d.%d.%d', lib:'%d.%d.%d.%d').\n"
            "\t[ResShaderArchive] %s\n",
            pHeader->version[0],
            pHeader->version[1],
            pHeader->version[2],
            pHeader->version[3],
            NW_G3D_VERSION_MAJOR,
            NW_G3D_VERSION_MINOR,
            NW_G3D_VERSION_MICRO,
            NW_G3D_VERSION_SHADERBUGFIX,
            GetArchiveName(ptr)
        );
        return false;
    }

#if NW_G3D_IS_HOST_CAFE
    // PC ではエンディアン反転して使用するので、エンディアンが違っても問題ない。
    if (pHeader->byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        NW_G3D_WARNING(false, "Endian check failed ('0x%X' must be '0x%X').\n"
            "\t[ResFile] %s\n",
            pHeader->byteOrder,
            BinaryFileHeader::BYTE_ORDER_MARK,
            GetArchiveName(ptr)
        );
        return false;
    }
#endif

    return true;
}

}}} // namespace nw::g3d::res
