﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResUtility.h>
#include <nw/g3d/fnd/g3d_GX2Utility.h>

#if NW_G3D_IS_HOST_CAFE
#include <nw/version.h>
#if NW_CURRENT_VERSION_NUMBER >= NW_VERSION_NUMBER(1, 10, 0, 0)
#define INSERT_G3D_SYMBOL 1
#else
#define INSERT_G3D_SYMBOL 0
#endif
#else
#define INSERT_G3D_SYMBOL 0
#endif

#if INSERT_G3D_SYMBOL
#include <nw/middlewareSymbol.h>
#endif

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

namespace {

#if !defined(NW_RELEASE) // 未使用関数に対する警告対策

inline
const char* GetFileName(const void* ptr)
{
    const ResFileData* pData = static_cast<const ResFileData*>(ptr);
    if (pData->fileHeader.byteOrder == BinaryFileHeader::BYTE_ORDER_MARK)
    {
        return pData->ofsName.to_ptr();
    }
    else
    {
        const u32* ofsName = reinterpret_cast<const u32*>(&pData->ofsName);
        return reinterpret_cast<const char*>(ofsName) + LoadRevU32(ofsName);
    }
}

#endif // !defined(NW_RELEASE)

} // anonymous namespace

bool ResFile::IsValid(const void* ptr)
{
    NW_G3D_ASSERT_NOT_NULL(ptr);
    const BinaryFileHeader* pHeader = static_cast<const BinaryFileHeader*>(ptr);

    if (pHeader->sigWord != SIGNATURE)
    {
        NW_G3D_WARNING(false, "Signature check failed ('%c%c%c%c' must be '%c%c%c%c').\n",
            pHeader->signature[0],
            pHeader->signature[1],
            pHeader->signature[2],
            pHeader->signature[3],
            NW_G3D_GET_SIGNATURE0(SIGNATURE),
            NW_G3D_GET_SIGNATURE1(SIGNATURE),
            NW_G3D_GET_SIGNATURE2(SIGNATURE),
            NW_G3D_GET_SIGNATURE3(SIGNATURE)
        );
        return false;
    }

    if (pHeader->verWord != NW_G3D_VERSION)
    {
        NW_G3D_WARNING(false, "Version check failed (bin:'%d.%d.%d.%d', lib:'%d.%d.%d.%d').\n"
            "\t[ResFile] %s\n",
            pHeader->version[0],
            pHeader->version[1],
            pHeader->version[2],
            pHeader->version[3],
            NW_G3D_VERSION_MAJOR,
            NW_G3D_VERSION_MINOR,
            NW_G3D_VERSION_MICRO,
            NW_G3D_VERSION_BINARYBUGFIX,
            GetFileName(ptr)
        );
        return false;
    }

#if NW_G3D_IS_HOST_CAFE
    // PC ではエンディアン反転して使用するので、エンディアンが違っても問題ない。
    if (pHeader->byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        NW_G3D_WARNING(false, "Endian check failed ('0x%X' must be '0x%X').\n"
            "\t[ResFile] %s\n",
            pHeader->byteOrder,
            BinaryFileHeader::BYTE_ORDER_MARK,
            GetFileName(ptr)
        );
        return false;
    }
#endif

    return true;
}

ResFile* ResFile::ResCast(void* ptr)
{
#if INSERT_G3D_SYMBOL
    // モジュール情報の埋め込み
    NW_PUT_MIDDLEWARE_SYMBOL(g3d);
#endif

    NW_G3D_ASSERT_NOT_NULL(ptr);
    ResFileData* pData = static_cast<ResFileData*>(ptr);
    ResFile* pFile = static_cast<ResFile*>(pData);
#if !NW_G3D_IS_HOST_CAFE
    // ptr が NULL の場合もあるので Valid かどうかチェックする。
    if ((pData != NULL) && pData->fileHeader.byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        // エンディアンを変換する。
        Endian<true>::Swap(pData);

#if !defined( NW_STRIP_GL )
        // タイリングを解除する。
        for (int idxTex = 0, numTex = pFile->GetTextureCount(); idxTex < numTex; ++idxTex)
        {
            ResTexture* pResTex = pFile->GetTexture(idxTex);
            GfxTexture* pGfxTex = pResTex->GetGfxTexture();
            GX2Surface& surface = pGfxTex->GetGX2Texture()->surface;
            if (surface.tileMode != GX2_TILE_MODE_LINEAR_SPECIAL)
            {
                pGfxTex->SetImagePtrs(pResTex->GetData(), pResTex->GetMipData());
                ConvertToGLSurface(surface, pGfxTex->arrayLength);
                pGfxTex->SetImagePtrs(NULL, NULL);
            }
        }
#endif // !defined( NW_STRIP_GL )
    }
#endif // !NW_G3D_IS_HOST_CAFE

#if !defined( NW_STRIP_GL ) // GL と一緒にはがす
    NW_G3D_ASSERT_ADDR_ALIGNMENT(ptr, pData->alignment);
#endif // !defined( NW_STRIP_GL )
    return pFile;
}

BindResult ResFile::Bind(const ResFile* pFile, bit32 typeMask)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFile, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    if (typeMask & BIND_TEXTURE)
    {
        for (int idxModel = 0, numModel = GetModelCount(); idxModel < numModel; ++idxModel)
        {
            ResModel* pModel = GetModel(idxModel);
            result |= pModel->Bind(pFile);
        }

        for (int idxTexPat = 0, numTexPat = GetTexPatternAnimCount();
            idxTexPat < numTexPat; ++idxTexPat)
        {
            ResTexPatternAnim* pTexPat = GetTexPatternAnim(idxTexPat);
            result |= pTexPat->Bind(pFile);
        }
    }

    return result;
}

BindResult ResFile::Bind(const ResFile* pFile, bit32 typeMask, BindCallback* pCallback)
{
    NW_G3D_ASSERT_NOT_NULL(pFile);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pCallback, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    pCallback->m_pFile = this;

    if (typeMask & BIND_TEXTURE)
    {
        for (int idxModel = 0, numModel = GetModelCount(); idxModel < numModel; ++idxModel)
        {
            ResModel* pModel = GetModel(idxModel);
            pCallback->m_pModel = pModel;
            result |= pModel->Bind(pFile, pCallback);
        }
        pCallback->m_pModel = NULL;

        for (int idxTexPat = 0, numTexPat = GetTexPatternAnimCount();
            idxTexPat < numTexPat; ++idxTexPat)
        {
            ResTexPatternAnim* pTexPat = GetTexPatternAnim(idxTexPat);
            pCallback->m_pTexPatternAnim = pTexPat;
            result |= pTexPat->Bind(pFile, pCallback);
        }
        pCallback->m_pTexPatternAnim = NULL;
    }

    return result;
}

nw::g3d::res::BindResult ResFile::Bind(const BindFuncTable& funcTable)
{
    BindResult result;

    for (int idxSceneAnim = 0, numSceneAnim = GetSceneAnimCount();
        idxSceneAnim < numSceneAnim; ++idxSceneAnim)
    {
        ResSceneAnim* pSceneAnim = GetSceneAnim(idxSceneAnim);
        result |= pSceneAnim->Bind(funcTable);
    }

    return result;
}

void ResFile::Release(bit32 typeMask)
{
    if (typeMask & BIND_TEXTURE)
    {
        for (int idxModel = 0, numModel = GetModelCount(); idxModel < numModel; ++idxModel)
        {
            ResModel* pModel = GetModel(idxModel);
            pModel->Release();
        }

        for (int idxTexPat = 0, numTexPat = GetTexPatternAnimCount();
            idxTexPat < numTexPat; ++idxTexPat)
        {
            ResTexPatternAnim* pTexPat = GetTexPatternAnim(idxTexPat);
            pTexPat->Release();
        }
    }
}

void ResFile::Setup()
{
    int numModel = this->GetModelCount();
    for (int i = 0; i < numModel; ++i)
    {
        this->GetModel(i)->Setup();
    }

    int numTexture = this->GetTextureCount();
    for (int i = 0; i < numTexture; ++i)
    {
        this->GetTexture(i)->Setup();
    }
}

void ResFile::Cleanup()
{
    int numModel = this->GetModelCount();
    for (int i = 0; i < numModel; ++i)
    {
        this->GetModel(i)->Cleanup();
    }

    int numTexture = this->GetTextureCount();
    for (int i = 0; i < numTexture; ++i)
    {
        this->GetTexture(i)->Cleanup();
    }
}

void ResFile::Reset()
{
    int numModel = this->GetModelCount();
    for (int i = 0; i < numModel; ++i)
    {
        this->GetModel(i)->Reset();
    }

    int numSkeletalAnim = this->GetSkeletalAnimCount();
    for (int i = 0; i < numSkeletalAnim; ++i)
    {
        this->GetSkeletalAnim(i)->Reset();
    }

    int numShaderparamAnim = this->GetShaderParamAnimCount();
    for (int i = 0; i < numShaderparamAnim; ++i)
    {
        this->GetShaderParamAnim(i)->Reset();
    }

    int numTexPatternAnim = this->GetTexPatternAnimCount();
    for (int i = 0; i < numTexPatternAnim; ++i)
    {
        this->GetTexPatternAnim(i)->Reset();
    }

    int numBoneVisAnim = this->GetBoneVisAnimCount();
    for (int i = 0; i < numBoneVisAnim; ++i)
    {
        this->GetBoneVisAnim(i)->Reset();
    }

    int numMatVisAnim = this->GetMatVisAnimCount();
    for (int i = 0; i < numMatVisAnim; ++i)
    {
        this->GetMatVisAnim(i)->Reset();
    }

    int numShapeAnim = this->GetShapeAnimCount();
    for (int i = 0; i < numShapeAnim; ++i)
    {
        this->GetShapeAnim(i)->Reset();
    }

    int numSceneAnim = this->GetSceneAnimCount();
    for (int i = 0; i < numSceneAnim; ++i)
    {
        this->GetSceneAnim(i)->Reset();
    }

    ref().pUserPtr.set_ptr(NULL);
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
