﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResDictionary.h>
#include <algorithm>
#include <limits.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

namespace {

u32 GetBit(const char* str, size_t len, u32 refBit)
{
    u32 wd = (refBit >> 3);
    u32 pos = (refBit & 7);
    // 参照ビットが name の外の場合は0
    return wd < len ? (str[wd] >> pos) & 0x1 : 0;
}

u32 GetBit(const ResName* name, u32 refBit)
{
    return GetBit(name->GetName(), name->GetLength(), refBit);
}

} // anonymous namespace

#if defined(__clang__) && defined(ANDROID)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wtautological-undefined-compare"
#endif

void* ResDicPatricia::Find(const char* str) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(str);
        Node* pNode = FindNode(str, strlen(str));
        if (pNode)
        {
            return pNode->ofsData.to_ptr();
        }
    }

    return NULL;
}

void* ResDicPatricia::Find(const ResName* name) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(name);
        Node* pNode = FindNode(name);
        if (pNode)
        {
            return pNode->ofsData.to_ptr();
        }
    }

    return NULL;
}

void* ResDicPatricia::Find(const char* str, size_t len) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(str);
        Node* pNode = FindNode(str, len);
        if (pNode)
        {
            return pNode->ofsData.to_ptr();
        }
    }

    return NULL;
}

int ResDicPatricia::FindIndex(const char* str) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(str);
        const Node* pNode = FindNode(str, strlen(str));
        if (pNode)
        {
            return static_cast<int>(std::distance(ref().node + 1, pNode));
        }
    }

    return -1;
}

int ResDicPatricia::FindIndex(const char* str, size_t len) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(str);
        const Node* pNode = FindNode(str, len);
        if (pNode)
        {
            return static_cast<int>(std::distance(ref().node + 1, pNode));
        }
    }

    return -1;
}

int ResDicPatricia::FindIndex(const ResName* name) const
{
    if (this != NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(name);
        const Node* pNode = FindNode(name);
        if (pNode)
        {
            return static_cast<int>(std::distance(ref().node + 1, pNode));
        }
    }

    return -1;
}

ResDicPatricia::BuildResult ResDicPatricia::Build()
{
    // numData と各ノードの ofsName が正しく設定されている状態から辞書を再構築する。
    // size と各ノードの ofsData は変更しない。

    if (this == NULL)
    {
        return SUCCESS; // 空の場合は何もしない。
    }

    Node* node = ref().node;
    Node& root = node[0];
    root.idxLeft = root.idxRight = 0;
    root.refBit = ~0u;

    // 長さを取得できるように一時的に長さ0の ResName を設定する。
    // 64ビット版でも利用できるように、スタックではなく ResDicPatriciaData::size の領域を借りる。
    ResNameData* empty = reinterpret_cast<ResNameData*>(&ref().size);
    ResName::LengthType len = empty->len;
    empty->len = 0;
    root.ofsName.set_ptr(empty->str);

    for (int nodeIndex = 1, end = ref().numData; nodeIndex <= end; ++nodeIndex)
    {
        Node& current = node[nodeIndex];
        const ResName* name = current.ofsName.GetResName();
        u32 refBit = 0;

        // 1度最後まで辿ってビット比較対象の文字列を得る。
        {
            const Node* parent = &root;
            const Node* child = &node[parent->idxLeft];

            while (parent->refBit > child->refBit)
            {
                // child->refBit目が1なら右のリンクを、0なら左のリンクを辿る。
                // 上のノードに戻ったらループを終了する。
                parent = child;
                child = GetBit(name, child->refBit) ?
                    &node[child->idxRight] : &node[child->idxLeft];
            }

            refBit = static_cast<u32>(std::max(
                name->GetLength(), child->ofsName.GetResName()->GetLength())) * CHAR_BIT;
            while (GetBit(child->ofsName.GetResName(), refBit) == GetBit(name, refBit))
            {
                // 戻った先のノードのエントリと違うビットを探す
                // refBit ビット目で判断することにする。
                if (refBit == 0)
                {
                    return ERR_NAME_DUPLICATION; // 名前が重複している場合は失敗。
                }
                --refBit;
            }
            current.refBit = refBit;
        }

        // 挿入
        {
            Node* parent = &root;
            Node* child = &node[parent->idxLeft];

            while (parent->refBit > child->refBit && child->refBit > refBit)
            {
                parent = child;
                child = GetBit(name, child->refBit) ?
                    &node[child->idxRight] : &node[child->idxLeft];
            }
            // この時点で、parent と child の間にノードを挿入することが確定している。

            u16 idxCurrent = static_cast<u16>(nodeIndex);
            u16 idxChild = static_cast<u16>(std::distance(node, child));

            if (GetBit(name, current.refBit))
            {
                // 当該ビットが1の場合は右にt(自身)をくっつける(current -> (current, child))。
                current.idxLeft = idxChild;
                current.idxRight = idxCurrent;
            }
            else
            {
                // 当該ビットが0の場合は左にt(自身)をくっつける(current -> (child, current))。
                current.idxLeft = idxCurrent;
                current.idxRight = idxChild;
            }

            // parent のどちらの子供にするか決める。
            if (GetBit(name, parent->refBit))
            {
                parent->idxRight = idxCurrent;
            }
            else
            {
                parent->idxLeft  = idxCurrent;
            }
        }
    }

    root.ofsName.set_ptr(NULL);
    empty->len = len;
    return SUCCESS;
}

#if defined(__clang__) && defined(ANDROID)
#pragma clang diagnostic pop
#endif

ResDicPatricia::Node* ResDicPatricia::FindNode(const char* str, size_t len) const
{
#ifndef NW_G3D_DIC_LINEAR
    const Node* node = ref().node;
    const Node* parent = &node[0];
    const Node* child = &node[parent->idxLeft];

    while(parent->refBit > child->refBit)
    {
        parent = child;
        // ビットが1なら右、0なら左をたどる
        child = GetBit(str, len, child->refBit) ? &node[child->idxRight] : &node[child->idxLeft];
    }

    if (child->ofsName.offset && child->ofsName.GetResName()->Equals(str, len))
    {
        return const_cast<Node*>(child);
    }
    else
    {
        return NULL;
    }
#else
    for (int idx = 1, end = GetDataCount(); idx <= end; ++idx)
    {
        const Node& node = ref().node[idx];
        if (child->ofsName.offset && node.ofsName.GetResName()->Equals(str, len))
        {
            return const_cast<Node*>(&node);
        }
    }
    return NULL;
#endif // NW_G3D_DIC_LINEAR
}

ResDicPatricia::Node* ResDicPatricia::FindNode(const ResName* name) const
{
#ifndef NW_G3D_DIC_LINEAR
    const Node* node = ref().node;
    const Node* parent = &node[0];
    const Node* child = &node[parent->idxLeft];

    while(parent->refBit > child->refBit)
    {
        parent = child;
        // ビットが1なら右、0なら左をたどる
        child = GetBit(name, child->refBit) ? &node[child->idxRight] : &node[child->idxLeft];
    }

    if (child->ofsName.offset && child->ofsName.GetResName()->Equals(name))
    {
        return const_cast<Node*>(child);
    }
    else
    {
        return NULL;
    }
#else
    for (int idx = 1, end = GetDataCount(); idx <= end; ++idx)
    {
        const Node& node = ref().node[idx];
        if (child->ofsName.offset && node.ofsName.GetResName()->Equals(name))
        {
            return const_cast<Node*>(&node);
        }
    }
    return NULL;
#endif // NW_G3D_DIC_LINEAR
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
