﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_TexPatternAnimObj.h>
#include <algorithm>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/g3d_ModelObj.h>

namespace nw { namespace g3d {

void TexPatternAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numBind = std::max(arg.GetMaxMatCount(), arg.GetMaxMatAnimCount());
    int numCurve = arg.GetMaxCurveCount();
    int numPatAnim = arg.GetMaxPatAnimCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = Align(sizeof(u16) * numPatAnim);
    chunk[idx++].size = sizeof(bit32) * numBind;
    chunk[idx++].size = Align(sizeof(s8) * numPatAnim);
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    chunk[idx++].size = sizeof(ResTexture*) * arg.GetMaxTextureCount();
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t TexPatternAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool TexPatternAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numBind = std::max(arg.GetMaxMatCount(), arg.GetMaxMatAnimCount());
    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Init(sizer.GetPtr<bit32>(ptr, Sizer::BIND_TABLE), numBind);
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));
    m_MaxMatAnim = arg.GetMaxMatAnimCount();
    m_MaxSubBind = arg.GetMaxPatAnimCount();
    m_MaxTexture = arg.GetMaxTextureCount();
    m_pSubBindIndexArray = sizer.GetPtr<s8>(ptr, Sizer::SUBBIND_TABLE);
    m_ppTextureArray = sizer.GetPtr<ResTexture*>(ptr, Sizer::TEXTURE_TABLE);

    return true;
}

void TexPatternAnimObj::SetResource(ResTexPatternAnim* pRes)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pRes), "%s\n", NW_G3D_RES_GET_NAME(pRes, GetName()));

    m_pRes = pRes;
    m_pMatAnimArray = pRes->ref().ofsMatAnimArray.to_ptr<ResTexPatternMatAnimData>();

    SetTargetUnbound();

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetMatAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());

    ClearTexture();
}

BindResult TexPatternAnimObj::Bind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(IsAcceptable(pModel), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pModel->GetMaterialCount());

    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxAnim);
        const ResName* pName = pMatAnim->ref().ofsName.GetResName();
        int idxTarget = pModel->GetMaterialIndex(pName);
        if (idxTarget >= 0)
        {
            // TexPattern のバインド
            const ResMaterial* pMaterial = pModel->GetMaterial(idxTarget);
            BindResult subResult = SubBind(pMatAnim, pMaterial);
            if (!subResult.IsMissed())
            {
                // 1つも関連付けられなかった場合はバインドしない。
                bindTable.Bind(idxAnim, idxTarget);
            }
            result |= subResult;
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    SetTargetBound();
    TexPatternAnimObj::ClearResult();
    return result;
}

BindResult TexPatternAnimObj::Bind(const ModelObj* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    return Bind(pModel->GetResource());
}

void TexPatternAnimObj::BindFast(const ResModel* pTarget)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pTarget), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(pTarget == m_pRes->GetBindModel(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pTarget->GetMaterialCount());
    bindTable.BindAll(m_pRes->ref().ofsBindIndexArray.to_ptr<u16>());

    // TexPattern のバインド
    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        if (idxTarget < AnimBindTable::NOT_BOUND)
        {
            ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxAnim);
            BindResult result = SubBindFast(pMatAnim);
            if (result.IsMissed())
            {
                // 1つも関連付けられなかった場合はバインド解除する。
                bindTable.Unbind(idxAnim, idxTarget);
            }
        }
    }

    SetTargetBound();
    TexPatternAnimObj::ClearResult();
}

BindResult TexPatternAnimObj::SubBind(
    const ResTexPatternMatAnim* pMatAnim, const ResMaterial* pMaterial)
{
    BindResult result;
    const ResTexPatternMatAnim::PatAnimInfo* pInfo =
        pMatAnim->ref().ofsPatAnimInfoArray.to_ptr<const ResTexPatternMatAnim::PatAnimInfo>();
    int beginPatAnim = pMatAnim->ref().beginPatAnim;

    for (int idxPatAnim = 0, numPatAnim = pMatAnim->GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = pMaterial->GetSamplerIndex(pName);
        m_pSubBindIndexArray[beginPatAnim + idxPatAnim] = static_cast<s8>(idxTarget);
        result |= idxTarget >= 0 ? BindResult::Bound() : BindResult::NotBound();
    }

    return result;
}

BindResult TexPatternAnimObj::SubBindFast(const ResTexPatternMatAnim* pMatAnim)
{
    BindResult result;
    const ResTexPatternMatAnim::PatAnimInfo* pInfo =
        pMatAnim->ref().ofsPatAnimInfoArray.to_ptr<const ResTexPatternMatAnim::PatAnimInfo>();
    int beginPatAnim = pMatAnim->ref().beginPatAnim;

    for (int idxPatAnim = 0, numPatAnim = pMatAnim->GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        m_pSubBindIndexArray[beginPatAnim + idxPatAnim] = pInfo->subbindIndex;
        result |= pInfo->subbindIndex == -1 ?
            BindResult::NotBound() : BindResult::Bound();
    }

    return result;
}

void TexPatternAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    // バインドされていなくても初期化してしまう。
    for (int idxAnim = 0, numAnim = GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxAnim);
        int beginPatAnim = pMatAnim->ref().beginPatAnim;
        u16* pResult = AddOffset<u16>(GetResultBuffer(), sizeof(u16) * beginPatAnim);
        pMatAnim->Init(pResult);
    }
}

void TexPatternAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        const AnimBindTable& bindTable = GetBindTable();
        AnimContext& context = GetContext();
        int numMatAnim = bindTable.GetAnimCount();
        if (context.IsFrameCacheValid())
        {
            for (int idxPatAnim = 0; idxPatAnim < numMatAnim; ++idxPatAnim)
            {
                const ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxPatAnim);
                int beginPatAnim = pMatAnim->ref().beginPatAnim;
                int beginCurve = pMatAnim->ref().beginCurve;

                if (!bindTable.IsCalcEnabled(idxPatAnim))
                {
                    continue;
                }

                u16* pResult = AddOffset<u16>(GetResultBuffer(), sizeof(u16) * beginPatAnim);
                pMatAnim->Eval(pResult, frame, &m_pSubBindIndexArray[beginPatAnim],
                    context.GetFrameCacheArray(beginCurve));
            }
        }
        else
        {
            for (int idxMatAnim = 0; idxMatAnim < numMatAnim; ++idxMatAnim)
            {
                const ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
                int beginPatAnim = pMatAnim->ref().beginPatAnim;

                if (!bindTable.IsCalcEnabled(idxMatAnim))
                {
                    continue;
                }

                u16* pResult = AddOffset<u16>(GetResultBuffer(), sizeof(u16) * beginPatAnim);
                pMatAnim->Eval(pResult, frame, &m_pSubBindIndexArray[beginPatAnim]);
            }
        }

        UpdateLastFrame();
    }
}

void TexPatternAnimObj::ApplyTo(ModelObj* pModelObj) const
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    const AnimBindTable& bindTable = GetBindTable();
    int numMatAnim = bindTable.GetAnimCount();
    for (int idxMatAnim = 0; idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        if (!bindTable.IsApplyEnabled(idxMatAnim))
        {
            continue;
        }

        int idxMat = bindTable.GetTargetIndex(idxMatAnim);
        MaterialObj* pMaterial = pModelObj->GetMaterial(idxMat);
        ApplyTo(pMaterial, idxMatAnim);
    }
}

void TexPatternAnimObj::ApplyTo(MaterialObj* pMaterialObj, int animIndex) const
{
    // カーブの評価結果
    const ResTexPatternMatAnim* pMatAnim = GetMatAnim(animIndex);
    int beginPatAnim = pMatAnim->ref().beginPatAnim;
    const u16* pResult = AddOffset<u16>(GetResultBuffer(), sizeof(u16) * beginPatAnim);

    // 書き込み先と Texture の情報
    const ResTexPatternMatAnim::PatAnimInfo* pInfo =
        pMatAnim->ref().ofsPatAnimInfoArray.to_ptr<ResTexPatternMatAnim::PatAnimInfo>();
    int numPatAnim = pMatAnim->GetPatAnimCount();
    for (int idxPatAnim = 0; idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        int subBindIndex = m_pSubBindIndexArray[beginPatAnim + idxPatAnim];
        if (subBindIndex == -1)
        {
            continue;
        }

        pMaterialObj->SetResTexture(subBindIndex, m_ppTextureArray[pResult[idxPatAnim]]);
    }
}

}} // namespace nw::g3d
