﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_ShaderUtility.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/res/g3d_ResShape.h>
#include <nw/g3d/g3d_MaterialObj.h>
#include <nw/g3d/g3d_ShaderObj.h>
#include <algorithm>

namespace nw { namespace g3d {

void ShaderUtility::InitShaderKey(bit32* pKey, int keyLength,
    const ResShadingModel* pResShadingModel, const ResShaderAssign* pResShaderAssign,
    bool materialPass /*= true*/)
{
    NW_G3D_ASSERT_NOT_NULL(pResShadingModel);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResShaderAssign, "%s\n", NW_G3D_RES_GET_NAME(pResShadingModel, GetName()));
    NW_G3D_ASSERT(keyLength >= pResShadingModel->GetKeyLength());
    // Release 版で未使用
    NW_G3D_UNUSED( keyLength );
    NW_G3D_UNUSED( materialPass );

    // シェーダキーをまずデフォルトで初期化します。
    int staticKeyLength = pResShadingModel->GetStaticKeyLength();
    pResShadingModel->WriteDefaultStaticKey(pKey);
    pResShadingModel->WriteDefaultDynamicKey(pKey + staticKeyLength);

    // モデルに記録された静的なシェーダオプションで上書きします。
    for (int idxOption = 0, numOption = pResShaderAssign->GetShaderOptionCount();
        idxOption < numOption; ++idxOption)
    {
        const char* optionName = pResShaderAssign->GetShaderOptionName(idxOption);
        if (const ResShaderOption* pOption = pResShadingModel->GetStaticOption(optionName))
        {
            const char* optionValue = pResShaderAssign->GetShaderOption(idxOption);
            int choice = pOption->GetChoiceIndex(optionValue);
            if (choice < 0)
            {
                // モデルが持っているオプションの値がシェーダに存在しないので、
                // モデルが最新のシェーダと関連付けられていない可能性があります。
                NW_G3D_WARNING(!materialPass, "Unknown choice. (%s)", optionValue);
                continue;
            }
            pOption->WriteStaticKey(pKey, choice);
        }
        else
        {
            // モデルが持っているオプションの名前がシェーダに存在しません。
            // モデルが最新のシェーダと関連付けられていない可能性があります。
            NW_G3D_WARNING(!materialPass, "Unknown option. (%s)", optionName);
        }
    }
}

void ShaderUtility::InitShaderKey(
    ShadingModelObj* pShadingModelObj, const ResShaderAssign* pResShaderAssign,
    bool materialPass /*= true*/)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModelObj);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResShaderAssign, "%s\n", NW_G3D_RES_GET_NAME(pShadingModelObj->GetResource(), GetName()));
    // Release 版で未使用
    NW_G3D_UNUSED( materialPass );

    // シェーダキーをまずデフォルトで初期化します。
    pShadingModelObj->ClearStaticKey();

    // モデルに記録された静的なシェーダオプションで上書きします。
    for (int idxAssignOption = 0, numAssignOption = pResShaderAssign->GetShaderOptionCount();
        idxAssignOption < numAssignOption; ++idxAssignOption)
    {
        const char* optionName = pResShaderAssign->GetShaderOptionName(idxAssignOption);
        int idxOption = pShadingModelObj->GetStaticOptionIndex(optionName);
        if (idxOption >= 0)
        {
            const ResShaderOption* pOption = pShadingModelObj->GetStaticOption(idxOption);
            const char* optionValue = pResShaderAssign->GetShaderOption(idxAssignOption);
            int choice = pOption->GetChoiceIndex(optionValue);
            if (choice < 0)
            {
                // モデルが持っているオプションの値がシェーダに存在しないので、
                // モデルが最新のシェーダと関連付けられていない可能性があります。
                NW_G3D_WARNING(!materialPass, "Unknown choice. (%s)", optionValue);
                continue;
            }
            pShadingModelObj->WriteStaticKey(idxOption, choice);
        }
        else
        {
            // モデルが持っているオプションの名前がシェーダに存在しません。
            // モデルが最新のシェーダと関連付けられていない可能性があります。
            NW_G3D_WARNING(!materialPass, "Unknown option. (%s)", optionName);
        }
    }
}

void ShaderUtility::BindShaderParam(
    ResMaterial* pResMaterial, const ResShadingModel* pResShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResMaterial);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResShadingModel, "%s\n", NW_G3D_RES_GET_NAME(pResMaterial, GetName()));
    int idxBlock = pResShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_MATERIAL);
    if (idxBlock == -1)
    {
        pResMaterial->SetRawParamSize(0);
        return;
    }

    const ResUniformBlockVar* pBlockVar = pResShadingModel->GetUniformBlock(idxBlock);
    pResMaterial->SetRawParamSize(pBlockVar->GetSize()); // UniformBlock 全体のサイズを設定します。
    for (int idxParam = 0, numParam = pResMaterial->GetShaderParamCount();
        idxParam < numParam; ++idxParam)
    {
        // シェーダパラメータの id でシェーダからルックアップし、オフセットを初期化します。
        ResShaderParam* pResShaderParam = pResMaterial->GetShaderParam(idxParam);
        const nw::g3d::ResUniformVar* pUniformVar = pBlockVar->GetUniform(pResShaderParam->GetId());
        pResShaderParam->SetOffset(pUniformVar ? pUniformVar->GetOffset() : -1);
    }
}

void ShaderUtility::InitShaderParam(
    MaterialObj* pMaterialObj, const ResShadingModel* pResShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pMaterialObj);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResShadingModel, "%s\n", NW_G3D_RES_GET_NAME(pMaterialObj->GetResource(), GetName()));
    NW_G3D_ASSERT(pMaterialObj->IsBlockBufferValid());

    int idxBlock = pResShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_MATERIAL);
    if (idxBlock == -1)
    {
        return;
    }

    const ResUniformBlockVar* pBlock = pResShadingModel->GetUniformBlock(idxBlock);
    GfxBuffer& buffer = pMaterialObj->GetMatBlock();
    NW_G3D_ASSERTMSG(buffer.GetSize() >= pBlock->GetSize(), "%s\n", NW_G3D_RES_GET_NAME(pResShadingModel, GetName()));

    for (int idxBuffer = 0, numBuffer = pMaterialObj->GetBufferingCount();
        idxBuffer < numBuffer; ++idxBuffer)
    {
        void* pData = buffer.GetData(idxBuffer);
        CPUCache::FillZero(pData, buffer.GetSize());
        memcpy(pData, pBlock->GetDefault(), pBlock->GetSize());
    }
}

}} // namespace nw::g3d
