﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_ShaderObj.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <algorithm>

namespace nw { namespace g3d {

void ShadingModelObj::Sizer::Calc(const InitArg& arg)
{
    ResShadingModel* pRes = arg.GetResource();
    size_t keySize = pRes->GetStaticKeyLength() * sizeof(bit32);
    int numDirtyFlag = pRes->GetStaticOptionCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = keySize;
    chunk[idx++].size = keySize;
    chunk[idx++].size = Align(numDirtyFlag, 32) >> 3; // 1 オプションあたり 1 ビット。
    NW_G3D_ASSERTMSG(idx == NUM_CHUNK, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    CalcOffset(chunk, NUM_CHUNK);
}

//--------------------------------------------------------------------------------------------------

size_t ShadingModelObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool ShadingModelObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pBuffer, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResShadingModel* pRes = arg.GetResource();

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pRes = pRes;
    m_Flag = 0;
    m_pShaderKey = sizer.GetPtr<bit32>(ptr, Sizer::SHADER_KEY);
    m_pOptionKey = sizer.GetPtr<bit32>(ptr, Sizer::OPTION_KEY);
    m_pDirtyFlagArray = sizer.GetPtr<bit32>(ptr, Sizer::DIRTY_FLAG_ARRAY);
    m_pUserPtr = NULL;
    m_pBufferPtr = pBuffer;
    m_pBlockBuffer = NULL;

    // デフォルトキーで初期化
    m_pRes->WriteDefaultStaticKey(m_pShaderKey);
    std::copy(m_pShaderKey, m_pShaderKey + m_pRes->GetStaticKeyLength(), m_pOptionKey);
    // バリエーションの全範囲で初期化します。
    m_ShaderRange.pBegin =
        m_pRes->ref().ofsKeyTable.to_ptr<const bit32>() + m_pRes->GetStaticKeyLength();
    m_ShaderRange.pEnd =
        m_ShaderRange.pBegin + m_pRes->GetKeyLength() * m_pRes->GetShaderProgramCount();

    return true;
}

size_t ShadingModelObj::CalcBlockBufferSize()
{
    int idxBlock = m_pRes->GetSystemBlockIndex(ResUniformBlockVar::TYPE_OPTION);
    if (idxBlock < 0)
    {
        return 0;
    }
    else
    {
        const ResUniformBlockVar* pBlock = m_pRes->GetUniformBlock(idxBlock);
        return Align(pBlock->GetSize(), BLOCK_BUFFER_ALIGNMENT);
    }
}

bool ShadingModelObj::SetupBlockBuffer(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERTMSG(bufferSize == 0 || pBuffer, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(pBuffer, BLOCK_BUFFER_ALIGNMENT, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG((m_Flag & BLOCK_BUFFER_VALID) == 0, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (CalcBlockBufferSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBlockBuffer = pBuffer;

    int idxBlock = m_pRes->GetSystemBlockIndex(ResUniformBlockVar::TYPE_OPTION);
    if (idxBlock >= 0)
    {
        const ResUniformBlockVar* pBlock = m_pRes->GetUniformBlock(idxBlock);
        if (pBlock->GetSize() > 0)
        {
            CPUCache::FillZero(pBuffer, Align(pBlock->GetSize(), CACHE_BLOCK_SIZE));
        }

        m_Flag &= ~BRANCH_DIRTY;
    }

    int numDirtyFlag = m_pRes->GetStaticOptionCount();
    memset(m_pDirtyFlagArray, 0, Align(numDirtyFlag, 32) >> 3); // 1 パラメータあたり 1 ビット。

    m_OptionBlock.SetData(pBuffer, bufferSize);
    m_OptionBlock.Setup();
    m_OptionBlock.DCFlush();
    m_Flag |= BLOCK_BUFFER_VALID;

    return true;
}

void ShadingModelObj::CleanupBlockBuffer()
{
    NW_G3D_ASSERTMSG(m_Flag & BLOCK_BUFFER_VALID, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_OptionBlock.Cleanup();
    m_Flag ^= BLOCK_BUFFER_VALID;
}

void ShadingModelObj::CalcOptionBlock()
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRes, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    if ((m_Flag & BRANCH_DIRTY) == 0)
    {
        return;
    }

    int numDirtyFlag = GetStaticOptionCount();
    for (int idxFlag32 = 0, numFlag32 = (numDirtyFlag + 31) >> 5;
        idxFlag32 < numFlag32; ++idxFlag32)
    {
        bit32 flag32 = m_pDirtyFlagArray[idxFlag32];
        while (flag32)
        {
            int bitIndex = 31 - CountLeadingZeros(flag32);
            int optionIndex = (idxFlag32 << 5) + bitIndex;
            ResShaderOption* pOption = GetStaticOption(optionIndex);
            if (pOption->GetBranchOffset() != ResShaderOption::INVALID_OFFSET)
            {
                // シェーダコンパイラがデッドストリップを行う場合があるので判定します。
                // s32 であっても u32 として扱う。
                const u32* pChoiceValues = pOption->ref().ofsChoiceValues.to_ptr<u32>();
                NW_G3D_ASSERT_NOT_NULL_DETAIL(pChoiceValues, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
                int idxChoice = pOption->ReadStaticKey(m_pOptionKey);
                u32* pBranch = AddOffset<u32>(m_OptionBlock.GetData(), pOption->GetBranchOffset());
#if NW_G3D_IS_HOST_CAFE
                StoreRevU32(pBranch, pChoiceValues[idxChoice]);
#else
                *pBranch = pChoiceValues[idxChoice];
#endif
            }
            flag32 ^= 0x1 << bitIndex;
        }
        m_pDirtyFlagArray[idxFlag32] = flag32;
    }

    m_OptionBlock.DCFlush();
    m_Flag ^= BRANCH_DIRTY;
}

void ShadingModelObj::ClearStaticKey()
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRes, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    for (int idxOption = 0, numOption = GetStaticOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        WriteStaticKey(idxOption, pOption->GetDefaultIndex());
    }
}

void ShadingModelObj::WriteStaticKey(int optionIndex, int choiceIndex)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRes, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(optionIndex, GetStaticOptionCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    const ResShaderOption* pOption = GetStaticOption(optionIndex);
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(choiceIndex, pOption->GetChoiceCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (!pOption->IsBranch())
    {
        pOption->WriteStaticKey(m_pShaderKey, choiceIndex);
    }
    else
    {
        // branch が true のオプションは m_pShaderKey をデフォルト値のままにします。
        // UniformBlock に書き込むための値を m_pOptionKey に記録します。
        pOption->WriteStaticKey(m_pOptionKey, choiceIndex);
        if (pOption->GetBranchOffset() != ResShaderOption::INVALID_OFFSET)
        {
            SetDirtyFlag(optionIndex);
        }
    }
}

int ShadingModelObj::ReadStaticKey(int optionIndex) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRes, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(optionIndex, GetStaticOptionCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    const ResShaderOption* pOption = GetStaticOption(optionIndex);
    if (!pOption->IsBranch())
    {
        return pOption->ReadStaticKey(m_pShaderKey);
    }
    else
    {
        // branch が true のオプションは m_pOptionKey に記録されています。
        return pOption->ReadStaticKey(m_pOptionKey);
    }
}

int ShadingModelObj::PrintRawKeyTo(char* pStr, int strLength) const
{
    return ResShadingModel::PrintKeyTo(pStr, strLength, m_pShaderKey, m_pRes->GetStaticKeyLength());
}

int ShadingModelObj::PrintKeyTo(char* pStr, int strLength) const
{
    u32 currentKey[NW_G3D_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, numOption = GetStaticOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        int choice = pOption->ReadStaticKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteStaticKey(currentKey, choice);
    }
    return ResShadingModel::PrintKeyTo(pStr, strLength, currentKey, m_pRes->GetStaticKeyLength());
}

int ShadingModelObj::PrintRawOptionTo(char* pStr, int strLength) const
{
    return m_pRes->PrintStaticOptionTo(pStr, strLength, m_pShaderKey);
}

int ShadingModelObj::PrintOptionTo(char* pStr, int strLength) const
{
    u32 currentKey[NW_G3D_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, numOption = GetStaticOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        int choice = pOption->ReadStaticKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteStaticKey(currentKey, choice);
    }
    return m_pRes->PrintStaticOptionTo(pStr, strLength, currentKey);
}

//--------------------------------------------------------------------------------------------------

void ShaderSelector::Sizer::Calc(const InitArg& arg)
{
    ResShadingModel* pRes = arg.GetShadingModel()->GetResource();
    size_t keySize = pRes->GetDynamicKeyLength() * sizeof(bit32);

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = keySize;
    chunk[idx++].size = keySize;
    chunk[idx++].size = keySize;
    NW_G3D_ASSERTMSG(idx == NUM_CHUNK, "%s\n", NW_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));

    CalcOffset(chunk, NUM_CHUNK);
}

//--------------------------------------------------------------------------------------------------

size_t ShaderSelector::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool ShaderSelector::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pBuffer, "%s\n", NW_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ShadingModelObj* pShadingModel = arg.GetShadingModel();
    ResShadingModel* pRes = pShadingModel->GetResource();

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pShadingModel = pShadingModel;
    m_pShaderKey = sizer.GetPtr<bit32>(ptr, Sizer::SHADER_KEY);
    m_pOptionKey = sizer.GetPtr<bit32>(ptr, Sizer::OPTION_KEY);
    m_pLastShaderKey = sizer.GetPtr<bit32>(ptr, Sizer::LAST_SHADER_KEY);
    m_pBufferPtr = pBuffer;

    int keyLength = m_pShadingModel->GetResource()->GetDynamicKeyLength();
    if (keyLength > 0)
    {
        pRes->WriteDefaultDynamicKey(m_pShaderKey);        // デフォルトキーで初期化。
        std::copy(m_pShaderKey, m_pShaderKey + keyLength, m_pOptionKey);
        pRes->WriteInvalidDynamicKey(m_pLastShaderKey);    // 無効なキーで初期化。
        m_pProgram = NULL;
    }
    else
    {
        // DynamicKey が存在しない場合は StaticKey で一意に決まっているはずです。
        const ShaderRange& range = m_pShadingModel->GetShaderRange();
        int idxProgram = pRes->FindProgramIndex(range, m_pShaderKey);
        NW_G3D_ASSERTMSG(idxProgram != SHADER_PROGRAM_NONE, "%s\n", NW_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));
        m_pProgram = pRes->GetShaderProgram(idxProgram);
    }

    return true;
}

bool ShaderSelector::UpdateVariation()
{
    NW_G3D_ASSERT_NOT_NULL(m_pShadingModel);

    // シェーダキーの dynamic 部分のみを使用します。
    // static 部分は ShadingModelObj で決定します。
    ResShadingModel* pRes = m_pShadingModel->GetResource();
    int keyLength = pRes->GetDynamicKeyLength();
    const bit32* pShaderKey = m_pShaderKey;
    bit32* pLastShaderKey = m_pLastShaderKey;

    if (!std::equal(pShaderKey, pShaderKey + keyLength, pLastShaderKey))
    {
        // 前回のシェーダキーの dynamic 部分に変更があった場合のみ更新を試みます。
        const ShaderRange& range = m_pShadingModel->GetShaderRange();
        int idxProgram = pRes->FindProgramIndex(range, pShaderKey);
        if (idxProgram == SHADER_PROGRAM_NONE)
        {
            // 見つからなかった場合は最後のキーが無効であるとみなします。
            pRes->WriteInvalidDynamicKey(m_pLastShaderKey);
            m_pProgram = NULL;
            return false;
        }
        else
        {
            // 見つかった場合は最後のキーとして記録します。
            std::copy(pShaderKey, pShaderKey + keyLength, pLastShaderKey);
            m_pProgram = pRes->GetShaderProgram(idxProgram);
        }

    }
    // 遅延コンパイルのため更新を試みます。
    // リアルタイム編集のために毎フレーム呼び出します。
    m_pProgram->Update();
    return true;
}

void ShaderSelector::ClearDynamicKey()
{
    NW_G3D_ASSERT_NOT_NULL(m_pShadingModel);
    for (int idxOption = 0, numOption = GetDynamicOptionCount(); idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        WriteDynamicKey(idxOption, pOption->GetDefaultIndex());
    }
}

void ShaderSelector::WriteDynamicKey(int optionIndex, int choiceIndex)
{
    NW_G3D_ASSERT_NOT_NULL(m_pShadingModel);

    const ResShaderOption* pOption = GetDynamicOption(optionIndex);
    if (!pOption->IsBranch())
    {
        pOption->WriteDynamicKey(m_pShaderKey, choiceIndex);
    }
    else
    {
        // branch が true のオプションは m_pShaderKey をデフォルト値のままにします。
        // ここではオプションの情報を記録しますが、シェーダには情報を送りません。
        // 別途情報を送信し、オプションがデフォルトの場合にそれを参照するように
        // シェーダを記述する必要があります。
        pOption->WriteDynamicKey(m_pOptionKey, choiceIndex);
    }
}

int ShaderSelector::ReadDynamicKey(int optionIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(m_pShadingModel);

    const ResShaderOption* pOption = GetDynamicOption(optionIndex);
    if (!pOption->IsBranch())
    {
        return pOption->ReadDynamicKey(m_pShaderKey);
    }
    else
    {
        // branch が true のオプションは m_pOptionKey に記録されています。
        return pOption->ReadDynamicKey(m_pOptionKey);
    }
}

int ShaderSelector::PrintRawKeyTo(char* pStr, int strLength) const
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    return ResShadingModel::PrintKeyTo(pStr, strLength, m_pShaderKey, pRes->GetDynamicKeyLength());
}

int ShaderSelector::PrintKeyTo(char* pStr, int strLength) const
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    u32 currentKey[NW_G3D_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, numOption = GetDynamicOptionCount();
        idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        int choice = pOption->ReadDynamicKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteDynamicKey(currentKey, choice);
    }
    return ResShadingModel::PrintKeyTo(pStr, strLength, currentKey, pRes->GetDynamicKeyLength());
}

int ShaderSelector::PrintRawOptionTo(char* pStr, int strLength) const
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    return pRes->PrintDynamicOptionTo(pStr, strLength, m_pShaderKey);
}

int ShaderSelector::PrintOptionTo(char* pStr, int strLength) const
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    u32 currentKey[NW_G3D_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, numOption = GetDynamicOptionCount();
        idxOption < numOption; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        int choice = pOption->ReadDynamicKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteDynamicKey(currentKey, choice);
    }
    return pRes->PrintDynamicOptionTo(pStr, strLength, currentKey);
}

}} // namespace nw::g3d

