﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// このソースの処理はコピー禁止です。

#include <nw/g3d/fnd/g3d_GfxState.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/ut/g3d_Flag.h>
#include <nw/g3d/math/g3d_MathCommon.h>

#include <algorithm>

#ifndef NW_STRIP_GL
#pragma warning(disable:4100)
using nw::g3d::fnd::detail::GL;
#endif

namespace nw { namespace g3d { namespace fnd {

namespace {

enum RegPolygonCtrl
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  1, CULL_FRONT),
    NW_G3D_FLAG_VALUE_DECLARE( 1,  1, CULL_BACK),
    NW_G3D_FLAG_VALUE_DECLARE( 2,  1, FACE),
    NW_G3D_FLAG_VALUE_DECLARE( 3,  2, POLY_MODE),
    NW_G3D_FLAG_VALUE_DECLARE( 5,  3, POLYMODE_FRONT_PTYPE),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  3, POLYMODE_BACK_PTYPE),
    NW_G3D_FLAG_VALUE_DECLARE(11,  1, POLY_OFFSET_FRONT_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE(12,  1, POLY_OFFSET_BACK_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE(13,  1, POLY_OFFSET_PARA_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 0,  2, CULL_FACE),
    NW_G3D_FLAG_VALUE_DECLARE(11,  2, POLY_OFFSET_ENABLE)
};

enum RegDepthCtrl
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  1, STENCIL_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 1,  1, Z_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 2,  1, Z_WRITE_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 4,  3, ZFUNC),
    NW_G3D_FLAG_VALUE_DECLARE( 7,  1, BACKFACE_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  3, STENCILFUNC),
    NW_G3D_FLAG_VALUE_DECLARE(11,  3, STENCILFAIL),
    NW_G3D_FLAG_VALUE_DECLARE(14,  3, STENCILZPASS),
    NW_G3D_FLAG_VALUE_DECLARE(17,  3, STENCILZFAIL),
    NW_G3D_FLAG_VALUE_DECLARE(20,  3, STENCILFUNC_BF),
    NW_G3D_FLAG_VALUE_DECLARE(23,  3, STENCILFAIL_BF),
    NW_G3D_FLAG_VALUE_DECLARE(26,  3, STENCILZPASS_BF),
    NW_G3D_FLAG_VALUE_DECLARE(29,  3, STENCILZFAIL_BF)
};

enum RegStencilMask
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  8, STENCILREF),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  8, STENCILMASK),
    NW_G3D_FLAG_VALUE_DECLARE(16,  8, STENCILWRITEMASK)
};

enum RegAlphaTest
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  3, ALPHA_FUNC),
    NW_G3D_FLAG_VALUE_DECLARE( 3,  1, ALPHA_TEST_ENABLE)
};

enum RegColorCtrl
{
    NW_G3D_FLAG_VALUE_DECLARE( 1,  1, MULTIWRITE_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE( 4,  3, SPECIAL_OP),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  8, TARGET_BLEND_ENABLE),
    NW_G3D_FLAG_VALUE_DECLARE(16,  8, ROP3)
};

enum RegBlendCtrl
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  5, COLOR_SRCBLEND),
    NW_G3D_FLAG_VALUE_DECLARE( 5,  3, COLOR_COMB_FCN),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  5, COLOR_DESTBLEND),
    NW_G3D_FLAG_VALUE_DECLARE(16,  5, ALPHA_SRCBLEND),
    NW_G3D_FLAG_VALUE_DECLARE(21,  3, ALPHA_COMB_FCN),
    NW_G3D_FLAG_VALUE_DECLARE(24,  5, ALPHA_DESTBLEND),
    NW_G3D_FLAG_VALUE_DECLARE(29,  1, SEPARATE_ALPHA_BLEND)
};

enum RegPointSize
{
    NW_G3D_FLAG_VALUE_DECLARE( 0, 16, HEIGHT),
    NW_G3D_FLAG_VALUE_DECLARE(16, 16, WIDTH)
};

enum RegPointLimits
{
    NW_G3D_FLAG_VALUE_DECLARE( 0, 16, MIN_SIZE),
    NW_G3D_FLAG_VALUE_DECLARE(16, 16, MAX_SIZE)
};

NW_G3D_INLINE
u32 FloatToU12_4(float x)
{
    // GX2 の挙動に合わせて切り捨てる。
    return FastCast<u16>(Math::Clamp(x, 0.0f, float(0xFFFF) / 16) * 16.0f);
}

NW_G3D_INLINE
float U12_4ToFloat(u32 x)
{
    return FastCast<float>(static_cast<u16>(x)) / 16.0f;
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

void GfxPolygonCtrl::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetPolygonControlReg(&gx2PolygonControl);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    u32 cullFace = NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, CULL_FACE, u32);
    if (cullFace)
    {
        glEnable(GL_CULL_FACE);
        NW_G3D_TABLE_FIELD u32 tblCullFace[] = { GL_FRONT, GL_BACK, GL_FRONT_AND_BACK };
        glCullFace(tblCullFace[cullFace - 1]);
    }
    else
    {
        glDisable(GL_CULL_FACE);
    }
    NW_G3D_TABLE_FIELD u32 tblFrontFace[] = { GL_CCW, GL_CW };
    glFrontFace(tblFrontFace[GetFrontFace()]);
#if !defined(NW_G3D_IS_GL_ES)
    if (GetPolygonModeEnable())
    {
        NW_G3D_TABLE_FIELD u32 tblPolygonMode[] = { GL_POINT, GL_LINE, GL_FILL };
        glPolygonMode(GL_FRONT, tblPolygonMode[GetPolygonModeFront()]);
        glPolygonMode(GL_BACK, tblPolygonMode[GetPolygonModeBack()]);
    }
    else
    {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
    if (NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_ENABLE, u32))
    {
        glEnable(GL_POLYGON_OFFSET_FILL);
    }
    else
    {
        glDisable(GL_POLYGON_OFFSET_FILL);
    }
    if (GetPointLineOffsetEnable())
    {
        glEnable(GL_POLYGON_OFFSET_POINT);
        glEnable(GL_POLYGON_OFFSET_LINE);
    }
    else
    {
        glDisable(GL_POLYGON_OFFSET_POINT);
        glDisable(GL_POLYGON_OFFSET_LINE);
    }
#endif
    NW_G3D_GL_ASSERT();
#endif
}

void GfxPolygonCtrl::SetDefault()
{
    // GX2SetPolygonControl(GX2_FRONT_FACE_CCW, GX2_DISABLE, GX2_DISABLE, GX2_DISABLE,
    //     GX2_POLYGON_MODE_TRIANGLE, GX2_POLYGON_MODE_TRIANGLE,
    //     GX2_DISABLE, GX2_DISABLE, GX2_DISABLE);
    gx2PolygonControl.reg = 0x00280240;
}

void GfxPolygonCtrl::SetCullFront(GX2Boolean cull)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, CULL_FRONT, cull);
}

void GfxPolygonCtrl::SetCullBack(GX2Boolean cull)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, CULL_BACK, cull);
}

void GfxPolygonCtrl::SetFrontFace(GX2FrontFaceMode mode)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, FACE, mode);
}

GX2Boolean GfxPolygonCtrl::GetCullFront() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, CULL_FRONT, GX2Boolean);
}

GX2Boolean GfxPolygonCtrl::GetCullBack() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, CULL_BACK, GX2Boolean);
}

GX2FrontFaceMode GfxPolygonCtrl::GetFrontFace() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, FACE, GX2FrontFaceMode);
}

void GfxPolygonCtrl::SetPolygonModeEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLY_MODE, enable);
}

void GfxPolygonCtrl::SetPolygonModeFront(GX2PolygonMode mode)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLYMODE_FRONT_PTYPE, mode);
}

void GfxPolygonCtrl::SetPolygonModeBack(GX2PolygonMode mode)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLYMODE_BACK_PTYPE, mode);
}

GX2Boolean GfxPolygonCtrl::GetPolygonModeEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLY_MODE, GX2Boolean);
}

GX2PolygonMode GfxPolygonCtrl::GetPolygonModeFront() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLYMODE_FRONT_PTYPE, GX2PolygonMode);
}

GX2PolygonMode GfxPolygonCtrl::GetPolygonModeBack() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLYMODE_BACK_PTYPE, GX2PolygonMode);
}

void GfxPolygonCtrl::SetPolygonOffsetFrontEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_FRONT_ENABLE, enable);
}

void GfxPolygonCtrl::SetPolygonOffsetBackEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_BACK_ENABLE, enable);
}

void GfxPolygonCtrl::SetPointLineOffsetEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_PARA_ENABLE, enable);
}

GX2Boolean GfxPolygonCtrl::GetPolygonOffsetFrontEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_FRONT_ENABLE, GX2Boolean);
}

GX2Boolean GfxPolygonCtrl::GetPolygonOffsetBackEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_BACK_ENABLE, GX2Boolean);
}

GX2Boolean GfxPolygonCtrl::GetPointLineOffsetEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2PolygonControl.reg, POLY_OFFSET_PARA_ENABLE, GX2Boolean);
}

//--------------------------------------------------------------------------------------------------

void GfxDepthCtrl::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetDepthStencilControlReg(&gx2DepthStencilControl);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    GL::Enable(GL_DEPTH_TEST, !!GetDepthTestEnable());
    glDepthMask(static_cast<GLboolean>(GetDepthWriteEnable()));
    glDepthFunc(GL_NEVER + GetDepthFunc());

    GL::Enable(GL_STENCIL_TEST, !!GetStencilTestEnable());
    NW_G3D_TABLE_FIELD u32 tblStencilOp[] = {
        GL_KEEP,
        GL_ZERO,
        GL_REPLACE,
        GL_INCR,
        GL_DECR,
        GL_INVERT,
        GL_INCR_WRAP,
        GL_DECR_WRAP
    };
    if (GetBackStencilEnable())
    {
        s32 refValue = 0;
        u32 valueMask = 0;
        glGetIntegerv(GL_STENCIL_REF, &refValue);
        glGetIntegerv(GL_STENCIL_VALUE_MASK, reinterpret_cast<s32*>(&valueMask));
        glStencilFuncSeparate(GL_FRONT, GL_NEVER + GetFrontStencilFunc(), refValue, valueMask);
        glGetIntegerv(GL_STENCIL_BACK_REF, &refValue);
        glGetIntegerv(GL_STENCIL_BACK_VALUE_MASK, reinterpret_cast<s32*>(&valueMask));
        glStencilFuncSeparate(GL_BACK, GL_NEVER + GetBackStencilFunc(), refValue, valueMask);
        glStencilOpSeparate(GL_FRONT, tblStencilOp[GetFrontStencilFail()],
            tblStencilOp[GetFrontStencilZFail()], tblStencilOp[GetFrontStencilZPass()]);
        glStencilOpSeparate(GL_BACK, tblStencilOp[GetFrontStencilFail()],
            tblStencilOp[GetFrontStencilZFail()], tblStencilOp[GetFrontStencilZPass()]);
    }
    else
    {
        s32 refValue = 0;
        u32 mask = 0;
        glGetIntegerv(GL_STENCIL_REF, &refValue);
        glGetIntegerv(GL_STENCIL_VALUE_MASK, reinterpret_cast<s32*>(&mask));
        glStencilFunc(GL_NEVER + GetFrontStencilFunc(), refValue, mask);
        glStencilOp(tblStencilOp[GetFrontStencilFail()],
            tblStencilOp[GetFrontStencilZFail()], tblStencilOp[GetFrontStencilZPass()]);
    }
    NW_G3D_GL_ASSERT();
#endif
}

void GfxDepthCtrl::SetDefault()
{
    //GX2InitDepthStencilControlReg(&reg, GX2_TRUE, GX2_TRUE, GX2_COMPARE_LESS,
    //     GX2_FALSE, GX2_FALSE,
    //     GX2_COMPARE_ALWAYS, GX2_STENCIL_REPLACE, GX2_STENCIL_REPLACE, GX2_STENCIL_REPLACE,
    //     GX2_COMPARE_ALWAYS, GX2_STENCIL_REPLACE, GX2_STENCIL_REPLACE, GX2_STENCIL_REPLACE);
    gx2DepthStencilControl.reg = 0x49749716;
}

void GfxDepthCtrl::SetDepthTestEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, Z_ENABLE, enable);
}

void GfxDepthCtrl::SetDepthWriteEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, Z_WRITE_ENABLE, enable);
}

void GfxDepthCtrl::SetDepthFunc(GX2CompareFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, ZFUNC, func);
}

void GfxDepthCtrl::SetStencilTestEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCIL_ENABLE, enable);
}

void GfxDepthCtrl::SetBackStencilEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, BACKFACE_ENABLE, enable);
}

GX2Boolean GfxDepthCtrl::GetDepthTestEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, Z_ENABLE, GX2Boolean);
}

GX2Boolean GfxDepthCtrl::GetDepthWriteEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, Z_WRITE_ENABLE, GX2Boolean);
}

GX2CompareFunction GfxDepthCtrl::GetDepthFunc() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, ZFUNC, GX2CompareFunction);
}

GX2Boolean GfxDepthCtrl::GetStencilTestEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCIL_ENABLE, GX2Boolean);
}

GX2Boolean GfxDepthCtrl::GetBackStencilEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, BACKFACE_ENABLE, GX2Boolean);
}

void GfxDepthCtrl::SetFrontStencilFunc(GX2CompareFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFUNC, func);
}

void GfxDepthCtrl::SetFrontStencilZPass(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZPASS, func);
}

void GfxDepthCtrl::SetFrontStencilZFail(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZFAIL, func);
}

void GfxDepthCtrl::SetFrontStencilFail(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFAIL, func);
}

GX2CompareFunction GfxDepthCtrl::GetFrontStencilFunc() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFUNC, GX2CompareFunction);
}

GX2StencilFunction GfxDepthCtrl::GetFrontStencilZPass() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZPASS, GX2StencilFunction);
}

GX2StencilFunction GfxDepthCtrl::GetFrontStencilZFail() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZFAIL, GX2StencilFunction);
}

GX2StencilFunction GfxDepthCtrl::GetFrontStencilFail() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFAIL, GX2StencilFunction);
}

void GfxDepthCtrl::SetBackStencilFunc(GX2CompareFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFUNC_BF, func);
}

void GfxDepthCtrl::SetBackStencilZPass(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZPASS_BF, func);
}

void GfxDepthCtrl::SetBackStencilZFail(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZFAIL_BF, func);
}

void GfxDepthCtrl::SetBackStencilFail(GX2StencilFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFAIL_BF, func);
}

GX2CompareFunction GfxDepthCtrl::GetBackStencilFunc() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFUNC_BF, GX2CompareFunction);
}

GX2StencilFunction GfxDepthCtrl::GetBackStencilZPass() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZPASS_BF, GX2StencilFunction);
}

GX2StencilFunction GfxDepthCtrl::GetBackStencilZFail() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILZFAIL_BF, GX2StencilFunction);
}

GX2StencilFunction GfxDepthCtrl::GetBackStencilFail() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthStencilControl.reg, STENCILFAIL_BF, GX2StencilFunction);
}

//--------------------------------------------------------------------------------------------------

void GfxStencilMask::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetStencilMaskReg(&gx2StencilMask);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    u32 func = 0;
    glGetIntegerv(GL_STENCIL_FUNC, reinterpret_cast<s32*>(&func));
    glStencilFuncSeparate(GL_FRONT, func, GetRefValueFront(), GetValueMaskFront());
    glStencilMaskSeparate(GL_FRONT, GetWriteMaskFront());
    glGetIntegerv(GL_STENCIL_BACK_FUNC, reinterpret_cast<s32*>(&func));
    glStencilFuncSeparate(GL_BACK, func, GetRefValueBack(), GetValueMaskBack());
    glStencilMaskSeparate(GL_BACK, GetWriteMaskBack());
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStencilMask::SetDefault()
{
    // GX2SetStencilMask(0xFF, 0xFF, 0x1, 0xFF, 0xFF, 0x1);
    gx2StencilMask.reg[0] = gx2StencilMask.reg[1] = 0x00FFFF01;
}

void GfxStencilMask::SetValueMaskFront(u8 mask)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILMASK, mask);
}

void GfxStencilMask::SetWriteMaskFront(u8 mask)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILWRITEMASK, mask);
}

void GfxStencilMask::SetRefValueFront(u8 value)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILREF, value);
}

u8 GfxStencilMask::GetValueMaskFront() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILMASK, u8);
}

u8 GfxStencilMask::GetWriteMaskFront() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILWRITEMASK, u8);
}

u8 GfxStencilMask::GetRefValueFront() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[0], STENCILREF, u8);
}

void GfxStencilMask::SetValueMaskBack(u8 mask)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILMASK, mask);
}

void GfxStencilMask::SetWriteMaskBack(u8 mask)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILWRITEMASK, mask);
}

void GfxStencilMask::SetRefValueBack(u8 value)
{
    NW_G3D_SET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILREF, value);
}

u8 GfxStencilMask::GetValueMaskBack() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILMASK, u8);
}

u8 GfxStencilMask::GetWriteMaskBack() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILWRITEMASK, u8);
}

u8 GfxStencilMask::GetRefValueBack() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2StencilMask.reg[1], STENCILREF, u8);
}

//--------------------------------------------------------------------------------------------------

void GfxAlphaTest::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetAlphaTestReg(&gx2AlphaTest);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    if (GetAlphaTestEnable())
    {
        glEnable(GL_ALPHA_TEST);
        glAlphaFunc(GL_NEVER + GetAlphaFunc(), GetRefValue());
    }
    else
    {
        glDisable(GL_ALPHA_TEST);
    }
#else
    if (GetAlphaTestEnable())
    {
        NW_G3D_WARNING(false, "NW: alpha test is not supported.");
    }
#endif
    NW_G3D_GL_ASSERT();
#endif
}

void GfxAlphaTest::SetDefault()
{
    // GX2SetAlphaTest(GX2_DISABLE, GX2_COMPARE_LESS, 0.0f);
    gx2AlphaTest.reg[0] = 0x00000001;
    gx2AlphaTest.reg[1] = 0x0;
}

void GfxAlphaTest::SetAlphaTestEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2AlphaTest.reg[0], ALPHA_TEST_ENABLE, enable);
}

void GfxAlphaTest::SetAlphaFunc(GX2CompareFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2AlphaTest.reg[0], ALPHA_FUNC, func);
}

void GfxAlphaTest::SetRefValue(f32 value)
{
    union
    {
        f32 f32;
        u32 u32;
    } reg = { value };
    gx2AlphaTest.reg[1] = reg.u32;
}

GX2Boolean GfxAlphaTest::GetAlphaTestEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2AlphaTest.reg[0], ALPHA_TEST_ENABLE, GX2Boolean);
}

GX2CompareFunction GfxAlphaTest::GetAlphaFunc() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2AlphaTest.reg[0], ALPHA_FUNC, GX2CompareFunction);
}

f32 GfxAlphaTest::GetRefValue() const
{
    union
    {
        u32 u32;
        f32 f32;
    } reg = { gx2AlphaTest.reg[1] };
    return reg.f32;
}

//--------------------------------------------------------------------------------------------------

void GfxColorCtrl::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetColorControlReg(&gx2ColorControl);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    if (GetLogicOp() == GX2_LOGIC_OP_COPY)
    {
        glDisable(GL_COLOR_LOGIC_OP);
    }
    else
    {
        NW_G3D_TABLE_FIELD u32 tblLogicOp[] = {
            GL_CLEAR,
            GL_NOR,
            GL_AND_INVERTED,
            GL_COPY_INVERTED,
            GL_AND_REVERSE,
            GL_INVERT,
            GL_XOR,
            GL_NAND,
            GL_AND,
            GL_EQUIV,
            GL_NOOP,
            GL_OR_INVERTED,
            GL_COPY,
            GL_OR_REVERSE,
            GL_OR,
            GL_SET
        };
        glEnable(GL_COLOR_LOGIC_OP);
        glLogicOp(tblLogicOp[GetLogicOp() & 0xF]);
    }
#endif
    u32 blendEnableMask = GetBlendEnableMask();
    for (int renderTarget = 0; renderTarget < 8; ++renderTarget)
    {
        GL::Enable(GL_BLEND, renderTarget, !!((blendEnableMask >> renderTarget) & 0x1));
    }
    NW_G3D_GL_ASSERT();
#endif
}

void GfxColorCtrl::SetDefault()
{
    // GX2SetColorControl(GX2_LOGIC_OP_COPY, 0, GX2_DISABLE, GX2_ENABLE);
    gx2ColorControl.reg = 0x00CC0000;
}

void GfxColorCtrl::SetLogicOp(GX2LogicOp lop)
{
    NW_G3D_SET_FLAG_VALUE(gx2ColorControl.reg, ROP3, lop);
}

void GfxColorCtrl::SetBlendEnableMask(u8 mask)
{
    NW_G3D_SET_FLAG_VALUE(gx2ColorControl.reg, TARGET_BLEND_ENABLE, mask);
}

void GfxColorCtrl::SetMultiWriteEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2ColorControl.reg, MULTIWRITE_ENABLE, enable);
}

void GfxColorCtrl::SetColorBufferEnable(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2ColorControl.reg, SPECIAL_OP, 1 - enable);
}

GX2LogicOp GfxColorCtrl::GetLogicOp() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2ColorControl.reg, ROP3, GX2LogicOp);
}

u8 GfxColorCtrl::GetBlendEnableMask() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2ColorControl.reg, TARGET_BLEND_ENABLE, u8);
}

GX2Boolean GfxColorCtrl::GetMultiWriteEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2ColorControl.reg, MULTIWRITE_ENABLE, GX2Boolean);
}

GX2Boolean GfxColorCtrl::GetColorBufferEnable() const
{
    return static_cast<GX2Boolean>(!NW_G3D_GET_FLAG_VALUE(gx2ColorControl.reg, SPECIAL_OP, GX2Boolean));
}

//--------------------------------------------------------------------------------------------------

void GfxBlendCtrl::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetBlendControlReg(&gx2BlendControl);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    Load(static_cast<GX2RenderTarget>(gx2BlendControl.reg[0]));
#endif
}

void GfxBlendCtrl::Load(GX2RenderTarget target) const
{
#if NW_G3D_IS_GX2
    GX2BlendControlReg tempReg;

    tempReg.reg[0] = static_cast<u32>(target);
    tempReg.reg[1] = gx2BlendControl.reg[1];

    GX2SetBlendControlReg(&tempReg);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_TABLE_FIELD u32 tblBlendEq[] = {
        GL_FUNC_ADD,
        GL_FUNC_SUBTRACT,
        GL_MIN,
        GL_MAX,
        GL_FUNC_REVERSE_SUBTRACT
    };
    NW_G3D_TABLE_FIELD u32 tblBlendFunc[] = {
        GL_ZERO,
        GL_ONE,
        GL_SRC_COLOR,
        GL_ONE_MINUS_SRC_COLOR,
        GL_SRC_ALPHA,
        GL_ONE_MINUS_SRC_ALPHA,
        GL_DST_ALPHA,
        GL_ONE_MINUS_DST_ALPHA,
        GL_DST_COLOR,
        GL_ONE_MINUS_DST_COLOR,
        GL_SRC_ALPHA_SATURATE,
        GL_NONE,
        GL_NONE,
        GL_CONSTANT_COLOR,
        GL_ONE_MINUS_CONSTANT_COLOR,
#if !defined(NW_G3D_IS_GL_ES)
        GL_SRC1_COLOR,
        GL_ONE_MINUS_SRC1_COLOR,
        GL_SRC1_ALPHA,
        GL_ONE_MINUS_SRC1_ALPHA,
#else
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
#endif
        GL_CONSTANT_ALPHA,
        GL_ONE_MINUS_CONSTANT_ALPHA
    };
    NW_G3D_ASSERTMSG(tblBlendFunc[GetColorDstBlend()] != GL_INVALID_INDEX, "NW: Unsupported blend function.");
    NW_G3D_ASSERTMSG(tblBlendFunc[GetAlphaDstBlend()] != GL_INVALID_INDEX, "NW: Unsupported blend function.");
    if (GetSeparateAlphaBlend())
    {
        GL::BlendEquation(target,
            tblBlendEq[GetColorCombine()], tblBlendEq[GetAlphaCombine()]);
        GL::BlendFunc(target,
            tblBlendFunc[GetColorSrcBlend()], tblBlendFunc[GetColorDstBlend()],
            tblBlendFunc[GetAlphaSrcBlend()], tblBlendFunc[GetAlphaDstBlend()]);
    }
    else
    {
        GL::BlendEquation(target,
            tblBlendEq[GetColorCombine()], tblBlendEq[GetColorCombine()]);
        GL::BlendFunc(target,
            tblBlendFunc[GetColorSrcBlend()], tblBlendFunc[GetColorDstBlend()],
            tblBlendFunc[GetColorSrcBlend()], tblBlendFunc[GetColorDstBlend()]);
    }
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED( target );
#endif
}

void GfxBlendCtrl::SetDefault()
{
    // GX2SetBlendControl(GX2_RENDER_TARGET_0,
    //     GX2_BLEND_SRC_ALPHA, GX2_BLEND_ONE_MINUS_SRC_ALPHA, GX2_BLEND_COMBINE_ADD,
    //     GX2_ENABLE,
    //     GX2_BLEND_SRC_ALPHA, GX2_BLEND_ONE_MINUS_SRC_ALPHA, GX2_BLEND_COMBINE_ADD);
    gx2BlendControl.reg[0] = 0x0;
    gx2BlendControl.reg[1] = 0x25040504;
}

void GfxBlendCtrl::SetColorSrcBlend(GX2BlendFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_SRCBLEND, func);
}

void GfxBlendCtrl::SetColorDstBlend(GX2BlendFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_DESTBLEND, func);
}

void GfxBlendCtrl::SetColorCombine(GX2BlendCombine combine)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_COMB_FCN, combine);
}

void GfxBlendCtrl::SetSeparateAlphaBlend(GX2Boolean enable)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], SEPARATE_ALPHA_BLEND, enable);
}

void GfxBlendCtrl::SetAlphaSrcBlend(GX2BlendFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_SRCBLEND, func);
}

void GfxBlendCtrl::SetAlphaDstBlend(GX2BlendFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_DESTBLEND, func);
}

void GfxBlendCtrl::SetAlphaCombine(GX2BlendCombine combine)
{
    NW_G3D_SET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_COMB_FCN, combine);
}

GX2BlendFunction GfxBlendCtrl::GetColorSrcBlend() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_SRCBLEND, GX2BlendFunction);
}

GX2BlendFunction GfxBlendCtrl::GetColorDstBlend() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_DESTBLEND, GX2BlendFunction);
}

GX2BlendCombine GfxBlendCtrl::GetColorCombine() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], COLOR_COMB_FCN, GX2BlendCombine);
}

GX2Boolean GfxBlendCtrl::GetSeparateAlphaBlend() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], SEPARATE_ALPHA_BLEND, GX2Boolean);
}

GX2BlendFunction GfxBlendCtrl::GetAlphaSrcBlend() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_SRCBLEND, GX2BlendFunction);
}

GX2BlendFunction GfxBlendCtrl::GetAlphaDstBlend() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_DESTBLEND, GX2BlendFunction);
}

GX2BlendCombine GfxBlendCtrl::GetAlphaCombine() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2BlendControl.reg[1], ALPHA_COMB_FCN, GX2BlendCombine);
}

//--------------------------------------------------------------------------------------------------

void GfxBlendColor::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetBlendConstantColorReg(&gx2BlendConstantColor);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    glBlendColor(
        gx2BlendConstantColor.reg[0].f32All, gx2BlendConstantColor.reg[1].f32All,
        gx2BlendConstantColor.reg[2].f32All, gx2BlendConstantColor.reg[3].f32All);
    NW_G3D_GL_ASSERT();
#endif
}

void GfxBlendColor::SetDefault()
{
    memset(gx2BlendConstantColor.reg, 0, sizeof(GX2BlendConstantColorReg));
}

void GfxBlendColor::SetConstantColor(const f32 color[4])
{
    this->gx2BlendConstantColor.reg[0].f32All = color[0];
    this->gx2BlendConstantColor.reg[1].f32All = color[1];
    this->gx2BlendConstantColor.reg[2].f32All = color[2];
    this->gx2BlendConstantColor.reg[3].f32All = color[3];
}

f32* GfxBlendColor::GetConstantColor()
{
    return &gx2BlendConstantColor.reg[0].f32All;
}

const f32* GfxBlendColor::GetConstantColor() const
{
    return &gx2BlendConstantColor.reg[0].f32All;
}

//--------------------------------------------------------------------------------------------------

void GfxPolygonOffset::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetPolygonOffsetReg(&gx2PolygonOffset);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    glPolygonOffset(GetFrontScale(), GetFrontOffset());
    NW_G3D_GL_ASSERT();
#endif
}

void GfxPolygonOffset::SetDefault()
{
    memset(gx2PolygonOffset.reg, 0, sizeof(GX2PolygonOffsetReg));
}

void GfxPolygonOffset::SetFrontScale(f32 scale)
{
    gx2PolygonOffset.reg[0].f32All = scale * 16.0f;
}

void GfxPolygonOffset::SetFrontOffset(f32 offset)
{
    gx2PolygonOffset.reg[1].f32All = offset;
}

void GfxPolygonOffset::SetBackScale(f32 scale)
{
    gx2PolygonOffset.reg[2].f32All = scale * 16.0f;
}

void GfxPolygonOffset::SetBackOffset(f32 offset)
{
    gx2PolygonOffset.reg[3].f32All = offset;
}

void GfxPolygonOffset::SetClamp(f32 clamp)
{
    gx2PolygonOffset.reg[4].f32All = clamp;
}

f32 GfxPolygonOffset::GetFrontScale() const
{
    return gx2PolygonOffset.reg[0].f32All / 16.0f;
}

f32 GfxPolygonOffset::GetFrontOffset() const
{
    return gx2PolygonOffset.reg[1].f32All;
}

f32 GfxPolygonOffset::GetBackScale() const
{
    return gx2PolygonOffset.reg[2].f32All / 16.0f;
}

f32 GfxPolygonOffset::GetBackOffset() const
{
    return gx2PolygonOffset.reg[3].f32All;
}

f32 GfxPolygonOffset::GetClamp() const
{
    return gx2PolygonOffset.reg[4].f32All;
}

//--------------------------------------------------------------------------------------------------

void GfxPointSize::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetPointSizeReg(&gx2PointSize);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    glPointSize(GetWidth());
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Point size is not supported.");
#endif
#endif
}

void GfxPointSize::SetDefault()
{
    // GX2SetPointSize(1.0f, 1.0f);
    gx2PointSize.reg = 0x00080008;
}

void GfxPointSize::SetWidth(f32 width)
{
    u32 val = FloatToU12_4(width * 0.5f);
    NW_G3D_SET_FLAG_VALUE(gx2PointSize.reg, WIDTH, val);
}

void GfxPointSize::SetHeight(f32 height)
{
    u32 val = FloatToU12_4(height * 0.5f);
    NW_G3D_SET_FLAG_VALUE(gx2PointSize.reg, HEIGHT, val);
}

f32 GfxPointSize::GetWidth() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2PointSize.reg, WIDTH, u32);
    return U12_4ToFloat(val) * 2.0f;
}

f32 GfxPointSize::GetHeight() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2PointSize.reg, HEIGHT, u32);
    return U12_4ToFloat(val) * 2.0f;
}

//--------------------------------------------------------------------------------------------------

void GfxPointLimits::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetPointLimitsReg(&gx2PointLimits);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    glPointParameterf(GL_POINT_SIZE_MIN, GetMinSize());
    glPointParameterf(GL_POINT_SIZE_MAX, GetMaxSize());
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Point size limit is not supported.");
#endif
#endif
}

void GfxPointLimits::SetDefault()
{
    // GX2SetPointLimits(0.0f, 8191.875f);
    gx2PointLimits.reg = 0xFFFF0000;
}

void GfxPointLimits::SetMinSize(f32 size)
{
    u32 val = FloatToU12_4(size * 0.5f);
    NW_G3D_SET_FLAG_VALUE(gx2PointLimits.reg, MIN_SIZE, val);
}

void GfxPointLimits::SetMaxSize(f32 size)
{
    u32 val = FloatToU12_4(size * 0.5f);
    NW_G3D_SET_FLAG_VALUE(gx2PointLimits.reg, MAX_SIZE, val);
}

f32 GfxPointLimits::GetMinSize() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2PointLimits.reg, MIN_SIZE, u32);
    return U12_4ToFloat(val) * 2.0f;
}

f32 GfxPointLimits::GetMaxSize() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2PointLimits.reg, MAX_SIZE, u32);
    return U12_4ToFloat(val) * 2.0f;
}

//--------------------------------------------------------------------------------------------------

void GfxLineWidth::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetLineWidthReg(&gx2LineWidth);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    glLineWidth(GetWidth());
    NW_G3D_GL_ASSERT();
#endif
}

void GfxLineWidth::SetDefault()
{
    // GX2SetLineWidth(1.0f);
    gx2LineWidth.reg = 0x00000008;
}

void GfxLineWidth::SetWidth(f32 width)
{
    gx2LineWidth.reg = FloatToU12_4(width * 0.5f);
}

f32 GfxLineWidth::GetWidth() const
{
    return U12_4ToFloat(gx2LineWidth.reg) * 2.0f;
}

//--------------------------------------------------------------------------------------------------

void GfxChannelMasks::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetTargetChannelMasksReg(&gx2TargetChannelMask);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    for (int target = 0; target < 8; ++target)
    {
        GX2ChannelMask mask = GetChannelMask(static_cast<GX2RenderTarget>(target));
        GL::ColorMask(target,
            static_cast<GLboolean>((mask >> 0) & 0x1), static_cast<GLboolean>((mask >> 1) & 0x1),
            static_cast<GLboolean>((mask >> 2) & 0x1), static_cast<GLboolean>((mask >> 3) & 0x1));
    }
    NW_G3D_GL_ASSERT();
#endif
}

void GfxChannelMasks::SetDefault()
{
    gx2TargetChannelMask.reg = 0xFFFFFFFF;
}

void GfxChannelMasks::SetChannelMask(GX2RenderTarget target, GX2ChannelMask mask)
{
    int shift = target * 4;
    gx2TargetChannelMask.reg &= ~(0xF << shift);
    gx2TargetChannelMask.reg |= mask << shift;
}

GX2ChannelMask GfxChannelMasks::GetChannelMask(GX2RenderTarget target) const
{
    int shift = target * 4;
    return static_cast<GX2ChannelMask>((gx2TargetChannelMask.reg >> shift) & 0xF);
}

//--------------------------------------------------------------------------------------------------

void GfxViewport::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetViewport(GetOriginX(), GetOriginY(), GetWidth(), GetHeight(), GetNearZ(), GetFarZ());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    glViewport(static_cast<GLint>(GetOriginX()), static_cast<GLint>(GetOriginY()),
        static_cast<GLsizei>(GetWidth()), static_cast<GLsizei>(GetHeight()));
#if !defined(NW_G3D_IS_GL_ES)
    glDepthRange(GetNearZ(), GetFarZ());
#else
    glDepthRangef(GetNearZ(), GetFarZ());
#endif
    NW_G3D_GL_ASSERT();
#endif
}

void GfxViewport::SetDefault()
{
    SetViewport(0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f);
}

void GfxViewport::SetViewport(
    float originX, float originY, float width, float height, float nearZ, float farZ)
{
    reg[0] = originX;
    reg[1] = originY;
    reg[2] = width;
    reg[3] = height;
    reg[4] = nearZ;
    reg[5] = farZ;
}

float GfxViewport::GetOriginX() const
{
    return reg[0];
}

float GfxViewport::GetOriginY() const
{
    return reg[1];
}

float GfxViewport::GetWidth() const
{
    return reg[2];
}

float GfxViewport::GetHeight() const
{
    return reg[3];
}

float GfxViewport::GetNearZ() const
{
    return reg[4];
}

float GfxViewport::GetFarZ() const
{
    return reg[5];
}

//--------------------------------------------------------------------------------------------------

void GfxScissor::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetScissor(GetOriginX(), GetOriginY(), GetWidth(), GetHeight());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    glEnable(GL_SCISSOR_TEST);
    glScissor(GetOriginX(), GetOriginY(), GetWidth(), GetHeight());
    NW_G3D_GL_ASSERT();
#endif
}

void GfxScissor::SetDefault()
{
    SetScissor(0, 0, 0, 0);
}

void GfxScissor::SetScissor(u32 originX, u32 originY, u32 width, u32 height)
{
    reg[0] = originX;
    reg[1] = originY;
    reg[2] = width;
    reg[3] = height;
}

u32 GfxScissor::GetOriginX() const
{
    return reg[0];
}

u32 GfxScissor::GetOriginY() const
{
    return reg[1];
}

u32 GfxScissor::GetWidth() const
{
    return reg[2];
}

u32 GfxScissor::GetHeight() const
{
    return reg[3];
}

}}} // namespace nw::g3d::fnd
