﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// このソースの処理はコピー禁止です。

#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <functional>
#include <algorithm>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/fnd/g3d_GX2Utility.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/ut/g3d_Flag.h>
#include <nw/g3d/math/g3d_MathCommon.h>

#if !defined( NW_STRIP_GL )
#pragma warning(disable:4100)
using nw::g3d::fnd::detail::GL;
using nw::g3d::fnd::detail::BufferBinder;
using nw::g3d::fnd::detail::TextureBinder;
#endif

namespace nw { namespace g3d { namespace fnd {

namespace {

enum RegDepthBuffer2
{
    NW_G3D_FLAG_VALUE_DECLARE(25,  1, TILE_SURFACE_ENABLE)
};

enum RegSampler0
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  3, CLAMP_X),
    NW_G3D_FLAG_VALUE_DECLARE( 3,  3, CLAMP_Y),
    NW_G3D_FLAG_VALUE_DECLARE( 6,  3, CLAMP_Z),
    NW_G3D_FLAG_VALUE_DECLARE( 9,  2, XY_MAG_FILTER),
    NW_G3D_FLAG_VALUE_DECLARE(12,  2, XY_MIN_FILTER),
    NW_G3D_FLAG_VALUE_DECLARE(15,  2, Z_FILTER),
    NW_G3D_FLAG_VALUE_DECLARE(17,  2, MIP_FILTER),
    NW_G3D_FLAG_VALUE_DECLARE(19,  3, MAX_ANISO_RATIO),
    NW_G3D_FLAG_VALUE_DECLARE(22,  2, BORDER_COLOR_TYPE),
    NW_G3D_FLAG_VALUE_DECLARE(26,  3, DEPTH_COMPARE_FUNCTION)
};

enum RegSampler1
{
    NW_G3D_FLAG_VALUE_DECLARE( 0, 10, MIN_LOD),
    NW_G3D_FLAG_VALUE_DECLARE(10, 10, MAX_LOD),
    NW_G3D_FLAG_VALUE_DECLARE(20, 12, LOD_BIAS)
};

enum RegSampler2
{
    NW_G3D_FLAG_VALUE_DECLARE(30,  1, DEPTH_COMPARE_ENABLE) // PC でのみ使用。実機では必ず 0
};

enum InstCFlow1
{
    NW_G3D_FLAG_VALUE_DECLARE(10,  3, COUNT),
    NW_G3D_FLAG_VALUE_DECLARE(19,  1, COUNT_3)
};

enum InstVFetch0
{
    NW_G3D_FLAG_VALUE_DECLARE( 5,  2, FETCH_TYPE),
    NW_G3D_FLAG_VALUE_DECLARE( 8,  4, BUFFER_ID),
    NW_G3D_FLAG_VALUE_DECLARE(24,  2, SRC_SEL_X),
    NW_G3D_FLAG_VALUE_DECLARE(26,  6, MEGA_FETCH_COUNT)
};

enum InstVFetch1
{
    NW_G3D_FLAG_VALUE_DECLARE( 0,  8, SEMANTIC_ID),
    NW_G3D_FLAG_VALUE_DECLARE( 9,  3, DST_SEL_X),
    NW_G3D_FLAG_VALUE_DECLARE(12,  3, DST_SEL_Y),
    NW_G3D_FLAG_VALUE_DECLARE(15,  3, DST_SEL_Z),
    NW_G3D_FLAG_VALUE_DECLARE(18,  3, DST_SEL_W),
    NW_G3D_FLAG_VALUE_DECLARE(22,  6, DATA_FORMAT),
    NW_G3D_FLAG_VALUE_DECLARE(28,  2, NUM_FORMAT_ALL),
    NW_G3D_FLAG_VALUE_DECLARE(30,  1, FORMAT_COMP_ALL)
};

enum InstVFetch2
{
    NW_G3D_FLAG_VALUE_DECLARE( 0, 16, OFFSET),
    NW_G3D_FLAG_VALUE_DECLARE(16,  2, ENDIAN_SWAP)
};

NW_G3D_FORCE_INLINE
u32 ReadInst(const u32* pInst, int instIndex)
{
    return LoadRevU32(pInst + instIndex);
}

NW_G3D_FORCE_INLINE
void WriteInst(u32* pInst, int instIndex, u32 value)
{
    StoreRevU32(pInst + instIndex, value);
}

NW_G3D_INLINE
u32 FloatToU4_6(float x)
{
    // GX2 の挙動に合わせて切り捨てる。
    return FastCast<u16>(Math::Clamp(x, 0.0f, 13.0f) * 64.0f);
}

NW_G3D_INLINE
float U4_6ToFloat(u32 x)
{
    return FastCast<float>(static_cast<u16>(x)) / 64.0f;
}

NW_G3D_INLINE
u32 FloatToS5_6(float x)
{
    // GX2 の挙動に合わせて切り捨てる。
    return FastCast<s16>(Math::Clamp(x, -32.0f, 31.984375f) * 64.0f);
}

NW_G3D_INLINE
float S5_6ToFloat(u32 x)
{
    union
    {
        u32 u;
        s32 s;
    } x32;
    x32.u = x << 20; // 負数に対応するために上位にシフトしてから戻す。
    return FastCast<float>(static_cast<s16>(x32.s >> 20)) / 64.0f;
}

//--------------------------------------------------------------------------------------------------

#if defined( _MSC_VER )
#pragma warning(push)
#pragma warning(disable:4305)
#pragma warning(disable:4309)
#elif defined( __ghs__ )
#pragma ghs nowarning 69
#endif

#define GX2_SURFACE_FORMAT_T_R32_G32_B32_UINT       0x0000012f
#define GX2_SURFACE_FORMAT_T_R32_G32_B32_SINT       0x0000032f
#define GX2_SURFACE_FORMAT_T_R32_G32_B32_FLOAT      0x00000830

#define NW_G3D_DEF_FMT(attrib, surface, ...) \
    { static_cast<u8>(GX2_ATTRIB_FORMAT_##attrib), static_cast<u8>(GX2_SURFACE_FORMAT_##surface), __VA_ARGS__ }

const struct InternalFormat
{
    u8 attribFormat;    // low byte only
    u8 surfaceFormat;   // low byte only
    u8 fetchSize;       // size of attrib per vertex - 1 in byte
    u8 endianSwap;      // 0: u8, 1: u16, 2: u32
    GX2CompSel compSel;
} s_InternalFormat[] = {
    NW_G3D_DEF_FMT(8_UNORM,             TC_R8_UNORM,                0,  0, GX2_COMP_SEL_X001),
    NW_G3D_DEF_FMT(4_4_UNORM,           T_R4_G4_UNORM,              0,  0, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(16_UNORM,            TCD_R16_UNORM,              1,  1, GX2_COMP_SEL_X001),
    NW_G3D_DEF_FMT(16_FLOAT,            TC_R16_FLOAT,               1,  1, GX2_COMP_SEL_X001),
    NW_G3D_DEF_FMT(8_8_UNORM,           TC_R8_G8_UNORM,             1,  0, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(32_UINT,             TC_R32_UINT,                3,  2, GX2_COMP_SEL_X001),
    NW_G3D_DEF_FMT(32_FLOAT,            TCD_R32_FLOAT,              3,  2, GX2_COMP_SEL_X001),
    NW_G3D_DEF_FMT(16_16_UNORM,         TC_R16_G16_UNORM,           3,  1, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(16_16_FLOAT,         TC_R16_G16_FLOAT,           3,  1, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(10_11_11_FLOAT,      TC_R11_G11_B10_FLOAT,       3,  2, GX2_COMP_SEL_XYZ1),
    NW_G3D_DEF_FMT(8_8_8_8_UNORM,       TCS_R8_G8_B8_A8_UNORM,      3,  0, GX2_COMP_SEL_XYZW),
    NW_G3D_DEF_FMT(10_10_10_2_UNORM,    TCS_R10_G10_B10_A2_UNORM,   3,  2, GX2_COMP_SEL_XYZW),
    NW_G3D_DEF_FMT(32_32_UINT,          TC_R32_G32_UINT,            7,  2, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(32_32_FLOAT,         TC_R32_G32_FLOAT,           7,  2, GX2_COMP_SEL_XY01),
    NW_G3D_DEF_FMT(16_16_16_16_UNORM,   TC_R16_G16_B16_A16_UNORM,   7,  1, GX2_COMP_SEL_XYZW),
    NW_G3D_DEF_FMT(16_16_16_16_FLOAT,   TC_R16_G16_B16_A16_FLOAT,   7,  1, GX2_COMP_SEL_XYZW),
    NW_G3D_DEF_FMT(32_32_32_UINT,       T_R32_G32_B32_UINT,         11, 2, GX2_COMP_SEL_XYZ1),
    NW_G3D_DEF_FMT(32_32_32_FLOAT,      T_R32_G32_B32_FLOAT,        11, 2, GX2_COMP_SEL_XYZ1),
    NW_G3D_DEF_FMT(32_32_32_32_UINT,    TC_R32_G32_B32_A32_UINT,    15, 2, GX2_COMP_SEL_XYZW),
    NW_G3D_DEF_FMT(32_32_32_32_FLOAT,   TC_R32_G32_B32_A32_FLOAT,   15, 2, GX2_COMP_SEL_XYZW)
};

#if !defined( NW_STRIP_GL )

struct GLAttribInternal
{
    u8 attribFormat;    // low byte only
    u8 surfaceFormat;   // low byte only
    u8 size;            // number of component
    u8 reserved;
    u32 type;           // float or unsigned integer
} s_GLAttribInternal[] = {
    NW_G3D_DEF_FMT(8_UNORM,             TC_R8_UNORM,                1,  0,  GL_UNSIGNED_BYTE),
    NW_G3D_DEF_FMT(4_4_UNORM,           T_R4_G4_UNORM,              2,  0,  GL_UNSIGNED_BYTE),
    NW_G3D_DEF_FMT(16_UNORM,            TCD_R16_UNORM,              1,  0,  GL_UNSIGNED_SHORT),
    NW_G3D_DEF_FMT(16_FLOAT,            TC_R16_FLOAT,               1,  0,  GL_HALF_FLOAT),
    NW_G3D_DEF_FMT(8_8_UNORM,           TC_R8_G8_UNORM,             2,  0,  GL_UNSIGNED_BYTE),
    NW_G3D_DEF_FMT(32_UINT,             TC_R32_UINT,                1,  0,  GL_UNSIGNED_INT),
    NW_G3D_DEF_FMT(32_FLOAT,            TCD_R32_FLOAT,              1,  0,  GL_FLOAT),
    NW_G3D_DEF_FMT(16_16_UNORM,         TC_R16_G16_UNORM,           2,  0,  GL_UNSIGNED_SHORT),
    NW_G3D_DEF_FMT(16_16_FLOAT,         TC_R16_G16_FLOAT,           2,  0,  GL_HALF_FLOAT),
    NW_G3D_DEF_FMT(10_11_11_FLOAT,      TC_R11_G11_B10_FLOAT,       3,  0,  GL_UNSIGNED_INT_10F_11F_11F_REV),
    NW_G3D_DEF_FMT(8_8_8_8_UNORM,       TCS_R8_G8_B8_A8_UNORM,      4,  0,  GL_UNSIGNED_BYTE),
    NW_G3D_DEF_FMT(10_10_10_2_UNORM,    TCS_R10_G10_B10_A2_UNORM,   4,  0,  GL_UNSIGNED_INT_2_10_10_10_REV),
    NW_G3D_DEF_FMT(32_32_UINT,          TC_R32_G32_UINT,            2,  0,  GL_UNSIGNED_INT),
    NW_G3D_DEF_FMT(32_32_FLOAT,         TC_R32_G32_FLOAT,           2,  0,  GL_FLOAT),
    NW_G3D_DEF_FMT(16_16_16_16_UNORM,   TC_R16_G16_B16_A16_UNORM,   4,  0,  GL_UNSIGNED_SHORT),
    NW_G3D_DEF_FMT(16_16_16_16_FLOAT,   TC_R16_G16_B16_A16_FLOAT,   4,  0,  GL_HALF_FLOAT),
    NW_G3D_DEF_FMT(32_32_32_UINT,       T_R32_G32_B32_UINT,         3,  0,  GL_UNSIGNED_INT),
    NW_G3D_DEF_FMT(32_32_32_FLOAT,      T_R32_G32_B32_FLOAT,        3,  0,  GL_FLOAT),
    NW_G3D_DEF_FMT(32_32_32_32_UINT,    TC_R32_G32_B32_A32_UINT,    4,  0,  GL_UNSIGNED_INT),
    NW_G3D_DEF_FMT(32_32_32_32_FLOAT,   TC_R32_G32_B32_A32_FLOAT,   4,  0,  GL_FLOAT)
};

#endif // !defined( NW_STRIP_GL )

#if defined( _MSC_VER )
#pragma warning(pop)
#elif defined( __ghs__ )
#pragma ghs endnowarning
#endif

struct InternalCmp : public std::binary_function<InternalFormat, u8, bool>
{
    bool operator ()(const InternalFormat& lhs, u8 rhs) const
    {
        return lhs.surfaceFormat == rhs;
    }
};

const InternalFormat& FindInternalFormat(u8 surfaceFormat)
{

    return *std::find_if(
        s_InternalFormat,
        s_InternalFormat + (sizeof(s_InternalFormat) / sizeof(InternalFormat) - 1),
        std::bind2nd(InternalCmp(), surfaceFormat));
}

#if !defined( NW_STRIP_GL )

struct AttribInternalCmp : public std::binary_function<GLAttribInternal, u8, bool>
{
    bool operator ()(const GLAttribInternal& lhs, u8 rhs) const
    {
        return lhs.surfaceFormat == rhs;
    }
};

const GLAttribInternal& FindGLAttribInternal(u8 surfaceFormat)
{

    return *std::find_if(
        s_GLAttribInternal,
        s_GLAttribInternal + (sizeof(s_GLAttribInternal) / sizeof(GLAttribInternal) - 1),
        std::bind2nd(AttribInternalCmp(), surfaceFormat));
}

struct TexImageArg
{
    GLenum target;
    GLenum detailFormat;
    GLenum format;
    GLenum type;
    GLboolean compressed;
    u32 minWidth;
    u32 minHeight;
    u32 minDepth;
};

void InitTexImageArg(TexImageArg* pArg, const GfxTexture& texture)
{
    const GX2Surface& surface = texture.GetGX2Texture()->surface;
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_TABLE_FIELD GLenum s_tblTarget[] = {
        GL_TEXTURE_1D,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_1D_ARRAY,
        GL_TEXTURE_2D_ARRAY,
        GL_TEXTURE_2D_MULTISAMPLE,
        GL_TEXTURE_2D_MULTISAMPLE_ARRAY
    };
#else
    NW_G3D_TABLE_FIELD GLenum s_tblTarget[] = {
        GL_INVALID_INDEX,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_INVALID_INDEX,
        GL_TEXTURE_2D_ARRAY,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX
    };

    NW_G3D_ASSERTMSG(s_tblTarget[surface.dim] != GL_INVALID_INDEX, "NW: Unsupported texture target. Target id is %d", surface.dim);
#endif
    pArg->target = s_tblTarget[surface.dim];
    if (surface.dim == GX2_SURFACE_DIM_CUBE && texture.arrayLength > 0)
    {
#if !defined(NW_G3D_IS_GL_ES)
        pArg->target = GL_TEXTURE_CUBE_MAP_ARRAY;
#else
        NW_G3D_ASSERTMSG(false, "NW: Unsupported texture target. Target id is %d", surface.dim);
#endif
    }
    GLSurfaceFormat glFormat =
        (surface.use == GX2_SURFACE_USE_DEPTH_BUFFER ||
        surface.use == GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE) ?
        FindGLDepthFormat(surface.format) : FindGLFormat(surface.format);
    pArg->detailFormat = glFormat.detailFormat;
    pArg->format = glFormat.format;
    pArg->type = glFormat.type;

    pArg->compressed = IsCompressed(surface.format);

    pArg->minWidth = pArg->minHeight = pArg->minDepth = 1;
    if (surface.dim == GX2_SURFACE_DIM_1D_ARRAY)
    {
        pArg->minHeight = surface.height;
    }
    else if (surface.dim >= GX2_SURFACE_DIM_CUBE)
    {
        pArg->minDepth = surface.depth;
    }
}

#endif // !defined( NW_STRIP_GL )

enum
{
    ATTRIB_BIT_INT          = 0x100,
    ATTRIB_BIT_SIGNED       = 0x200,
    ATTRIB_BIT_FLOAT        = 0x800,
    ATTRIB_FMT_NORMALIZED   = 0,
    ATTRIB_FMT_INT          = 1,
    ATTRIB_FMT_SCALED       = 2
};

#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
bool IsLayeredSurface(const nw::g3d::fnd::detail::GX2SurfaceData& surface)
{
    // NOTE: GX2SurfaceDim の定義に GX2_SURFACE_DIM_CUBE_ARRAY が追加されたら修正
    return surface.dim == GX2_SURFACE_DIM_3D ||
        surface.dim == GX2_SURFACE_DIM_1D_ARRAY ||
        surface.dim == GX2_SURFACE_DIM_2D_ARRAY ||
        surface.dim == GX2_SURFACE_DIM_2D_MSAA_ARRAY ||
        (surface.dim == GX2_SURFACE_DIM_CUBE && surface.depth > 0);
}
#endif

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

GfxBuffer::GfxBuffer()
{
    memset(this, 0, sizeof(*this));
}

void GfxBuffer::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle == 0)
    {
        glGenBuffers(1, &handle);
        UpdateRegs();
    }
#endif
}

void GfxBuffer::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle != 0)
    {
        glDeleteBuffers(1, &handle);
        handle = 0;
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GfxBuffer::UpdateRegs()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glNamedBufferDataEXT(handle, size * numBuffering, NULL, GL_DYNAMIC_DRAW);
#else
    BufferBinder binder(GL_ARRAY_BUFFER, handle);
    glBufferData(GL_ARRAY_BUFFER, size * numBuffering, NULL, GL_DYNAMIC_DRAW);
#endif // NW_G3D_GL_PORTABILITY
    NW_G3D_GL_ASSERT();
#endif
}

void GfxBuffer::DCFlush(int bufferIndex /*= 0*/) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
#if NW_G3D_IS_GX2
#if NW_G3D_FORCE_PPC_SYNC
    DCFlushRange(GetData(bufferIndex), size);
#else
    DCFlushRangeNoSync(GetData(bufferIndex), size);
#endif // NW_G3D_FORCE_PPC_SYNC
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    u32 offset = stride > 0 ? 0 : size * bufferIndex; // 頂点バッファは範囲指定できないので上書き。
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glNamedBufferSubDataEXT(handle, offset, size, GetData(bufferIndex));
#else
    BufferBinder binder(GL_ARRAY_BUFFER, handle);
    glBufferSubData(GL_ARRAY_BUFFER, offset, size, GetData(bufferIndex));
#endif // NW_G3D_GL_PORTABILITY
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::FlushExportBuffer(int bufferIndex /*= 0*/) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2Invalidate(GX2_INVALIDATE_EXPORT_BUFFER, AddOffset(pData, size * bufferIndex), size);
#else
    NW_G3D_UNUSED(bufferIndex);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    void* pSrc = glMapNamedBufferRangeEXT(handle, 0, size, GL_MAP_READ_BIT);
    memcpy(AddOffset(pData, size * bufferIndex), pSrc, size);
    glUnmapNamedBufferEXT(handle);
#else
#if !defined(NW_G3D_IS_GL_ES)
    BufferBinder binder(GL_SHADER_STORAGE_BUFFER, handle);
    void* pSrc = glMapBufferRange(GL_SHADER_STORAGE_BUFFER, 0, size, GL_MAP_READ_BIT);
    memcpy(AddOffset(pData, size * bufferIndex), pSrc, size);
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER);
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported to flush to exported buffer.");
#endif
#endif // NW_G3D_GL_PORTABILITY
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::DCRefresh()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glGetNamedBufferSubDataEXT(handle, 0, size, pData);
#else
#if !defined(NW_G3D_IS_GL_ES)
    BufferBinder binder(GL_ARRAY_BUFFER, handle);
    glGetBufferSubData(GL_ARRAY_BUFFER, 0, size, pData);
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported to refresh the data.");
#endif
#endif // NW_G3D_GL_PORTABILITY
    NW_G3D_GL_ASSERT();
#endif
}

void GfxBuffer::LoadIndices() const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, handle);
    NW_G3D_GL_ASSERT();
#endif
}

void GfxBuffer::LoadVertices(u32 slot, int bufferIndex /*= 0*/) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
#if NW_G3D_IS_GX2
    GX2SetAttribBuffer(slot, size, stride, GetData(bufferIndex));
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    NW_G3D_UNUSED(slot);
#else
    NW_G3D_UNUSED(slot);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::LoadVertexUniforms(u32 location, int bufferIndex /*= 0*/) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
    NW_G3D_ASSERT(size > 0); // 外側で判定した方が他の処理も省き得るのでここではアサートにします。
#if NW_G3D_IS_GX2
    GX2SetVertexUniformBlock(location, size, GetData(bufferIndex));
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glBindBufferRange(GL_UNIFORM_BUFFER, location, handle, size * bufferIndex, size);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(location);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::LoadGeometryUniforms(u32 location, int bufferIndex /*= 0*/) const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
    NW_G3D_ASSERT(size > 0); // 外側で判定した方が他の処理も省き得るのでここではアサートにします。
    GX2SetGeometryUniformBlock(location, size, GetData(bufferIndex));
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexUniforms(location, bufferIndex);
#else
    NW_G3D_UNUSED(location);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::LoadFragmentUniforms(u32 location, int bufferIndex /*= 0*/) const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
    NW_G3D_ASSERT(size > 0); // 外側で判定した方が他の処理も省き得るのでここではアサートにします。
    GX2SetPixelUniformBlock(location, size, GetData(bufferIndex));
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexUniforms(location, bufferIndex);
#else
    NW_G3D_UNUSED(location);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::LoadComputeUniforms(u32 location, int bufferIndex /*= 0*/) const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
    NW_G3D_ASSERT(size > 0); // 外側で判定した方が他の処理も省き得るのでここではアサートにします。
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2SetComputeUniformBlock(location, size, GetData(bufferIndex));
#else
    NW_G3D_UNUSED(location);
    NW_G3D_UNUSED(bufferIndex);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexUniforms(location, bufferIndex);
#else
    NW_G3D_UNUSED(location);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

void GfxBuffer::LoadStreamOutBuffer(u32 location)
{
#if NW_G3D_IS_GX2
    GX2StreamOutBuffer buffer;
    buffer.size = size;
    buffer.dataPtr = pData;
    buffer.vertexStride = stride;
    GX2SetStreamOutBuffer(location, &buffer);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, location, handle);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(location);
#endif
}

void GfxBuffer::LoadExportBuffer(int bufferIndex /*= 0*/) const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, numBuffering);
    NW_G3D_ASSERT(size > 0); // 外側で判定した方が他の処理も省き得るのでここではアサートにします。
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2SetShaderExportBuffer(AddOffset(pData, bufferIndex * size), size);
#else
    NW_G3D_UNUSED(bufferIndex);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT(handle != 0);
    glBindBufferRange(GL_SHADER_STORAGE_BUFFER, 0, handle, size * bufferIndex, size);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported to load exported buffer.");
#endif
#else
    NW_G3D_UNUSED(bufferIndex);
#endif
}

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

void GfxBuffer::SetData(void* pData, u32 size, int bufferingCount /*= 1*/)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    NW_G3D_ASSERT(bufferingCount > 0);
    this->pData = size > 0 ? pData : NULL;
    this->size = size;
    this->numBuffering = static_cast<u16>(bufferingCount);
#else
    NW_G3D_UNUSED(pData);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(bufferingCount);
#endif
}

NW_G3D_PRAGMA_POP_WARNINGS

void* GfxBuffer::GetData(int bufferIndex /*= 0*/)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    NW_G3D_ASSERT_INDEX_BOUNDS( bufferIndex, numBuffering );
    return AddOffset(pData, size * bufferIndex);
#else
    NW_G3D_UNUSED(bufferIndex);
    return NULL;
#endif
}

const void* GfxBuffer::GetData(int bufferIndex /*= 0*/) const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    NW_G3D_ASSERT_INDEX_BOUNDS( bufferIndex, numBuffering );
    return AddOffset(pData, size * bufferIndex);
#else
    NW_G3D_UNUSED(bufferIndex);
    return NULL;
#endif
}

void GfxBuffer::SetStreamOutContext(GX2StreamOutContext* pCtx)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    pCtxPtr = pCtx;
#else
    NW_G3D_UNUSED(pCtx);
#endif
}

GX2StreamOutContext* GfxBuffer::GetStreamOutContext()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return pCtxPtr;
#else
    return NULL;
#endif
}

const GX2StreamOutContext* GfxBuffer::GetStreamOutContext() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return pCtxPtr;
#else
    return NULL;
#endif
}

//--------------------------------------------------------------------------------------------------

GfxSampler::GfxSampler()
{
    SetDefault();
    handle = 0;
}

void GfxSampler::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle == 0)
    {
        glGenSamplers(1, &handle);
        NW_G3D_GL_ASSERT();
        UpdateRegs();
    }
#endif
}

void GfxSampler::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle != 0)
    {
        glDeleteSamplers(1, &handle);
        handle = 0;
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GfxSampler::UpdateRegs()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    // clamp
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_TABLE_FIELD u32 tblClamp[] = {
        GL_REPEAT,
        GL_MIRRORED_REPEAT,
        GL_CLAMP_TO_EDGE,
        GL_MIRROR_CLAMP_TO_EDGE_EXT,
        GL_CLAMP,
        GL_MIRROR_CLAMP_EXT,
        GL_CLAMP_TO_BORDER,
        GL_MIRROR_CLAMP_TO_BORDER_EXT
    };
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_S, tblClamp[GetClampX()]);
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_T, tblClamp[GetClampY()]);
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_R, tblClamp[GetClampZ()]);
#else
    NW_G3D_TABLE_FIELD u32 tblClamp[] = {
        GL_REPEAT,
        GL_MIRRORED_REPEAT,
        GL_CLAMP_TO_EDGE,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX
    };
    NW_G3D_ASSERTMSG(tblClamp[GetClampX()] != GL_INVALID_INDEX, "NW: Unsupported wrap mode.");
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_S, tblClamp[GetClampX()]);
    NW_G3D_ASSERTMSG(tblClamp[GetClampY()] != GL_INVALID_INDEX, "NW: Unsupported wrap mode.");
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_T, tblClamp[GetClampY()]);
    NW_G3D_ASSERTMSG(tblClamp[GetClampZ()] != GL_INVALID_INDEX, "NW: Unsupported wrap mode.");
    glSamplerParameteri(handle, GL_TEXTURE_WRAP_R, tblClamp[GetClampZ()]);
#endif
    GX2TexBorderType borderType = GetBorderType();
    if (borderType < GX2_TEX_BORDER_USE_REGISTER)
    {
#if !defined(NW_G3D_IS_GL_ES)
        NW_G3D_TABLE_FIELD float tblBorderColor[][4] = {
            { 0.0f, 0.0f, 0.0f, 0.0f },
            { 0.0f, 0.0f, 0.0f, 1.0f },
            { 1.0f, 1.0f, 1.0f, 1.0f }
        };
        glSamplerParameterfv(handle, GL_TEXTURE_BORDER_COLOR, tblBorderColor[borderType]);
#else
    NW_G3D_WARNING(false, "NW: Unsupported border color.");
#endif
    }

    // filter
    NW_G3D_TABLE_FIELD u32 tblFilter[] = {
        GL_NEAREST,
        GL_LINEAR,
        GL_NEAREST_MIPMAP_NEAREST,
        GL_LINEAR_MIPMAP_NEAREST,
        GL_NEAREST_MIPMAP_LINEAR,
        GL_LINEAR_MIPMAP_LINEAR
    };
    NW_G3D_TABLE_FIELD u32 tblAniso[] = {
        1, 2, 4, 8, 16
    };
    glSamplerParameteri(handle, GL_TEXTURE_MAG_FILTER, tblFilter[GetMagFilter()]);
    glSamplerParameteri(handle, GL_TEXTURE_MIN_FILTER,
        tblFilter[GetMipFilter() << 1 | GetMinFilter()]);
#if !defined(NW_G3D_IS_GL_ES)
    glSamplerParameteri(handle, GL_TEXTURE_MAX_ANISOTROPY_EXT, tblAniso[GetMaxAniso()]);
#endif
    // LOD
    glSamplerParameterf(handle, GL_TEXTURE_MIN_LOD, GetMinLOD());
    glSamplerParameterf(handle, GL_TEXTURE_MAX_LOD, GetMaxLOD());
#if !defined(NW_G3D_IS_GL_ES)
    glSamplerParameterf(handle, GL_TEXTURE_LOD_BIAS, GetLODBias());
#else
    NW_G3D_WARNING(false, "NW: Unsupported LOD bias.");
#endif
    // compare
    if (GetCompareEnable())
    {
        NW_G3D_TABLE_FIELD u32 tblCompare[] = {
            GL_NEVER,
            GL_LESS,
            GL_EQUAL,
            GL_LEQUAL,
            GL_GREATER,
            GL_NOTEQUAL,
            GL_GEQUAL,
            GL_ALWAYS
        };
        glSamplerParameteri(handle, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE);
        glSamplerParameteri(handle, GL_TEXTURE_COMPARE_FUNC, tblCompare[GetCompareFunc()]);
    }
    else
    {
        glSamplerParameteri(handle, GL_TEXTURE_COMPARE_MODE, GL_NONE);
    }
    NW_G3D_GL_ASSERT();
#endif
}

void GfxSampler::LoadVertexSampler(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetVertexSampler(GetGX2Sampler(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glBindSampler(unit, handle);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxSampler::LoadGeometrySampler(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetGeometrySampler(GetGX2Sampler(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexSampler(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxSampler::LoadFragmentSampler(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetPixelSampler(GetGX2Sampler(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexSampler(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxSampler::LoadComputeSampler(u32 unit) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2SetComputeSampler(GetGX2Sampler(), unit);
#else
    NW_G3D_UNUSED(unit);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexSampler(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxSampler::SetDefault()
{
    // GX2InitSampler(&sampler, GX2_TEX_CLAMP_WRAP, GX2_TEX_XY_FILTER_POINT);
    // GX2InitSamplerZMFilter(&sampler, GX2_TEX_Z_FILTER_USE_XY, GX2_TEX_MIP_FILTER_LINEAR);
    gx2Sampler.samplerReg[0] = 0x02040000;
    gx2Sampler.samplerReg[1] = 0x000FFC00;
    gx2Sampler.samplerReg[2] = 0x80000000;
}

void GfxSampler::SetClampX(GX2TexClamp clamp)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_X, clamp);
}

void GfxSampler::SetClampY(GX2TexClamp clamp)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_Y, clamp);
}

void GfxSampler::SetClampZ(GX2TexClamp clamp)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_Z, clamp);
}

void GfxSampler::SetBorderType(GX2TexBorderType type)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], BORDER_COLOR_TYPE, type);
}

GX2TexClamp GfxSampler::GetClampX() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_X, GX2TexClamp);
}

GX2TexClamp GfxSampler::GetClampY() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_Y, GX2TexClamp);
}

GX2TexClamp GfxSampler::GetClampZ() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], CLAMP_Z, GX2TexClamp);
}

GX2TexBorderType GfxSampler::GetBorderType() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], BORDER_COLOR_TYPE, GX2TexBorderType);
}

void GfxSampler::SetMagFilter(GX2TexXYFilterType filter)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], XY_MAG_FILTER, filter);
}

void GfxSampler::SetMinFilter(GX2TexXYFilterType filter)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], XY_MIN_FILTER, filter);
}

void GfxSampler::SetZFilter(GX2TexZFilterType filter)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], Z_FILTER, filter);
}

void GfxSampler::SetMipFilter(GX2TexMipFilterType filter)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], MIP_FILTER, filter);
}

void GfxSampler::SetMaxAniso(GX2TexAnisoRatio ratio)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], MAX_ANISO_RATIO, ratio);
    const bit32 anisoMask = 0x00004800;
    if (ratio == GX2_TEX_ANISO_1_TO_1)
    {
        gx2Sampler.samplerReg[0] &= ~anisoMask;
    }
    else
    {
        gx2Sampler.samplerReg[0] |= anisoMask;
    }
}

GX2TexXYFilterType GfxSampler::GetMagFilter() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], XY_MAG_FILTER, GX2TexXYFilterType);
}

GX2TexXYFilterType GfxSampler::GetMinFilter() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], XY_MIN_FILTER, GX2TexXYFilterType);
}

GX2TexZFilterType GfxSampler::GetZFilter() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], Z_FILTER, GX2TexZFilterType);
}

GX2TexMipFilterType GfxSampler::GetMipFilter() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], MIP_FILTER, GX2TexMipFilterType);
}

GX2TexAnisoRatio GfxSampler::GetMaxAniso() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0], MAX_ANISO_RATIO, GX2TexAnisoRatio);
}

void GfxSampler::SetMinLOD(float minLOD)
{
    u32 val = FloatToU4_6(minLOD);
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[1], MIN_LOD, val);
}

void GfxSampler::SetMaxLOD(float maxLOD)
{
    u32 val = FloatToU4_6(maxLOD);
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[1], MAX_LOD, val);
}

void GfxSampler::SetLODBias(float bias)
{
    u32 val = FloatToS5_6(bias);
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[1], LOD_BIAS, val);
}

float GfxSampler::GetMinLOD() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[1], MIN_LOD, u32);
    return U4_6ToFloat(val);
}

float GfxSampler::GetMaxLOD() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[1], MAX_LOD, u32);
    return U4_6ToFloat(val);
}

float GfxSampler::GetLODBias() const
{
    u32 val = NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[1], LOD_BIAS, u32);
    return S5_6ToFloat(val);
}

void GfxSampler::SetCompareEnable(GX2Boolean enable)
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[2], DEPTH_COMPARE_ENABLE, enable);
#else
    NW_G3D_UNUSED(enable);
#endif
}

GX2Boolean GfxSampler::GetCompareEnable() const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[2], DEPTH_COMPARE_ENABLE, GX2Boolean);
#else
    return GX2_FALSE;
#endif
}

void GfxSampler::SetCompareFunc(GX2CompareFunction func)
{
    NW_G3D_SET_FLAG_VALUE(gx2Sampler.samplerReg[0], DEPTH_COMPARE_FUNCTION, func);
}

GX2CompareFunction GfxSampler::GetCompareFunc() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2Sampler.samplerReg[0],
        DEPTH_COMPARE_FUNCTION, GX2CompareFunction);
}

//--------------------------------------------------------------------------------------------------

GfxTexture::GfxTexture()
{
    SetDefault();
    handle = 0;
}

void GfxTexture::Setup()
{
#if NW_G3D_IS_GX2
    UpdateRegs();
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle == 0)
    {
        glGenTextures(1, &handle);
        NW_G3D_GL_ASSERT();
        UpdateRegs();
    }
#endif
}

void GfxTexture::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle)
    {
        glDeleteTextures(1, &handle);
        handle = 0;
        NW_G3D_GL_ASSERT();
    }
#endif
}
void GfxTexture::UpdateRegs()
{
#if NW_G3D_IS_GX2
    GX2InitTextureRegs(GetGX2Texture());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);

    const GX2Texture& texture = *GetGX2Texture();
    const GX2Surface& surface = texture.surface;

    TexImageArg arg;
    InitTexImageArg(&arg, *this);

    for (u32 mipLevel = 0, numMips = surface.numMips; mipLevel < numMips; ++mipLevel)
    {
        GLsizei width = std::max<u32>(arg.minWidth, surface.width >> mipLevel);
        GLsizei height = std::max<u32>(arg.minHeight, surface.height >> mipLevel);
        GLsizei depth = std::max<u32>(arg.minDepth, surface.depth >> mipLevel);
        GLsizei imageSize = CalcImageSize(surface.format, width, height, depth);
        GL::TexImage(handle, arg.target, mipLevel, arg.detailFormat, width, height, depth,
            arg.format, arg.type, arg.compressed, imageSize, NULL);
    }

    NW_G3D_TABLE_FIELD GLenum tblSwizzle[] = {
        GL_RED, GL_GREEN, GL_BLUE, GL_ALPHA, GL_ZERO, GL_ONE
    };
    GX2CompSel compSel = texture.compSel;
    GLint swizzle[] = {
        static_cast<GLint>(tblSwizzle[GX2_GET_COMPONENT_X_R(compSel)]),
        static_cast<GLint>(tblSwizzle[GX2_GET_COMPONENT_Y_G(compSel)]),
        static_cast<GLint>(tblSwizzle[GX2_GET_COMPONENT_Z_B(compSel)]),
        static_cast<GLint>(tblSwizzle[GX2_GET_COMPONENT_W_A(compSel)])
    };

    NW_G3D_GL_BIND_TEXTURE(arg.target, handle);

    GL::TexParameter(handle, arg.target, GL_TEXTURE_BASE_LEVEL, texture.viewFirstMip);
    GL::TexParameter(handle, arg.target, GL_TEXTURE_MAX_LEVEL,
        texture.viewFirstMip + texture.viewNumMips - 1);
#if !defined(NW_G3D_IS_GL_ES)
    GL::TexParameter(handle, arg.target, GL_TEXTURE_SWIZZLE_RGBA, swizzle);
#else
    GL::TexParameter(handle, arg.target, GL_TEXTURE_SWIZZLE_R, swizzle[0]);
    GL::TexParameter(handle, arg.target, GL_TEXTURE_SWIZZLE_G, swizzle[1]);
    GL::TexParameter(handle, arg.target, GL_TEXTURE_SWIZZLE_B, swizzle[2]);
    GL::TexParameter(handle, arg.target, GL_TEXTURE_SWIZZLE_A, swizzle[3]);
    NW_G3D_GL_ASSERT();
#endif
#endif
}

void GfxTexture::DCFlush() const
{
#if NW_G3D_IS_GX2
#if NW_G3D_FORCE_PPC_SYNC
    DCFlushRange(GetBasePtr(), GetBaseSize());
    if (GetMipPtr())
    {
        DCFlushRange(GetMipPtr(), GetMipSize());
    }
#else
    DCFlushRangeNoSync(GetBasePtr(), GetBaseSize());
    if (GetMipPtr())
    {
        DCFlushRangeNoSync(GetMipPtr(), GetMipSize());
    }
#endif // NW_G3D_FORCE_PPC_SYNC
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);

    const GX2Texture& texture = *GetGX2Texture();
    const GX2Surface& surface = texture.surface;

    TexImageArg arg;
    InitTexImageArg(&arg, *this);

    for (u32 mipLevel = 0, numMips = surface.numMips; mipLevel < numMips; ++mipLevel)
    {
        GLsizei width = std::max<u32>(arg.minWidth, surface.width >> mipLevel);
        GLsizei height = std::max<u32>(arg.minHeight, surface.height >> mipLevel);
        GLsizei depth = std::max<u32>(arg.minDepth, surface.depth >> mipLevel);
        GLsizei imageSize = CalcImageSize(surface.format, width, height, depth);
        const GLvoid* imagePtr = GetImagePtr(mipLevel);
        GL::TexSubImage(handle, arg.target, mipLevel, arg.detailFormat, width, height, depth,
            arg.format, arg.type, arg.compressed, imageSize, imagePtr);
    }

    NW_G3D_GL_ASSERT();
#endif
}

void GfxTexture::DCRefresh()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);

    GX2Texture& texture = *GetGX2Texture();
    GX2Surface& surface = texture.surface;

    TexImageArg arg;
    InitTexImageArg(&arg, *this);

    for (u32 mipLevel = 0, numMips = surface.numMips; mipLevel < numMips; ++mipLevel)
    {
        GLsizei width = std::max<u32>(arg.minWidth, surface.width >> mipLevel);
        GLsizei height = std::max<u32>(arg.minHeight, surface.height >> mipLevel);
        GLsizei depth = std::max<u32>(arg.minDepth, surface.depth >> mipLevel);
        GLsizei imageSize = CalcImageSize(surface.format, width, height, depth);
        GLvoid* imagePtr = GetImagePtr(mipLevel);
        GL::GetTexImage(handle, arg.target, mipLevel, arg.format, arg.type, imageSize, imagePtr);
    }

    NW_G3D_GL_ASSERT();
#endif
}

void GfxTexture::CalcSize()
{
#if NW_G3D_IS_GX2
    GX2CalcSurfaceSizeAndAlignment(&GetGX2Texture()->surface);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    CalcSurfaceSizeAndAlignment(GetGX2Texture()->surface);
#endif
}

void GfxTexture::SetDefault()
{
    GX2Texture* pTexture = GetGX2Texture();
    GX2Surface& surface = pTexture->surface;
    surface.dim = GX2_SURFACE_DIM_2D;
    surface.width = 1;
    surface.height = 1;
    surface.depth = 1;
    surface.numMips = 1;
    surface.format = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
    surface.aa = GX2_AA_MODE_1X;
    surface.use = GX2_SURFACE_USE_COLOR_BUFFER_TEXTURE;
    surface.imageSize = 0;
    surface.imagePtr = NULL;
    surface.mipSize = 0;
    surface.mipPtr = NULL;
    surface.tileMode = GX2_TILE_MODE_DEFAULT;
    surface.swizzle = 0;
    surface.alignment = 0;
    surface.pitch = 0;
    std::memset(surface.mipOffset, 0, sizeof(surface.mipOffset));
    pTexture->viewFirstMip = 0;
    pTexture->viewNumMips = 1;
    pTexture->viewFirstSlice = 0;
    pTexture->viewNumSlices = 1;
    pTexture->compSel = GX2_COMP_SEL_XYZW;
    this->arrayLength = 0;
}

void GfxTexture::SetImagePtrs(void* basePtr, void* mipPtr)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    GX2Texture* pTexture = GetGX2Texture();
    pTexture->surface.imagePtr = basePtr;
    if (pTexture->surface.numMips > 1 && mipPtr == NULL && basePtr) {
        pTexture->surface.mipPtr = AddOffset(basePtr, pTexture->surface.mipOffset[0]);
    } else {
        pTexture->surface.mipPtr = mipPtr;
    }
#else
    NW_G3D_UNUSED(basePtr);
    NW_G3D_UNUSED(mipPtr);
    gx2Texture.surface.imagePtr = 0;
    gx2Texture.surface.mipPtr = 0;
#endif
}

void* GfxTexture::GetBasePtr()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2Texture()->surface.imagePtr;
#else
    return NULL;
#endif
}

const void* GfxTexture::GetBasePtr() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2Texture()->surface.imagePtr;
#else
    return NULL;
#endif
}

void* GfxTexture::GetMipPtr()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2Texture()->surface.mipPtr;
#else
    return NULL;
#endif
}

const void* GfxTexture::GetMipPtr() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2Texture()->surface.mipPtr;
#else
    return NULL;
#endif
}

void* GfxTexture::GetImagePtr(int mipLevel)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return fnd::GetImagePtr(GetGX2Texture()->surface, mipLevel);
#else
    NW_G3D_UNUSED(mipLevel);
    return NULL;
#endif
}

const void* GfxTexture::GetImagePtr(int mipLevel) const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return fnd::GetImagePtr(GetGX2Texture()->surface, mipLevel);
#else
    NW_G3D_UNUSED(mipLevel);
    return NULL;
#endif
}

void GfxTexture::LoadVertexTexture(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetVertexTexture(GetGX2Texture(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_TABLE_FIELD u32 tblDimension[] = {
        GL_TEXTURE_1D,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_1D_ARRAY,
        GL_TEXTURE_2D_ARRAY,
        GL_TEXTURE_2D_MULTISAMPLE,
        GL_TEXTURE_2D_MULTISAMPLE_ARRAY
    };
#else
    NW_G3D_TABLE_FIELD u32 tblDimension[] = {
        GL_INVALID_INDEX,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_INVALID_INDEX,
        GL_TEXTURE_2D_ARRAY,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX
    };
#endif
    NW_G3D_ASSERTMSG(tblDimension[GetDimension()] != GL_INVALID_INDEX, "NW: Unsupported texture target.");
    u32 dimension = tblDimension[GetDimension()];

    // NOTE: GX2SurfaceDim の定義に GX2_SURFACE_DIM_CUBE_ARRAY が追加されたら修正
    if (GetDimension() == GX2_SURFACE_DIM_CUBE &&
        arrayLength > 0)
    {
#if !defined(NW_G3D_IS_GL_ES)
        dimension = GL_TEXTURE_CUBE_MAP_ARRAY;
#else
        NW_G3D_ASSERTMSG(false, "NW: Unsupported texture target.");
#endif
    }
    GL::BindTexture(unit, dimension, handle);

    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxTexture::LoadGeometryTexture(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetGeometryTexture(GetGX2Texture(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexTexture(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxTexture::LoadFragmentTexture(u32 unit) const
{
#if NW_G3D_IS_GX2
    GX2SetPixelTexture(GetGX2Texture(), unit);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexTexture(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

void GfxTexture::LoadComputeTexture(u32 unit) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2SetComputeTexture(GetGX2Texture(), unit);
#else
    NW_G3D_UNUSED(unit);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    LoadVertexTexture(unit);
#else
    NW_G3D_UNUSED(unit);
#endif
}

//--------------------------------------------------------------------------------------------------

GfxColorBuffer::GfxColorBuffer()
{
    memset(this, 0, sizeof(*this));
}

void GfxColorBuffer::UpdateRegs()
{
#if NW_G3D_IS_GX2
    GX2InitColorBufferRegs(GetGX2ColorBuffer());
#endif
}

void GfxColorBuffer::UpdateRegsFTV()
{
#if NW_G3D_IS_GX2
    GX2Surface& surface = GetGX2ColorBuffer()->surface;
    GX2AAMode aa = surface.aa;
    GX2InitColorBufferRegs(GetGX2ColorBuffer());
    if (surface.aa != aa)
    {
        GX2Surface paddedSurf = surface;
        paddedSurf.aa = aa;
        paddedSurf.use = GX2_SURFACE_USE_COLOR_BUFFER_TEXTURE;
        GX2CalcSurfaceSizeAndAlignment(&paddedSurf);
        NW_G3D_ASSERT(surface.imageSize <= paddedSurf.imageSize);
        NW_G3D_ASSERT(surface.alignment <= paddedSurf.alignment);
        surface.imageSize = paddedSurf.imageSize;
        surface.alignment = paddedSurf.alignment;
    }
#endif
}

void GfxColorBuffer::CalcAuxSize()
{
#if NW_G3D_IS_GX2
    if (GetGX2ColorBuffer()->surface.aa == GX2_AA_MODE_1X)
    {
        gx2ColorBuffer.auxSize = alignmentAux = 0;
    }
    else
    {
        GX2CalcColorBufferAuxInfo(GetGX2ColorBuffer(), &gx2ColorBuffer.auxSize, &alignmentAux);
    }
#else
    gx2ColorBuffer.auxSize = 0;
#endif
}

void GfxColorBuffer::Load(GX2RenderTarget target)
{
#if NW_G3D_IS_GX2
    GX2SetColorBuffer(GetGX2ColorBuffer(), target);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    if (IsLayeredSurface(gx2ColorBuffer.surface))
    {
        glFramebufferTextureLayer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + target,
            handle, gx2ColorBuffer.viewMip, gx2ColorBuffer.viewFirstSlice);
    }
    else
    {
#if !defined(NW_G3D_IS_GL_ES)
        glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + target, handle,
            gx2ColorBuffer.viewMip);
#else
        if (gx2ColorBuffer.surface.dim == GX2_SURFACE_DIM_2D)
        {
            glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + target, GL_TEXTURE_2D, handle,
            gx2ColorBuffer.viewMip);
        }
        else
        {
            NW_G3D_ASSERTMSG(false, "NW: Unsupported texture target for framebuffer.");
        }
#endif
    }
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED( target );
#endif
}

void GfxColorBuffer::Load(GX2RenderTarget target, u32 hFrameBuffer)
{
#if NW_G3D_IS_GX2
    NW_G3D_UNUSED(hFrameBuffer);
    GX2SetColorBuffer(GetGX2ColorBuffer(), target);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(hFrameBuffer != 0);
    NW_G3D_ASSERT(handle != 0);
#if NW_G3D_GL_PORTABILITY >= NW_G3D_GL_LEVEL2
    glBindFramebuffer(GL_FRAMEBUFFER, hFrameBuffer);
#endif
    if (IsLayeredSurface(gx2ColorBuffer.surface))
    {
        GL::FrameBufferTexture(hFrameBuffer, GL_COLOR_ATTACHMENT0 + target,
            handle, gx2ColorBuffer.viewMip, gx2ColorBuffer.viewFirstSlice);
    }
    else
    {
        GL::FrameBufferTexture(
            hFrameBuffer, GL_COLOR_ATTACHMENT0 + target, handle, gx2ColorBuffer.viewMip);
    }
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED( target );
    NW_G3D_UNUSED( hFrameBuffer );
#endif
}

void GfxColorBuffer::Clear(float r, float g, float b, float a)
{
#if NW_G3D_IS_GX2
    GX2ClearColor(GetGX2ColorBuffer(), r, g, b, a);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    system.ctx.Activate();
    float color[4] = { r, g, b, a };
    int layer = -1;
    int endLayer = 0;
    if (IsLayeredSurface(gx2ColorBuffer.surface))
    {
        layer = gx2ColorBuffer.viewFirstSlice;
        endLayer = static_cast<int>(gx2ColorBuffer.viewFirstSlice + gx2ColorBuffer.viewNumSlices);
    }
    for (; layer < endLayer; ++layer)
    {
        system.BindFrameBuffer(this, layer);
        glClearBufferfv(GL_COLOR, 0, color);
    }

    NW_G3D_GL_ASSERT();
    GfxContext::Invalidate(); // GX2 の挙動に合わせて無効化する。
#else
    NW_G3D_UNUSED( r );
    NW_G3D_UNUSED( g );
    NW_G3D_UNUSED( b );
    NW_G3D_UNUSED( a );
#endif
}

void GfxColorBuffer::ExpandAux()
{
#if NW_G3D_IS_GX2
    GX2ExpandAAColorBuffer(GetGX2ColorBuffer());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    GfxContext::Invalidate();
#endif
}

void GfxColorBuffer::Resolve(GfxTexture* pTexture, u32 mipLevel, u32 slice) const
{
    NW_G3D_ASSERT_NOT_NULL(pTexture);
#if NW_G3D_IS_GX2
    GX2ResolveAAColorBuffer(
        GetGX2ColorBuffer(), &pTexture->GetGX2Texture()->surface, mipLevel, slice);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    system.ctx.Activate();

    const nw::g3d::fnd::detail::SystemShader& shader =
            system.SelectCopyShader(gx2ColorBuffer.surface.dim, true);
    glUseProgram(shader.hShader);
    if (IsLayeredSurface(gx2ColorBuffer.surface))
    {
        NW_G3D_ASSERT(pTexture->gx2Texture.surface.depth >= slice + gx2ColorBuffer.viewFirstSlice);

        for (int endLayer = gx2ColorBuffer.viewFirstSlice + gx2ColorBuffer.viewNumSlices,
            layer = gx2ColorBuffer.viewFirstSlice; layer < endLayer; ++layer)
        {
            system.BindFrameBuffer(pTexture, mipLevel, slice + layer);
            system.BindTexture(handle, shader.texUnit, mipLevel, gx2ColorBuffer.surface.dim);
            glUniform1i(shader.locations[shader.LOCATION_SLICE], layer);
            system.DrawScreenQuad();
        }
    }
    else
    {
        system.BindFrameBuffer(pTexture, mipLevel, slice);
        system.BindTexture(handle, shader.texUnit, mipLevel);
        system.DrawScreenQuad();
    }

    NW_G3D_GL_ASSERT();
    GfxContext::Invalidate(); // GX2 の挙動に合わせて無効化する。
#else
    NW_G3D_UNUSED( pTexture );
    NW_G3D_UNUSED( mipLevel );
    NW_G3D_UNUSED( slice );
#endif
}

void GfxColorBuffer::SetTexture(GfxTexture* pTexture)
{
    NW_G3D_ASSERT_NOT_NULL(pTexture);
    memcpy(&gx2ColorBuffer.surface, &pTexture->GetGX2Texture()->surface, sizeof(GX2Surface));
    gx2ColorBuffer.surface.use = GX2_SURFACE_USE_COLOR_BUFFER_TEXTURE;
    gx2ColorBuffer.viewMip = 0;
    gx2ColorBuffer.viewFirstSlice = pTexture->GetGX2Texture()->viewFirstSlice;
    gx2ColorBuffer.viewNumSlices = pTexture->GetGX2Texture()->viewNumSlices;
#if NW_G3D_IS_GL
    handle = pTexture->handle;
#endif
}

void GfxColorBuffer::SetImagePtrs(GfxTexture* pTexture)
{
    NW_G3D_ASSERT(0 == memcmp(&pTexture->GetGX2Texture()->surface,
        &GetGX2ColorBuffer()->surface, sizeof(u32) * 7));

    GetGX2ColorBuffer()->surface.imagePtr = pTexture->GetBasePtr();
    GetGX2ColorBuffer()->surface.mipPtr = pTexture->GetMipPtr();
#if NW_G3D_IS_GL
    handle = pTexture->handle;
#endif
}

void GfxColorBuffer::SetAuxPtr(void* auxPtr)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    GetGX2ColorBuffer()->auxPtr = auxPtr;
#else
    NW_G3D_UNUSED( auxPtr );
    gx2ColorBuffer.auxPtr = NULL;
#endif
}

void* GfxColorBuffer::GetAuxPtr()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2ColorBuffer()->auxPtr;
#else
    return NULL;
#endif
}

const void* GfxColorBuffer::GetAuxPtr() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2ColorBuffer()->auxPtr;
#else
    return NULL;
#endif
}

u32 GfxColorBuffer::GetAuxSize() const
{
#if NW_G3D_IS_GX2
    return gx2ColorBuffer.auxSize;
#else
    return 0;
#endif
}

u32 GfxColorBuffer::GetAuxAlignment() const
{
#if NW_G3D_IS_GX2
    return alignmentAux;
#else
    return 0;
#endif
}

//--------------------------------------------------------------------------------------------------

GfxDepthBuffer::GfxDepthBuffer()
{
    memset(this, 0, sizeof(*this));
}

void GfxDepthBuffer::UpdateRegs()
{
#if NW_G3D_IS_GX2
    GX2InitDepthBufferRegs(GetGX2DepthBuffer());
#endif
}

void GfxDepthBuffer::CalcHiZSize()
{
#if NW_G3D_IS_GX2
    GX2CalcDepthBufferHiZInfo(GetGX2DepthBuffer(), &gx2DepthBuffer.hiZSize, &alignmentHiZ);
#else
    gx2DepthBuffer.hiZSize = 0;
#endif
}

void GfxDepthBuffer::DCFlush() const
{
#if NW_G3D_IS_GX2
#if NW_G3D_FORCE_PPC_SYNC
    DCFlushRange(GetHiZPtr(), GetHiZSize());
#else
    DCFlushRangeNoSync(GetHiZPtr(), GetHiZSize());
#endif // NW_G3D_FORCE_PPC_SYNC
#endif
}

void GfxDepthBuffer::Load()
{
#if NW_G3D_IS_GX2
    GX2SetDepthBuffer(GetGX2DepthBuffer());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    GX2SurfaceFormat format = GetGX2DepthBuffer()->surface.format;
    GLenum attachment =
        format == GX2_SURFACE_FORMAT_D_D24_S8_UNORM ||
        format == GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24 ?
        GL_DEPTH_STENCIL_ATTACHMENT : GL_DEPTH_ATTACHMENT;
    if (IsLayeredSurface(gx2DepthBuffer.surface))
    {
        glFramebufferTextureLayer(GL_FRAMEBUFFER, attachment, handle,
            gx2DepthBuffer.viewMip, gx2DepthBuffer.viewFirstSlice);
    }
    else
    {
#if !defined(NW_G3D_IS_GL_ES)
        glFramebufferTexture(GL_FRAMEBUFFER, attachment, handle, gx2DepthBuffer.viewMip);
#else
        if (gx2DepthBuffer.surface.dim == GX2_SURFACE_DIM_2D)
        {
            glFramebufferTexture2D(GL_FRAMEBUFFER, attachment, GL_TEXTURE_2D, handle,
            gx2DepthBuffer.viewMip);
        }
        else
        {
            NW_G3D_ASSERTMSG(false, "NW: Unsupported texture target for framebuffer.");
        }
#endif
    }
    NW_G3D_GL_ASSERT();
#endif
}

void GfxDepthBuffer::Load(u32 hFrameBuffer)
{
#if NW_G3D_IS_GX2
    NW_G3D_UNUSED(hFrameBuffer);
    GX2SetDepthBuffer(GetGX2DepthBuffer());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(hFrameBuffer != 0);
    NW_G3D_ASSERT(handle != 0);
    GX2SurfaceFormat format = GetGX2DepthBuffer()->surface.format;
    GLenum attachment =
        format == GX2_SURFACE_FORMAT_D_D24_S8_UNORM ||
        format == GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24 ?
        GL_DEPTH_STENCIL_ATTACHMENT : GL_DEPTH_ATTACHMENT;
#if NW_G3D_GL_PORTABILITY >= NW_G3D_GL_LEVEL2
    glBindFramebuffer(GL_FRAMEBUFFER, hFrameBuffer);
#endif
    if (IsLayeredSurface(gx2DepthBuffer.surface))
    {
        GL::FrameBufferTexture(hFrameBuffer, attachment, handle,
            gx2DepthBuffer.viewMip, gx2DepthBuffer.viewFirstSlice);
    }
    else
    {
        GL::FrameBufferTexture(hFrameBuffer, attachment, handle, gx2DepthBuffer.viewMip);
    }
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED( hFrameBuffer );
#endif
}

#if defined(__clang__)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wswitch"
#endif

void GfxDepthBuffer::Clear(GX2ClearMode mode)
{
#if NW_G3D_IS_GX2
    GX2ClearDepthStencilEx(
        GetGX2DepthBuffer(), gx2DepthBuffer.clearDepth, gx2DepthBuffer.clearStencil, mode);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    system.ctx.Activate();

    int iStencil = gx2DepthBuffer.clearStencil;
    int layer = -1;
    int endLayer = 0;
    if (IsLayeredSurface(gx2DepthBuffer.surface))
    {
        layer = gx2DepthBuffer.viewFirstSlice;
        endLayer = static_cast<int>(gx2DepthBuffer.viewFirstSlice + gx2DepthBuffer.viewNumSlices);
    }
    for (; layer < endLayer; ++layer)
    {
        system.BindFrameBuffer(this, layer);
        switch (mode)
        {
        case GX2_CLEAR_DEPTH:
            glClearBufferfv(GL_DEPTH, 0, &gx2DepthBuffer.clearDepth);
            break;
        case GX2_CLEAR_STENCIL:
            glClearBufferiv(GL_STENCIL, 0, &iStencil);
            break;
        case GX2_CLEAR_BOTH:
            glClearBufferfi(GL_DEPTH_STENCIL, 0, gx2DepthBuffer.clearDepth, iStencil);
            break;
        }
    }

    NW_G3D_GL_ASSERT();
    GfxContext::Invalidate(); // GX2 の挙動に合わせて無効化する。
#else
    NW_G3D_UNUSED( mode );
#endif
}

#if defined(__clang__)
#pragma clang diagnostic pop
#endif

void GfxDepthBuffer::ExpandHiZ()
{
#if NW_G3D_IS_GX2
    GX2ExpandDepthBuffer(GetGX2DepthBuffer());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    GfxContext::Invalidate();
#endif
}

void GfxDepthBuffer::ConvertToTexture(GfxTexture* pTexture, u32 mipLevel, u32 slice) const
{
    NW_G3D_ASSERT_NOT_NULL(pTexture);
#if NW_G3D_IS_GX2
    GX2ConvertDepthBufferToTextureSurface(
        GetGX2DepthBuffer(), &pTexture->GetGX2Texture()->surface, mipLevel, slice);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    system.ctx.Activate();

    const nw::g3d::fnd::detail::SystemShader& shader =
            system.SelectCopyShader(gx2DepthBuffer.surface.dim, true);
    glUseProgram(shader.hShader);
    if (IsLayeredSurface(gx2DepthBuffer.surface))
    {
        NW_G3D_ASSERT(pTexture->gx2Texture.surface.depth >= slice + gx2DepthBuffer.viewFirstSlice);

        for (int endLayer = gx2DepthBuffer.viewFirstSlice + gx2DepthBuffer.viewNumSlices,
            layer = gx2DepthBuffer.viewFirstSlice; layer < endLayer; ++layer)
        {
            system.BindFrameBuffer(pTexture, mipLevel, slice + layer);
            system.BindTexture(handle, shader.texUnit, mipLevel, gx2DepthBuffer.surface.dim);
            glUniform1i(shader.locations[shader.LOCATION_SLICE], layer);
            system.DrawScreenQuad();
        }
    }
    else
    {
        system.BindFrameBuffer(pTexture, mipLevel, slice);
        system.BindTexture(handle, shader.texUnit, mipLevel);
        system.DrawScreenQuad();
    }

    NW_G3D_GL_ASSERT();
    GfxContext::Invalidate(); // GX2 の挙動に合わせて無効化する。
#else
    NW_G3D_UNUSED( pTexture );
    NW_G3D_UNUSED( mipLevel );
    NW_G3D_UNUSED( slice );
#endif
}

void GfxDepthBuffer::SetTexture(GfxTexture* pTexture)
{
    NW_G3D_ASSERT_NOT_NULL(pTexture);
    memcpy(&gx2DepthBuffer.surface, &pTexture->GetGX2Texture()->surface, sizeof(gx2DepthBuffer.surface));
    gx2DepthBuffer.surface.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
    gx2DepthBuffer.viewMip = 0;
    gx2DepthBuffer.viewFirstSlice = pTexture->GetGX2Texture()->viewFirstSlice;
    gx2DepthBuffer.viewNumSlices = pTexture->GetGX2Texture()->viewNumSlices;
#if NW_G3D_IS_GL
    handle = pTexture->handle;
#endif
}

void GfxDepthBuffer::SetImagePtrs(GfxTexture* pTexture)
{
    NW_G3D_ASSERT(0 == memcmp(&pTexture->GetGX2Texture()->surface,
        &GetGX2DepthBuffer()->surface, sizeof(u32) * 7));

    GetGX2DepthBuffer()->surface.imagePtr = pTexture->GetBasePtr();
    GetGX2DepthBuffer()->surface.mipPtr = pTexture->GetMipPtr();
#if NW_G3D_IS_GL
    handle = pTexture->handle;
#endif
}

void GfxDepthBuffer::SetHiZPtr(void* hiZPtr)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    GetGX2DepthBuffer()->hiZPtr = hiZPtr;
#else
    NW_G3D_UNUSED( hiZPtr );
    gx2DepthBuffer.hiZPtr = NULL;
#endif
}

void* GfxDepthBuffer::GetHiZPtr()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2DepthBuffer()->hiZPtr;
#else
    return NULL;
#endif
}

const void* GfxDepthBuffer::GetHiZPtr() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2DepthBuffer()->hiZPtr;
#else
    return NULL;
#endif
}

u32 GfxDepthBuffer::GetHiZSize() const
{
#if NW_G3D_IS_GX2
    return gx2DepthBuffer.hiZSize;
#else
    return 0;
#endif
}

u32 GfxDepthBuffer::GetHiZAlignment() const
{
#if NW_G3D_IS_GX2
    return alignmentHiZ;
#else
    return 0;
#endif
}

void GfxDepthBuffer::SetHiZEnable(GX2Boolean enable)
{
    //GX2InitDepthBufferHiZEnable(GetGX2DepthBuffer(), enable);
    NW_G3D_SET_FLAG_VALUE(gx2DepthBuffer._regs[2], TILE_SURFACE_ENABLE, enable);

}

GX2Boolean GfxDepthBuffer::GetHiZEnable() const
{
    return NW_G3D_GET_FLAG_VALUE(gx2DepthBuffer._regs[2], TILE_SURFACE_ENABLE, GX2Boolean);
}

void GfxDepthBuffer::SetClearDepthStencil(float depth, u8 stencil)
{
#if NW_G3D_IS_GX2
    GX2SetClearDepthStencil(GetGX2DepthBuffer(), depth, stencil);
#else
    gx2DepthBuffer.clearDepth = depth;
    gx2DepthBuffer.clearStencil = stencil;
#endif
}

float GfxDepthBuffer::GetClearDepth() const
{
    return gx2DepthBuffer.clearDepth;
}

u8 GfxDepthBuffer::GetClearStencil() const
{
    return static_cast<u8>(gx2DepthBuffer.clearStencil);
}

//--------------------------------------------------------------------------------------------------

GfxStreamOut::GfxStreamOut()
{
    memset(this, 0, sizeof(*this));
}

void GfxStreamOut::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle == 0)
    {
        glGenTransformFeedbacks(1, &handle);
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GfxStreamOut::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle != 0)
    {
        glDeleteTransformFeedbacks(1, &handle);
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GfxStreamOut::Load()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, handle);
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStreamOut::Begin()
{
#if NW_G3D_IS_GX2
    GX2StreamOutBuffer buffer;
    for (int location = 0; location < GX2_MAX_STREAMOUT_BUFFERS; ++location)
    {
        if (GfxBuffer* pBuffer = GetStreamOutBuffer(location))
        {
            buffer.ctxPtr = pBuffer->GetStreamOutContext();
            GX2SetStreamOutContext(location, &buffer, GX2_TRUE);
        }
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    const GLPrimitiveType& type = FindGLPrimitiveType(primType);
    NW_G3D_ASSERT(type.varyingType != 0xFFFFFFFF);
    glBeginTransformFeedback(type.varyingType);
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStreamOut::End()
{
#if NW_G3D_IS_GX2
    GX2StreamOutBuffer buffer;
    for (int location = 0; location < GX2_MAX_STREAMOUT_BUFFERS; ++location)
    {
        if (GfxBuffer* pBuffer = GetStreamOutBuffer(location))
        {
            buffer.ctxPtr = pBuffer->GetStreamOutContext();
            GX2SaveStreamOutContext(location, &buffer);
        }
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glEndTransformFeedback();
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStreamOut::Pause()
{
#if NW_G3D_IS_GX2
    End();
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glPauseTransformFeedback();
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStreamOut::Resume()
{
#if NW_G3D_IS_GX2
    GX2StreamOutBuffer buffer;
    for (int location = 0; location < GX2_MAX_STREAMOUT_BUFFERS; ++location)
    {
        if (GfxBuffer* pBuffer = GetStreamOutBuffer(location))
        {
            buffer.ctxPtr = pBuffer->GetStreamOutContext();
            GX2SetStreamOutContext(location, &buffer, GX2_FALSE);
        }
    }
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glResumeTransformFeedback();
    NW_G3D_GL_ASSERT();
#endif
}

void GfxStreamOut::Draw(u32 location)
{
    NW_G3D_ASSERT_NOT_NULL(GetStreamOutBuffer(location));
#if NW_G3D_IS_GX2
    GfxBuffer* pBuffer = GetStreamOutBuffer(location);
    GX2StreamOutBuffer buffer;
    buffer.size = pBuffer->GetSize();
    buffer.dataPtr = pBuffer->GetData();
    buffer.vertexStride = pBuffer->GetStride();
    buffer.ctxPtr = pBuffer->GetStreamOutContext();
    GX2DrawStreamOut(primType, &buffer);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT(handle != 0);
    const GLPrimitiveType& type = FindGLPrimitiveType(primType);
    NW_G3D_ASSERT(type.drawType != GL_NONE);
    glDrawTransformFeedbackStream(type.drawType, handle, location);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported stream out.");
#endif
#else
    NW_G3D_UNUSED( location );
#endif
}

void GfxStreamOut::SetStreamOutBuffer(u32 location, GfxBuffer* pBuffer)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(location, GX2_MAX_STREAMOUT_BUFFERS);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    pStream[location] = pBuffer;
#else
    NW_G3D_UNUSED( pBuffer );
    streamUIntPtr[location] = 0;
#endif
}

GfxBuffer* GfxStreamOut::GetStreamOutBuffer(u32 location)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(location, GX2_MAX_STREAMOUT_BUFFERS);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return pStream[location];
#else
    NW_G3D_UNUSED(location);
    return NULL;
#endif
}

const GfxBuffer* GfxStreamOut::GetStreamOutBuffer(u32 location) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(location, GX2_MAX_STREAMOUT_BUFFERS);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return pStream[location];
#else
    NW_G3D_UNUSED(location);
    return NULL;
#endif
}


//--------------------------------------------------------------------------------------------------

GfxFetchShader::GfxFetchShader()
{
    memset(this, 0, sizeof(*this));
}

void GfxFetchShader::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3
    if (handle == 0)
    {
        glGenVertexArrays(1, &handle);
        NW_G3D_GL_ASSERT();
        UpdateRegs();
    }
#endif // NW_G3D_GL_PORTABILITY
#endif
}

void GfxFetchShader::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3
    if (handle)
    {
        glDeleteVertexArrays(1, &handle);
        handle = 0;
        NW_G3D_GL_ASSERT();
    }
#endif // NW_G3D_GL_PORTABILITY
#endif
}

void GfxFetchShader::UpdateRegs()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    // ストライドが 0 のバッファを固定頂点属性として扱う。
    bit32 fixedAttribs = 0;
    for (u32 idxAttrib = 0; idxAttrib < gx2FetchShader.num_attribs; ++idxAttrib)
    {
        const GfxBuffer* pBuffer = GetVertexBuffer(idxAttrib);
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        if (pBuffer->stride == 0)
        {
            fixedAttribs |= 0x1 << idxAttrib;
        }
    }
    gx2FetchShader._regs[0] = fixedAttribs; // この領域を流用する。
#endif
}

void GfxFetchShader::DCFlush() const
{
#if NW_G3D_IS_GX2
#if NW_G3D_FORCE_PPC_SYNC
    DCFlushRange(GetShaderPtr(), GetShaderSize());
#else
    DCFlushRangeNoSync(GetShaderPtr(), GetShaderSize());
#endif // NW_G3D_FORCE_PPC_SYNC
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3
    NW_G3D_ASSERT(handle != 0);
    LoadVertexAttribArray();
    NW_G3D_GL_ASSERT();
#endif  // NW_G3D_GL_PORTABILITY
#endif
}

void GfxFetchShader::CalcSize()
{
    u32 sizeCFInst = static_cast<u32>((Align(gx2FetchShader.num_attribs, MAX_INST_PER_FETCH_CLAUSE) /
        MAX_INST_PER_FETCH_CLAUSE + 1) * CF_INST_SIZE); // 終端命令の分 +1 する。
    ofsVFInst = static_cast<u32>(Align(sizeCFInst, FETCH_INST_ALIGNMENT));
    gx2FetchShader.shaderSize = ofsVFInst + gx2FetchShader.num_attribs * VF_INST_SIZE;
}

void GfxFetchShader::Load() const
{
#if NW_G3D_IS_GX2
    GX2SetFetchShader(GetGX2FetchShader());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3
    NW_G3D_ASSERT(handle != 0);
    glBindVertexArray(handle);
#else
    // ストリームアウトバッファが残っているとエラーになるので明示的に外します。
    for (GLuint location = 0; location < 16; ++location)
    {
        glDisableVertexAttribArray(location);
    }
    LoadVertexAttribArray();
#endif // NW_G3D_GL_PORTABILITY
    LoadVertexAttribValue();
    NW_G3D_GL_ASSERT();
#endif
}

void GfxFetchShader::SetVertexBuffer(int attribIndex, const GfxBuffer* pBuffer)
{
#if NW_G3D_IS_GX2
    (void)pBuffer;
    u32* pVFetchInst = GetVFInst(attribIndex);
    pVFetchInst[3] = 0;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    // 64 ビット版では問題があるので GL と一緒にはがしておく。
    u32* pVFetchInst = GetVFInst(attribIndex);
    pVFetchInst[3] = reinterpret_cast<u32>(pBuffer); // この領域は空いているはず。
#else
    NW_G3D_UNUSED(attribIndex);
    NW_G3D_UNUSED(pBuffer);
#endif
}

const GfxBuffer* GfxFetchShader::GetVertexBuffer(int attribIndex) const
{
#if NW_G3D_IS_GX2
    const u32* pVFetchInst = GetVFInst(attribIndex);
    return reinterpret_cast<const GfxBuffer*>(pVFetchInst[3]);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    // 64 ビット版では問題があるので GL と一緒にはがしておく。
    const u32* pVFetchInst = GetVFInst(attribIndex);
    return reinterpret_cast<const GfxBuffer*>(pVFetchInst[3]);
#else
    NW_G3D_UNUSED(attribIndex);
    return NULL;
#endif
}

void GfxFetchShader::ReplaceVertexBuffer(u32 slot, const GfxBuffer* pBuffer) const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    if (pBuffer->stride == 0)
    {
        return;
    }
    for (u32 idxAttrib = 0; idxAttrib < gx2FetchShader.num_attribs; ++idxAttrib)
    {
        if (slot == GetBufferSlot(idxAttrib))
        {
            LoadVertexAttribArray(idxAttrib, pBuffer);
        }
    }
    glBindVertexArray(handle);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(slot);
    NW_G3D_UNUSED(pBuffer);
#endif
}

void GfxFetchShader::ResetVertexBuffer() const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3
    NW_G3D_ASSERT(handle != 0);
    LoadVertexAttribArray();
    NW_G3D_GL_ASSERT();
#endif // NW_G3D_GL_PORTABILITY
#endif
}

void GfxFetchShader::SetShaderPtr(void* ptr)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    NW_G3D_ASSERT_ADDR_ALIGNMENT(ptr, SHADER_ALIGNMENT);
    GetGX2FetchShader()->shaderPtr = ptr;
#else
    NW_G3D_UNUSED(ptr);
    gx2FetchShader.shaderPtr = 0;
#endif
}

void* GfxFetchShader::GetShaderPtr()
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2FetchShader()->shaderPtr;
#else
    return NULL;
#endif
}

const void* GfxFetchShader::GetShaderPtr() const
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return GetGX2FetchShader()->shaderPtr;
#else
    return NULL;
#endif
}

void GfxFetchShader::SetDivisors(u32 divisor2, u32 divisor3)
{
    u32 numDivisor = 0;
    if (divisor2 > 0)
    {
        ++numDivisor;
    }
    if (divisor3 > 0)
    {
        ++numDivisor;
    }
    gx2FetchShader._num_divisors = numDivisor;
    gx2FetchShader._divisors[0] = divisor2;
    gx2FetchShader._divisors[1] = divisor3;
}

void GfxFetchShader::SetDefault(void* pShader)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pShader, SHADER_ALIGNMENT);

    u32 numAttrib = gx2FetchShader.num_attribs;
    u32 numCFlowInst = (numAttrib + MAX_INST_PER_FETCH_CLAUSE - 1) >> 4;
    u32* pCFlowInst = static_cast<u32*>(pShader);
    for (u32 idxCFlowInst = 0; idxCFlowInst < numCFlowInst; ++idxCFlowInst)
    {
        u32 inst0 = (ofsVFInst + idxCFlowInst * MAX_INST_PER_FETCH_CLAUSE * VF_INST_SIZE) >> 3;
        WriteInst(pCFlowInst++, 0, inst0);
        if (numAttrib > (idxCFlowInst + 1) * MAX_INST_PER_FETCH_CLAUSE)
        {
            WriteInst(pCFlowInst++, 0, 0x01881C00);
        }
        else
        {
            u32 inst1 = 0x01800000;
            NW_G3D_SET_FLAG_VALUE(inst1, COUNT, (numAttrib - 1));
            NW_G3D_SET_FLAG_VALUE(inst1, COUNT_3, (numAttrib - 1) >> 3);
            WriteInst(pCFlowInst++, 0, inst1);
        }
    }
    WriteInst(pCFlowInst++, 0, 0);
    WriteInst(pCFlowInst++, 0, 0x8A000000);
    memset(pCFlowInst, 0, CF_INST_SIZE); // pCFlowInst += 2;

    for (u32 idxAttrib = 0; idxAttrib < numAttrib; ++idxAttrib)
    {
        // GX2InitAttribStream(&attribs[i], 0, 0, 0, GX2_ATTRIB_FORMAT_32_32_32_32_FLOAT);
        // attribs[i].dest_sel = GX2_COMP_SEL_XYZW
        u32* pVFetchInst = GetVFInst(pShader, idxAttrib);
        WriteInst(pVFetchInst, 0, 0x3C00A001);
        WriteInst(pVFetchInst, 1, 0x28D64800);
        WriteInst(pVFetchInst, 2, 0x000A0000);
        WriteInst(pVFetchInst, 3, 0);
    }
}

void GfxFetchShader::SetLocation(void* pShader, int attribIndex, u32 location)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst1 = ReadInst(pVFetchInst, 1);
    NW_G3D_SET_FLAG_VALUE(inst1, SEMANTIC_ID, location);
    WriteInst(pVFetchInst, 1, inst1);
}

u32 GfxFetchShader::GetLocation(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst1 = ReadInst(pVFetchInst, 1);
    return NW_G3D_GET_FLAG_VALUE(inst1, SEMANTIC_ID, u32);
}

void GfxFetchShader::SetBufferSlot(void* pShader, int attribIndex, u32 slot)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    NW_G3D_ASSERT_INDEX_BOUNDS(slot, GX2_MAX_ATTRIB_BUFFERS);
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst0 = ReadInst(pVFetchInst, 0);
    NW_G3D_SET_FLAG_VALUE(inst0, BUFFER_ID, slot);
    WriteInst(pVFetchInst, 0, inst0);
}

u32 GfxFetchShader::GetBufferSlot(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst0 = ReadInst(pVFetchInst, 0);
    return NW_G3D_GET_FLAG_VALUE(inst0, BUFFER_ID, u32);
}

void GfxFetchShader::SetFormat(void* pShader, int attribIndex, GX2AttribFormat format)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    const InternalFormat& fmt = s_InternalFormat[format & 0x3F];

    u32 inst0 = ReadInst(pVFetchInst, 0);
    NW_G3D_SET_FLAG_VALUE(inst0, MEGA_FETCH_COUNT, fmt.fetchSize);
    WriteInst(pVFetchInst, 0, inst0);

    u32 inst1 = ReadInst(pVFetchInst, 1);
    u32 fmtType = (format & ATTRIB_BIT_FLOAT) ? ATTRIB_FMT_SCALED :
        (format & ATTRIB_BIT_INT) ? ATTRIB_FMT_INT : ATTRIB_FMT_NORMALIZED;
    NW_G3D_SET_FLAG_VALUE(inst1, NUM_FORMAT_ALL, fmtType);
    NW_G3D_SET_FLAG_VALUE(inst1, DST_SEL_X, GX2_GET_COMPONENT_X_R(fmt.compSel));
    NW_G3D_SET_FLAG_VALUE(inst1, DST_SEL_Y, GX2_GET_COMPONENT_Y_G(fmt.compSel));
    NW_G3D_SET_FLAG_VALUE(inst1, DST_SEL_Z, GX2_GET_COMPONENT_Z_B(fmt.compSel));
    NW_G3D_SET_FLAG_VALUE(inst1, DST_SEL_W, GX2_GET_COMPONENT_W_A(fmt.compSel));
    NW_G3D_SET_FLAG_VALUE(inst1, FORMAT_COMP_ALL,  format & ATTRIB_BIT_SIGNED ? 1 : 0);
    NW_G3D_SET_FLAG_VALUE(inst1, DATA_FORMAT, fmt.surfaceFormat);
    WriteInst(pVFetchInst, 1, inst1);

    u32 inst2 = ReadInst(pVFetchInst, 2);
    NW_G3D_SET_FLAG_VALUE(inst2, ENDIAN_SWAP, fmt.endianSwap);
    WriteInst(pVFetchInst, 2, inst2);
}

GX2AttribFormat GfxFetchShader::GetFormat(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst1 = ReadInst(pVFetchInst, 1);

    u32 format = 0;
    u8 surfaceFormat = NW_G3D_GET_FLAG_VALUE(inst1, DATA_FORMAT, u8);
    // surfaceFormat を線形探査して attribformat に変換する。
    format |= FindInternalFormat(surfaceFormat).attribFormat;

    u32 fmtType = NW_G3D_GET_FLAG_VALUE(inst1, NUM_FORMAT_ALL, u32);
    format |= (fmtType == ATTRIB_FMT_SCALED) ? ATTRIB_BIT_FLOAT :
        (fmtType == ATTRIB_FMT_INT) ? ATTRIB_BIT_INT : 0;
    u8 signedType = NW_G3D_GET_FLAG_VALUE(inst1, FORMAT_COMP_ALL, u8);
    format |= signedType == 1 ? ATTRIB_BIT_SIGNED : 0;

    return static_cast<GX2AttribFormat>(format);
}

void GfxFetchShader::SetEndianSwapMode(void* pShader, int attribIndex, GX2EndianSwapMode mode)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst2 = ReadInst(pVFetchInst, 2);
    NW_G3D_SET_FLAG_VALUE(inst2, ENDIAN_SWAP, mode);
    WriteInst(pVFetchInst, 2, inst2);
}

GX2EndianSwapMode GfxFetchShader::GetEndianSwapMode(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst2 = ReadInst(pVFetchInst, 2);
    return NW_G3D_GET_FLAG_VALUE(inst2, ENDIAN_SWAP, GX2EndianSwapMode);
}

void GfxFetchShader::SetDivisorSlot(void* pShader, int attribIndex, u32 slot)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    static const u8 tblDivisor[] = { 0, 3, 1, 2 }; // 予約値が先頭に並ぶようマッピングを変える。
    GX2AttribIndexType fetchType = slot ? GX2_ATTRIB_INDEX_INSTANCE_ID : GX2_ATTRIB_INDEX_VERTEX_ID;
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst0 = ReadInst(pVFetchInst, 0);
    NW_G3D_SET_FLAG_VALUE(inst0, FETCH_TYPE, fetchType);
    NW_G3D_SET_FLAG_VALUE(inst0, SRC_SEL_X, tblDivisor[slot]);
    WriteInst(pVFetchInst, 0, inst0);

}

u32 GfxFetchShader::GetDivisorSlot(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    static const u8 tblDivisor[] = { 0, 2, 3, 1 }; // 予約値が先頭に並ぶようマッピングを変える。
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst0 = ReadInst(pVFetchInst, 0);
    return tblDivisor[NW_G3D_GET_FLAG_VALUE(inst0, SRC_SEL_X, u32)];
}

void GfxFetchShader::SetOffset(void* pShader, int attribIndex, u32 offset)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst2 = ReadInst(pVFetchInst, 2);
    NW_G3D_SET_FLAG_VALUE(inst2, OFFSET, offset);
    WriteInst(pVFetchInst, 2, inst2);
}

u32 GfxFetchShader::GetOffset(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    const u32* pVFetchInst = GetVFInst(pShader, attribIndex);
    u32 inst2 = ReadInst(pVFetchInst, 2);
    return NW_G3D_GET_FLAG_VALUE(inst2, OFFSET, u32);
}

u32* GfxFetchShader::GetVFInst(void* pShader, int attribIndex)
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    return AddOffset<u32>(pShader, ofsVFInst + attribIndex * VF_INST_SIZE);
}

const u32* GfxFetchShader::GetVFInst(const void* pShader, int attribIndex) const
{
    NW_G3D_ASSERT_NOT_NULL(pShader);
    return AddOffset<u32>(pShader, ofsVFInst + attribIndex * VF_INST_SIZE);
}

void GfxFetchShader::LoadVertexAttribArray(u32 idxAttrib, const GfxBuffer* pBuffer) const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    u32 location = GetLocation(idxAttrib);

    GL::EnableVertexAttribArray(handle, location, GL_TRUE);

    const u32* pVFetchInst = GetVFInst(idxAttrib);
    u32 inst1 = ReadInst(pVFetchInst, 1);
    u8 surfaceFormat = NW_G3D_GET_FLAG_VALUE(inst1, DATA_FORMAT, u8);
    const GLAttribInternal& fmt = FindGLAttribInternal(surfaceFormat);
    u32 type;
    if (fmt.type == GL_UNSIGNED_INT_2_10_10_10_REV)
    {
        type = NW_G3D_GET_FLAG_VALUE(inst1, FORMAT_COMP_ALL, u8) ?
            GL_INT_2_10_10_10_REV : GL_UNSIGNED_INT_2_10_10_10_REV;
    }
    else
    {
        type = fmt.type - NW_G3D_GET_FLAG_VALUE(inst1, FORMAT_COMP_ALL, u8);
    }
    u32 formatAll = NW_G3D_GET_FLAG_VALUE(inst1, NUM_FORMAT_ALL, u32);
    u32 offset = GetOffset(idxAttrib);
    GL::VertexAttribOffset(handle, pBuffer->handle, location,
        fmt.size, type, formatAll == ATTRIB_FMT_INT, formatAll == ATTRIB_FMT_NORMALIZED,
        pBuffer->stride, reinterpret_cast<const GLvoid*>(offset));

    GL::VertexAttribDivisor(handle, location, GetDivisor(GetDivisorSlot(idxAttrib)));
#else
    NW_G3D_UNUSED(idxAttrib);
    NW_G3D_UNUSED(pBuffer);
#endif
}

void GfxFetchShader::LoadVertexAttribArray() const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    for (u32 idxAttrib = 0; idxAttrib < gx2FetchShader.num_attribs; ++idxAttrib)
    {
        const GfxBuffer* pBuffer = GetVertexBuffer(idxAttrib);
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        if (pBuffer->stride > 0)
        {
            // 固定頂点属性は VertexArrayObject に入りません。
            LoadVertexAttribArray(idxAttrib, pBuffer);
        }
    }
#endif
}

void GfxFetchShader::LoadVertexAttribValue() const
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (bit32 fixedAttribs = gx2FetchShader._regs[0])
    {
        // 固定頂点属性は VertexArrayObject に入らないので毎回設定します。
        for (u32 idxAttrib = 0; idxAttrib < gx2FetchShader.num_attribs; ++idxAttrib)
        {
            if (fixedAttribs & (0x1 << idxAttrib))
            {
                const GfxBuffer* pBuffer = GetVertexBuffer(idxAttrib);
                NW_G3D_ASSERT_NOT_NULL(pBuffer);

                u32 location = GetLocation(idxAttrib);
                GL::EnableVertexAttribArray(handle, location, GL_FALSE);

                const u32* pVFetchInst = GetVFInst(GetGX2FetchShader()->shaderPtr, idxAttrib);
                u32 inst2 = ReadInst(pVFetchInst, 2);

                u8 endianSwap = NW_G3D_GET_FLAG_VALUE(inst2, ENDIAN_SWAP, u8);
                u8 cmpSize = 0x1 << endianSwap;
                if (cmpSize == 4)
                {
                    u32 inst0 = ReadInst(pVFetchInst, 0);
                    u32 inst1 = ReadInst(pVFetchInst, 1);
                    u8 vertexSize = NW_G3D_GET_FLAG_VALUE(inst0, MEGA_FETCH_COUNT, u8) + 1;
                    u32 fmtType = NW_G3D_GET_FLAG_VALUE(inst1, NUM_FORMAT_ALL, u32);
                    u8 signedType = NW_G3D_GET_FLAG_VALUE(inst1, FORMAT_COMP_ALL,  u8);

                    u8 cmpCount = vertexSize / cmpSize;
                    if (fmtType == ATTRIB_FMT_SCALED)
                    {
                        // float
                        float vertices[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
                        memcpy(vertices, pBuffer->pData, sizeof(u32) * cmpCount);
                        glVertexAttrib4fv(location, vertices);
                    }
                    else
                    {
                        // int/uint
                        u32 vertices[4] = { 0, 0, 0, 1 };
                        memcpy(vertices, pBuffer->pData, sizeof(u32) * cmpCount);
                        if (signedType)
                        {
#if !defined(NW_G3D_IS_GL_ES)
                            glVertexAttrib4iv(location, reinterpret_cast<s32*>(vertices));
#else
                            NW_G3D_NOT_SUPPORTED();
#endif
                        }
                        else
                        {
#if !defined(NW_G3D_IS_GL_ES)
                            glVertexAttrib4uiv(location, vertices);
#else
                            NW_G3D_NOT_SUPPORTED();
#endif
                        }
                    }
                }
                else
                {
                    // float/int/uint 以外の固定頂点属性は未対応です。
                    NW_G3D_NOT_IMPLEMENTED();
                }
            }
        }
    }
#endif
}

//--------------------------------------------------------------------------------------------------

GfxDispatchParams::GfxDispatchParams()
{
    memset(this, 0, sizeof(*this));
}

void GfxDispatchParams::SetParams(uint num_groups_x /*= 1*/,
                               uint num_groups_y /*= 1*/, uint num_groups_z /*= 1*/)
{
    NW_G3D_WARNING(IsAligned(this, CLASS_ALIGNMENT), "GfxDispatchParams must be aligned.");

    params.num_groups_x = num_groups_x;
    params.num_groups_y = num_groups_y;
    params.num_groups_z = num_groups_z;
    params.padding = 0;
}

void GfxDispatchParams::CPUFlush()
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2Invalidate(GX2_INVALIDATE_CPU, &params, sizeof(GX2DispatchParams));
#endif
#endif
}

void GfxDispatchParams::Dispatch()
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2DispatchCompute(&params);
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    glDispatchCompute(params.num_groups_x, params.num_groups_y, params.num_groups_z);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported compute shader.");
#endif
#endif
}

//--------------------------------------------------------------------------------------------------

void SetStreamOutEnable(GX2Boolean enable)
{
#if NW_G3D_IS_GX2
    GX2SetStreamOutEnable(enable);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (!enable)
    {
        glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, 0);
    }
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(enable);
#endif
}

void SetRasterizerClipControl(GX2Boolean rasterizerEnable, GX2Boolean zClipEnable)
{
#if NW_G3D_IS_GX2
    GX2SetRasterizerClipControl(rasterizerEnable, zClipEnable);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    GL::Enable(GL_RASTERIZER_DISCARD, !rasterizerEnable);
#if !defined(NW_G3D_IS_GL_ES)
    GL::Enable(GL_DEPTH_CLAMP, !zClipEnable);
#endif
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_UNUSED(rasterizerEnable);
    NW_G3D_UNUSED(zClipEnable);
#endif
}

void SetPrimitiveRestartIndex(u32 restartIndex)
{
#if NW_G3D_IS_GX2
    GX2SetPrimitiveRestartIndex(restartIndex);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    glPrimitiveRestartIndex(restartIndex);
    NW_G3D_GL_ASSERT();
#else
    NW_G3D_ASSERTMSG(false, "NW: Unsupported primitive restart index.");
#endif
#else
    NW_G3D_UNUSED( restartIndex );
#endif
}

void ClearBuffers(GfxColorBuffer* pColorBuffer, GfxDepthBuffer* pDepthBuffer,
    float r, float g, float b, float a, GX2ClearMode mode)
{
    NW_G3D_ASSERT_NOT_NULL(pColorBuffer);
    NW_G3D_ASSERT_NOT_NULL(pDepthBuffer);
#if NW_G3D_IS_GX2
    GX2ClearBuffers(
        pColorBuffer->GetGX2ColorBuffer(), pDepthBuffer->GetGX2DepthBuffer(), r, g, b, a, mode);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    pColorBuffer->Clear(r, g, b, a);
    pDepthBuffer->Clear(mode);
#else
    NW_G3D_UNUSED( pColorBuffer );
    NW_G3D_UNUSED( pDepthBuffer );
    NW_G3D_UNUSED( r );
    NW_G3D_UNUSED( g );
    NW_G3D_UNUSED( b );
    NW_G3D_UNUSED( a );
    NW_G3D_UNUSED( mode );
#endif
}

}}} // namespace nw::g3d::fnd
