﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/edit/g3d_InputCapture.h>

#if NW_G3D_CONFIG_USE_HOSTIO || ( NW_G3D_CONFIG_USE_INPUTCAPTURE && !defined(ANDROID) && !defined(__APPLE__) )

#include <cstring>

// #define DEBUG_LOG

namespace nw { namespace g3d { namespace edit {

// 実機側のチャンネル名です。
static const char INPUT_CAPTURE_CHANNEL_NAME[] = "NW_INPUT_CAP";

enum MESSAGETYPE
{
    MESSAGETYPE_PING,
    MESSAGETYPE_ACK,

    MESSAGETYPE_CAPTURESTART,
    MESSAGETYPE_CAPTURESTOP,

    MESSAGETYPE_KEYDOWN,
    MESSAGETYPE_KEYUP,

    MESSAGETYPE_MOUSEMOVE,
    MESSAGETYPE_MOUSEDOWN,
    MESSAGETYPE_MOUSEUP,
    MESSAGETYPE_MOUSEDOUBLECLICK,
    MESSAGETYPE_MOUSEWHEEL,

    MESSAGETYPE_MAX
};

enum BORDER_TYPE
{
    BORDER_TYPE_NONE,
    BORDER_TYPE_RIGHT,
    BORDER_TYPE_LEFT,
    BORDER_TYPE_TOP,
    BORDER_TYPE_BOTTOM
};

struct PingData
{
    u8      msg[8];
};

struct CaptureData
{
    u8      borderType;
    u8      captureOnMousePosition;
    u16     borderLength;
    s16     pointerPosX;
    s16     pointerPosY;
};

struct KeyStatusData
{
    u8      modifier;
    u8      attribute;
    u8      category;
    u8      keyCode;
    u32     keyboardLanguageId;
};

struct MouseStatusData
{
    u8      button[2];
    s16     wheel;
    s16     scrollLines;
    u8      reserved[2];
};

struct PointerStatusData
{
    s16     moveX;
    s16     moveY;
    u8      reserved[4];
};

struct MessageChunk
{
    u8      type;
    u8      reserved[3];

    union
    {
        PingData          pingData;
        CaptureData       captureData;
        KeyStatusData     keyStatus;
        MouseStatusData   mouseStatus;
        PointerStatusData pointerStatus;
    };
};


#if NW_G3D_IS_HOST_CAFE
template <typename T> T NetToHost(T value)
{
    return value;
}
template <typename T> T HostToNet(T value)
{
    return value;
}
#else
template <typename T> T NetToHost(T value)
{
    return ReverseEndian(value);
}
template <typename T> T HostToNet(T value)
{
    return ReverseEndian(value);
}
#endif

//---------------------------------------------------------------------------
template <typename OutputT>
inline OutputT
BitExtract(OutputT bits, int pos, int len=1)
{
    const OutputT mask = static_cast<OutputT>( ( 1UL << len ) - 1 );
    return static_cast<OutputT>( (bits >> pos) & mask );
}

//---------------------------------------------------------------------------
inline u16
ReverseEndian( u16 x )
{
    return static_cast<u16>(
        BitExtract( x,  0, 8 ) << 8 |
        BitExtract( x,  8, 8 ) << 0
    );
}

//---------------------------------------------------------------------------
inline s16
ReverseEndian( s16 x )
{
    return static_cast<s16>(
        BitExtract( x,  0, 8 ) << 8 |
        BitExtract( x,  8, 8 ) << 0
    );
}

#if NW_G3D_IS_HOST_CAFE

HIOHandle InputCapture::s_Handle = HIO_HANDLE_INVALID;

//---------------------------------------------------------------------------
/*static*/ void
InputCapture::ConnectionCallback(HIOStatus status, void* context)
{
    InputCapture* inputCapture = static_cast<InputCapture*>( context );

    if (status == HIO_STATUS_NEW_CONNECTION)
    {
        inputCapture->m_IsConnected = true;
    #if defined DEBUG_LOG
        OSReport("Connected.\n");
    #endif
    }
    else if (status == HIO_STATUS_NO_CONNECTIONS)
    {
        inputCapture->m_IsConnected = false;
    #if defined DEBUG_LOG
        OSReport("Disconnected.\n");
    #endif
    }
}

//---------------------------------------------------------------------------
/*static*/ void
InputCapture::ReadCallback(HIOStatus status, void* context)
{
    InputCapture* capture = static_cast<InputCapture*>( context );

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        if (status >= 0)
        {
            capture->m_ReadBuffer[capture->m_ReadBufferCount].readSize = static_cast<u32>( status );
            capture->m_ReadBuffer[capture->m_ReadBufferCount].isReadDone = true;

            capture->m_ReadBufferCount = (capture->m_ReadBufferCount + 1) % BUFFER_COUNT;
        }
        break;
    }
}

//---------------------------------------------------------------------------
/*static*/ void
InputCapture::WriteCallback(HIOStatus status, void* context)
{
    (void)context;

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        break;
    }
}
#endif

//---------------------------------------------------------------------------
/* ctor */
InputCapture::InputCapture( const InitArg& arg ) :
    m_IsConnected(false),
    m_IsCapturing(false),
    m_ReadBufferCount(0),
    m_AnalyzeBufferCount(0),
    m_KeyboardLanguageId(0),
    m_MouseButton(0),
    m_LastMouseButton(0),
    m_MouseButtonTriggered(0),
    m_MouseButtonReleased(0),
    m_MousePosX(0),
    m_MousePosY(0),
    m_MouseWheelRaw(0),
    m_MouseWheelScrollLines(0),
    m_ScreenWidth(arg.width),
    m_ScreenHeight(arg.height),
    m_BorderType(BORDER_TYPE_NONE),
    m_BorderLength(m_ScreenHeight),
    m_IsPointerClampEnabled(arg.isPointerClampEnabled),
    m_IsPointerOnScreen(false)
{
}

//---------------------------------------------------------------------------
bool
InputCapture::Open()
{
#if NW_G3D_IS_HOST_CAFE
    if ( s_Handle != HIO_HANDLE_INVALID )
    {
        return false;
    }

    s_Handle = HIOOpen( INPUT_CAPTURE_CHANNEL_NAME, ConnectionCallback, this, HIO_CHANNEL_OPTION_READ_WRITE, 0 );

    if ( s_Handle == HIO_HANDLE_INVALID )
    {
        // NW_G3D_WARNING(true, "HIO Handle open error %d\n", s_Handle);
        return false;
    }

    return true;

#else
    // PC 版は未対応です。
    return false;

#endif
}

//---------------------------------------------------------------------------
void
InputCapture::Read()
{
#if NW_G3D_IS_HOST_CAFE
    // ハンドルが無効ならば、処理を抜ける
    if ( s_Handle == HIO_HANDLE_INVALID )
    {
        return;
    }

    if (!m_IsConnected)
    {
        return;
    }

    if (!m_ReadBuffer[m_ReadBufferCount].isReadStarted)
    {
        HIOReadAsync( s_Handle, BUFFER_SIZE, &(m_ReadBuffer[m_ReadBufferCount].pReadBuffer), ReadCallback, this );
        m_ReadBuffer[m_ReadBufferCount].isReadStarted = true;
    }

    m_LastKeyState = m_KeyState;
    m_LastMouseButton = m_MouseButton;
    m_MouseWheelRaw = 0;
    m_MouseWheelScrollLines = 0;

    if (m_ReadBuffer[m_AnalyzeBufferCount].isReadDone)
    {
        Analyze();

        m_ReadBuffer[m_AnalyzeBufferCount].isReadStarted = false;
        m_ReadBuffer[m_AnalyzeBufferCount].isReadDone = false;
        m_ReadBuffer[m_AnalyzeBufferCount].readSize = 0;

        m_AnalyzeBufferCount = (m_AnalyzeBufferCount + 1) % BUFFER_COUNT;
    }

    BitSet256 keyChanged = m_KeyState ^ m_LastKeyState;
    m_KeyTriggered = keyChanged & m_KeyState;
    m_KeyReleased = keyChanged & m_LastKeyState;

    u32 mouseButtonChanged = m_MouseButton ^ m_LastMouseButton;
    m_MouseButtonTriggered = mouseButtonChanged & m_MouseButton;
    m_MouseButtonReleased = mouseButtonChanged & m_LastMouseButton;

#else
    // PC 版は未対応です。
    return;

#endif
}

//---------------------------------------------------------------------------
void
InputCapture::Write(void* buf, u32 size)
{
#if NW_G3D_IS_HOST_CAFE
    // ハンドルが無効ならば、処理を抜ける
    if ( s_Handle == HIO_HANDLE_INVALID )
    {
        return;
    }

    if ( ! m_IsConnected )
    {
        return;
    }

    HIOWriteAsync( s_Handle, size, buf, WriteCallback, this );

#else
    // PC 版は未対応です。
    (void)buf;
    (void)size;
    return;

#endif
}

//---------------------------------------------------------------------------
void
InputCapture::Close()
{
#if NW_G3D_IS_HOST_CAFE
    if ( s_Handle == HIO_HANDLE_INVALID )
    {
        return;
    }

    HIOClose( s_Handle );
    s_Handle = HIO_HANDLE_INVALID;

#else
    // PC 版は未対応です。
    return;

#endif
}

//---------------------------------------------------------------------------
void
InputCapture::Analyze()
{
    int msgNum = m_ReadBuffer[m_AnalyzeBufferCount].readSize / sizeof( MessageChunk );

    for ( int i = 0; i < msgNum; ++i )
    {
        MessageChunk& data = reinterpret_cast<MessageChunk*>( m_ReadBuffer[m_AnalyzeBufferCount].pReadBuffer )[i];

        switch (data.type)
        {
        case MESSAGETYPE_PING:
            {
                u8 readMsg[8] = { 'p', 'i', 'n', 'g', 0, 0, 0, 0 };
                bool result = true;
                for ( int idxMsg = 0; idxMsg < sizeof(readMsg); ++idxMsg )
                {
                    if ( data.pingData.msg[idxMsg] != readMsg[idxMsg] )
                    {
                        result = false;
                        break;
                    }
                }

                if ( result )
                {
                    SendAckMsg();
                }
            }
            break;

        case MESSAGETYPE_CAPTURESTART:
            {
                ResetState();

                m_BorderType = data.captureData.borderType;
                m_BorderLength = NetToHost( data.captureData.borderLength );

                if ( data.captureData.captureOnMousePosition != 0 )
                {
                    // マウスポインタ移動からキャプチャーを開始します。
                    m_MousePosX = NetToHost( data.captureData.pointerPosX );
                    m_MousePosY = NetToHost( data.captureData.pointerPosY );

                    if ( data.captureData.borderType == BORDER_TYPE_RIGHT )
                    {
                        m_MousePosX = 1;
                        m_MousePosY = static_cast<int>( static_cast<f32>(m_ScreenHeight) / m_BorderLength * m_MousePosY );
                    }
                    else if ( data.captureData.borderType == BORDER_TYPE_LEFT )
                    {
                        m_MousePosX = m_ScreenWidth - 1;
                        m_MousePosY = static_cast<int>( static_cast<f32>(m_ScreenHeight) / m_BorderLength * m_MousePosY );
                    }
                    else if ( data.captureData.borderType == BORDER_TYPE_TOP )
                    {
                        m_MousePosX = static_cast<int>( static_cast<f32>(m_ScreenWidth) / m_BorderLength * m_MousePosX );
                        m_MousePosY = m_ScreenHeight - 1;
                    }
                    else if ( data.captureData.borderType == BORDER_TYPE_BOTTOM )
                    {
                        m_MousePosX = static_cast<int>( static_cast<f32>(m_ScreenWidth) / m_BorderLength * m_MousePosX );
                        m_MousePosY = 1;
                    }
                }
                else
                {
                    // ホットキーからキャプチャーを開始します。
                    int width = m_ScreenWidth;
                    int height = m_ScreenHeight;

                    if ( m_MousePosX < 0 )
                    {
                        m_MousePosX = 0;
                    }

                    if ( m_MousePosX > width )
                    {
                        m_MousePosX = width;
                    }

                    if ( m_MousePosY < 0 )
                    {
                        m_MousePosY = 0;
                    }

                    if ( m_MousePosY > height )
                    {
                        m_MousePosY = height;
                    }
                }

                m_IsCapturing = true;
            }
            break;

        case MESSAGETYPE_CAPTURESTOP:
            {
                ResetState();
                m_IsCapturing = false;
            }
            break;

        case MESSAGETYPE_KEYDOWN:
            if ( m_IsCapturing && data.keyStatus.keyCode > 0 )
            {
                m_KeyboardLanguageId = data.keyStatus.keyboardLanguageId;
                m_KeyState.Set( data.keyStatus.keyCode, 1 );
            }
            break;

        case MESSAGETYPE_KEYUP:
            if ( m_IsCapturing && data.keyStatus.keyCode > 0 )
            {
                m_KeyboardLanguageId = data.keyStatus.keyboardLanguageId;
                m_KeyState.Set( data.keyStatus.keyCode, 0 );
            }
            break;

        case MESSAGETYPE_MOUSEMOVE:
            if ( m_IsCapturing )
            {
                m_MousePosX += NetToHost( data.pointerStatus.moveX );
                m_MousePosY += NetToHost( data.pointerStatus.moveY );
            }
            break;

        case MESSAGETYPE_MOUSEDOWN:
        case MESSAGETYPE_MOUSEUP:
            if ( m_IsCapturing )
            {
                m_MouseButton = data.mouseStatus.button[0];
            }
            break;

        case MESSAGETYPE_MOUSEDOUBLECLICK:
            if ( m_IsCapturing )
            {
            }
            break;

        case MESSAGETYPE_MOUSEWHEEL:
            if ( m_IsCapturing )
            {
                m_MouseWheelRaw += NetToHost( data.mouseStatus.wheel );
                m_MouseWheelScrollLines += NetToHost( data.mouseStatus.scrollLines );
            }
            break;

        default:
            break;
        }
    }

    AdjustMousePos();
} // NOLINT (readability/fn_size)

//---------------------------------------------------------------------------
void
InputCapture::AdjustMousePos()
{
    if ( ! m_IsCapturing )
    {
        return;
    }

    bool captureStop = false;
    int pointerPosX = 0;
    int pointerPosY = 0;
    int width = m_ScreenWidth;
    int height = m_ScreenHeight;
    f32 length = static_cast<f32>( m_BorderLength );

    m_IsPointerOnScreen = true;

    // マウスポインタのスクリーン移動範囲を調節します。
    if ( m_MousePosX < 0 )
    {
        m_IsPointerOnScreen = false;

        if ( m_IsPointerClampEnabled )
        {
            m_MousePosX = 0;
        }

        if ( m_BorderType == BORDER_TYPE_RIGHT )
        {
            pointerPosX = 0;
            pointerPosY = static_cast<int>( length / height * m_MousePosY );
            captureStop = true;
        }
    }

    if ( m_MousePosX > width )
    {
        m_IsPointerOnScreen = false;

        if ( m_IsPointerClampEnabled )
        {
            m_MousePosX = width;
        }

        if ( m_BorderType == BORDER_TYPE_LEFT )
        {
            pointerPosX = width;
            pointerPosY = static_cast<int>( length / height * m_MousePosY );
            captureStop = true;
        }
    }

    if ( m_MousePosY < 0 )
    {
        m_IsPointerOnScreen = false;

        if ( m_IsPointerClampEnabled )
        {
            m_MousePosY = 0;
        }

        if ( m_BorderType == BORDER_TYPE_BOTTOM )
        {
            pointerPosX = static_cast<int>( length / width * m_MousePosX );
            pointerPosY = 0;
            captureStop = true;
        }
    }

    if ( m_MousePosY > height )
    {
        m_IsPointerOnScreen = false;

        if ( m_IsPointerClampEnabled )
        {
            m_MousePosY = height;
        }

        if ( m_BorderType == BORDER_TYPE_TOP )
        {
            pointerPosX = static_cast<int>( length / width * m_MousePosX );
            pointerPosY = m_MousePosY;
            captureStop = true;
        }
    }

    // 境界に達した時にキャプチャの中断を要求するメッセージを送信します。
    if ( m_BorderType != BORDER_TYPE_NONE &&
         captureStop && !m_KeyState.Any() && m_MouseButton == 0 )
    {
        SendStopCaptureMsg( pointerPosX, pointerPosY );
        ResetState();
    }
}

//---------------------------------------------------------------------------
void
InputCapture::SendStopCaptureMsg( int pointerPosX, int pointerPosY )
{
    MessageChunk sendData;
    memset( &sendData, 0, sizeof(MessageChunk) );

    sendData.type = MESSAGETYPE_CAPTURESTOP;
    sendData.captureData.pointerPosX = HostToNet( static_cast<s16>( pointerPosX ) );
    sendData.captureData.pointerPosY = HostToNet( static_cast<s16>( pointerPosY ) );

    Write( &sendData, sizeof(MessageChunk) );

    m_IsCapturing = false;
}

//---------------------------------------------------------------------------
void
InputCapture::SendAckMsg()
{
    MessageChunk sendData;
    memset( &sendData, 0, sizeof(MessageChunk) );
    sendData.type = MESSAGETYPE_ACK;
    sendData.pingData.msg[0] = 'a';
    sendData.pingData.msg[1] = 'c';
    sendData.pingData.msg[2] = 'k';
    sendData.pingData.msg[3] = ' ';

    Write( &sendData, sizeof(MessageChunk) );
}

//---------------------------------------------------------------------------
void
InputCapture::ResetState()
{
    m_KeyState.Reset();
    m_LastKeyState.Reset();
    m_KeyTriggered.Reset();
    m_KeyReleased.Reset();
    m_MouseButton = 0;
    m_LastMouseButton = 0;
    m_MouseButtonTriggered = 0;
    m_MouseButtonReleased = 0;
    m_MouseWheelRaw = 0;
    m_MouseWheelScrollLines = 0;
}

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
