﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace ut {

namespace detail {

template<typename TCharType>
class DynamicLengthStringBase
{
public:
    static const TCharType NULL_CHAR = static_cast<TCharType>(0);
    static const size_t MAXIMU_LENGTH = 1024; // 1024文字まで

    explicit DynamicLengthStringBase(IAllocator* allocator)
        : m_pString(NULL)
        , m_StringLength(0)
        , m_pAllocator(allocator)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
    }

    ~DynamicLengthStringBase()
    {
        Clear();
        m_pAllocator = NULL;
    }

    void Clear()
    {
        if (m_pString != NULL)
        {
            m_pAllocator->Free(m_pString);
            m_pString = NULL;
        }
    }

    bool Assign(const TCharType* str)
    {
        NW_G3D_EDIT_ASSERT_NOT_NULL(m_pAllocator);
        size_t length = CalcStrLength(str);
        if (length == 0)
        {
            return false;
        }

        size_t bufferSize = (length + 1) * sizeof(TCharType);
        TCharType* stringBuffer = static_cast<TCharType*>(m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT));
        if (stringBuffer == NULL)
        {
            return false;
        }

        if (m_pString != NULL)
        {
            m_pAllocator->Free(m_pString);
            m_pString = NULL;
        }

        m_pString = stringBuffer;
        std::memcpy(m_pString, str, (length * sizeof(TCharType)));

        m_pString[length] = DynamicLengthStringBase<TCharType>::NULL_CHAR;
        m_StringLength = length;

        return true;
    }

    int Compare( const DynamicLengthStringBase<TCharType>& rhs) const
    {
        NW_G3D_EDIT_ASSERT_NOT_NULL(m_pAllocator);
        // アドレスがいっしょなので同一
        if (GetStr() == rhs.GetStr())
        {
            return 0;
        }

        size_t length = GetLength();
        if (length < rhs.GetLength())
        {
            length = rhs.GetLength();
        }

        for (size_t i = 0; i <= length; ++i)
        {
            if (m_pString[i] == NULL_CHAR && rhs.m_pString[i] == NULL_CHAR)
            {
                return 0;
            }
            else if (m_pString[i] == NULL_CHAR)
            {
                return -1;
            }
            else if (rhs.m_pString[i] == NULL_CHAR)
            {
                return 1;
            }
            else if (m_pString[i] < rhs.m_pString[i])
            {
                return -1;
            }
            else if (m_pString[i] > rhs.m_pString[i])
            {
                return 1;
            }
        }
        return 0;
    }

    const TCharType* GetStr() const
    {
        return m_pString;
    }
    size_t GetLength() const
    {
        return m_StringLength;
    }

private:
    inline size_t CalcStrLength(const TCharType* str) const;

    TCharType* m_pString;
    size_t m_StringLength;
    IAllocator* m_pAllocator;
};

template<typename TCharType>
size_t DynamicLengthStringBase<TCharType>::CalcStrLength(const TCharType* str) const
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(m_pAllocator);
    NW_G3D_ASSERT_NOT_NULL(str);

    size_t length = 0;
    while (NW_G3D_STATIC_CONDITION(1))
    {
        if ( length > MAXIMU_LENGTH || str[length] == NULL_CHAR)
        {
            break;
        }
        ++length;
    }
    if (length > MAXIMU_LENGTH)
    {
        NW_G3D_WARNING(0, "string is too long.");
        return 0;
    }
    return length;
}

typedef DynamicLengthStringBase<char> DynamicLengthString;

} // namespace detail

}}}} // namespace nw::g3d::edit::ut

#endif // NW_G3D_CONFIG_USE_HOSTIO
