﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditUtility.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#if NW_G3D_IS_HOST_WIN

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <winsock.h>
#pragma comment(lib, "wsock32.lib ")

#else

#include <cafe/hio.h>
#include <cafe/fs.h>

#endif

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_ShaderParamAnimObj.h>
#include <nw/g3d/g3d_SkeletonObj.h>
#include <nw/g3d/g3d_res.h>

namespace nw { namespace g3d { namespace edit {

namespace detail
{
struct MatEditArg
{
    ModelObj* modelObj;
    const int* indices;
    int indexSize;
};

#define EDIT_KIND_TO_TYPE(kind, type) case kind: return type;
g3d::res::ResShaderParam::Type GetResShaderParamType(EditTargetKind kind)
{
    using namespace g3d::res;
    switch(kind)
    {
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL,   ResShaderParam::TYPE_BOOL)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2,  ResShaderParam::TYPE_BOOL2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3,  ResShaderParam::TYPE_BOOL3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4,  ResShaderParam::TYPE_BOOL4)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT,    ResShaderParam::TYPE_INT)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2,   ResShaderParam::TYPE_INT2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3,   ResShaderParam::TYPE_INT3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4,   ResShaderParam::TYPE_INT4)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT,   ResShaderParam::TYPE_UINT)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2,  ResShaderParam::TYPE_UINT2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3,  ResShaderParam::TYPE_UINT3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4,  ResShaderParam::TYPE_UINT4)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT,  ResShaderParam::TYPE_FLOAT)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2, ResShaderParam::TYPE_FLOAT2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3, ResShaderParam::TYPE_FLOAT3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4, ResShaderParam::TYPE_FLOAT4)

    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2, ResShaderParam::TYPE_FLOAT2x2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3, ResShaderParam::TYPE_FLOAT2x3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4, ResShaderParam::TYPE_FLOAT2x4)

    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2, ResShaderParam::TYPE_FLOAT3x2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3, ResShaderParam::TYPE_FLOAT3x3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4, ResShaderParam::TYPE_FLOAT3x4)

    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2, ResShaderParam::TYPE_FLOAT4x2)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3, ResShaderParam::TYPE_FLOAT4x3)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4, ResShaderParam::TYPE_FLOAT4x4)

    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D, ResShaderParam::TYPE_SRT3D)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D, ResShaderParam::TYPE_SRT2D)

    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA,      ResShaderParam::TYPE_TEXSRT)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX,    ResShaderParam::TYPE_TEXSRT)
    EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE, ResShaderParam::TYPE_TEXSRT)

    default:
        break;
    }
    return ResShaderParam::NUM_TYPE;
}

#define EDIT_TYPE_TO_SIZE(type, size) case type: return size;
int GetArraySize(g3d::res::ResShaderParam::Type type)
{
    using namespace g3d::res;
    switch(type)
    {
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_BOOL,   1)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_BOOL2,  2)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_BOOL3,  3)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_BOOL4,  4)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_INT,    1)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_INT2,   2)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_INT3,   3)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_INT4,   4)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_UINT,   1)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_UINT2,  2)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_UINT3,  3)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_UINT4,  4)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT,  1)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT2, 2)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT3, 3)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT4, 4)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT2x2, 4)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT2x3, 6)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT2x4, 8)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT3x2, 6)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT3x3, 9)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT3x4, 12)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT4x2, 8)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT4x3, 12)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_FLOAT4x4, 16)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_SRT3D, 9)
    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_SRT2D, 5)

    EDIT_TYPE_TO_SIZE(ResShaderParam::TYPE_TEXSRT,           6)

    default:
        break;
    }
    return 0;
}

void SetMatVisibility(ModelObj* modelObj, const int indices[], int indexSize, bool visible)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        modelObj->SetMatVisibility(indices[i], visible);
    }
}

void SetMatDisplayFace(ModelObj* modelObj, const int indices[], int indexSize, bool isFront, bool isBack)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetPolygonCtrl().SetCullFront(static_cast<GX2Boolean>(isFront));
        matObj->GetResRenderState()->GetPolygonCtrl().SetCullBack(static_cast<GX2Boolean>(isBack));
    }
}

void SetMatRenderStateMode(ModelObj* modelObj, const int indices[], int indexSize, res::ResRenderState::Mode mode)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->AdjustMode(mode);
    }
}

void SetMatRenderStateBlendMode(ModelObj* modelObj, const int indices[], int indexSize, res::ResRenderState::BlendMode blendMode)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->AdjustBlendMode(blendMode);
    }
}

void SetMatDepthTestEnable(ModelObj* modelObj, const int indices[], int indexSize, bool enable)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetDepthCtrl().SetDepthTestEnable(static_cast<GX2Boolean>(enable));
    }
}

void SetMatDepthTestWriteEnable(ModelObj* modelObj, const int indices[], int indexSize, bool enable)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetDepthCtrl().SetDepthWriteEnable(static_cast<GX2Boolean>(enable));
    }
}

void SetMatDepthTestFunc(ModelObj* modelObj, const int indices[], int indexSize, GX2CompareFunction func)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetDepthCtrl().SetDepthFunc(func);
    }
}

void SetMatAlphaTestEnable(ModelObj* modelObj, const int indices[], int indexSize, bool enable)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetAlphaTest().SetAlphaTestEnable(static_cast<GX2Boolean>(enable));
    }
}

void SetMatAlphaTestFunc(ModelObj* modelObj, const int indices[], int indexSize, GX2CompareFunction func)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetAlphaTest().SetAlphaFunc(func);
    }
}

void SetMatAlphaTestValue(ModelObj* modelObj, const int indices[], int indexSize, float value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetAlphaTest().SetRefValue(value);
    }
}

void SetMatColorCombine(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendCombine blendCombine)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetColorCombine(blendCombine);
    }
}

void SetMatAlphaCombine(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendCombine blendCombine)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetAlphaCombine(blendCombine);
    }
}

void SetMatColorSrcBlend(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendFunction blendFunc)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetColorSrcBlend(blendFunc);
    }
}

void SetMatColorDstBlend(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendFunction blendFunc)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetColorDstBlend(blendFunc);
    }
}

void SetMatAlphaSrcBlend(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendFunction blendFunc)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetAlphaSrcBlend(blendFunc);
    }
}

void SetMatAlphaDstBlend(ModelObj* modelObj, const int indices[], int indexSize, GX2BlendFunction blendFunc)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendCtrl().SetAlphaDstBlend(blendFunc);
    }
}

void SetMatConstantColor(ModelObj* modelObj, const int indices[], int indexSize, const f32 color[4])
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetBlendColor().SetConstantColor(color);
    }
}

void SetMatLogicOp(ModelObj* modelObj, const int indices[], int indexSize, GX2LogicOp logicOp)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        matObj->GetResRenderState()->GetColorCtrl().SetLogicOp(logicOp);
    }
}

void SetMatSamplerWrapU(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexClamp value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetClampX(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetMatSamplerWrapV(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexClamp value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetClampY(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}
void SetMatSamplerWrapW(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexClamp value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetClampZ(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetMatSamplerMagFilter(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexXYFilterType value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMagFilter(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetMatSamplerMinFilter(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexXYFilterType value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMinFilter(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetMatSamplerMipFilter(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexMipFilterType value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMipFilter(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetMatSamplerMaxAniso(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, GX2TexAnisoRatio value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMaxAniso(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

template<typename TValue>
void SetMatShaderParamValue(
    ModelObj* modelObj,
    const char* paramName,
    const int indices[],
    int indexSize,
    ResShaderParam::Type paramType,
    const TValue value[],
    int valueSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(paramName, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        int shaderParamCount = matObj->GetShaderParamCount();
        int paramIndex = matObj->GetShaderParamIndex(paramName);

        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(paramIndex, shaderParamCount);
        if (paramIndex != -1 && shaderParamCount > paramIndex)
        {
            ResShaderParam* resShaderParam = matObj->GetResShaderParam(paramIndex);
            // この判定で良いかはあとで考える
            if (resShaderParam->GetType() == paramType)
            {
                TValue* editValue = matObj->EditShaderParam<TValue>(paramIndex);
                for(int idxValue = 0; idxValue < valueSize; ++idxValue)
                {
                    editValue[idxValue] = value[idxValue];
                }
            }
        }
    }
}

void SetMatShaderParamTexSrt(
    ModelObj* modelObj,
    const char* paramName,
    const int indices[],
    int indexSize,
    TexSrt::Mode mode,
    ResShaderParam::Type paramType,
    const f32 value[],
    int valueSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(paramName, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERT(paramType == ResShaderParam::TYPE_TEXSRT);

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);
        int shaderParamCount = matObj->GetShaderParamCount();
        int paramIndex = matObj->GetShaderParamIndex(paramName);

        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(paramIndex, shaderParamCount);
        if (paramIndex != -1 && shaderParamCount > paramIndex)
        {
            ResShaderParam* resShaderParam = matObj->GetResShaderParam(paramIndex);
            if (resShaderParam->GetType() == paramType)
            {
                TexSrt* texSrtValue = matObj->EditShaderParam<TexSrt>(paramIndex);
                texSrtValue->mode = mode;
                f32* editValue = &texSrtValue->sx;
                for(int idxValue = 0; idxValue < valueSize; ++idxValue)
                {
                    editValue[idxValue] = value[idxValue];
                }
            }
        }
    }
}

void SetMatSamplerMinLOD(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, float value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMinLOD(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}
void SetMatSamplerMaxLOD(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, float value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetMaxLOD(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}
void SetMatSamplerLODBias(ModelObj* modelObj, const int indices[], int indexSize, int samplerIndex, float value)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(samplerIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        MaterialObj* matObj = modelObj->GetMaterial(indices[i]);

        if (samplerIndex >= matObj->GetSamplerCount() )
        {
            continue;
        }

        ResSampler* resSampler = matObj->GetResSampler(samplerIndex);
        if (resSampler != NULL)
        {
            resSampler->GetGfxSampler()->SetLODBias(value);
            resSampler->GetGfxSampler()->UpdateRegs();
        }
    }
}

void SetBoneVisibility(ModelObj* modelObj, const int indices[], int indexSize, bool visible)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    for (int i = 0; i < indexSize; ++i)
    {
        modelObj->SetBoneVisibility(indices[i], visible);
    }
}

void SetBoneBillboard(ModelObj* modelObj, const int indices[], int indexSize, bit32 flag)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));

    const bit32 clearMask = static_cast<bit32>(~ResBone::BILLBOARD_MASK);
    SkeletonObj* skeletonObj = modelObj->GetSkeleton();
    for (int i = 0; i < indexSize; ++i)
    {
        ResBone* resBone = skeletonObj->GetBone(indices[i]);
        if (resBone != NULL)
        {
            resBone->ref().flag &= clearMask;
            resBone->ref().flag |= flag;
        }
    }
    skeletonObj->GetResource()->UpdateBillboardMode();
}

bool EditMaterials(ModelObj* modelObj, const EditMaterialArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL(modelObj);
    bool result = true;
    switch(arg.editTargetKind)
    {
    case EDIT_TARGET_MATERIAL_VISIBILITY:
    case EDIT_TARGET_MATERIAL_RENDER_STATE_MODE:
    case EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC:
    case EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE:
    case EDIT_TARGET_MATERIAL_ALPHATEST_FUNC:
    case EDIT_TARGET_MATERIAL_ALPHATEST_VALUE:
    case EDIT_TARGET_MATERIAL_COLOR_COMBINE:
    case EDIT_TARGET_MATERIAL_ALPHA_COMBINE:
    case EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND:
    case EDIT_TARGET_MATERIAL_COLOR_DST_BLEND:
    case EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND:
    case EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND:
    case EDIT_TARGET_MATERIAL_LOGIC_OP:
        {
            const EditSimpleValue* editValue =
                static_cast<const EditSimpleValue*>(arg.value);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_VISIBILITY:
                SetMatVisibility(modelObj, arg.index, arg.indexSize, editValue->bValue);
                break;
            case EDIT_TARGET_MATERIAL_RENDER_STATE_MODE:
                SetMatRenderStateMode(modelObj, arg.index, arg.indexSize, editValue->renderStateMode);
                break;
            case EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE:
                SetMatRenderStateBlendMode(modelObj, arg.index, arg.indexSize, editValue->renderStateBlendMode);
                break;
            case EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE:
                SetMatDepthTestEnable(modelObj, arg.index, arg.indexSize, editValue->bValue);
                break;
            case EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE:
                SetMatDepthTestWriteEnable(modelObj, arg.index, arg.indexSize, editValue->bValue);
                break;
            case EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC:
                SetMatDepthTestFunc(modelObj, arg.index, arg.indexSize, editValue->depthTestFunc);
                break;
            case EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE:
                SetMatAlphaTestEnable(modelObj, arg.index, arg.indexSize, editValue->bValue);
                break;
            case EDIT_TARGET_MATERIAL_ALPHATEST_FUNC:
                SetMatAlphaTestFunc(modelObj, arg.index, arg.indexSize, editValue->alphaTestFunc);
                break;
            case EDIT_TARGET_MATERIAL_ALPHATEST_VALUE:
                SetMatAlphaTestValue(modelObj, arg.index, arg.indexSize, editValue->fValue);
                break;
            case EDIT_TARGET_MATERIAL_COLOR_COMBINE:
                SetMatColorCombine(modelObj, arg.index, arg.indexSize, editValue->blendCombine);
                break;
            case EDIT_TARGET_MATERIAL_ALPHA_COMBINE:
                SetMatAlphaCombine(modelObj, arg.index, arg.indexSize, editValue->blendCombine);
                break;
            case EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND:
                SetMatColorSrcBlend(modelObj, arg.index, arg.indexSize, editValue->colorBlendFunc);
                break;
            case EDIT_TARGET_MATERIAL_COLOR_DST_BLEND:
                SetMatColorDstBlend(modelObj, arg.index, arg.indexSize, editValue->colorBlendFunc);
                break;
            case EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND:
                SetMatAlphaSrcBlend(modelObj, arg.index, arg.indexSize, editValue->alphaBlendFunc);
                break;
            case EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND:
                SetMatAlphaDstBlend(modelObj, arg.index, arg.indexSize, editValue->alphaBlendFunc);
                break;
            case EDIT_TARGET_MATERIAL_LOGIC_OP:
                SetMatLogicOp(modelObj, arg.index, arg.indexSize, editValue->logicOp);
                break;

            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_TARGET_MATERIAL_CONSTANT_COLOR:
        {
            const EditVector4Value* editValue =
                static_cast<const EditVector4Value*>(arg.value);
            SetMatConstantColor(modelObj, arg.index, arg.indexSize, editValue->fValue);
        }
        break;
    case EDIT_TARGET_MATERIAL_DISPLAYFACE:
        {
            const EditSimpleValue* edit_value =
                static_cast<const EditSimpleValue*>(arg.value);
            // ここは、最適化予定
            bool isFront = false;
            if (edit_value->uHigh > 0)
            {
                isFront = true;
            }

            bool isBack = false;
            if (edit_value->uLow > 0)
            {
                isBack = true;
            }
            SetMatDisplayFace( modelObj, arg.index, arg.indexSize, isFront, isBack );
        }
        break;
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
        {
            const EditSamplerValue* edit_value =
                static_cast<const EditSamplerValue*>(arg.value);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
                SetMatSamplerWrapU(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->clamp);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
                SetMatSamplerWrapV(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->clamp);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
                SetMatSamplerWrapW(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->clamp);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
                SetMatSamplerMagFilter(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->filter);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
                SetMatSamplerMinFilter(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->filter);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
                SetMatSamplerMipFilter(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->mipFilter);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
                SetMatSamplerMaxAniso(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->ratio);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
                SetMatSamplerMinLOD(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->lod);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
                SetMatSamplerMaxLOD(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->lod);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
                SetMatSamplerLODBias(modelObj, arg.index, arg.indexSize, edit_value->samplerIndex, edit_value->bias);
                break;

            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
        {
            const EditShaderParamVectorValue* edit_value =
                static_cast<const EditShaderParamVectorValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
                SetMatShaderParamValue<u32>(
                    modelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexSize,
                    type,
                    edit_value->value.uValue, valueSize);
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
                SetMatShaderParamValue<s32>(
                    modelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexSize,
                    type,
                    edit_value->value.sValue, valueSize);
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
                SetMatShaderParamValue<f32>(
                    modelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexSize,
                    type,
                    edit_value->value.fValue, valueSize);
                break;

            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D:
        {
            const EditShaderParamMatrixValue* editValue =
                static_cast<const EditShaderParamMatrixValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type);
            SetMatShaderParamValue<f32>(
                modelObj, reinterpret_cast<const char*>(editValue->paramName),
                arg.index, arg.indexSize,
                type,
                editValue->value.a, valueSize);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
        {
            // あとでパケットに含める形に変更
            TexSrt::Mode mode;
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
                mode = TexSrt::MODE_3DSMAX;
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
                mode = TexSrt::MODE_SOFTIMAGE;
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
                mode = TexSrt::MODE_MAYA;
                break;
            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
            const EditShaderParamMatrixValue* editValue =
                static_cast<const EditShaderParamMatrixValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type) - 1; // パケットに含めていないのでサイズが一つ余分なので減らす
            SetMatShaderParamTexSrt(
                modelObj, reinterpret_cast<const char*>(editValue->paramName),
                arg.index, arg.indexSize,
                mode,
                type,
                editValue->value.a, valueSize);
        }
        break;
    default:
        result = false;
    }
    return result;
} // NOLINT (readability/fn_size)

bool EditBones(ModelObj* modelObj, const EditBoneArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(modelObj, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    bool result = true;
    switch(arg.editTargetKind)
    {
    case EDIT_TARGET_BONE_VISIBILITY:
    case EDIT_TARGET_BONE_BILLBOARD:
        {
            const EditSimpleValue* editValue =
                static_cast<const EditSimpleValue*>(arg.value);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_BONE_VISIBILITY:
                SetBoneVisibility(modelObj, arg.index, arg.indexSize, editValue->bValue);
                break;
            case EDIT_TARGET_BONE_BILLBOARD:
                SetBoneBillboard(modelObj, arg.index, arg.indexSize, editValue->flag);
                break;

            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    default:
        result = false;
    }
    return result;
}

bool EditRenderInfoArraySize(ModelObj* modelObj, const char* labelName, const RenderInfoEditInfo* info)
{
    NW_G3D_ASSERT_NOT_NULL(modelObj);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(labelName, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(info, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    MaterialObj* materialObj = modelObj->GetMaterial(info->materialIndex);

    ResRenderInfo* resRenderInfo = materialObj->GetResource()->GetRenderInfo(labelName);
    if (resRenderInfo == NULL)
    {
        return false;
    }
    resRenderInfo->ref().arrayLength = static_cast<u16>(info->arraySize);
    return true;
}

void SetShaderParamAnimationCurve(ResShaderParamAnim* resShaderParamAnim, int materialIndex)
{
    NW_G3D_ASSERT_NOT_NULL(resShaderParamAnim);
    ResShaderParamMatAnim* resShaderParamMatAnim = resShaderParamAnim->GetMatAnim(materialIndex);
    if (resShaderParamMatAnim == NULL)
    {
        return;
    }
    NW_G3D_EDIT_PRINT("%s\n", resShaderParamMatAnim->GetName());

    ResAnimCurve* resAnimCurve = resShaderParamMatAnim->GetCurve(0);
    (void)resAnimCurve;
}

#define CASE_RETURN_STRING(cmd) case cmd: return #cmd;

const char* GetEditCommandString(CommandFlag command)
{
    switch (command)
    {
        CASE_RETURN_STRING(FILEDATA_LOAD_FILE_COMMAND_FLAG);
        CASE_RETURN_STRING(FILEDATA_UNLOAD_FILE_COMMAND_FLAG);
        CASE_RETURN_STRING(FILEDATA_RELOAD_FILE_COMMAND_FLAG);
        CASE_RETURN_STRING(FILEDATA_UNLOAD_ALL_COMMAND_FLAG);

        CASE_RETURN_STRING(SYSTEM_PING_RECV_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_PING_SEND_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_BEGIN_FREEZE_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_END_FREEZE_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_RUNTIME_STATE_NORMAL_COMMAND_FLAG);
        CASE_RETURN_STRING(SYSTEM_RUNTIME_ERROR_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_MATERIAL_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_SEND_ATTACH_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RECV_ATTACH_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SEND_DETACH_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_FILE_LOADED_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_FILE_RELOADED_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SEND_RENDER_INFO_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RECV_RENDER_INFO_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG);

        CASE_RETURN_STRING(ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG);
        CASE_RETURN_STRING(ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG);
        CASE_RETURN_STRING(ANIMATION_PLAY_POLICY_COMMAND_FLAG);
        CASE_RETURN_STRING(ANIMATION_FRAME_STEP_COMMAND_FLAG);
        CASE_RETURN_STRING(ANIMATION_FRAME_COUNT_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_SHADER_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_BONE_COMMAND_FLAG);

        CASE_RETURN_STRING(EDIT_MODEL_BONE_BIND_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_MODEL_LAYOUT_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG);
        CASE_RETURN_STRING(EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG);

        CASE_RETURN_STRING(MODEL_ANIMATION_BIND_COMMAND_FLAG);
        CASE_RETURN_STRING(MODEL_ANIMATION_UNBIND_COMMAND_FLAG);
        CASE_RETURN_STRING(MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG);
        CASE_RETURN_STRING(MODEL_ANIMATION_PLAY_COMMAND_FLAG);
        CASE_RETURN_STRING(MODEL_ANIMATION_STOP_COMMAND_FLAG);
        CASE_RETURN_STRING(MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG);

        CASE_RETURN_STRING(SCENE_ANIMATION_BIND_COMMAND_FLAG);
        CASE_RETURN_STRING(SCENE_ANIMATION_UNBIND_COMMAND_FLAG);
        CASE_RETURN_STRING(SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG);

        CASE_RETURN_STRING(PICK_RUNTIME_MODEL_COMMAND_FLAG);
        CASE_RETURN_STRING(PICK_RUNTIME_MATERIAL_COMMAND_FLAG);
        CASE_RETURN_STRING(PICK_RUNTIME_BONE_COMMAND_FLAG);
        CASE_RETURN_STRING(PICK_RUNTIME_SHAPE_COMMAND_FLAG);

        CASE_RETURN_STRING(OTHER_EXECUTE_USER_SCRIPT_FLAG);
    default:
        return "Unkown Edit Command";
    }
}

const char* GetFileDataKindString(FileDataKind kind)
{
    switch(kind)
    {
    CASE_RETURN_STRING(FILEDATA_MODEL);
    CASE_RETURN_STRING(FILEDATA_TEXTURE);
    CASE_RETURN_STRING(FILEDATA_SHADER_PARAM_ANIM);
    CASE_RETURN_STRING(FILEDATA_MAT_COLOR_ANIM);
    CASE_RETURN_STRING(FILEDATA_TEXTURE_SRT_ANIM);
    CASE_RETURN_STRING(FILEDATA_TEXTURE_PATTERN_ANIM);
    CASE_RETURN_STRING(FILEDATA_SKELETAL_ANIM);
    CASE_RETURN_STRING(FILEDATA_BONE_VISIBILITY_ANIM);
    CASE_RETURN_STRING(FILEDATA_MAT_VISIBILITY_ANIM);
    CASE_RETURN_STRING(FILEDATA_SHAPE_ANIM);
    CASE_RETURN_STRING(FILEDATA_SCENE_ANIM);
    CASE_RETURN_STRING(FILEDATA_SHADER_DEFINE);
    CASE_RETURN_STRING(FILEDATA_SHADER_ARCHIVE);
    CASE_RETURN_STRING(FILEDATA_SHADER_PROGRAM);
    default:
        return "Unknown FileDataKind";
    }
}

const char* GetAnimKindString(EditAnimKind kind)
{
    switch(kind)
    {
    CASE_RETURN_STRING(EDIT_SHADER_PARAM_ANIM);
    CASE_RETURN_STRING(EDIT_COLOR_ANIM);
    CASE_RETURN_STRING(EDIT_TEX_SRT_ANIM);
    CASE_RETURN_STRING(EDIT_TEX_PATTERN_ANIM);
    CASE_RETURN_STRING(EDIT_SKELETAL_ANIM);
    CASE_RETURN_STRING(EDIT_BONE_VIS_ANIM);
    CASE_RETURN_STRING(EDIT_MAT_VIS_ANIM);
    CASE_RETURN_STRING(EDIT_SHAPE_ANIM);
    default:
        return "Unknown EditAnimKind";
    }
}

const char* GetResName(ResFile* pResFile, FileDataKind kind)
{
    switch(kind)
    {
    case FILEDATA_MODEL:
        return pResFile->GetModelName(0);
    case FILEDATA_TEXTURE:
        return pResFile->GetTextureName(0);
    case FILEDATA_SHADER_PARAM_ANIM:
        return pResFile->GetShaderParamAnimName(0);
    case FILEDATA_MAT_COLOR_ANIM:
        return pResFile->GetColorAnimName(0);
    case FILEDATA_TEXTURE_SRT_ANIM:
        return pResFile->GetTexSrtAnimName(0);
    case FILEDATA_TEXTURE_PATTERN_ANIM:
        return pResFile->GetTexPatternAnimName(0);
    case FILEDATA_SKELETAL_ANIM:
        return pResFile->GetSkeletalAnimName(0);
    case FILEDATA_BONE_VISIBILITY_ANIM:
        return pResFile->GetBoneVisAnimName(0);
    case FILEDATA_MAT_VISIBILITY_ANIM:
        return pResFile->GetMatVisAnimName(0);
    case FILEDATA_SHAPE_ANIM:
        return pResFile->GetShapeAnimName(0);
    case FILEDATA_SCENE_ANIM:
        return pResFile->GetSceneAnimName(0);
    case FILEDATA_SHADER_ARCHIVE:
    case FILEDATA_SHADER_DEFINE:
    case FILEDATA_SHADER_PROGRAM:
        return pResFile->GetName();
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    return NULL;
}

FileDataKind ConvertToFileDataKind(nw::g3d::edit::EditAnimKind kind)
{
    switch (kind)
    {
    case nw::g3d::edit::EDIT_SHADER_PARAM_ANIM: return FILEDATA_SHADER_PARAM_ANIM;
    case nw::g3d::edit::EDIT_COLOR_ANIM       : return FILEDATA_MAT_COLOR_ANIM;
    case nw::g3d::edit::EDIT_TEX_SRT_ANIM     : return FILEDATA_TEXTURE_SRT_ANIM;
    case nw::g3d::edit::EDIT_TEX_PATTERN_ANIM : return FILEDATA_TEXTURE_PATTERN_ANIM;
    case nw::g3d::edit::EDIT_SKELETAL_ANIM    : return FILEDATA_SKELETAL_ANIM;
    case nw::g3d::edit::EDIT_BONE_VIS_ANIM    : return FILEDATA_BONE_VISIBILITY_ANIM;
    case nw::g3d::edit::EDIT_MAT_VIS_ANIM     : return FILEDATA_MAT_VISIBILITY_ANIM;
    case nw::g3d::edit::EDIT_SHAPE_ANIM       : return FILEDATA_SHAPE_ANIM;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    return FILEDATA_KIND_ERROR;
}

} // namespace nw::g3d::edit::detail

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
