﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderProgram.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResShader.h>
#include "g3d_EditSocket.h"
#include "g3d_EditUtility.h"
#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace edit::ut::detail;

EditShaderProgram::EditShaderProgram(
    IAllocator* allocator,
    u32 shaderArchiveKey,
    int shadingModelIndex,
    int shaderProgramIndex,
    ResShaderProgram* resShaderProgram)
    : m_pAllocator(allocator)
    , m_pTargetResShaderProgram(resShaderProgram)
    , m_pEditResShaderProgramData(NULL)
    , m_pUpdateResShaderArchive(NULL)
    , m_OriginalHandle(resShaderProgram->ref().handle)

    , m_OriginalFlag(resShaderProgram->ref().flag)
    , m_OriginalNumSampler(resShaderProgram->ref().numSampler)
    , m_OriginalNumUniformBlock(resShaderProgram->ref().numUniformBlock)
    , m_OriginalAttribuActiveFlag(resShaderProgram->ref().attribActiveFlag)

    , m_WorkBuffer(allocator)
    , m_OptionInfos(allocator, DEFAULT_ALIGNMENT)
    , m_ShaderArchiveKey(shaderArchiveKey)
    , m_ShadingModelIndex(shadingModelIndex)
    , m_ShaderProgramIndex(shaderProgramIndex)
    , m_PacketEnableFlag(false)
    , m_SendInfoCallbackFlag(false)
{
    NW_G3D_ASSERT_NOT_NULL(allocator);
    NW_G3D_ASSERT_NOT_NULL(resShaderProgram);

    NW_G3D_EDIT_ASSERT(NUM_STAGE==ResShaderProgram::NUM_STAGE); // とりあえず

    for (int stage = 0; stage < NUM_STAGE; ++stage)
    {
        m_OriginalNumALUInst[stage] = resShaderProgram->ref().numALUInst[stage];
        m_OriginalNumTfetchInst[stage] = resShaderProgram->ref().numTfetchInst[stage];
        m_OriginalNumVfetchInst[stage] = resShaderProgram->ref().numVfetchInst[stage];
        m_OriginalNumCflowInst[stage] = resShaderProgram->ref().numCflowInst[stage];
    }

    m_OriginalOffsetSamplerTable = resShaderProgram->ref().ofsSamplerTable.to_ptr();
    m_OriginalOffsetUniformBlockTable = resShaderProgram->ref().ofsUniformBlockTable.to_ptr();
    m_OriginalOffsetVertexShader = resShaderProgram->ref().ofsVertexShader.to_ptr();
    m_OriginalOffsetGeometryShader = resShaderProgram->ref().ofsGeometryShader.to_ptr();
    m_OriginalOffsetGeometryCopyShader = resShaderProgram->ref().ofsGeometryCopyShader.to_ptr();
    m_OriginalOffsetFragmentShader = resShaderProgram->ref().ofsFragmentShader.to_ptr();
    m_OriginalOffsetComputeShader = resShaderProgram->ref().ofsComputeShader.to_ptr();
}

void
EditShaderProgram::Cleanup()
{
    // 初期値に戻す
    {
        // オリジナルのプログラムの場合は Cleanup しない。
        // Cafe 版は GX2 構造体がプログラム間で共有されている可能性があるため。
        // GL 版は オリジナルのハンドルを破棄しないようにするため。
        if (m_pUpdateResShaderArchive != NULL)
        {
            m_pTargetResShaderProgram->Cleanup();
        }

        m_pTargetResShaderProgram->ref().flag = m_OriginalFlag;
        m_pTargetResShaderProgram->ref().numSampler = m_OriginalNumSampler;
        m_pTargetResShaderProgram->ref().numUniformBlock = m_OriginalNumUniformBlock;
        m_pTargetResShaderProgram->ref().attribActiveFlag = m_OriginalAttribuActiveFlag;
        for (int stage = 0; stage < NUM_STAGE; ++stage)
        {
            m_pTargetResShaderProgram->ref().numALUInst[stage] = m_OriginalNumALUInst[stage];
            m_pTargetResShaderProgram->ref().numTfetchInst[stage] = m_OriginalNumTfetchInst[stage];
            m_pTargetResShaderProgram->ref().numVfetchInst[stage] = m_OriginalNumVfetchInst[stage];
            m_pTargetResShaderProgram->ref().numCflowInst[stage] = m_OriginalNumCflowInst[stage];
        }

        m_pTargetResShaderProgram->ref().ofsComputeShader.set_ptr(m_OriginalOffsetComputeShader);
        m_pTargetResShaderProgram->ref().ofsFragmentShader.set_ptr(m_OriginalOffsetFragmentShader);
        m_pTargetResShaderProgram->ref().ofsGeometryCopyShader.set_ptr(m_OriginalOffsetGeometryCopyShader);
        m_pTargetResShaderProgram->ref().ofsGeometryShader.set_ptr(m_OriginalOffsetGeometryShader);
        m_pTargetResShaderProgram->ref().ofsVertexShader.set_ptr(m_OriginalOffsetVertexShader);
        m_pTargetResShaderProgram->ref().ofsSamplerTable.set_ptr(m_OriginalOffsetSamplerTable);
        m_pTargetResShaderProgram->ref().ofsUniformBlockTable.set_ptr(m_OriginalOffsetUniformBlockTable);

        m_pTargetResShaderProgram->ref().handle = m_OriginalHandle;

#if NW_G3D_IS_HOST_WIN
        m_pTargetResShaderProgram->Setup();
#endif
    }

    ClearOptionInfo();

    m_WorkBuffer.Clear();
}

void
EditShaderProgram::Destroy()
{
    if (m_pUpdateResShaderArchive != NULL)
    {
        m_pAllocator->Free(m_pUpdateResShaderArchive);
        m_pUpdateResShaderArchive = NULL;
    }

    m_OptionInfos.Destroy();

    m_pAllocator = NULL;
}

bool
EditShaderProgram::SendModifiedShaderProgram(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);

    if (!m_PacketEnableFlag)
    {
        return false;
    }

    bool result = socket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_ModifiedShaderProgramBufferSize);

    if (result)
    {
        m_PacketEnableFlag = false;
    }
    return result;
}

bool
EditShaderProgram::MakeModifiedShaderProgramPacket()
{
    m_PacketEnableFlag = false;
    size_t bufferSize = CalcWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_ModifiedShaderProgramBufferSize = bufferSize;
    MakeModifiedShaderProgramBuffer();
    m_PacketEnableFlag = true;
    return true;
}

bool
EditShaderProgram::PushOptionInfo(const char* option, const char* choice)
{
    NW_G3D_ASSERT_NOT_NULL(option);
    NW_G3D_ASSERT_NOT_NULL(choice);

    size_t bufferSize = sizeof(OptionInfo);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    // オプション情報登録作業
    {
        OptionInfo* optionInfo = new (buffer) OptionInfo(m_pAllocator);
        if (!optionInfo->Setup(option, choice))
        {
            optionInfo->~OptionInfo();
            m_pAllocator->Free(optionInfo);
            return false;
        }

        if (!m_OptionInfos.PushBack(optionInfo))
        {
            optionInfo->~OptionInfo();
            m_pAllocator->Free(optionInfo);
            return false;
        }
    }
    return true;
}

void
EditShaderProgram::UpdateShaderProgram(ResShaderArchive* resShaderArchive)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(resShaderArchive);
    NW_G3D_EDIT_ASSERTMSG(resShaderArchive->GetShadingModelCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
    ResShadingModel* resShadingModel = resShaderArchive->GetShadingModel(0);

    NW_G3D_EDIT_ASSERTMSG(resShadingModel->GetShaderProgramCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(resShadingModel, GetName()));
    ResShaderProgram* resShaderProgram = resShadingModel->GetShaderProgram(0);
    ResShaderProgramData* resShaderProgramData = resShaderProgram->ptr();

    // オリジナルのプログラムの場合は Cleanup しない。
    // Cafe 版は GX2 構造体がプログラム間で共有されている可能性があるため。
    // GL 版は オリジナルのハンドルを破棄しないようにするため。
    if (m_pUpdateResShaderArchive != NULL)
    {
        m_pTargetResShaderProgram->Cleanup();
    }
    else
    {
        m_pTargetResShaderProgram->ref().handle = 0;
    }

    m_pTargetResShaderProgram->ref().flag = resShaderProgramData->flag & ~ResShaderProgram::INITIALIZED;

    m_pTargetResShaderProgram->ref().numSampler = resShaderProgramData->numSampler;
    m_pTargetResShaderProgram->ref().numUniformBlock = resShaderProgramData->numUniformBlock;
    m_pTargetResShaderProgram->ref().attribActiveFlag = resShaderProgramData->attribActiveFlag;
    for (int stage = 0; stage < NUM_STAGE; ++stage)
    {
        m_pTargetResShaderProgram->ref().numALUInst[stage] = resShaderProgramData->numALUInst[stage];
        m_pTargetResShaderProgram->ref().numTfetchInst[stage] = resShaderProgramData->numTfetchInst[stage];
        m_pTargetResShaderProgram->ref().numVfetchInst[stage] = resShaderProgramData->numVfetchInst[stage];
        m_pTargetResShaderProgram->ref().numCflowInst[stage] = resShaderProgramData->numCflowInst[stage];
    }

    m_pTargetResShaderProgram->ref().ofsComputeShader.set_ptr(resShaderProgramData->ofsComputeShader.to_ptr());
    m_pTargetResShaderProgram->ref().ofsFragmentShader.set_ptr(resShaderProgramData->ofsFragmentShader.to_ptr());
    m_pTargetResShaderProgram->ref().ofsGeometryCopyShader.set_ptr(resShaderProgramData->ofsGeometryCopyShader.to_ptr());
    m_pTargetResShaderProgram->ref().ofsGeometryShader.set_ptr(resShaderProgramData->ofsGeometryShader.to_ptr());
    m_pTargetResShaderProgram->ref().ofsVertexShader.set_ptr(resShaderProgramData->ofsVertexShader.to_ptr());
    m_pTargetResShaderProgram->ref().ofsSamplerTable.set_ptr(resShaderProgramData->ofsSamplerTable.to_ptr());
    m_pTargetResShaderProgram->ref().ofsUniformBlockTable.set_ptr(resShaderProgramData->ofsUniformBlockTable.to_ptr());

    m_pEditResShaderProgramData = resShaderProgramData;

    if (m_pUpdateResShaderArchive != NULL)
    {
        m_pAllocator->Free(m_pUpdateResShaderArchive);
    }
    m_pUpdateResShaderArchive = resShaderArchive;

    m_pTargetResShaderProgram->Setup();
}

void
EditShaderProgram::ClearOptionInfo()
{
    int size = m_OptionInfos.Size();
    for (int i = 0; i < size; ++i)
    {
        OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
        optionInfo->~OptionInfo();
        m_pAllocator->Free(optionInfo);
        optionInfo = NULL;
    }
    m_OptionInfos.Clear();
}

size_t
EditShaderProgram::CalcWorkBufferSize()
{
    int optionInfoSize = m_OptionInfos.Size();

    size_t size = sizeof(PacketHeader);
    size += sizeof(ShaderProgramSendInfo);

    if (optionInfoSize == 0)
    {
        return size;
    }

    size += sizeof(ShaderProgramOptionInfo) * optionInfoSize;
    {
        for (int i = 0; i < optionInfoSize; ++i)
        {
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            optionInfo->m_OptionOffset = size;
            size += optionInfo->m_OptionBinarySize;
            optionInfo->m_ChoiceOffset = size;
            size += optionInfo->m_ChoiceBinarySize;
        }
    }
    return size;
}

void
EditShaderProgram::MakeModifiedShaderProgramBuffer()
{
    // CalcWorkBufferSize が成功している前提で実行

    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(m_pUpdateResShaderArchive, GetName()));

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NW_G3D_EDIT_MAGIC;
    header->verWord = NW_G3D_EDIT_VERSION;
    header->command = EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG;
    header->dataSize = m_ModifiedShaderProgramBufferSize - sizeof(PacketHeader);

    int optionInfoCount = m_OptionInfos.Size();

    ShaderProgramSendInfo* sendInfo = g3d::ut::AddOffset<ShaderProgramSendInfo>(buffer, sizeof(PacketHeader));
    sendInfo->shaderArchiveKey = m_ShaderArchiveKey;
    sendInfo->shadingModelIndex = m_ShadingModelIndex;
    sendInfo->shaderProgramIndex = m_ShaderProgramIndex;
    sendInfo->optionInfoNum = static_cast<u32>(optionInfoCount);

    size_t optionInfoSize = sizeof(ShaderProgramOptionInfo);
    ShaderProgramOptionInfo* firstShaderProgramOptionInfo = g3d::ut::AddOffset<ShaderProgramOptionInfo>(sendInfo, sizeof(ShaderProgramSendInfo));
    {
        for (int i = 0; i < optionInfoCount; ++i)
        {
            ShaderProgramOptionInfo* shaderProgramOptionInfo = g3d::ut::AddOffset<ShaderProgramOptionInfo>(firstShaderProgramOptionInfo, i * optionInfoSize);
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            shaderProgramOptionInfo->optionOffset = optionInfo->m_OptionOffset;
            shaderProgramOptionInfo->choiceOffset = optionInfo->m_ChoiceOffset;
        }
    }

    {
        for (int i = 0; i < optionInfoCount; ++i)
        {
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            {
                char* name = g3d::ut::AddOffset<char>(buffer, optionInfo->m_OptionOffset);
                memcpy(name, optionInfo->m_Option.GetStr(), optionInfo->m_Option.GetLength());
                name[optionInfo->m_Option.GetLength()] = 0;
            }

            {
                char* name = g3d::ut::AddOffset<char>(buffer, optionInfo->m_ChoiceOffset);
                memcpy(name, optionInfo->m_Choice.GetStr(), optionInfo->m_Choice.GetLength());
                name[optionInfo->m_Choice.GetLength()] = 0;
            }
        }
    }
}

bool
EditShaderProgram::OptionInfo::Setup(const char* option, const char* choice)
{
    bool isFailed = false;
    isFailed |= !m_Option.Assign(option);
    isFailed |= !m_Choice.Assign(choice);

    if (isFailed)
    {
        m_Option.Clear();
        m_Choice.Clear();
    }

    m_OptionBinarySize = nw::g3d::ut::Align(m_Option.GetLength() + 1, DEFAULT_ALIGNMENT);
    m_ChoiceBinarySize = nw::g3d::ut::Align(m_Choice.GetLength() + 1, DEFAULT_ALIGNMENT);
    return !isFailed;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
