﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditSceneAnimObj.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace ut::detail;

bool
EditSceneAnimObj::ReloadSceneAnimObj(nw::g3d::res::ResFile* resFile)
{
    NW_G3D_ASSERT_NOT_NULL(resFile);

    NW_G3D_ASSERTMSG(resFile->GetSceneAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(resFile, GetName()));
    nw::g3d::res::ResSceneAnim* resAnim = resFile->GetSceneAnim(0);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resAnim, "%s\n", NW_G3D_RES_GET_NAME(resFile, GetName()));

    Detach();
    m_pResAnim = resAnim;
    m_pResFile = resFile;

    bool result = Attach();
    if (result)
    {
        result = Setup();
        if (!result)
        {
            Detach();
        }
    }
    return result;
}

bool
EditSceneAnimObj::Attach()
{
    int cameraAnimCount = m_pResAnim->GetCameraAnimCount();
    int lightAnimCount = m_pResAnim->GetLightAnimCount();
    int fogAnimCount = m_pResAnim->GetFogAnimCount();

    bool isFailed = false;
    // コールバックに渡すために、配列を作成
    {
        if (cameraAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<CameraAnimObj*>::CalcBufferSize(cameraAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                m_CameraAnimObjPtrArray.SetArrayBuffer(buffer, cameraAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (lightAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<LightAnimObj*>::CalcBufferSize(lightAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                m_LightAnimObjPtrArray.SetArrayBuffer(buffer, lightAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (fogAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<FogAnimObj*>::CalcBufferSize(fogAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                m_FogAnimObjPtrArray.SetArrayBuffer(buffer, fogAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }
    }
    // 配列作成が一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    isFailed = false;
    // Edit***AnimObj を生成
    {
        if (cameraAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditCameraAnimObj>::CalcBufferSize(cameraAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                 m_EditCameraAnimArray.SetArrayBuffer(buffer, cameraAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (lightAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditLightAnimObj>::CalcBufferSize(lightAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                m_EditLightAnimArray.SetArrayBuffer(buffer, lightAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (fogAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditFogAnimObj>::CalcBufferSize(fogAnimCount);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer != NULL) // バッファ確保失敗
            {
                m_EditFogAnimArray.SetArrayBuffer(buffer, fogAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }
    }
    // Edit***AnimObj生成が一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    isFailed = false;
    {
        for(int i = 0, end = cameraAnimCount; i < end; ++i)
        {
            ResCameraAnim* resAnim = m_pResAnim->GetCameraAnim(i);
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            new (editAnimObj) EditCameraAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    {
        for(int i = 0, end = lightAnimCount; i < end; ++i)
        {
            ResLightAnim* resAnim = m_pResAnim->GetLightAnim(i);
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            new (editAnimObj) EditLightAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    {
        for(int i = 0, end = fogAnimCount; i < end; ++i)
        {
            ResFogAnim* resAnim = m_pResAnim->GetFogAnim(i);
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            new (editAnimObj) EditFogAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    // Init に一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    return true;
} // NOLINT (readability/fn_size)

void
EditSceneAnimObj::Detach()
{
    {
        for(int i = 0, end = m_EditFogAnimArray.Size(); i < end; ++i)
        {
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            editAnimObj->Destroy();
        }

        void* buffer = m_EditFogAnimArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_EditFogAnimArray.Clear();
        }
    }

    {
        void* buffer = m_FogAnimObjPtrArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_FogAnimObjPtrArray.Clear();
        }
    }
    {
        for(int i = 0, end = m_EditLightAnimArray.Size(); i < end; ++i)
        {
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            editAnimObj->Destroy();
        }

        void* buffer = m_EditLightAnimArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_EditLightAnimArray.Clear();
        }
    }

    {
        void* buffer = m_LightAnimObjPtrArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_LightAnimObjPtrArray.Clear();
        }
    }

    {
        for(int i = 0, end = m_EditCameraAnimArray.Size(); i < end; ++i)
        {
            EditCameraAnimObj* editCameraAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            editCameraAnimObj->Destroy();
        }

        void* buffer = m_EditCameraAnimArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_EditCameraAnimArray.Clear();
        }
    }

    {
        void* buffer = m_CameraAnimObjPtrArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_CameraAnimObjPtrArray.Clear();
        }
    }
}

void
EditSceneAnimObj::EditCameraAnimCurve(
    int cameraIndex,
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize
)
{
    if (cameraIndex >= m_pResAnim->GetCameraAnimCount())
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(cameraIndex, m_pResAnim->GetCameraAnimCount());
        return;
    }

    EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(cameraIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve, resAnimCurveSize);
}

void
EditSceneAnimObj::EditLightAnimCurve(
    int lightIndex,
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    if (lightIndex >= m_pResAnim->GetLightAnimCount())
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(lightIndex, m_pResAnim->GetLightAnimCount());
        return;
    }
    EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(lightIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve, resAnimCurveSize);
}

void
EditSceneAnimObj::EditFogAnimCurve(
    int fogIndex,
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    if (fogIndex >= m_pResAnim->GetFogAnimCount())
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(fogIndex, m_pResAnim->GetFogAnimCount());
        return;
    }
    EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(fogIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve, resAnimCurveSize);
}

void
EditSceneAnimObj::Calc()
{
    {
        int animCount = m_CameraAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calc();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calc();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calc();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
}

void
EditSceneAnimObj::SetFrame(f32 frame)
{
    {
        int animCount = m_CameraAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
}

void
EditSceneAnimObj::SetStep(f32 step)
{
    {
        int animCount = m_CameraAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
}

void
EditSceneAnimObj::SetPlayPolicy(bool isLoopAnim)
{
    {
        int animCount = m_EditCameraAnimArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            EditCameraAnimObj* editAnim = m_EditCameraAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    {
        int animCount = m_EditLightAnimArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            EditLightAnimObj* editAnim = m_EditLightAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    {
        int animCount = m_EditFogAnimArray.Size();
        for (int i = 0; i < animCount; ++i)
        {
            EditFogAnimObj* editAnim = m_EditFogAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    m_IsLoopAnim = isLoopAnim;
}

bool
EditSceneAnimObj::Setup()
{
    bool isFailed = false;
    {
        for(int i = 0, end = m_EditCameraAnimArray.Size(); i < end; ++i)
        {
            EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    {
        for(int i = 0, end = m_EditLightAnimArray.Size(); i < end; ++i)
        {
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    {
        for(int i = 0, end = m_EditFogAnimArray.Size(); i < end; ++i)
        {
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    return !isFailed;
}

bool
EditCameraAnimObj::Init()
{
    int curveCount = m_pResAnim->GetCurveCount();

    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalcBufferSize(curveCount);

        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(i, m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditCameraAnimObj::Destroy()
{
    for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditCameraAnimObj::Setup()
{
    CameraAnimObj::InitArg initArg;
    initArg.DisableContext();
    initArg.Reserve(m_pResAnim);

    size_t bufferSize = CameraAnimObj::CalcBufferSize(initArg);
    bool result = m_WorkBuffer.Resize(bufferSize);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));//今は止める

    m_AnimObj.Init(initArg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditCameraAnimObj::EditCurve(
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    NW_G3D_ASSERTMSG(curveIndex != -1, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve->ptr(), resAnimCurveSize);
}

void
EditCameraAnimObj::SetPlayPolicy(bool isLoopAnim)
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Loop);
    }
    else
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Onetime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool
EditLightAnimObj::Init()
{
    int curveCount = m_pResAnim->GetCurveCount();
    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalcBufferSize(curveCount);

        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(i, m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditLightAnimObj::Destroy()
{
    for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditLightAnimObj::Setup()
{
    LightAnimObj::InitArg initArg;
    initArg.DisableContext();
    initArg.Reserve(m_pResAnim);

    size_t bufferSize = LightAnimObj::CalcBufferSize(initArg);
    bool result = m_WorkBuffer.Resize(bufferSize);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));//今は止める

    m_AnimObj.Init(initArg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditLightAnimObj::EditCurve(
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    NW_G3D_ASSERTMSG(curveIndex != -1, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve->ptr(), resAnimCurveSize);
}

void
EditLightAnimObj::SetPlayPolicy(bool isLoopAnim)
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Loop);
    }
    else
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Onetime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool
EditFogAnimObj::Init()
{
    int curveCount = m_pResAnim->GetCurveCount();

    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalcBufferSize(curveCount);

        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(i, m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditFogAnimObj::Destroy()
{
    for(int i = 0, end = m_EditAnimCurveArray.Size(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditFogAnimObj::Setup()
{
    FogAnimObj::InitArg initArg;
    initArg.DisableContext();
    initArg.Reserve(m_pResAnim);

    size_t bufferSize = FogAnimObj::CalcBufferSize(initArg);
    bool result = m_WorkBuffer.Resize(bufferSize);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));//今は止める

    m_AnimObj.Init(initArg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditFogAnimObj::EditCurve(
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    NW_G3D_ASSERTMSG(curveIndex != -1, "%s\n", NW_G3D_RES_GET_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve->ptr(), resAnimCurveSize);
}

void
EditFogAnimObj::SetPlayPolicy(bool isLoopAnim)
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Loop);
    }
    else
    {
        m_AnimFrameCtrl.Init(start, end, PlayPolicy_Onetime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool EditSceneAnimObj::ContainsLoopAnimInCameraAnim() const
{
    for(int i = 0, end = m_EditCameraAnimArray.Size(); i < end; ++i)
    {
        const EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

bool EditSceneAnimObj::ContainsLoopAnimInLightAnim() const
{
    for(int i = 0, end = m_EditLightAnimArray.Size(); i < end; ++i)
    {
        const EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

bool EditSceneAnimObj::ContainsLoopAnimInFogAnim() const
{
    for(int i = 0, end = m_EditFogAnimArray.Size(); i < end; ++i)
    {
        const EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
