﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditPickup.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/ut/g3d_Inlines.h>
#include "g3d_EditSocket.h"
#include "g3d_EditUtility.h"

namespace nw { namespace g3d { namespace edit { namespace detail {

bool
EditPickup::PushMaterialPickup(const ModelObj* modelObj, s32 materialIndex)
{
    size_t bufferSize = sizeof(MaterialPickup);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    // マテリアルピックアップ登録作業
    {
        MaterialPickup* pickup = new (buffer) MaterialPickup(modelObj, materialIndex);
        if (!m_MaterialPickups.PushBack(pickup))
        {
            m_pAllocator->Free(pickup);
            return false;
        }
    }
    return true;
}

void
EditPickup::Clear()
{
    {
        int size = m_MaterialPickups.Size();
        for (int i = 0;i < size; ++i)
        {
            MaterialPickup* pickup = m_MaterialPickups.UnsafeAt(i);
            m_pAllocator->Free(pickup);
        }
        m_MaterialPickups.Clear();
    }
}

void
EditPickup::Destroy()
{
    Clear();
    m_WorkBuffer.Clear();
    m_MaterialPickups.Destroy();
}

bool
EditPickup::MakePickupPacket()
{
    size_t bufferSize = CalcWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_PickupBufferSize = bufferSize;
    MakePickupPacketBuffer();
    return true;
}

bool
EditPickup::SendPickup(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);

    return socket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_PickupBufferSize);
}

size_t
EditPickup::CalcWorkBufferSize()
{
    int materialPickupCount = m_MaterialPickups.Size();

    if (materialPickupCount <= 0)
    {
        return 0;
    }

    size_t size = sizeof(PacketHeader);
    size += sizeof(PickupSendInfo);

    size_t materialPickupBufferSize = sizeof(MaterialPickup) * materialPickupCount;
    size += materialPickupBufferSize;

    return size;
}

void
EditPickup::MakePickupPacketBuffer()
{
    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NW_G3D_ASSERT_NOT_NULL(buffer);

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NW_G3D_EDIT_MAGIC;
    header->verWord = NW_G3D_EDIT_VERSION;
    header->command = PICK_TOOL_MATERIAL_COMMAND_FLAG;
    header->dataSize = m_PickupBufferSize - sizeof(PacketHeader);

    PickupSendInfo* sendInfo = g3d::ut::AddOffset<PickupSendInfo>(buffer, sizeof(PacketHeader));

    int materialPickupCount = m_MaterialPickups.Size();

    sendInfo->materialPickupNum = static_cast<u32>(materialPickupCount);
    sendInfo->padding[0] = 0;
    sendInfo->padding[1] = 0;
    sendInfo->padding[2] = 0;

    size_t materialPickupInfoSize = sizeof(MaterialPickupInfo);
    MaterialPickupInfo* firstInfo = g3d::ut::AddOffset<MaterialPickupInfo>(sendInfo, sizeof(PickupSendInfo));
    {
        for (int i = 0; i < materialPickupCount; ++i)
        {
            MaterialPickupInfo* info = g3d::ut::AddOffset<MaterialPickupInfo>(firstInfo, i * materialPickupInfoSize);
            MaterialPickup* pickup = m_MaterialPickups.UnsafeAt(i);

            info->modelObjKey = reinterpret_cast<u32>(pickup->m_pModelObj);
            info->materialIndex = pickup->m_MaterialIndex;
        }
    }
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
