﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include "ut/g3d_DynamicPtrArray.h"
#include "ut/g3d_DynamicArray.h"
#include "ut/g3d_RingBuffer.h"
#include "ut/g3d_DynamicLengthString.h"
#include "g3d_EditAnimObj.h"

namespace nw { namespace g3d {

class ModelObj;
class ModelAnimObj;

namespace res {

class ResModel;
class ResShaderArchive;
class ResFile;

} // namespace res

namespace edit {

class EditCallback;
class EditServer;
class IAllocator;

namespace detail {

class EditSocket;
class EditModelObj;
class EditAnimObj;
class EditShaderParamAnimObj;
class EditTexPatternAnimObj;
class EditMatVisibilityAnimObj;
class EditBoneVisibilityAnimObj;
class EditSceneAnimObj;
class EditShaderArchive;

class EditManager
{
public:
    struct ResFileInfo
    {
        FileDataKind kind;
        ResFile* pResFile;
        char retargetingHostModelName[NW_G3D_EDIT_FILENAME_MAX];
        u32 resFileKey;

        ResFileInfo()
            : kind(FILEDATA_KIND_ERROR)
            , pResFile(NULL)
            , resFileKey(0)
        {
            memset(retargetingHostModelName, 0, NW_G3D_EDIT_FILENAME_MAX);
        }
    };

public:
    EditManager(IAllocator* allocator, EditCallback* editCallback);
    ~EditManager();

    // モデル関連操作
    RuntimeErrorCode AttachModelObj(EditModelObj* pTarget, ModelObj* pModelObj);

    // ロード用の　EditModelObj を作成します。
    RuntimeErrorCode CreateEditModelObj(EditModelObj** pOut, nw::g3d::res::ResFile* pTargetResFile, nw::g3d::res::ResFile*pFirstLoadResFile);

    // アタッチ用の EditModelObj を作成します。
    RuntimeErrorCode CreateAttachedEditModelObj(nw::g3d::ModelObj* pAttachTargetModelObj, nw::g3d::res::ResFile*pFirstLoadResFile);

    bool AddEditModelObj(EditModelObj* editModelObj)
    {
        NW_G3D_ASSERT_NOT_NULL(editModelObj);
        return m_EditModelArray.PushBack(editModelObj);
    }

    bool IsAttaching(const ModelObj* pTarget) const
    {
        return m_AttachingModelObjArray.IndexOf(pTarget) >= 0;
    }

    void RemoveAttachingModelObj(const ModelObj* pTarget)
    {
        int index = m_AttachingModelObjArray.IndexOf(pTarget);
        NW_G3D_ASSERT(index >= 0);
        m_AttachingModelObjArray.Erase(index);
    }

    void DeleteEditModelObj(u32 key);
    void EditMaterials(const EditMaterialArg& arg);
    void EditBones(const EditBoneArg& arg);

    void UpdateBlink();
    void StartBlinking(const SelectTargetArg& arg);

    int GetActiveEditAnimIndex(const ModelObj* modelObj, int boundAnimIndex) const;
    int GetModelAnimBoundCount(const ModelObj* modelObj) const;

    bool HasModelObj(const nw::g3d::ModelObj* modelObj) const
    {
        NW_G3D_ASSERT_NOT_NULL(modelObj);
        if (FindSameModelObj(modelObj) != NULL)
        {
            return true;
        }

        if (FindSameAttachModelObjQueue(modelObj) != NULL)
        {
            // キューにたまっているものか判定
            // NULLの場合は、遅延処理に積まれてないので存在しない。
            return true;
        }

        if (IsAttaching(modelObj))
        {
            return true;
        }

        return false;
    }

    bool AddModelLayoutQueue(u32 modelKey, const f32 scale[3], const f32 rotate[3], const f32 translate[3]);

    EditModelObj* FindEditModelObj(u32 key) const;
    bool AddModelFileLoadedQueue(const EditModelObj* editModelObj, u32 toolKey);

    // アニメ関連操作
    EditAnimObj* AddEditAnimObj(nw::g3d::res::ResFile* pResFile, FileDataKind kind);
    EditAnimObj* AddEditShaderParamAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditColorAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditTexSrtAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditSkeletalAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditTexPatternAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditBoneVisibilityAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditMatVisibilityAnimObj(nw::g3d::res::ResFile* targetResFile);
    EditAnimObj* AddEditShapeAnimObj(nw::g3d::res::ResFile* targetResFile);

    void DeleteEditAnimObj(u32 key);
    void DeleteEditAnimObj(EditAnimObj* pEditAnimObj);

    void EditAnimCurve(const EditAnimCurveArg& arg);

    bool HasEditAnim() const;
    int GetEditAnimCount() const;
    float GetEditAnimFrameCount(int animIndex) const;
    const char* GetEditAnimName(int animIndex) const;
    EditAnimKind GetEditAnimKind(int animIndex) const;
    EditAnimObj* FindEditAnimObj(u32 animationResFileKey) const;
    EditAnimObj* FindBoundEditAnimObj(u32 modelKey, u32 boundAnimationResFileKey);
    bool AddAnimFileLoadedQueue(FileDataKind kind, u32 resFileKey, u32 toolKey);
    bool ReloadAnimResource(u32 resFileKey, nw::g3d::res::ResFile* resFile, FileDataKind kind);
    bool AddAnimFileReloadedQueue(FileDataKind kind, u32 newResFileKey, u32 oldResFileKey);

    // シーンアニメ操作
    EditSceneAnimObj* AddEditSceneAnimObj(nw::g3d::res::ResFile* targetResFile);
    void DeleteEditSceneAnimObj(u32 key);
    void EditSceneAnimCurve(const EditAnimCurveArg& arg);

    bool HasEditSceneAnim() const;
    void BindSceneAnimations(const BindAnimArg& arg);
    void UnbindSceneAnimations(const BindAnimArg& arg);
    bool ReloadEditSceneAnimObj(u32 resFileKey, nw::g3d::res::ResFile* resFile);
    EditSceneAnimObj* FindEditSceneAnimObj(u32 key) const;
    bool AddSceneAnimFileLoadedQueue(const EditSceneAnimObj* editSceneAnimObj, u32 toolKey);
    bool AddSceneAnimFileReloadedQueue(const EditSceneAnimObj* editAnimObj, u32 resFileKey);
    static void ExecuteBindSceneAnimCallback(EditSceneAnimObj* editAnimObj, EditCallback* callback);
    static void ExecuteUnbindSceneAnimCallback(EditSceneAnimObj* editAnimObj, EditCallback* callback);


    // シェーダ関連操作
    EditShaderArchive* CreateAttachEditShaderArchive(nw::g3d::res::ResShaderArchive* targetResShaderArchive);

    bool AddEditShaderArchive(EditShaderArchive* editShaderArchive)
    {
        NW_G3D_ASSERT_NOT_NULL(editShaderArchive);
        return m_EditShaderArchiveArray.PushBack(editShaderArchive);
    }

    void DeleteEditShaderArchive(u32 key);

    bool HasShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive) const
    {
        NW_G3D_ASSERT_NOT_NULL(resShaderArchive);
        if (FindSameResShaderArchive(resShaderArchive) == NULL &&
            FindSameAttachShaderArchiveQueue(resShaderArchive) == NULL)
        {
            // キューにたまっているものか判定
            // NULLの場合は、遅延処理に積まれてないので存在しない。
            return false;
        }
        return true;
    }
    void UpdateEditShaderArchive(u32 key, int shadingModelIndices[], u32 indexSize);
    EditShaderArchive* FindEditShaderArchive(u32 key) const;


    // その他
    void Cleanup();
    void CalcAnimations();
    void CalcSkeletalAnimations(const nw::g3d::ModelObj* modelObj);
    void Reset();
    void ClearEditManager(bool isCallbackEnabled = false);

    void ClearCalcFlagModelAnims();

    void DeleteAll(bool callbackEnabled);

    bool QueueAttachModel(nw::g3d::ModelObj* targetModel, const char* path);
    bool QueueDetachModel(const nw::g3d::ModelObj* targetModel);
    bool QueueDeleteEditModelObj(EditModelObj* editModelObj);

    void ExecuteCommandQueue(EditSocket* socket);
    void SendModifiedShaderPrograms(EditSocket* socket);

    bool QueueAttachShaderArchive(nw::g3d::res::ResShaderArchive* resShaderArchive, const char* path);
    bool QueueDetachShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive);
    bool QueueDeleteEditShaderArchive(EditShaderArchive* editShaderArchive);

    void SetAnimFlag(bool enable);
    bool GetAnimFlag() const
    {
        return m_AnimationFlag;
    }

    bool IsLoopAnim() const
    {
        return m_IsLoopAnim;
    }

    void SetFrame(float frame)
    {
        m_Frame = frame;
        SetAnimFlag(false);// フレームを設定した場合は、アニメーションを止める
    }

    float GetFrame() const
    {
        return m_Frame;
    }

    void SetFrameStep(f32 frameStep);
    float GetFrameStep() const
    {
        return m_FrameStep;
    }

    void SetPreviewFrameStepRate(f32 rate);
    float GetPreviewFrameStepRate() const
    {
        return m_PreviewFrameStepRate;
    }

    void SetFrameCount(float frameCount)
    {
        m_FrameCount = frameCount;
    }

    float GetFrameCount() const
    {
        return m_FrameCount;
    }

    void SetPlayPolicy(EditPlayPolicyKind kind)
    {
        // 同じ値なら処理しない
        if (kind == m_PlayPolicy)
        {
            return;
        }
        m_PlayPolicy = kind;
        ResetPlayPolicy();
    }

    void ResetPlayPolicy()
    {
        m_IsLoopAnim = CanLoopAnim();
        SetAnimLoopFlag(m_IsLoopAnim);
    }

    void EditShaderParamAnimCurve(EditShaderParamAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditTexPatternAnimCurve(EditTexPatternAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditMatVisibilityAnimCurve(EditMatVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditBoneVisibilityAnimCurve(EditBoneVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditShapeAnimCurve(EditShapeAnimObj* editAnimObj, const EditAnimCurveArg& arg);

    void BindAnimations(const BindAnimArg& arg);
    void UnbindAnimations(const BindAnimArg& arg);
    void ForceUnbindAnimations(EditModelObj* editModelObj);

    bool QueueErrorCommand(RuntimeErrorCode errorCode);

    nw::g3d::res::ResFile* LoadResFileWithoutCopy(void** pFileBuffer, size_t fileSize, FileDataKind kind);
    void DeleteUnusedResFiles();

    nw::g3d::res::ResFile* LoadAnimResFileWithoutCopy(void** pFileBuffer, size_t fileSize, FileDataKind kind);
    void DeleteAnimResFile(u32 resFileKey);

    //! @brief animResFileKeyを持つアニメのリターゲッティングホストモデルを設定します。
    void SetRetargetingHostModel(u32 animResFileKey, u32 retargetHostModelObjKey);

    ModelObj* GetModelObjFromEditModelObjs(const char* name);

    //! @brief animResFileKeyを持つアニメのリターゲッティングを解除します。
    void UnsetRetargetingHostModel(u32 animResFileKey);

    void PrintAllEditAnimObjs();
    void PrintEditAnimObjs(u32 resFileKey, u32 boundModelKey);
    void PrintAllResFiles();

    void ExecuteRuntimeErrorCommandQueue(EditSocket* pSocket);


private:
    struct AttachCommand
    {
        union
        {
            u32 key;
            ModelObj* modelObjPtr;
        } instance;

        union
        {
            u32 key;
            ResModel* resModelPtr;
            ResShaderArchive *resShaderArchivePtr;
        } resource;

        u32 kind;
        ut::detail::DynamicLengthString* pathPtr;
    };

    // シーンアニメ関連
    void EditCameraAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditLightAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg);
    void EditFogAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg);

    EditModelObj* FindSameModelObj(const ModelObj* modelObj) const;
    EditShaderArchive* FindSameResShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive) const;

    AttachCommand* FindSameAttachModelObjQueue(const ModelObj* modelObj) const;
    AttachCommand* FindSameAttachShaderArchiveQueue(const ResShaderArchive* resShaderArchive) const;

    bool AddModelFileReloadedQueue(const EditModelObj* editModelObj, u32 resFileKey);

    void ExecuteAttachCommandQueueImpl(EditSocket* socket);
    void ExecuteFileLoadCommandQueueImpl(EditSocket* socket);
    void ExecuteModelLayoutCommandQueue(EditSocket* socket);

    void UnbindAnimation(EditModelObj* pTargetEditModelObj, u32 targetAnimResFileKey);

    bool CanLoopAnim() const;
    void SetAnimLoopFlag(bool enable);

    void SetPreviewStep();
    float GetPreviewStep() const
    {
        return m_FrameStep * m_PreviewFrameStepRate;
    }

    bool IsBound(EditModelObj* pEditModelObj, u32 animResFileKey) const;

    ResFile* GetAnimResFile(u32 resFileKey)
    {
        for (int i = 0, end = m_AnimResFileInfoArray.Size(); i < end; ++i)
        {
            ResFileInfo info = m_AnimResFileInfoArray[i];
            if (resFileKey == GetResFileKeyFromResFile(info.pResFile))
            {
                return info.pResFile;
            }
        }

        return NULL;
    }

    ResFileInfo* GetAnimResFileInfo(u32 resFileKey)
    {
        for (int i = 0, end = m_AnimResFileInfoArray.Size(); i < end; ++i)
        {
            ResFileInfo* pInfo = &m_AnimResFileInfoArray[i];
            if (resFileKey == GetResFileKeyFromResFile(pInfo->pResFile))
            {
                return pInfo;
            }
        }

        return NULL;
    }

    int GetIndexOfAnimResFile(ResFile* pResFile) const
    {
        for (int i = 0, end = m_AnimResFileInfoArray.Size(); i < end; ++i)
        {
            ResFileInfo info = m_AnimResFileInfoArray[i];
            if (pResFile == info.pResFile)
            {
                return i;
            }
        }

        return -1;
    }

    void DeleteEditAnimObjBindingForAllModels(EditAnimObj* pTarget);

    static bool SendAttach(AttachPacket* packet, EditSocket* socket, const AttachCommand* attachCommand);

    static int CompareSameResModel(const EditModelObj* a, const ResModel* b);
    static int CompareSameModelObj(const EditModelObj* a, const ModelObj* b);
    static int CompareSameAttachModelObj(const AttachCommand* a, const ModelObj* b);
    static int CompareSameAttachShaderArchive(const AttachCommand* a, const ResShaderArchive* b);

    static int CompareSameKey(const EditModelObj* a, const u32* b);
    static int CompareSameResShaderArchive(const EditShaderArchive* a, const ResShaderArchive* b);
    static int CompareSameKey(const EditAnimObj* a, const u32* b);
    static int CompareSameKey(const EditSceneAnimObj* a, const u32* b);
    static int CompareSameAttachPath(const ut::detail::DynamicLengthString* a, const char* b);

private:
    static const int ATTACH_MODEL_COMMAND_QUEUE_MAX = 16;
    static const int FILE_LOADED_QUEUE_MAX = 16;
    static const int MODEL_LAYOUT_QUEUE_MAX = 16;
    static const int RUNTIME_ERROR_QUEUE_MAX = 16;

    bool m_IsWriteStarted;
    IAllocator* m_pAllocator;
    ut::detail::DynamicPtrArray<EditModelObj> m_EditModelArray;
    ut::detail::DynamicPtrArray<EditAnimObj> m_EditAnimArray;
    ut::detail::DynamicPtrArray<EditShaderArchive> m_EditShaderArchiveArray;
    ut::detail::DynamicPtrArray<EditSceneAnimObj> m_EditSceneAnimArray;
    EditCallback* m_pEditCallback;
    bool m_AnimationFlag;
    f32 m_Frame;
    f32 m_FrameStep;
    f32 m_PreviewFrameStepRate;
    EditPlayPolicyKind m_PlayPolicy;
    f32 m_FrameCount;
    bool m_IsLoopAnim;

    AttachPacket            m_AttachPacket;
    AttachCommand m_AttachCommandArray[ATTACH_MODEL_COMMAND_QUEUE_MAX];
    ut::detail::RingBuffer<AttachCommand> m_AttachCommandQueue;

    FileLoadedPacket        m_FileLoadedPacket;
    FileLoadedBlock m_FileLoadedBlockArray[FILE_LOADED_QUEUE_MAX];
    ut::detail::RingBuffer<FileLoadedBlock> m_FileLoadedBlockQueue;

    ModelLayoutEditPacket m_ModelLayoutPacket;
    ModelLayoutEditBlock m_ModelLayoutEditBlockArray[MODEL_LAYOUT_QUEUE_MAX];
    ut::detail::RingBuffer<ModelLayoutEditBlock> m_ModelLayoutEditBlockQueue;

    RuntimeErrorNotificationPacket m_RuntimeErrorPacket;
    RuntimeErrorNotificationInfo m_RuntimeErrorInfoArray[RUNTIME_ERROR_QUEUE_MAX];
    ut::detail::RingBuffer<RuntimeErrorNotificationInfo> m_RuntimeErrorInfoQueue;

    // デタッチが来た時にアタッチを中断できるようにアタッチ中のモデルを保持しておく
    ut::detail::DynamicPtrArray<ModelObj> m_AttachingModelObjArray;

    // EditManagerが確保したリソースを管理するための配列
    ut::detail::DynamicArray<ResFileInfo> m_ResFileInfoArray;

    // アニメーションは他のリソースと違ってモデルにバインドされたタイミングで
    // EditObjが作られるので、取り回しの違いの都合で別管理にする
    ut::detail::DynamicArray<ResFileInfo> m_AnimResFileInfoArray;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(EditManager);
};

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
