﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditManager.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include "g3d_EditUtility.h"
#include "g3d_EditModelObj.h"
#include "g3d_EditSkeletalAnimObj.h"
#include "g3d_EditTexPatternAnimObj.h"
#include "g3d_EditShaderParamAnimObj.h"
#include "g3d_EditVisibilityAnimObj.h"
#include "g3d_EditShapeAnimObj.h"
#include "g3d_EditSocket.h"
#include "g3d_EditSceneAnimObj.h"
#include "g3d_EditShaderArchive.h"

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/edit/g3d_IAllocator.h>
#include <nw/g3d/edit/g3d_EditServer.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

namespace {
    bool IsAnimFileData(FileDataKind kind)
    {
        return FILEDATA_SHADER_PARAM_ANIM <= kind && kind <= FILEDATA_SCENE_ANIM;
    }

    bool BindAnimation(
        EditModelObj* pEditModelObj,
        EditAnimObj* pEditAnimObj,
        float currentFrame,
        float previewStep)
    {
        bool success = pEditModelObj->AddBoundAnim(pEditAnimObj);
        if (!success)
        {
            return false;
        }

        success = pEditAnimObj->BindModelObj(pEditModelObj->GetTargetModelObj());
        if (!success)
        {
            int index = pEditModelObj->GetIndexOfBoundAnim(pEditAnimObj);
            NW_G3D_EDIT_ASSERT(index >= 0);
            pEditModelObj->RemoveBoundAnim(index);
            return false;
        }

        pEditAnimObj->SetFrame(pEditModelObj->GetTargetModelObj(), currentFrame);
        pEditAnimObj->SetStep(previewStep);
        return true;
    }

    template<typename TEditObjType>
    TEditObjType* FindEditObj(
        ut::detail::DynamicPtrArray<TEditObjType>& editObjArray,
        ResFile* pResFile)
    {
        int editAnimObjCount = editObjArray.Size();
        for (int animIndex = 0; animIndex < editAnimObjCount; ++animIndex)
        {
            TEditObjType* pObj = editObjArray.UnsafeAt(animIndex);
            if (pObj->IsResFileBound(pResFile))
            {
                return pObj;
            }
        }

        return NULL;
    }

    nw::g3d::res::ResFile* SetupBufferAsResFile(void* buffer, size_t fileSize)
    {
        NW_G3D_ASSERT(nw::g3d::res::ResFile::IsValid(buffer));
        ResFile* pResFile = nw::g3d::ResFile::ResCast(buffer);
        nw::g3d::fnd::CPUCache::Flush(buffer, fileSize);

        NW_G3D_ASSERT_NOT_NULL(pResFile);

        // edit 側で管理する ResFile なので、ここでSetupを行う
        pResFile->Setup();

        return pResFile;
    }
}

EditManager::EditManager(IAllocator* allocator, EditCallback* editCallback)
    : m_IsWriteStarted(false)
    , m_pAllocator(allocator)
    , m_pEditCallback(editCallback)
    , m_EditModelArray(DEFAULT_ALIGNMENT)
    , m_EditAnimArray(DEFAULT_ALIGNMENT)
    , m_EditShaderArchiveArray(DEFAULT_ALIGNMENT)
    , m_EditSceneAnimArray(DEFAULT_ALIGNMENT)
    , m_AnimationFlag(false)
    , m_Frame(0.f)
    , m_FrameStep(1.f)
    , m_PreviewFrameStepRate(1.f)
    , m_PlayPolicy(EDIT_PLAY_POLICY_AUTO)
    , m_FrameCount(0.f)
    , m_IsLoopAnim(false)
    , m_AttachCommandQueue(m_AttachCommandArray, ATTACH_MODEL_COMMAND_QUEUE_MAX)
    , m_FileLoadedBlockQueue(m_FileLoadedBlockArray, FILE_LOADED_QUEUE_MAX)
    , m_ModelLayoutEditBlockQueue(m_ModelLayoutEditBlockArray, MODEL_LAYOUT_QUEUE_MAX)
    , m_RuntimeErrorInfoQueue(m_RuntimeErrorInfoArray, RUNTIME_ERROR_QUEUE_MAX)
    , m_AttachingModelObjArray(DEFAULT_ALIGNMENT)
    , m_ResFileInfoArray(DEFAULT_ALIGNMENT)
    , m_AnimResFileInfoArray(DEFAULT_ALIGNMENT)
{
    NW_G3D_ASSERT_NOT_NULL(allocator);

    memset(&m_AttachCommandArray, 0, sizeof(AttachCommand) * ATTACH_MODEL_COMMAND_QUEUE_MAX);
    memset(&m_FileLoadedBlockArray, 0, sizeof(FileLoadedBlock) * FILE_LOADED_QUEUE_MAX);
    memset(&m_AttachPacket, 0, sizeof(AttachPacket));
    memset(&m_ModelLayoutEditBlockArray, 0, sizeof(ModelLayoutEditBlock) * MODEL_LAYOUT_QUEUE_MAX);

    m_ModelLayoutPacket.header.command = EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG;
    m_ModelLayoutPacket.header.magic = NW_G3D_EDIT_MAGIC;
    m_ModelLayoutPacket.header.verWord = NW_G3D_EDIT_VERSION;
    m_ModelLayoutPacket.header.dataSize = sizeof(ModelLayoutEditBlock);

    m_EditModelArray.SetAllocator(allocator);
    m_EditAnimArray.SetAllocator(allocator);
    m_EditShaderArchiveArray.SetAllocator(allocator);
    m_EditSceneAnimArray.SetAllocator(allocator);
    m_AttachingModelObjArray.SetAllocator(allocator);
    m_ResFileInfoArray.SetAllocator(allocator);
    m_AnimResFileInfoArray.SetAllocator(allocator);
}

EditManager::~EditManager()
{
    Cleanup();
}

bool EditManager::HasEditAnim() const
{
    return m_EditAnimArray.Size() > 0;
}

bool EditManager::HasEditSceneAnim() const
{
    return m_EditSceneAnimArray.Size() > 0;
}

int EditManager::GetEditAnimCount() const
{
    return m_EditAnimArray.Size();
}

float EditManager::GetEditAnimFrameCount(int animIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
    return m_EditAnimArray.At(animIndex)->GetFrameCount();
}

const char* EditManager::GetEditAnimName(int animIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
    EditAnimObj* pEditAnimObj = m_EditAnimArray.At(animIndex);
    return pEditAnimObj->GetName();
}

EditAnimKind EditManager::GetEditAnimKind(int animIndex) const
{
    NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
    EditAnimObj* pEditAnimObj = m_EditAnimArray.At(animIndex);
    return pEditAnimObj->GetAnimKind();
}

void
EditManager::DeleteAll(bool callbackEnabled)
{
    // アニメ
    {
        while (m_EditAnimArray.Size() > 0)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(0);
            DeleteEditAnimObj(pEditAnimObj);
        }
    }

    // モデル
    {
        int size = m_EditModelArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditModelObj* editModel = m_EditModelArray.UnsafeAt(i);

            // フラグ有効時は、コールバックを実行
            if (callbackEnabled)
            {
                editModel->ResetToOriginal(m_pEditCallback);
            }
            else
            {
                editModel->ResetToOriginal(NULL);
            }

            m_pAllocator->Free(editModel);
        }
        m_EditModelArray.Clear();

        m_AttachingModelObjArray.Clear();
    }

    // シェーダ
    {
        int size = m_EditShaderArchiveArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditShaderArchive* editShaderArchive = m_EditShaderArchiveArray.UnsafeAt(i);
            DeleteEditShaderArchive(GetKeyFromResShaderArchive(editShaderArchive->GetTargetResShaderArchive()));
        }
        m_EditShaderArchiveArray.Clear();
    }

    // シーン
    {
        int animCount = m_EditSceneAnimArray.Size();
        if (callbackEnabled && m_pEditCallback != NULL)
        {
            for (int i = 0; i < animCount; ++i)
            {
                EditSceneAnimObj* editAnimObj = m_EditSceneAnimArray.UnsafeAt(i);
                if (editAnimObj->IsBound())
                {
                    EditManager::ExecuteUnbindSceneAnimCallback(editAnimObj, m_pEditCallback);
                }
            }
        }

        for (int i = 0; i < animCount; ++i)
        {
            EditSceneAnimObj* editAnimObj = m_EditSceneAnimArray.UnsafeAt(i);
            editAnimObj->Detach();
            m_pAllocator->Free(editAnimObj);
            editAnimObj = NULL;
        }
        m_EditSceneAnimArray.Clear();
    }

    // リソースファイル
    {
        for (int i = 0, size = m_ResFileInfoArray.Size(); i < size; ++i)
        {
            ResFile* pResFile = m_ResFileInfoArray[i].pResFile;
            m_pAllocator->Free(pResFile);
        }

        m_ResFileInfoArray.Clear();

        for (int i = 0, size = m_AnimResFileInfoArray.Size(); i < size; ++i)
        {
            ResFileInfo resFileInfo = m_AnimResFileInfoArray[i];
            m_pAllocator->Free(resFileInfo.pResFile);
        }

        m_AnimResFileInfoArray.Clear();
    }
}

bool
EditManager::QueueAttachModel(nw::g3d::ModelObj* targetModel, const char* path)
{
    NW_G3D_ASSERT_NOT_NULL(targetModel);
    AttachCommand command;
    command.instance.modelObjPtr = targetModel;
    command.resource.resModelPtr = targetModel->GetResource();
    command.kind = ATTACH_MODEL;
    command.pathPtr = NULL;
    if (path != NULL)
    {
        size_t length = strlen(path);
        NW_G3D_ASSERTMSG(length < NW_G3D_EDIT_FILENAME_MAX, "%s\n", NW_G3D_RES_GET_NAME(targetModel->GetResource(), GetName()));
        void* buffer = m_pAllocator->Alloc(sizeof(ut::detail::DynamicLengthString), DEFAULT_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetModel->GetResource(), GetName()));
        ut::detail::DynamicLengthString* pathString = new (buffer) ut::detail::DynamicLengthString(m_pAllocator);
        bool result = pathString->Assign(path);
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetModel->GetResource(), GetName()));
        command.pathPtr = pathString;
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetModel->GetResource(), GetName()));
    }

    m_AttachingModelObjArray.PushBack(targetModel);

    return m_AttachCommandQueue.PushBack(command);
}

bool
EditManager::QueueDetachModel(const nw::g3d::ModelObj* targetModel)
{
    NW_G3D_ASSERT_NOT_NULL(targetModel);

    int index = m_AttachingModelObjArray.IndexOf(targetModel);
    if (index >= 0)
    {
        // アタッチ中の場合はアタッチをキャンセル
        RemoveAttachingModelObj(targetModel);
        return true;
    }

    // 多重削除防止のため指定モデルが存在しない場合は処理を抜ける
    EditModelObj* editModelObj = m_EditModelArray.Find<ModelObj>(targetModel, CompareSameModelObj);
    if (editModelObj == NULL)
    {
        return false;
    }

    return QueueDeleteEditModelObj(editModelObj);
}

bool
EditManager::QueueDeleteEditModelObj(EditModelObj* editModelObj)
{
    AttachCommand command;
    command.instance.modelObjPtr = editModelObj->GetTargetModelObj();
    command.resource.resModelPtr = editModelObj->GetTargetModelObj()->GetResource();
    command.kind = DETACH_MODEL;
    command.pathPtr = NULL;
    return m_AttachCommandQueue.PushBack(command);
}

bool
EditManager::QueueAttachShaderArchive(nw::g3d::res::ResShaderArchive* resShaderArchive, const char* path)
{
    NW_G3D_ASSERT_NOT_NULL(resShaderArchive);
    AttachCommand command;
    command.resource.resShaderArchivePtr = resShaderArchive;
    command.instance.key = command.resource.key;
    command.kind = ATTACH_SHADER_ARCHIVE;
    command.pathPtr = NULL;
    if (path != NULL)
    {
        size_t length = strlen(path);
        NW_G3D_ASSERTMSG(length < NW_G3D_EDIT_FILENAME_MAX, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
        void* buffer = m_pAllocator->Alloc(sizeof(ut::detail::DynamicLengthString), DEFAULT_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
        ut::detail::DynamicLengthString* pathString = new (buffer) ut::detail::DynamicLengthString(m_pAllocator);
        bool result = pathString->Assign(path);
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
        command.pathPtr = pathString;
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
    }
    return m_AttachCommandQueue.PushBack(command);
}

bool
EditManager::QueueDetachShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL(resShaderArchive);

    // 多重削除防止のため指定シェーダアーカイブが存在しない場合は処理を抜ける
    EditShaderArchive* editShaderArchive = m_EditShaderArchiveArray.Find<ResShaderArchive>(resShaderArchive, CompareSameResShaderArchive);

    if (editShaderArchive == NULL)
    {
        return false;
    }

    return QueueDeleteEditShaderArchive(editShaderArchive);
}

bool
EditManager::QueueDeleteEditShaderArchive(EditShaderArchive* editShaderArchive)
{
    AttachCommand command;
    command.resource.resShaderArchivePtr = editShaderArchive->GetTargetResShaderArchive();
    command.instance.key = command.resource.key;
    command.kind = DETACH_SHADER_ARCHIVE;
    command.pathPtr = NULL;
    return m_AttachCommandQueue.PushBack(command);
}

RuntimeErrorCode
EditManager::CreateEditModelObj(EditModelObj** pOut, nw::g3d::res::ResFile* pOriginalResFile, nw::g3d::res::ResFile* pFirstLoadResFile)
{
    NW_G3D_ASSERT_NOT_NULL(pOriginalResFile);
    NW_G3D_ASSERTMSG(pOriginalResFile->GetModelCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(pOriginalResFile, GetName()));

    nw::g3d::res::ResModel* pOriginalResModel = pOriginalResFile->GetModel(0);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pOriginalResModel, "%s\n", NW_G3D_RES_GET_NAME(pOriginalResFile, GetName()));

    size_t size = sizeof(EditModelObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
    }

    EditModelObj* pEditModelObj = new (buffer) EditModelObj(m_pAllocator, pOriginalResFile, pOriginalResModel, pFirstLoadResFile);
    NW_G3D_ASSERT_NOT_NULL(pEditModelObj);

    bool result = this->AddEditModelObj(pEditModelObj);
    NW_G3D_ASSERT(result);

    *pOut = pEditModelObj;
    return RUNTIME_ERROR_CODE_NO_ERROR;
}

RuntimeErrorCode
EditManager::CreateAttachedEditModelObj(nw::g3d::ModelObj* pAttachTargetModelObj, ResFile* pFirstLoadResFile)
{
    NW_G3D_ASSERT_NOT_NULL(pAttachTargetModelObj);

    nw::g3d::res::ResModel* pAttachTargetResModel = pAttachTargetModelObj->GetResource();
    NW_G3D_ASSERT_NOT_NULL(pAttachTargetResModel);

    size_t size = sizeof(EditModelObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
    }

    EditModelObj* pEditModelObj = new (buffer) EditModelObj(m_pAllocator, pAttachTargetResModel, pFirstLoadResFile);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pEditModelObj, "%s\n", NW_G3D_RES_GET_NAME(pAttachTargetModelObj->GetResource(), GetName()));

    bool result = this->AddEditModelObj(pEditModelObj);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(pAttachTargetModelObj->GetResource(), GetName()));

    RuntimeErrorCode errorCode = this->AttachModelObj(pEditModelObj, pAttachTargetModelObj);
    if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
    {
        this->DeleteEditModelObj(pEditModelObj->GetKey());
        return errorCode;
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

void
EditManager::DeleteEditModelObj(u32 key)
{
    EditModelObj* deleteEditModel = FindEditModelObj(key);
    if (deleteEditModel != NULL)
    {
        NW_G3D_EDIT_PRINT("Unload Target Model Name: %s\n", deleteEditModel->GetResModel()->GetName());
        {
            // バインドされているEditAnimObj をアンバインド
            ForceUnbindAnimations(deleteEditModel);

            // 状態を初期状態へ戻す
            deleteEditModel->ResetToOriginal(m_pEditCallback);
        }

        int index = m_EditModelArray.IndexOf(deleteEditModel);
        NW_G3D_ASSERTMSG(index >= 0, "%s\n", NW_G3D_RES_GET_NAME(deleteEditModel->GetTargetModelObj()->GetResource(), GetName()));
        m_EditModelArray.Erase(index);
        m_pAllocator->Free(deleteEditModel);
    }
    else
    {
        // EditModelObj が見つからない場合はアタッチ中の可能性あり
        ModelObj* pModelObj = GetModelObjFromKey(key);
        if (IsAttaching(pModelObj))
        {
            RemoveAttachingModelObj(pModelObj);
            return;
        }
    }
}

void
EditManager::EditMaterials(const EditMaterialArg& arg)
{
    EditModelObj* editModel = FindEditModelObj(arg.key);
    if (editModel == NULL)
    {
        return;
    }

    if (arg.editTargetKind == EDIT_TARGET_MATERIAL_VISIBILITY)
    {
        const EditSimpleValue* editValue =static_cast<const EditSimpleValue*>(arg.value);
        if (!editValue->bValue)
        {
            SelectTargetArg selectTargetArg;
            selectTargetArg.modelObj = editModel->GetTargetModelObj();
            selectTargetArg.targetKind = SelectTargetArg::TARGET_MATERIAL;
            selectTargetArg.index = arg.index;
            selectTargetArg.indexSize = arg.indexSize;

            editModel->BreakMaterialBlinking(selectTargetArg);
        }
    }

    if (nw::g3d::edit::detail::EditMaterials(editModel->GetTargetModelObj(), arg))
    {
        if (m_pEditCallback != NULL)
        {
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_VISIBILITY:
            case EDIT_TARGET_MATERIAL_DISPLAYFACE:
            case EDIT_TARGET_MATERIAL_RENDER_STATE_MODE:
            case EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE:
            case EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE:
            case EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE:
            case EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC:
            case EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE:
            case EDIT_TARGET_MATERIAL_ALPHATEST_FUNC:
            case EDIT_TARGET_MATERIAL_ALPHATEST_VALUE:
            case EDIT_TARGET_MATERIAL_COLOR_COMBINE:
            case EDIT_TARGET_MATERIAL_ALPHA_COMBINE:
            case EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND:
            case EDIT_TARGET_MATERIAL_COLOR_DST_BLEND:
            case EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND:
            case EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND:
            case EDIT_TARGET_MATERIAL_CONSTANT_COLOR:
            case EDIT_TARGET_MATERIAL_LOGIC_OP:
                {
                    m_pEditCallback->UpdateRenderState(editModel->GetTargetModelObj());
                }
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_NAME:
            case EDIT_TARGET_MATERIAL_SAMPLER_TEXTURE_REF:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
            case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
                {
                    m_pEditCallback->UpdateSamplerParam(editModel->GetTargetModelObj());
                }
                break;

            default:
                break;
            }
        }
    }
}

void
EditManager::EditBones(const EditBoneArg& arg)
{
    EditModelObj* editModel = FindEditModelObj(arg.modelKey);
    if (editModel == NULL)
    {
        return;
    }

    nw::g3d::edit::detail::EditBones(editModel->GetTargetModelObj(), arg);
}

void
EditManager::UpdateBlink()
{
    int size = m_EditModelArray.Size();
    for (int i = 0; i < size; ++i)
    {
        EditModelObj* pEditModel = m_EditModelArray.UnsafeAt(i);
        pEditModel->UpdateMaterialBlinking();
        pEditModel->UpdateShapeBlinking();
    }
}

void
EditManager::StartBlinking(const SelectTargetArg& arg)
{
    EditModelObj* editModel = FindSameModelObj(arg.modelObj);
    if (editModel == NULL)
    {
        return;
    }

    switch (arg.targetKind)
    {
    case SelectTargetArg::TARGET_MODEL:
        {
            editModel->StartModelBlinking();
        }

        return;
    case SelectTargetArg::TARGET_SHAPE:
        {
            editModel->StartShapeBlinking(arg);
        }

        return;
    case SelectTargetArg::TARGET_MATERIAL:
        {
            editModel->StartMaterialBlinking(arg);
        }

        return;
    default:
        return;
    }
}

EditAnimObj* EditManager::AddEditAnimObj(nw::g3d::res::ResFile* pResFile, FileDataKind kind)
{
    EditAnimObj* pEditAnimObj = NULL;
    switch (kind)
    {
    case FILEDATA_SHADER_PARAM_ANIM:
        pEditAnimObj = AddEditShaderParamAnimObj(pResFile);
        break;

    case FILEDATA_MAT_COLOR_ANIM:
        pEditAnimObj = AddEditColorAnimObj(pResFile);
        break;

    case FILEDATA_TEXTURE_SRT_ANIM:
        pEditAnimObj = AddEditTexSrtAnimObj(pResFile);
        break;

    case FILEDATA_TEXTURE_PATTERN_ANIM:
        pEditAnimObj = AddEditTexPatternAnimObj(pResFile);
        break;

    case FILEDATA_SKELETAL_ANIM:
        pEditAnimObj = AddEditSkeletalAnimObj(pResFile);
        break;

    case FILEDATA_BONE_VISIBILITY_ANIM:
        pEditAnimObj = AddEditBoneVisibilityAnimObj(pResFile);
        break;

    case FILEDATA_MAT_VISIBILITY_ANIM:
        pEditAnimObj = AddEditMatVisibilityAnimObj(pResFile);
        break;

    case FILEDATA_SHAPE_ANIM:
        pEditAnimObj = AddEditShapeAnimObj(pResFile);
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    return pEditAnimObj;
}

EditAnimObj*
EditManager::AddEditShaderParamAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetShaderParamAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditShaderParamAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditShaderParamAnimObj* editAnimObj = new (buffer) EditShaderParamAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditColorAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetColorAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditColorAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditColorAnimObj* editAnimObj = new (buffer) EditColorAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditTexSrtAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetTexSrtAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditTexSrtAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditTexSrtAnimObj* editAnimObj = new(buffer) EditTexSrtAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditSkeletalAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetSkeletalAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditSkeletalAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditSkeletalAnimObj* editAnimObj = new (buffer) EditSkeletalAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditTexPatternAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetTexPatternAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditTexPatternAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditTexPatternAnimObj* editAnimObj = new(buffer) EditTexPatternAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditBoneVisibilityAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERT(targetResFile->GetBoneVisAnimCount() > 0);

    size_t size = sizeof(EditBoneVisibilityAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditBoneVisibilityAnimObj* editAnimObj = new(buffer) EditBoneVisibilityAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditMatVisibilityAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERT(targetResFile->GetMatVisAnimCount() > 0);

    size_t size = sizeof(EditMatVisibilityAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditMatVisibilityAnimObj* editAnimObj = new(buffer) EditMatVisibilityAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

EditAnimObj*
EditManager::AddEditShapeAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetShapeAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditShapeAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditShapeAnimObj* editAnimObj = new (buffer) EditShapeAnimObj(m_pAllocator, targetResFile);
    {
        bool result = editAnimObj->CreateDataForEditingAnimCurve();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    if (m_EditAnimArray.PushBack(editAnimObj))
    {
        return editAnimObj;
    }
    return NULL;
}

void EditManager::DeleteEditAnimObj(EditAnimObj* pDeleteTargetEditAnim)
{
    NW_G3D_ASSERT_NOT_NULL(pDeleteTargetEditAnim);

    // 対象アニメがバインドされているモデルからアニメのバインドを外す
    for (int i = 0, end = pDeleteTargetEditAnim->GetBoundModelObjCount(); i < end; ++i)
    {
        ModelObj* pBoundModelObj = pDeleteTargetEditAnim->GetBoundModelObj(i);
        if (pBoundModelObj)
        {
            EditModelObj* pEditModelObj = FindSameModelObj(pBoundModelObj);
            NW_G3D_ASSERT_NOT_NULL(pEditModelObj);

            bool removed = pEditModelObj->RemoveBoundAnim(pDeleteTargetEditAnim);
            NW_G3D_UNUSED(removed);

            // モデルからバインド情報が消えているのに、アニメにだけ残っていた場合、内部実装の不具合なので警告しておく
            NW_G3D_EDIT_WARNING(removed, "inconsistency of animation binding state was detected");
        }
    }

    int index = m_EditAnimArray.IndexOf(pDeleteTargetEditAnim);
    NW_G3D_ASSERT(index >= 0);
    m_EditAnimArray.Erase(index);

    pDeleteTargetEditAnim->~EditAnimObj();
    m_pAllocator->Free(pDeleteTargetEditAnim);
}

void
EditManager::DeleteEditAnimObj(u32 resFileKey)
{
    EditAnimObj* pDeleteTargetEditAnim = FindEditAnimObj(resFileKey);
    while (pDeleteTargetEditAnim != NULL)
    {
        NW_G3D_EDIT_DEBUG_PRINT("Deleting...\n");
        pDeleteTargetEditAnim->PrintAnimInfo();
        DeleteEditAnimObj(pDeleteTargetEditAnim);
        pDeleteTargetEditAnim = FindEditAnimObj(resFileKey);
    }
}

void EditManager::PrintAllEditAnimObjs()
{
    for (int i = 0, end = m_EditAnimArray.Size(); i < end; ++i)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(i);
        NW_G3D_EDIT_DEBUG_PRINT("[%d] : key = %d\n", i, pEditAnimObj->GetResFileKey());
        pEditAnimObj->PrintAnimInfo();
    }
}

void EditManager::PrintEditAnimObjs(u32 resFileKey, u32 boundModelKey)
{
    for (int animIndex = 0, animCount = m_EditAnimArray.Size(); animIndex < animCount; ++animIndex)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
        if (pEditAnimObj->GetResFileKey() == resFileKey)
        {
            for (int modelIndex = 0, modelCount = pEditAnimObj->GetBoundModelObjCount(); modelIndex < modelCount; ++modelIndex)
            {
                ModelObj* pBoundModelObj = pEditAnimObj->GetBoundModelObj(modelIndex);
                if (pBoundModelObj == NULL)
                {
                    continue;
                }

                EditModelObj* pBoundEditModelObj = FindSameModelObj(pBoundModelObj);
                if (pBoundEditModelObj == NULL || pBoundEditModelObj->GetKey() != boundModelKey)
                {
                    continue;
                }

                NW_G3D_EDIT_DEBUG_PRINT("[%d] : key = %d\n", animIndex, pEditAnimObj->GetResFileKey());
                pEditAnimObj->PrintAnimInfo();
                return;
            }
        }
    }
}

void EditManager::PrintAllResFiles()
{
    for (int i = 0, end = m_ResFileInfoArray.Size(); i < end; ++i)
    {
        ResFileInfo info = m_ResFileInfoArray[i];
        NW_G3D_UNUSED(info);
        NW_G3D_EDIT_DEBUG_PRINT("[%d] : name = %s (%s)\n", i, GetResName(info.pResFile, info.kind), GetFileDataKindString(info.kind));
    }

    for (int i = 0, end = m_AnimResFileInfoArray.Size(); i < end; ++i)
    {
        ResFileInfo info = m_AnimResFileInfoArray[i];
        NW_G3D_UNUSED(info);
        NW_G3D_EDIT_DEBUG_PRINT("[%d] : name = %s (%s)\n", i, GetResName(info.pResFile, info.kind), GetFileDataKindString(info.kind));
    }
}

void
EditManager::EditAnimCurve(const EditAnimCurveArg& arg)
{
    EditAnimObj* targetEditAnim = FindEditAnimObj(arg.animationKey);
    if (targetEditAnim == NULL)
    {
        return;
    }

    switch(arg.animationKind)
    {
    case EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE:
        {
            EditShaderParamAnimObj* editAnim = static_cast<EditShaderParamAnimObj*>(targetEditAnim);
            EditShaderParamAnimCurve(editAnim, arg);
        }
        break;
    case EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE:
        {
            EditTexPatternAnimObj* editAnim = static_cast<EditTexPatternAnimObj*>(targetEditAnim);
            EditTexPatternAnimCurve(editAnim, arg);
        }
        break;
    case EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE:
        {
            EditMatVisibilityAnimObj* editAnim = static_cast<EditMatVisibilityAnimObj*>(targetEditAnim);
            EditMatVisibilityAnimCurve(editAnim, arg);
        }
        break;
    case EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE:
        {
            EditBoneVisibilityAnimObj* editAnim = static_cast<EditBoneVisibilityAnimObj*>(targetEditAnim);
            EditBoneVisibilityAnimCurve(editAnim, arg);
        }
        break;
    case EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE:
        {
            EditShapeAnimObj* editAnim = static_cast<EditShapeAnimObj*>(targetEditAnim);
            EditShapeAnimCurve(editAnim, arg);
        }
        break;
    default:
        NW_G3D_EDIT_PRINT("Unknown animation kind(=%x)\n", arg.animationKind);
        break;
    }
}

EditSceneAnimObj*
EditManager::AddEditSceneAnimObj(nw::g3d::res::ResFile* targetResFile)
{
    NW_G3D_ASSERT_NOT_NULL(targetResFile);
    NW_G3D_ASSERTMSG(targetResFile->GetSceneAnimCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));
    nw::g3d::res::ResSceneAnim* resAnim = targetResFile->GetSceneAnim(0);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resAnim, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));

    size_t size = sizeof(EditSceneAnimObj);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める

    EditSceneAnimObj* editAnimObj = new(buffer) EditSceneAnimObj(m_pAllocator, targetResFile, resAnim);
    {
        bool result = editAnimObj->Attach();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
        result = editAnimObj->Setup();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(targetResFile, GetName()));//今は止める
    }

    bool success = m_EditSceneAnimArray.PushBack(editAnimObj);
    if (!success)
    {
        return NULL;
    }

    // アニメーションが追加されたのでグローバルプレイポリシーの再設定を行う
    ResetPlayPolicy();

    return editAnimObj;
}

void
EditManager::DeleteEditSceneAnimObj(u32 key)
{
    EditSceneAnimObj* pDeleteTargetEditAnim = FindEditSceneAnimObj(key);
    if (pDeleteTargetEditAnim == NULL)
    {
        return;
    }

    // EditSceneAnimObj の中でUpdate した方が良いかも
    if (m_pEditCallback != NULL)
    {
        ExecuteUnbindSceneAnimCallback(pDeleteTargetEditAnim, m_pEditCallback);
    }

    pDeleteTargetEditAnim->Detach();

    int index = m_EditSceneAnimArray.IndexOf(pDeleteTargetEditAnim);
    NW_G3D_ASSERT(index >= 0);
    m_EditSceneAnimArray.Erase(index);
    m_pAllocator->Free(pDeleteTargetEditAnim);
}

void
EditManager::EditSceneAnimCurve(const EditAnimCurveArg& arg)
{
    EditSceneAnimObj* editSceneAnim = FindEditSceneAnimObj(arg.animationKey);
    if (editSceneAnim == NULL)
    {
        return;
    }

    switch(arg.animationKind)
    {
    case EDIT_TARGET_SCENE_ANIMATION_CAMERA_CURVE:
        {
            EditCameraAnimCurve(editSceneAnim, arg);
        }
        break;
    case EDIT_TARGET_SCENE_ANIMATION_LIGHT_CURVE:
        {
            EditLightAnimCurve(editSceneAnim, arg);
        }
        break;
    case EDIT_TARGET_SCENE_ANIMATION_FOG_CURVE:
        {
            EditFogAnimCurve(editSceneAnim, arg);
        }
        break;
    default:
        NW_G3D_EDIT_PRINT("Unknown scene animation kind(=%x)\n", arg.animationKind);
        break;
    }
}

EditShaderArchive*
EditManager::CreateAttachEditShaderArchive(nw::g3d::res::ResShaderArchive* targetResShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL(targetResShaderArchive);
    size_t size = sizeof(EditShaderArchive);
    void* buffer = m_pAllocator->Alloc(size, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return NULL;
    }

    EditShaderArchive* editShaderArchive =
        new(buffer) EditShaderArchive(m_pAllocator, targetResShaderArchive);
    return editShaderArchive;
}

void
EditManager::DeleteEditShaderArchive(u32 key)
{
    const ResShaderArchive* targetResShaderArchive = reinterpret_cast<const ResShaderArchive*>(key);
    EditShaderArchive* deleteEditShaderArchive =
        m_EditShaderArchiveArray.Find<ResShaderArchive>(targetResShaderArchive, CompareSameResShaderArchive);
    if (deleteEditShaderArchive == NULL)
    {
        return;
    }

    deleteEditShaderArchive->Detach();
    int index = m_EditShaderArchiveArray.IndexOf(deleteEditShaderArchive);
    NW_G3D_ASSERT(index >= 0);
    m_EditShaderArchiveArray.Erase(index);
    m_pAllocator->Free(deleteEditShaderArchive);

    if (m_pEditCallback)
    {
        DetachShaderArg arg;
        arg.shaderArchive = reinterpret_cast<ResShaderArchive*>(key);
        m_pEditCallback->DetachShader(arg);
    }
}

void
EditManager::Cleanup()
{
    ClearEditManager();
    m_EditAnimArray.Destroy();
    m_EditModelArray.Destroy();
    m_EditShaderArchiveArray.Destroy();
    m_EditSceneAnimArray.Destroy();
    m_AttachingModelObjArray.Destroy();

    m_pAllocator = NULL;
}

void
EditManager::Reset()
{
    m_AnimationFlag = false;
    m_IsWriteStarted = false;
    m_AttachCommandQueue.Clear();
    m_FileLoadedBlockQueue.Clear();
    m_ModelLayoutEditBlockQueue.Clear();
    m_RuntimeErrorInfoQueue.Clear();
}

void
EditManager::ClearEditManager(bool isCallbackEnabled)
{
    DeleteAll(isCallbackEnabled);
    Reset();
}

void
EditManager::ClearCalcFlagModelAnims()
{
    int animCount = m_EditAnimArray.Size();
    for (int i = 0; i < animCount; ++i)
    {
        EditAnimObj* animObj = m_EditAnimArray.UnsafeAt(i);
        NW_G3D_EDIT_ASSERT_NOT_NULL(animObj);
        animObj->ClearCalcFlag();
    }
}

void
EditManager::EditShaderParamAnimCurve(EditShaderParamAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCurve(
        value->materialIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::EditTexPatternAnimCurve(EditTexPatternAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCurve(
        value->materialIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::EditMatVisibilityAnimCurve(EditMatVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCurve(
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}
void
EditManager::EditBoneVisibilityAnimCurve(EditBoneVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCurve(
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::EditShapeAnimCurve(EditShapeAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCurve(
        value->vertexShapeIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);

}

bool EditManager::IsBound(EditModelObj* pEditModelObj, u32 animResFileKey) const
{
    EditAnimObj* pEditAnimObj = FindEditAnimObj(animResFileKey);
    if (pEditAnimObj == NULL)
    {
        return false;
    }

    return pEditAnimObj->IsModelBound(pEditModelObj->GetTargetModelObj());
}

void
EditManager::BindAnimations(const BindAnimArg& arg)
{
    EditModelObj* pEditModelObj = FindEditModelObj(arg.modelKey);
    if (pEditModelObj == NULL)
    {
        return;
    }

    // バインド時に、PlayPolicy を設定
    SetPlayPolicy(m_PlayPolicy);

    for (u32 animKeyIndex = 0; animKeyIndex < arg.animationKeySize; ++animKeyIndex)
    {
        u32 animResFileKey = arg.animationKeys[animKeyIndex];
        bool isAlreadyBound = IsBound(pEditModelObj, animResFileKey);
        if (isAlreadyBound)
        {
            // アニメが既にモデルにバインド済みであれば無視
            continue;
        }

        EditAnimObj* pEditAnimObj = FindEditAnimObj(animResFileKey);
        if (pEditAnimObj == NULL)
        {
            // 見つからなければ無視
            continue;
        }

        bool success = BindAnimation(pEditModelObj, pEditAnimObj, m_Frame, GetPreviewStep());
        if (!success)
        {
            // バインドに失敗しても無視するが、正しくバインドできない不具合に備えて警告ログを出しておく
            NW_G3D_WARNING(
                success,
                "Failed to bind animation \"%s\" to model \"%s\"\n",
                pEditAnimObj->GetName(), pEditModelObj->GetResModel()->GetName());
        }
    }

    // 元からバインドされているものもAutoの場合は変わる可能性があるので全体を設定しなおす。
    ResetPlayPolicy();
}

void EditManager::UnbindAnimation(EditModelObj* pTargetEditModelObj, u32 animResFileKey)
{
    EditAnimObj* pEditAnimObj = FindBoundEditAnimObj(pTargetEditModelObj->GetKey(), animResFileKey);
    while(pEditAnimObj != NULL)
    {
        // 対象アニメがバインドされているモデルからアニメのバインドを外す
        for (int i = 0, end = pEditAnimObj->GetBoundModelObjCount(); i < end; ++i)
        {
            ModelObj* pBoundModelObj = pEditAnimObj->GetBoundModelObj(i);
            NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(pBoundModelObj, "%s\n", pTargetEditModelObj->GetResModel()->GetName());
            if (pBoundModelObj == pTargetEditModelObj->GetTargetModelObj())
            {
                EditModelObj* pEditModelObj = FindSameModelObj(pBoundModelObj);
                NW_G3D_ASSERT_NOT_NULL(pEditModelObj);

                bool removed = pEditModelObj->RemoveBoundAnim(pEditAnimObj);
                NW_G3D_ASSERT(removed);

                pEditAnimObj->UnbindModelObj(pBoundModelObj);
                break;
            }
        }

        //DeleteEditAnimObj(pEditAnimObj);
        pEditAnimObj = FindBoundEditAnimObj(pTargetEditModelObj->GetKey(), animResFileKey);
    }
}

void
EditManager::UnbindAnimations(const BindAnimArg& arg)
{
    EditModelObj* pEditModelObj = FindEditModelObj(arg.modelKey);
    if (pEditModelObj == NULL)
    {
        return;
    }

    for (u32 animKeyIndex = 0; animKeyIndex < arg.animationKeySize; ++animKeyIndex)
    {
        u32 animResFileKey = arg.animationKeys[animKeyIndex];
        UnbindAnimation(pEditModelObj, animResFileKey);
    }
}

void
EditManager::ForceUnbindAnimations(EditModelObj* pEditModelObj)
{
    NW_G3D_ASSERT_NOT_NULL(pEditModelObj);
    NW_G3D_ASSERT_NOT_NULL(pEditModelObj->GetTargetModelObj());

    ModelObj* modelObj = pEditModelObj->GetTargetModelObj();
    int size = m_EditAnimArray.Size();
    for (int i = 0; i < size; ++i)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(i);
        for (int animIndex = 0, end = pEditAnimObj->GetBoundModelObjCount(); animIndex < end; ++animIndex)
        {
            if (pEditAnimObj->GetBoundModelObj(animIndex) == modelObj)
            {
                pEditAnimObj->UnbindModelObj(modelObj);
            }
        }
    }

    while (0 != pEditModelObj->GetBoundAnimCount())
    {
        bool removed = pEditModelObj->RemoveBoundAnim(0);
        NW_G3D_ASSERT(removed);
    }
}

void
EditManager::EditCameraAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditCameraAnimCurve(
        value->cameraIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::EditLightAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditLightAnimCurve(
        value->lightIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::EditFogAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);

    AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
    const nw::g3d::res::ResAnimCurve* resAnimCurve = static_cast<const nw::g3d::res::ResAnimCurve*>(arg.curveData);
    editAnimObj->EditFogAnimCurve(
        value->fogIndex,
        value->curveIndex,
        resAnimCurve,
        arg.curveDataSize);
}

void
EditManager::ExecuteBindSceneAnimCallback(EditSceneAnimObj* editAnimObj, EditCallback* callback)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);
    NW_G3D_EDIT_ASSERT_NOT_NULL(callback);

    BindSceneAnimArg arg;

    arg.cameraAnimObjPtrs = editAnimObj->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
    arg.numCameraAnimObj = editAnimObj->GetCameraAnimObjPtrArray().Size();

    arg.lightAnimObjPtrs = editAnimObj->GetLightAnimObjPtrArray().GetArrayBufferPtr();
    arg.numLightAnimObj = editAnimObj->GetLightAnimObjPtrArray().Size();

    arg.fogAnimObjPtrs = editAnimObj->GetFogAnimObjPtrArray().GetArrayBufferPtr();
    arg.numFogAnimObj = editAnimObj->GetFogAnimObjPtrArray().Size();

    callback->BindSceneAnim(arg);
    editAnimObj->SetBindFlag(true);
}

void
EditManager::ExecuteUnbindSceneAnimCallback(EditSceneAnimObj* editAnimObj, EditCallback* callback)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(editAnimObj);
    NW_G3D_EDIT_ASSERT_NOT_NULL(callback);

    UnbindSceneAnimArg arg;

    arg.cameraAnimObjPtrs = editAnimObj->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
    arg.numCameraAnimObj = editAnimObj->GetCameraAnimObjPtrArray().Size();

    arg.lightAnimObjPtrs = editAnimObj->GetLightAnimObjPtrArray().GetArrayBufferPtr();
    arg.numLightAnimObj = editAnimObj->GetLightAnimObjPtrArray().Size();

    arg.fogAnimObjPtrs = editAnimObj->GetFogAnimObjPtrArray().GetArrayBufferPtr();
    arg.numFogAnimObj = editAnimObj->GetFogAnimObjPtrArray().Size();

    callback->UnbindSceneAnim(arg);
    editAnimObj->SetBindFlag(false);
}

void
EditManager::BindSceneAnimations(const BindAnimArg& arg)
{
    if (m_pEditCallback == NULL)
    {
        return;
    }

    for (u32 i = 0; i < arg.animationKeySize; ++i)
    {
        EditSceneAnimObj* editAnimObj = FindEditSceneAnimObj(arg.animationKeys[i]);
        if (editAnimObj != NULL)
        {
            EditManager::ExecuteBindSceneAnimCallback(editAnimObj, m_pEditCallback);
        }
    }
}

void
EditManager::UnbindSceneAnimations(const BindAnimArg& arg)
{
    if (m_pEditCallback == NULL)
    {
        return;
    }

    for (u32 i = 0; i < arg.animationKeySize; ++i)
    {
        EditSceneAnimObj* editAnimObj = FindEditSceneAnimObj(arg.animationKeys[i]);
        if (editAnimObj != NULL)
        {
            EditManager::ExecuteUnbindSceneAnimCallback(editAnimObj, m_pEditCallback);
        }
    }
}

void
EditManager::UpdateEditShaderArchive(u32 key, int shadingModelIndices[], u32 indexSize)
{
    EditShaderArchive* editShaderArchive = FindEditShaderArchive(key);
    if (editShaderArchive != NULL)
    {
        editShaderArchive->UpdateShadingModels(shadingModelIndices, indexSize);
    }
}

EditModelObj*
EditManager::FindEditModelObj(u32 key) const
{
    return m_EditModelArray.Find<u32>(&key, CompareSameKey);
}

EditAnimObj*
EditManager::FindEditAnimObj(u32 key) const
{
    return m_EditAnimArray.Find<u32>(&key, CompareSameKey);
}

EditAnimObj* EditManager::FindBoundEditAnimObj(u32 modelKey, u32 boundAnimationResFileKey)
{
    EditModelObj* pEditModelObj = FindEditModelObj(modelKey);
    if (pEditModelObj == NULL)
    {
        return NULL;
    }

    for (int i = 0, end = pEditModelObj->GetBoundAnimCount(); i < end; ++i)
    {
        EditAnimObj* pIterEditAnimObj = pEditModelObj->GetBoundAnimAt(i);
        if (pIterEditAnimObj->GetResFileKey() == boundAnimationResFileKey)
        {
            return pIterEditAnimObj;
        }
    }

    return NULL;
}

EditShaderArchive*
EditManager::FindEditShaderArchive(u32 key) const
{
    const ResShaderArchive* resShaderArchive = reinterpret_cast<const ResShaderArchive*>(key);
    return m_EditShaderArchiveArray.Find<ResShaderArchive>(resShaderArchive, CompareSameResShaderArchive);
}

EditSceneAnimObj*
EditManager::FindEditSceneAnimObj(u32 key) const
{
    return m_EditSceneAnimArray.Find<u32>(&key, CompareSameKey);
}

EditManager::AttachCommand*
EditManager::FindSameAttachModelObjQueue(const ModelObj* modelObj) const
{
    return m_AttachCommandQueue.Find<ModelObj>(modelObj, CompareSameAttachModelObj);
}

EditManager::AttachCommand*
EditManager::FindSameAttachShaderArchiveQueue(const ResShaderArchive* resShaderArchive) const
{
    return m_AttachCommandQueue.Find<ResShaderArchive>(resShaderArchive, CompareSameAttachShaderArchive);
}

EditModelObj*
EditManager::FindSameModelObj(const ModelObj* modelObj) const
{
    NW_G3D_ASSERT_NOT_NULL(modelObj);
    return m_EditModelArray.Find<ModelObj>(modelObj, CompareSameModelObj);
}

EditShaderArchive*
EditManager::FindSameResShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive) const
{
    NW_G3D_ASSERT_NOT_NULL(resShaderArchive);
    return m_EditShaderArchiveArray.Find<ResShaderArchive>(resShaderArchive, CompareSameResShaderArchive);
}

void
EditManager::CalcAnimations()
{
    // モデルアニメーションの計算
    {
        int size = m_EditModelArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditModelObj* editModelObj = m_EditModelArray.UnsafeAt(i);
            int animCount = editModelObj->GetBoundAnimCount();
            for (int j = 0; j < animCount; ++j)
            {
                EditAnimObj* editAnim = editModelObj->GetBoundAnimAt(j);
                ModelObj* pModelObj = editModelObj->GetTargetModelObj();

                // CalcSkeletalAnimations で計算済みの場合は処理をスキップ
                // バインド対象が設定されていないので処理をスキップ
                if (editAnim->IsCalculated(pModelObj) || !editAnim->HasBoundModel())
                {
                    continue;
                }

                if (editAnim->IsPaused(pModelObj))
                {
                    editAnim->InvalidateContext(pModelObj);
                    editAnim->SetFrame(pModelObj, editAnim->GetPauseFrame(pModelObj));
                }
                else if (!m_AnimationFlag)
                {
                    editAnim->InvalidateContext(pModelObj);
                    editAnim->SetFrame(pModelObj, m_Frame);
                }

                editAnim->Calc(pModelObj);
                editAnim->ApplyAnimTo(pModelObj);
            }
        }
    }

    // コールバックが設定されていない場合は、シーンアニメの処理はスキップ
    if (m_pEditCallback == NULL)
    {
        return;
    }

    // シーンアニメーションの計算
    {
        int size = m_EditSceneAnimArray.Size();
        if (!m_AnimationFlag)
        {
            for (int i = 0; i < size; ++i)
            {
                EditSceneAnimObj* editAnim = m_EditSceneAnimArray.UnsafeAt(i);

                // バインドされていない場合は、スキップ
                if (!editAnim->IsBound())
                {
                    continue;
                }
                editAnim->SetFrame(m_Frame);
            }
        }

        for (int i = 0; i < size; ++i)
        {
            EditSceneAnimObj* editAnim = m_EditSceneAnimArray.UnsafeAt(i);

            // バインドされていない場合は、スキップ
            if (!editAnim->IsBound())
            {
                continue;
            }
            editAnim->Calc();
            ApplySceneAnimArg arg;

            arg.cameraAnimObjPtrs = editAnim->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
            arg.numCameraAnimObj = editAnim->GetCameraAnimObjPtrArray().Size();

            arg.lightAnimObjPtrs = editAnim->GetLightAnimObjPtrArray().GetArrayBufferPtr();
            arg.numLightAnimObj = editAnim->GetLightAnimObjPtrArray().Size();

            arg.fogAnimObjPtrs = editAnim->GetFogAnimObjPtrArray().GetArrayBufferPtr();
            arg.numFogAnimObj = editAnim->GetFogAnimObjPtrArray().Size();

            m_pEditCallback->ApplySceneAnim(arg);
        }
    }

    if (m_AnimationFlag)
    {
        if (m_IsLoopAnim)
        {
            m_Frame = PlayPolicy_Loop(m_Frame + GetPreviewStep(), 0.0f, m_FrameCount, NULL);
        }
        else
        {
            m_Frame = PlayPolicy_Onetime(m_Frame + GetPreviewStep(), 0.0f, m_FrameCount, NULL);
            if (m_Frame >= m_FrameCount)
            {
                m_AnimationFlag = false;
            }
        }
    }
}

void
EditManager::CalcSkeletalAnimations(const nw::g3d::ModelObj* modelObj)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(modelObj);
    EditModelObj* editModelObj = FindSameModelObj(modelObj);
    if (editModelObj == NULL)
    {
        return;
    }

    // モデルアニメーションの計算
    {
        int size = editModelObj->GetBoundAnimCount();
        for (int i = 0; i < size; ++i)
        {
            EditAnimObj* editAnim = editModelObj->GetBoundAnimAt(i);
            bool isSkeletalAnim = editAnim->GetAnimKind() == EDIT_SKELETAL_ANIM;

            // スケルタルアニメでない場合は処理をスキップ
            // バインド対象が設定されていないので処理をスキップ
            if (!isSkeletalAnim || !editAnim->HasBoundModel())
            {
                continue;
            }

            ModelObj* pModelObj = editModelObj->GetTargetModelObj();
            if (editAnim->IsPaused(pModelObj))
            {
                editAnim->InvalidateContext(pModelObj);
                editAnim->SetFrame(pModelObj, editAnim->GetPauseFrame(pModelObj));
            }
            else if (!m_AnimationFlag)
            {
                editAnim->InvalidateContext(pModelObj);
                editAnim->SetFrame(pModelObj, m_Frame);
            }

            editAnim->Calc(pModelObj);
            editAnim->ApplyAnimTo(pModelObj);
        }
    }
}

void
EditManager::ExecuteCommandQueue(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);

    if (m_IsWriteStarted)
    {
        if (!socket->IsWriting())
        {
            socket->ResetWriteFlag();
            m_IsWriteStarted = false;
        }
        else
        {
            return;
        }
    }

    // 書き込み不可能な場合は処理を抜ける
    if (socket->IsWriting())
    {
        return;
    }

    ExecuteRuntimeErrorCommandQueue(socket);

    if (socket->IsWriting())
    {
        return;
    }

    ExecuteModelLayoutCommandQueue(socket);

    if (socket->IsWriting())
    {
        return;
    }

    ExecuteAttachCommandQueueImpl(socket);

    if (socket->IsWriting())
    {
        return;
    }

    ExecuteFileLoadCommandQueueImpl(socket);
}

void
EditManager::SendModifiedShaderPrograms(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);
    int shaderArchiveCount = m_EditShaderArchiveArray.Size();
    for (int i = 0; i < shaderArchiveCount; ++i)
    {
        EditShaderArchive* editShaderArchive = m_EditShaderArchiveArray.UnsafeAt(i);
        editShaderArchive->SendModifiedShaderPrograms(socket);
    }
}

void
EditManager::SetAnimFlag(bool enable)
{
    m_AnimationFlag = enable;

    {
        int size = m_EditAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
            editAnim->SetFrameForAllBoundModels(m_Frame);
            editAnim->SetStep(GetPreviewStep());
        }
    }

    {
        int size = m_EditSceneAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
            editSceneAnim->SetFrame(m_Frame);
            editSceneAnim->SetStep(GetPreviewStep());
        }
    }
}

void
EditManager::SetFrameStep(f32 frameStep)
{
    m_FrameStep = frameStep;
    SetPreviewStep();
}

void
EditManager::SetPreviewFrameStepRate(f32 rate)
{
    m_PreviewFrameStepRate = rate;
    SetPreviewStep();
}

bool
EditManager::AddModelFileLoadedQueue(const EditModelObj* editModelObj, u32 toolKey)
{
    NW_G3D_ASSERT_NOT_NULL(editModelObj);

    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = FILEDATA_MODEL;
    block.toolKey = toolKey;
    block.resFileKey = editModelObj->GetResFileKey();
    block.resModelKey = editModelObj->GetResModelKey();
    block.modelObjKey = 0;
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddAnimFileLoadedQueue(
    FileDataKind kind,
    u32 resFileKey,
    u32 toolKey
    )
{
    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = kind;
    block.toolKey = toolKey;
    block.resFileKey = resFileKey;
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddSceneAnimFileLoadedQueue(const EditSceneAnimObj* editSceneAnimObj, u32 toolKey)
{
    NW_G3D_ASSERT_NOT_NULL(editSceneAnimObj);

    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = FILEDATA_SCENE_ANIM;
    block.toolKey = toolKey;
    block.resFileKey = editSceneAnimObj->GetResFileKey();
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddModelFileReloadedQueue(const EditModelObj* editModelObj, u32 pResFileKey)
{
    NW_G3D_ASSERT_NOT_NULL(editModelObj);

    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = FILEDATA_MODEL;
    block.resFileKey = pResFileKey;
    block.resModelKey = editModelObj->GetResModelKey();
    block.modelObjKey = reinterpret_cast<u32>(editModelObj->GetTargetModelObj());
    block.newResFileKey = editModelObj->GetResFileKey();
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddAnimFileReloadedQueue(FileDataKind kind, u32 newResFileKey, u32 oldResFileKey)
{
    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = kind;
    block.resFileKey = oldResFileKey;
    block.newResFileKey = newResFileKey;
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddSceneAnimFileReloadedQueue(const EditSceneAnimObj* editAnimObj, u32 pResFileKey)
{
    NW_G3D_ASSERT_NOT_NULL(editAnimObj);

    FileLoadedBlock block;
    memset(&block, 0, sizeof(block));
    block.fileDataKind = FILEDATA_SCENE_ANIM;
    block.resFileKey = pResFileKey;
    block.newResFileKey = editAnimObj->GetResFileKey();
    return m_FileLoadedBlockQueue.PushBack(block);
}

bool
EditManager::AddModelLayoutQueue(u32 modelKey, const f32 scale[3], const f32 rotate[3], const f32 translate[3])
{
    ModelLayoutEditBlock block;
    memset(&block, 0, sizeof(block));
    block.modelKey = modelKey;
    for (int i = 0; i < 3; ++i)
    {
        block.scale.a[i] = scale[i];
        block.rotate.a[i] = rotate[i];
        block.translate.a[i] = translate[i];
    }
    return m_ModelLayoutEditBlockQueue.PushBack(block);
}

void EditManager::DeleteEditAnimObjBindingForAllModels(EditAnimObj* pTarget)
{
    for (int modelIndex = 0, modelEnd = m_EditModelArray.Size(); modelIndex < modelEnd; ++modelIndex)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.At(modelIndex);
        pEditModelObj->RemoveBoundAnim(pTarget);
    }
}


bool
EditManager::ReloadAnimResource(u32 resFileKey, nw::g3d::res::ResFile* pResFile, FileDataKind kind)
{
    NW_G3D_ASSERT_NOT_NULL(pResFile);

    for (int i = 0, end = m_EditAnimArray.Size(); i < end; ++i)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(i);
        if (pEditAnimObj->GetResFileKey() == resFileKey)
        {
            bool success = pEditAnimObj->ReloadResource(pResFile);
            if (!success)
            {
                // リロードに失敗した場合はEditAnimObjのモデルバインド情報は消えるので、
                // すべてのモデルから該当のEditAnimObjのバインド情報を消す
                DeleteEditAnimObjBindingForAllModels(pEditAnimObj);
                return false;
            }
        }
    }

    u32 newResFileKey = GetResFileKeyFromResFile(pResFile);

    // 古いリソースを破棄
    {
        DeleteAnimResFile(resFileKey);
    }

    {
        bool success = AddAnimFileReloadedQueue(
            kind,
            newResFileKey,
            resFileKey);
        NW_G3D_ASSERT(success);
    }

    // アニメーションが更新されたのでグローバルプレイポリシーの再設定を行う
    ResetPlayPolicy();

    // リロード後に未使用のリソースがあったら解放する
    DeleteUnusedResFiles();

    return true;
}

bool
EditManager::ReloadEditSceneAnimObj(u32 resFileKey, nw::g3d::res::ResFile* pResFile)
{
    NW_G3D_ASSERT_NOT_NULL(pResFile);

    for (int i = 0, end = m_EditSceneAnimArray.Size(); i < end; ++i)
    {
        EditSceneAnimObj* pEditSceneAnimObj = m_EditSceneAnimArray.UnsafeAt(i);
        if (pEditSceneAnimObj->GetResFileKey() == resFileKey)
        {
            bool isBound = pEditSceneAnimObj->IsBound();
            if (m_pEditCallback && isBound)
            {
                EditManager::ExecuteUnbindSceneAnimCallback(pEditSceneAnimObj, m_pEditCallback);
            }

            bool success = pEditSceneAnimObj->ReloadSceneAnimObj(pResFile);
            if (!success)
            {
                return false;
            }

            if (m_pEditCallback && isBound)
            {
                EditManager::ExecuteBindSceneAnimCallback(pEditSceneAnimObj, m_pEditCallback);
            }

            success = AddSceneAnimFileReloadedQueue(
                pEditSceneAnimObj,
                resFileKey);
            NW_G3D_ASSERT(success);
        }
    }

    // アニメーションが更新されたのでグローバルプレイポリシーの再設定を行う
    ResetPlayPolicy();

    // リロード後に未使用のリソースがあったら解放する
    DeleteUnusedResFiles();

    return true;
}

RuntimeErrorCode EditManager::AttachModelObj(EditModelObj* pTarget, ModelObj* pModelObj)
{
    bool isModelObjAlreadyAttached = this->FindEditModelObj(GetKeyFromModelObj(pModelObj)) != NULL;
    if (isModelObjAlreadyAttached)
    {
        return RUNTIME_ERROR_CODE_DUPLICATE_MODEL_OBJ_KEY;
    }

    return pTarget->AttachEditModelObj(pModelObj, m_pEditCallback);
}

void
EditManager::ExecuteAttachCommandQueueImpl(EditSocket* socket)
{
    // この関数を実行する所でチェックしているので、socket の NULL チェックは行わない
    // Attach*** の遅延処理

    int queueSize;
    do
    {
        queueSize = m_AttachCommandQueue.Size();
        for (int i = 0; i < queueSize; ++i)
        {
            AttachCommand* command = m_AttachCommandQueue.Get(i);
            bool isLoadFile = false;
            for (int j = 0, fileBlockEnd = m_FileLoadedBlockQueue.Size(); j < fileBlockEnd; ++j)
            {
                FileLoadedBlock* block = m_FileLoadedBlockQueue.Get(j);
                if (command->resource.key == block->resModelKey && command->kind == ATTACH_MODEL)
                {
                    nw::g3d::ModelObj* modelObj = command->instance.modelObjPtr;
                    nw::g3d::res::ResModel* resModel = command->resource.resModelPtr;
                    EditModelObj* editModelObj = m_EditModelArray.Find<ResModel>(resModel, CompareSameResModel);
                    if (editModelObj != NULL)
                    {
                        NW_G3D_EDIT_PRINT("Load File AttachModel [%x]\n", command->instance.key);

                        block->modelObjKey = command->instance.key;
                        RemoveAttachingModelObj(GetModelObjFromKey(block->modelObjKey));

                        NW_G3D_ASSERTMSG(this->FindEditModelObj(GetKeyFromModelObj(modelObj)) == NULL,
                            "%s has already attached\n", NW_G3D_RES_GET_NAME(resModel, GetName()));
                        RuntimeErrorCode errorCode = this->AttachModelObj(editModelObj, modelObj);
                        if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
                        {
                            QueueErrorCommand(errorCode);

                            m_FileLoadedBlockQueue.EraseAt(j);

                            DeleteEditModelObj(block->modelObjKey);
                            m_AttachCommandQueue.EraseAt(i);
                            return;
                        }

                        if (command->pathPtr != NULL) // パスが含まれている場合は削除
                        {
                            command->pathPtr->Clear();
                        }

                        m_AttachCommandQueue.EraseAt(i);
                        isLoadFile = true;
                        break;
                    }
                }
            }
            if (!isLoadFile)
            {
                bool result = SendAttach(&m_AttachPacket, socket, command);
                if (result)
                {
                    // Detach 処理の場合は、ここで編集用のインスタンスを削除
                    switch(command->kind)
                    {
                    case ATTACH_MODEL:
                        NW_G3D_EDIT_PRINT("Attach Model CommandQueue : [%x]\n", command->instance.key);
                        break;
                    case ATTACH_SHADER_ARCHIVE:
                        NW_G3D_EDIT_PRINT("Attach ShaderArchive CommandQueue : [%x]\n", command->instance.key);
                        break;
                    case DETACH_MODEL:
                        DeleteEditModelObj(command->instance.key);
                        NW_G3D_EDIT_PRINT("Detach Model CommandQueue : [%x]\n", command->instance.key);
                        break;
                    case DETACH_SHADER_ARCHIVE:
                        DeleteEditShaderArchive(command->instance.key);
                        NW_G3D_EDIT_PRINT("Detach ShaderArchive CommandQueue : [%x]\n", command->instance.key);
                        break;
                    default:
                        NW_G3D_EDIT_ASSERTMSG(false, "Invalid command kind");
                    }

                    if (command->pathPtr != NULL) // パスが含まれている場合は削除
                    {
                        command->pathPtr->Clear();
                        m_pAllocator->Free(command->pathPtr);
                    }
                    m_AttachCommandQueue.EraseAt(i);
                    m_IsWriteStarted = true;
                    break;
                }
            }
        }

        // 送信処理を書き込み中であれば、処理を抜ける、処理が終わっていれば、
        // このままアタッチ処理のキューをなくなるまで処理
        if (m_IsWriteStarted)
        {
            if (!socket->IsWriting())
            {
                socket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }
            else
            {
                return;
            }
        }
    } while ( 0 < queueSize && socket->IsConnected() );
}

void
EditManager::ExecuteFileLoadCommandQueueImpl(EditSocket* socket)
{
    // FileLoaded の遅延処理
    int queueSize;
    do
    {
        queueSize = m_FileLoadedBlockQueue.Size();
        for (int idxFileBlockQueue = 0; idxFileBlockQueue < queueSize; ++idxFileBlockQueue)
        {
            FileLoadedBlock* block = m_FileLoadedBlockQueue.Get(idxFileBlockQueue);
            bool writeFlag = false;
            if (block->fileDataKind == FILEDATA_MODEL)
            {
                if (block->modelObjKey > 0 && block->resFileKey > 0 && block->resModelKey > 0)
                {
                    writeFlag = true;
                }
            }
            else if (IsAnimFileData(static_cast<FileDataKind>(block->fileDataKind)))
            {
                if (block->resFileKey > 0)
                {
                    writeFlag = true;
                }
            }

            if (writeFlag)
            {
                m_FileLoadedPacket.header.magic = NW_G3D_EDIT_MAGIC;
                m_FileLoadedPacket.header.verWord = NW_G3D_EDIT_VERSION;

                u32 command = EDIT_FILE_LOADED_COMMAND_FLAG;
                if (block->newResFileKey > 0)
                {
                    command = EDIT_FILE_RELOADED_COMMAND_FLAG;
                }

                m_FileLoadedPacket.header.command = command;
                m_FileLoadedPacket.header.dataSize = sizeof(FileLoadedBlock);
                m_FileLoadedPacket.block = *block;

                bool result = socket->WriteSync(&m_FileLoadedPacket, sizeof(FileLoadedPacket));
                if (result)
                {
                    NW_G3D_EDIT_PRINT("FileLoaded CommandQueue : %s [%x:%x]\n",
                        GetEditCommandString(static_cast<CommandFlag>(command)),
                        block->resFileKey,
                        block->newResFileKey);

                    m_FileLoadedBlockQueue.EraseAt(idxFileBlockQueue);
                    m_IsWriteStarted = true;
                    return;
                }
            }
        }
        // 送信処理を書き込み中であれば、処理を抜ける、処理が終わっていれば、
        // このままアタッチ処理のキューをなくなるまで処理
        if (m_IsWriteStarted)
        {
            if (!socket->IsWriting())
            {
                socket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }
            else
            {
                return;
            }
        }
    } while ( 0 < queueSize && socket->IsConnected() );
}

void
EditManager::ExecuteModelLayoutCommandQueue(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);

    int queueSize;
    do
    {
        queueSize = m_ModelLayoutEditBlockQueue.Size();

        // 送信処理を書き込み中であれば、処理を抜ける、処理が終わっていれば、
        // このままキューがなくなるまで処理
        if (m_IsWriteStarted)
        {
            if (!socket->IsWriting())
            {
                socket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }
            else
            {
                return;
            }
        }

        for (int idx = 0; idx < queueSize; ++idx)
        {
            ModelLayoutEditBlock* block = m_ModelLayoutEditBlockQueue.Get(idx);

            NW_G3D_EDIT_PRINT("ModelLayoutEdit CommandQueue : %s [%d:%x]\n",
                GetEditCommandString(static_cast<CommandFlag>(m_ModelLayoutPacket.header.command)),
                block->isBind,
                block->modelKey);

            memcpy(&m_ModelLayoutPacket.block, block, sizeof(m_ModelLayoutPacket.block));

            bool result = socket->WriteSync(&m_ModelLayoutPacket, sizeof(m_ModelLayoutPacket));
            if (result)
            {
                m_ModelLayoutEditBlockQueue.EraseAt(idx);
                m_IsWriteStarted = true;
                break;
            }
        }
    } while ( queueSize > 0 && socket->IsConnected() );
}

void EditManager::ExecuteRuntimeErrorCommandQueue(EditSocket* pSocket)
{
    NW_G3D_ASSERT_NOT_NULL(pSocket);

    int queueSize;
    do
    {
        queueSize = m_RuntimeErrorInfoQueue.Size();

        // 送信処理を書き込み中であれば、処理を抜ける、処理が終わっていれば、
        // このままキューがなくなるまで処理
        if (m_IsWriteStarted)
        {
            if (!pSocket->IsWriting())
            {
                pSocket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }
            else
            {
                return;
            }
        }

        for (int idx = 0; idx < queueSize; ++idx)
        {
            RuntimeErrorNotificationInfo* info = m_RuntimeErrorInfoQueue.Get(idx);

            m_RuntimeErrorPacket.header.magic = NW_G3D_EDIT_MAGIC;
            m_RuntimeErrorPacket.header.verWord = NW_G3D_EDIT_VERSION;

            u32 command = SYSTEM_RUNTIME_ERROR_COMMAND_FLAG;

            m_RuntimeErrorPacket.header.command = command;
            m_RuntimeErrorPacket.header.dataSize = sizeof(RuntimeErrorNotificationInfo);
            memcpy(&m_RuntimeErrorPacket.info, info, sizeof(m_RuntimeErrorPacket.info));

            NW_G3D_EDIT_PRINT("RuntimeErrorNotificationCommand CommandQueue : %s [%d]\n",
                GetEditCommandString(static_cast<CommandFlag>(command)), m_RuntimeErrorPacket.info.runtimeErrorCode);

            bool result = pSocket->WriteSync(&m_RuntimeErrorPacket, sizeof(m_RuntimeErrorPacket));
            if (result)
            {
                m_RuntimeErrorInfoQueue.EraseAt(idx);
                m_IsWriteStarted = true;
                break;
            }
        }
    } while ( queueSize > 0 && pSocket->IsConnected() );
}

bool
EditManager::CanLoopAnim() const
{
    bool hasLoopAnim = false;
    {
        int size = m_EditAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
            if (editAnim->HasBoundModel())
            {
                hasLoopAnim |= editAnim->IsLoopAnim();
            }
        }
    }

    // シーンアニメでループ設定があるものがあるか探す
    {
        int size = m_EditSceneAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
            hasLoopAnim |= editSceneAnim->ContainsLoopAnim();
        }
    }

    bool isLoopAnim = false;
    if (m_PlayPolicy == EDIT_PLAY_POLICY_AUTO)
    {
        if (hasLoopAnim)
        {
            isLoopAnim = true;
        }
    }
    else if (m_PlayPolicy == EDIT_PLAY_POLICY_LOOP)
    {
        isLoopAnim = true;
    }
    return isLoopAnim;
}

void
EditManager::SetAnimLoopFlag(bool enable)
{
    {
        int count = m_EditAnimArray.Size();
        for (int i = 0; i < count; ++i)
        {
            EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(enable);
            editAnim->SetStep(GetPreviewStep());
        }
    }

    {
        int count = m_EditSceneAnimArray.Size();
        for (int i = 0; i < count; ++i)
        {
            EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
            editSceneAnim->SetPlayPolicy(enable);
            editSceneAnim->SetStep(GetPreviewStep());
        }
    }
}

void
EditManager::SetPreviewStep()
{
    {
        int size = m_EditAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
            editAnim->SetStep(GetPreviewStep());
        }
    }

    {
        int size = m_EditSceneAnimArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
            editSceneAnim->SetStep(GetPreviewStep());
        }
    }
}

int
EditManager::GetModelAnimBoundCount(const ModelObj* modelObj) const
{
    EditModelObj* editModelObj = m_EditModelArray.Find<ModelObj>(modelObj, CompareSameModelObj);
    return editModelObj->GetBoundAnimCount();
}

int
EditManager::GetActiveEditAnimIndex(const ModelObj* modelObj, int boundAnimIndex) const
{
    int boundCount = 0;
    int count = m_EditAnimArray.Size();
    int index = -1;
    for (int i = 0; i < count; ++i)
    {
        EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
        for (int animIndex = 0, end = editAnim->GetBoundModelObjCount(); animIndex < end; ++animIndex)
        {
            if (modelObj == editAnim->GetBoundModelObj(animIndex))
            {
                if (boundCount == boundAnimIndex)
                {
                    index = i;
                    break;
                }
                ++boundCount;
                break;
            }
        }
    }
    return index;
}

/*static*/bool
EditManager::SendAttach(AttachPacket* packet, EditSocket* socket, const AttachCommand* attachCommand)
{
    NW_G3D_ASSERT_NOT_NULL(packet);
    NW_G3D_ASSERT_NOT_NULL(socket);
    NW_G3D_ASSERT_NOT_NULL(attachCommand);

    packet->header.magic = NW_G3D_EDIT_MAGIC;
    packet->header.verWord = NW_G3D_EDIT_VERSION;
    packet->block.attachedKey = attachCommand->instance.key;

    packet->block.attachKind = static_cast<u16>(attachCommand->kind);
    packet->block.attachFileName[0] = '\0'; // 全体を0で埋めるのはコストが高いので0番目で終わるように設定
    packet->block.flag = 0;

    switch(attachCommand->kind)
    {
    case ATTACH_MODEL:
    case ATTACH_SHADER_ARCHIVE:
        packet->header.command = EDIT_SEND_ATTACH_COMMAND_FLAG;
        break;
    case DETACH_MODEL:
    case DETACH_SHADER_ARCHIVE:
        packet->header.command = EDIT_SEND_DETACH_COMMAND_FLAG;
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    packet->header.dataSize = sizeof(AttachBlock);
    const char* name = NULL;

    switch(attachCommand->kind)
    {
    case ATTACH_MODEL:
    case DETACH_MODEL:
        {
            nw::g3d::ModelObj* modelObj =
                reinterpret_cast<nw::g3d::ModelObj*>(packet->block.attachedKey);
            name = modelObj->GetResource()->GetName();
        }
        break;
    case ATTACH_SHADER_ARCHIVE:
        {
            nw::g3d::res::ResShaderArchive* resShaderArchive =
                reinterpret_cast<nw::g3d::res::ResShaderArchive*>(packet->block.attachedKey);
            name = resShaderArchive->GetName();
            packet->block.flag |= ( attachCommand->resource.resShaderArchivePtr->ref().flag
                & ResShaderArchive::GL_BINARY_AVAILABLE ) ? ATTACH_SHADER_ARCHIVE_IS_BINARY : 0;
        }
        break;
    case DETACH_SHADER_ARCHIVE:
        {
            nw::g3d::res::ResShaderArchive* resShaderArchive =
                reinterpret_cast<nw::g3d::res::ResShaderArchive*>(packet->block.attachedKey);
            name = resShaderArchive->GetName();
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    size_t length = strlen(name);
    NW_G3D_ASSERT(length < NW_G3D_EDIT_FILENAME_MAX);
#if NW_G3D_IS_HOST_WIN
    strncpy_s(reinterpret_cast<char*>(packet->block.fileName), NW_G3D_EDIT_FILENAME_MAX, name, length);
#else
    memset(packet->block.fileName, 0, NW_G3D_EDIT_FILENAME_MAX);
    strncpy(reinterpret_cast<char*>(packet->block.fileName), name, length);
#endif

    if (attachCommand->pathPtr != NULL)
    {
        const char* attachPath = attachCommand->pathPtr->GetStr();
        size_t attachPathLength = attachCommand->pathPtr->GetLength();
#if NW_G3D_IS_HOST_WIN
        strncpy_s(reinterpret_cast<char*>(packet->block.attachFileName), NW_G3D_EDIT_FILENAME_MAX, attachPath, attachPathLength);
#else
        memset(packet->block.attachFileName, 0, NW_G3D_EDIT_FILENAME_MAX);
        strncpy(reinterpret_cast<char*>(packet->block.attachFileName), attachPath, attachPathLength);
#endif
    }
    return socket->WriteSync(packet, sizeof(AttachPacket));
}

/*static*/int
EditManager::CompareSameResModel(const EditModelObj* a, const ResModel* b)
{
    if (a->GetResModel() < b)
    {
        return -1;
    }
    else if (a->GetResModel() > b)
    {
        return 1;
    }
    return 0;
}

/*static*/int
EditManager::CompareSameModelObj(const EditModelObj* a, const ModelObj* b)
{
    if (a->GetTargetModelObj() < b)
    {
        return -1;
    }
    else if (a->GetTargetModelObj() > b)
    {
        return 1;
    }
    return 0;
}

/*static*/int
EditManager::CompareSameAttachModelObj(const AttachCommand* a, const ModelObj* b)
{
    int result = 0;
    if (a->instance.modelObjPtr < b)
    {
        --result;
    }
    else if (a->instance.modelObjPtr > b)
    {
        ++result;
    }

    // 適当に加減算
    if (a->kind != ATTACH_MODEL)
    {
        if (result < 0)
        {
            --result;
        }
        else
        {
            ++result;
        }
    }
    return result;
}

/*static*/int
EditManager::CompareSameAttachShaderArchive(const AttachCommand* a, const ResShaderArchive* b)
{
    int result = 0;
    if (a->resource.resShaderArchivePtr < b)
    {
        --result;
    }
    else if (a->resource.resShaderArchivePtr > b)
    {
        ++result;
    }

    // 適当に加減算
    if (a->kind != ATTACH_SHADER_ARCHIVE)
    {
        if (result < 0)
        {
            --result;
        }
        else
        {
            ++result;
        }
    }
    return result;
}

/*static*/int
EditManager::CompareSameKey(const EditModelObj* a, const u32* b)
{
    if (a->GetKey() < *b )
    {
        return -1;
    }
    else if (a->GetKey() > *b)
    {
        return 1;
    }
    return 0;
}

/*static*/int
EditManager::CompareSameAttachPath( const ut::detail::DynamicLengthString* a, const char* b)
{
    return strcmp(a->GetStr(), b);
}

/*static*/int
EditManager::CompareSameResShaderArchive(const EditShaderArchive* a, const ResShaderArchive* b)
{
    if (a->GetTargetResShaderArchive() < b)
    {
        return -1;
    }
    else if (a->GetTargetResShaderArchive() > b)
    {
        return 1;
    }
    return 0;
}

/*static*/int
EditManager::CompareSameKey(const EditAnimObj* a, const u32* b)
{
    if (a->GetResFileKey() < *b)
    {
        return -1;
    }
    else if (a->GetResFileKey() > *b)
    {
        return 1;
    }
    return 0;
}
/*static*/int
EditManager::CompareSameKey(const EditSceneAnimObj* a, const u32* b)
{
    if (a->GetResFileKey() < *b)
    {
        return -1;
    }
    else if (a->GetResFileKey() > *b)
    {
        return 1;
    }
    return 0;
}

bool EditManager::QueueErrorCommand(RuntimeErrorCode errorCode)
{
    RuntimeErrorNotificationInfo info;
    info.runtimeErrorCode = errorCode;
    return m_RuntimeErrorInfoQueue.PushBack(info);
}

nw::g3d::res::ResFile* EditManager::LoadResFileWithoutCopy(void** pFileBuffer, size_t fileSize, FileDataKind kind)
{
    ResFile* pResFile = SetupBufferAsResFile(*pFileBuffer, fileSize);
    *pFileBuffer = NULL;
    ResFileInfo info;
    info.kind = kind;
    info.pResFile = pResFile;
    m_ResFileInfoArray.PushBack(info);

    return pResFile;
}

nw::g3d::res::ResFile* EditManager::LoadAnimResFileWithoutCopy(void** pFileBuffer, size_t fileSize, FileDataKind kind)
{
    ResFile* pResFile = SetupBufferAsResFile(*pFileBuffer, fileSize);
    *pFileBuffer = NULL;
    ResFileInfo info;
    info.kind = kind;
    info.pResFile = pResFile;
    info.resFileKey = GetResFileKeyFromResFile(pResFile);

    m_AnimResFileInfoArray.PushBack(info);

    return pResFile;
}

void EditManager::DeleteAnimResFile(u32 resFileKey)
{
    ResFile* pResFile = GetAnimResFile(resFileKey);
    if (pResFile == NULL)
    {
        return;
    }

    m_pAllocator->Free(pResFile);
    int index = GetIndexOfAnimResFile(pResFile);
    m_AnimResFileInfoArray.Erase(index);
}

void EditManager::DeleteUnusedResFiles()
{
    bool isAllResFileUsed;
    do
    {
        isAllResFileUsed = true;
        int pResFileCount = m_ResFileInfoArray.Size();
        for (int pResFileIndex = 0; pResFileIndex < pResFileCount; ++pResFileIndex)
        {
            ResFile* pResFile = m_ResFileInfoArray[pResFileIndex].pResFile;
            EditAnimObj* pUsedEditAnimObj = FindEditObj(m_EditAnimArray, pResFile);
            EditSceneAnimObj* pUsedEditSceneAnimObj = FindEditObj(m_EditSceneAnimArray, pResFile);
            EditModelObj* pUsedEditModelObj = FindEditObj(m_EditModelArray, pResFile);
            if (pUsedEditAnimObj == NULL &&
                pUsedEditSceneAnimObj == NULL &&
                pUsedEditModelObj == NULL)
            {
                m_pAllocator->Free(pResFile);
                m_ResFileInfoArray.Erase(pResFileIndex);
                isAllResFileUsed = false;
                break;
            }
        }
    } while (isAllResFileUsed == false);
}

void EditManager::SetRetargetingHostModel(u32 animResFileKey, u32 retargetHostModelObjKey)
{
    EditModelObj* pRetargetHostEditModelObj = FindEditModelObj(retargetHostModelObjKey);
    if (pRetargetHostEditModelObj == NULL)
    {
        this->QueueErrorCommand(RUNTIME_ERROR_CODE_RETARGET_HOST_MODEL_NOT_FOUND);
        return;
    }

    // 今後バインドされたアニメにもリターゲティングを反映できるようにリソースに名前を記録しておく
    {
        ResFileInfo* pInfo = GetAnimResFileInfo(animResFileKey);
        NW_G3D_EDIT_ASSERT_NOT_NULL(pInfo);
        strcpy(pInfo->retargetingHostModelName, pRetargetHostEditModelObj->GetResModel()->GetName());
    }

    for (int animIndex = 0, end = m_EditAnimArray.Size(); animIndex < end; ++animIndex)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
        if (pEditAnimObj->GetResFileKey() == animResFileKey)
        {
            if (pEditAnimObj->GetAnimKind() == EDIT_SKELETAL_ANIM)
            {
                EditSkeletalAnimObj* pEditSkeletalAnimObj = static_cast<EditSkeletalAnimObj*>(pEditAnimObj);
                pEditSkeletalAnimObj->SetRetargetingHostModel(pRetargetHostEditModelObj->GetTargetModelObj());
            }
        }
    }
}

void EditManager::UnsetRetargetingHostModel(u32 animResFileKey)
{
    for (int animIndex = 0, end = m_EditAnimArray.Size(); animIndex < end; ++animIndex)
    {
        EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
        if (pEditAnimObj->GetResFileKey() == animResFileKey)
        {
            if (pEditAnimObj->GetAnimKind() == EDIT_SKELETAL_ANIM)
            {
                EditSkeletalAnimObj* pEditSkeletalAnimObj = static_cast<EditSkeletalAnimObj*>(pEditAnimObj);
                pEditSkeletalAnimObj->UnsetRetargetingHostModel();
            }
        }
    }
}

ModelObj* EditManager::GetModelObjFromEditModelObjs(const char* name)
{
    for (int idxModel = 0, numModel = m_EditModelArray.Size(); idxModel < numModel; ++idxModel)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(idxModel);
        ResModel* pResModel = pEditModelObj->GetTargetModelObj()->GetResource();
        const char* pName = pResModel->GetName();
        if (0 == strcmp(pName, name))
        {
            return pEditModelObj->GetTargetModelObj();
        }
    }

    return NULL;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
