﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include "g3d_IEditObj.h"

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_SkeletalAnimObj.h>
#include <nw/g3d/g3d_ShaderParamAnimObj.h>
#include <nw/g3d/g3d_TexPatternAnimObj.h>
#include <nw/g3d/g3d_VisibilityAnimObj.h>
#include <nw/g3d/g3d_ShapeAnimObj.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include "g3d_EditAnimObj.h"
#include "g3d_EditAnimCurve.h"
#include "g3d_EditUtility.h"
#include "ut/g3d_DynamicArray.h"

#include <stdint.h>

namespace nw { namespace g3d {

class ModelAnimObj;

namespace res {

class ResFile;
class ResSkeletalAnim;
class ResShaderParamAnim;
class ResShaderParamMatAnim;
class ResTexPatternMatAnim;
class ResTexPatternAnim;
class ResVisibilityAnim;

} // namespace res

namespace edit {

class IAllocator;

namespace detail {

class EditManager;
class EditAnimCurve;
class EditShaderParamAnimObj;
class EditTexPatternAnimObj;
class EditShapeAnimObj;

//! @brief アニメを編集するためのクラスです。
// このクラスではリソースひとつに対して複数のModelObjとModelAnimObjをバインドします。
class EditAnimObj : IEditObj
{
public:
    explicit EditAnimObj(
        IAllocator* pAllocator,
        nw::g3d::res::ResFile* pResFile,
        EditAnimKind animKind)
        : m_pAllocator(pAllocator)
        , m_pResFile(pResFile)
        , m_SetupFlag(false)
        , m_IsLoopAnim(false)
        , m_AnimKind(animKind)
        , m_BoundModelInfos(DEFAULT_ALIGNMENT)
    {
        NW_G3D_ASSERT_NOT_NULL(pResFile);
        NW_G3D_EDIT_ASSERT(IsValidResFile(pResFile));
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pAllocator, "%s\n", NW_G3D_RES_GET_NAME(m_pResFile, GetName()));
        m_BoundModelInfos.SetAllocator(pAllocator);
    }

    virtual ~EditAnimObj()
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.Size(); animIndex < animCount; ++animIndex)
        {
            DestroyAnimObj(animIndex);
        }
    }

    //! @brief モデルをバインドします。
    bool BindModelObj(ModelObj* pBindTargetModelObj)
    {
        NW_G3D_ASSERTMSG(!IsModelBound(pBindTargetModelObj), "%s", pBindTargetModelObj->GetResource()->GetName());
        AddBoundModelObj(pBindTargetModelObj);
        ModelAnimObj* pBoundAnimObj = m_BoundModelInfos[m_BoundModelInfos.Size() - 1]->pAnimObj;
        bool result = SetupInternal(pBindTargetModelObj, pBoundAnimObj);
        ResetPlayPolicy();
        m_SetupFlag = result;
        return result;
    }

    //! @brief モデルをアンバインドします。
    void UnbindModelObj(ModelObj* pModelObj)
    {
        ResetToOriginalValue(pModelObj);

        int index = IndexOf(pModelObj);
        DestroyAnimObj(index);
        BoundModelInfo* pInfo = m_BoundModelInfos[index];
        m_BoundModelInfos.Erase(index);
        pInfo->~BoundModelInfo();
        m_pAllocator->Free(pInfo);
    }

    //! @brief アニメカーブ編集用のデータを作成します。
    virtual bool CreateDataForEditingAnimCurve() = 0;

    //! @brief アニメカーブ編集用のデータを破棄します。
    virtual void DestroyDataForEditingAnimCurve() = 0;

    //! @biref バインドされているモデルにアニメーションを適用します。
    virtual void ApplyAnimTo(ModelObj* pModelObj)
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());

        ModelAnimObj* pAssociatedAnimObj = m_BoundModelInfos[index]->pAnimObj;
        pAssociatedAnimObj->ApplyTo(pModelObj);
    }

    //! @brief アニメリソースの名前を取得します。
    const char* GetName() const
    {
        return GetResName(m_pResFile, ConvertToFileDataKind(m_AnimKind));
    }

    //! @brief アニメの種類を取得します。
    EditAnimKind GetAnimKind() const
    {
        return m_AnimKind;
    }

    //! @brief アニメリソースをリロードします。
    bool ReloadResource(nw::g3d::res::ResFile* resFile);

    void InvalidateContext(ModelObj* pModelObj)
    {
        ModelAnimObj* pAnimObj = GetAssociatedAnimObj(pModelObj);
        if (pAnimObj == NULL)
        {
            return;
        }

        pAnimObj->InvalidateContext();
    }

    bool IsResFileBound(ResFile* pResFile) const
    {
        return m_pResFile == pResFile;
    }

    float GetFrameCount() const
    {
        if (m_BoundModelInfos.Size() == 0)
        {
            return 0.0f;
        }

        return m_BoundModelInfos[0]->pAnimObj->GetFrameCtrl().GetEndFrame();
    }

    bool IsLoopAnim() const
    {
        if (m_BoundModelInfos.Size() == 0)
        {
            return m_IsLoopAnim;
        }

        return m_BoundModelInfos[0]->pAnimObj->GetDefaultFrameCtrl().GetPlayPolicy() == PlayPolicy_Loop;
    }

    bool IsCalculated(ModelObj* pModelObj) const
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());

        return m_BoundModelInfos[index]->isCalculated;
    }

    bool IsPaused(ModelObj* pModelObj) const
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());
        return m_BoundModelInfos[index]->isPaused;
    }

    f32 GetPauseFrame(ModelObj* pModelObj) const
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());
        return m_BoundModelInfos[index]->pauseFrame;
    }

    void Calc(ModelObj* pModelObj)
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());

        if (m_BoundModelInfos[index]->isCalculated)
        {
            return;
        }

        ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
        pAnimObj->Calc();
        pAnimObj->GetFrameCtrl().UpdateFrame();
        m_BoundModelInfos[index]->isCalculated = true;
    }

    void SetFrame(ModelObj* pModelObj, f32 frame)
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());
        m_BoundModelInfos[index]->pAnimObj->GetFrameCtrl().SetFrame(frame);
    }

    void SetFrameForAllBoundModels(f32 frame)
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.Size(); animIndex < animCount; ++animIndex)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[animIndex]->pAnimObj;
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }

    void SetStep(f32 step)
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.Size(); animIndex < animCount; ++animIndex)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[animIndex]->pAnimObj;
            pAnimObj->GetFrameCtrl().SetStep(step);
        }
    }

    void SetPlayPolicy(bool isLoopAnim)
    {
        m_IsLoopAnim = isLoopAnim;
        ResetPlayPolicy();
    }

    void ResetPlayPolicy()
    {
        if (m_BoundModelInfos.Size() == 0)
        {
            return;
        }

        for (int index = 0, size = m_BoundModelInfos.Size(); index < size; ++index)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
            float start = pAnimObj->GetDefaultFrameCtrl().GetStartFrame();
            float end = pAnimObj->GetDefaultFrameCtrl().GetEndFrame();
            if (m_IsLoopAnim)
            {
                pAnimObj->GetFrameCtrl().Init(start, end, PlayPolicy_Loop);
            }
            else
            {
                pAnimObj->GetFrameCtrl().Init(start, end, PlayPolicy_Onetime);
            }
        }
    }

    void ClearCalcFlag()
    {
        for (int index = 0, end = m_BoundModelInfos.Size(); index < end; ++index)
        {
            m_BoundModelInfos[index]->isCalculated = false;
        }
    }

    void SetPauseFlag(ModelObj* pModelObj, bool enable, f32 frame)
    {
        int index = IndexOf(pModelObj);
        NW_G3D_EDIT_ASSERTMSG(index >= 0, "%s is not bound", pModelObj->GetResource()->GetName());

        BoundModelInfo* pInfo = m_BoundModelInfos[index];
        pInfo->isPaused = enable;
        pInfo->pauseFrame = frame;
    }

    ModelObj* GetBoundModelObj(int index)
    {
        return m_BoundModelInfos[index]->pBoundModelObj;
    }

    int GetBoundModelObjCount() const
    {
        return m_BoundModelInfos.Size();
    }

    uint32_t GetResFileKey() const
    {
        return GetResFileKeyFromResFile(m_pResFile);
    }

    void PrintAnimInfo() const
    {
        ResFile* pResFile = m_pResFile;
        NW_G3D_UNUSED(pResFile);
        NW_G3D_EDIT_ASSERT_NOT_NULL(pResFile);

        NW_G3D_EDIT_DEBUG_PRINT("%s (%s)\n",
            GetResName(pResFile, ConvertToFileDataKind(this->GetAnimKind())),
            GetAnimKindString(this->GetAnimKind()));

        NW_G3D_EDIT_DEBUG_PRINT("  Bound to:\n");
        for (int i = 0, end = m_BoundModelInfos.Size(); i < end; ++i)
        {
            ModelObj* pModelObj = m_BoundModelInfos[i]->pBoundModelObj;
            ResModel* pResModel = pModelObj->GetResource();
            NW_G3D_UNUSED(pModelObj);
            NW_G3D_UNUSED(pResModel);
            NW_G3D_EDIT_DEBUG_PRINT("    %s\n", pResModel->GetName());
        }
    }

    bool IsModelBound(ModelObj* pTargetModelObj) const
    {
        for (int i = 0, end = m_BoundModelInfos.Size(); i < end; ++i)
        {
            ModelObj* pBoundModelObj = m_BoundModelInfos[i]->pBoundModelObj;
            if (pBoundModelObj == pTargetModelObj)
            {
                return true;
            }
        }

        return false;
    }

    bool HasBoundModel() const
    {
        return m_BoundModelInfos.Size() > 0;
    }

protected:
    virtual bool SetupInternal(ModelObj* pBindTargetModelObj, ModelAnimObj* pBoundAnimObj) = 0;
    virtual ModelAnimObj* CreateAnimObj() = 0;
    virtual void ResetToOriginalValue(ModelObj* pBoundModelObj) = 0;

    void UnbindAllModelObjs()
    {
        while (m_BoundModelInfos.Size() > 0)
        {
            UnbindModelObj(m_BoundModelInfos[0]->pBoundModelObj);
        }
    }

    bool IsValidResFile(ResFile* pResFile)
    {
        switch (GetAnimKind())
        {
        case EDIT_SKELETAL_ANIM:
            return pResFile->GetSkeletalAnimCount() > 0;
        case EDIT_SHADER_PARAM_ANIM:
            return pResFile->GetShaderParamAnimCount() > 0;
        case EDIT_COLOR_ANIM:
            return pResFile->GetColorAnimCount() > 0;
        case EDIT_TEX_SRT_ANIM:
            return pResFile->GetTexSrtAnimCount() > 0;
        case EDIT_TEX_PATTERN_ANIM:
            return pResFile->GetTexPatternAnimCount() > 0;
        case EDIT_BONE_VIS_ANIM:
            return pResFile->GetBoneVisAnimCount() > 0;
        case EDIT_MAT_VIS_ANIM:
            return pResFile->GetMatVisAnimCount() > 0;
        case EDIT_SHAPE_ANIM:
            return pResFile->GetShapeAnimCount() > 0;
        default:
            NW_G3D_EDIT_UNEXPECTED_DEFAULT;
        }

        return false;
    }

    void* GetResAnim()
    {
        switch (GetAnimKind())
        {
        case EDIT_SKELETAL_ANIM:
            return m_pResFile->GetSkeletalAnim(0);
        case EDIT_SHADER_PARAM_ANIM:
            return m_pResFile->GetShaderParamAnim(0);
        case EDIT_COLOR_ANIM:
            return m_pResFile->GetColorAnim(0);
        case EDIT_TEX_SRT_ANIM:
            return m_pResFile->GetTexSrtAnim(0);
        case EDIT_TEX_PATTERN_ANIM:
            return m_pResFile->GetTexPatternAnim(0);
        case EDIT_BONE_VIS_ANIM:
            return m_pResFile->GetBoneVisAnim(0);
        case EDIT_MAT_VIS_ANIM:
            return m_pResFile->GetMatVisAnim(0);
        case EDIT_SHAPE_ANIM:
            return m_pResFile->GetShapeAnim(0);
        default:
            NW_G3D_EDIT_UNEXPECTED_DEFAULT;
        }

        return NULL;
    }

    ModelAnimObj* GetAnimObj(int index)
    {
        return m_BoundModelInfos[index]->pAnimObj;
    }

    ModelAnimObj* GetAssociatedAnimObj(ModelObj* pModelObj)
    {
        int index = IndexOf(pModelObj);
        if (index == -1)
        {
            return NULL;
        }

        return m_BoundModelInfos[index]->pAnimObj;
    }

    bool                    m_SetupFlag;
    IAllocator*             m_pAllocator;
    nw::g3d::res::ResFile*  m_pResFile;

private:
    struct BoundModelInfo
    {
        ModelObj* pBoundModelObj;
        ModelAnimObj* pAnimObj;
        AnimFrameCtrl animFrameCtrl;
        f32  pauseFrame;
        bool isPaused;
        bool isCalculated;

        BoundModelInfo()
            : pBoundModelObj(NULL)
            , pAnimObj(NULL)
            , pauseFrame(0.0f)
            , isPaused(false)
            , isCalculated(false)
        {
        }

        BoundModelInfo(const BoundModelInfo& rhs)
        {
            operator = (rhs);
        }

        BoundModelInfo& operator=(const BoundModelInfo& rhs)
        {
            pBoundModelObj = rhs.pBoundModelObj;
            pAnimObj = rhs.pAnimObj;
            pauseFrame = rhs.pauseFrame;
            isPaused = rhs.isPaused;
            isCalculated = rhs.isCalculated;
            animFrameCtrl.Init(
                rhs.animFrameCtrl.GetStartFrame(),
                rhs.animFrameCtrl.GetEndFrame(),
                rhs.animFrameCtrl.GetPlayPolicy());

            return *this;
        }
    };

    void FreeAnimObjBuffers();

    void SetResource(nw::g3d::res::ResFile* resFile)
    {
        m_pResFile = resFile;
    }

    void GetBoundModelObjs(ut::detail::DynamicArray<BoundModelInfo>* pDestination)
    {
        for (int i = 0, end = GetBoundModelObjCount(); i < end; ++i)
        {
            (*pDestination).PushBack(*m_BoundModelInfos[i]);
        }
    }

    void AddBoundModelObjs(ut::detail::DynamicArray<BoundModelInfo>& source)
    {
        for (int i = 0, end = GetBoundModelObjCount(); i < end; ++i)
        {
            AddBoundModelObj((source)[i].pBoundModelObj);
        }
    }

    void AddBoundModelObj(ModelObj* pModelObj)
    {
        BoundModelInfo* pInfo;
        {
            void* buffer = m_pAllocator->Alloc(sizeof(BoundModelInfo), DEFAULT_ALIGNMENT);
            pInfo = new (buffer) BoundModelInfo();
            NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(pInfo, "%s\n", pModelObj->GetResource()->GetName());
            m_BoundModelInfos.PushBack(pInfo);
        }
        pInfo->pBoundModelObj = pModelObj;

        ModelAnimObj* pModelAnimObj = CreateAnimObj();
        NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(pModelAnimObj, "%s\n", pModelObj->GetResource()->GetName());

        pModelAnimObj->SetFrameCtrl(&pInfo->animFrameCtrl);

        pInfo->pAnimObj = pModelAnimObj;
    }

    void DestroyAnimObj(int index)
    {
        ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
        void* buffer = pAnimObj->GetBufferPtr();
        if (buffer)
        {
            m_pAllocator->Free(pAnimObj->GetBufferPtr());
        }
        pAnimObj->~ModelAnimObj();
        m_pAllocator->Free(pAnimObj);
    }

    int IndexOf(ModelObj* pModelObj) const
    {
        for (int index = 0, size = m_BoundModelInfos.Size(); index < size; ++index)
        {
            if (m_BoundModelInfos[index]->pBoundModelObj == pModelObj)
            {
                return index;
            }
        }

        return -1;
    }

    ut::detail::DynamicArray<BoundModelInfo*> m_BoundModelInfos;
    bool          m_IsLoopAnim;
    EditAnimKind m_AnimKind;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(EditAnimObj);
};

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
