﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_CommandMonitor.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>
#include "g3d_EditUtility.h"

using namespace nw::g3d::edit::detail;
using namespace nw::g3d::edit::ut::detail;
namespace
{
bool IsAbnormalPacket(const PacketHeader* pPacketHeader)
{
    if (pPacketHeader->magic != NW_G3D_EDIT_MAGIC ||
        pPacketHeader->verWord != NW_G3D_EDIT_VERSION)
    {
        NW_G3D_WARNING(false, "Version check failed (packet:'%d.%d.%d.%d' lib:'%d.%d.%d.%d').\n",
            pPacketHeader->version[0],
            pPacketHeader->version[1],
            pPacketHeader->version[2],
            pPacketHeader->version[3],
            NW_G3D_VERSION_EDITMAJOR,
            NW_G3D_VERSION_EDITMINOR,
            NW_G3D_VERSION_EDITMICRO,
            NW_G3D_VERSION_EDITBUGFIX);
        return true;
    }
    return false;
}
}

namespace nw {
namespace g3d {
namespace edit {
namespace detail {

CommandMonitor::CommandMonitor(
    bool (*pAnalyzeCommandCallback)(AnalyzeCommandArg&),
    bool (*pProcessCommandCallback)(ProcessCommandArg&))
    : m_pAllocator(NULL)
    , m_FileLoadIndex(0)
    , m_IsHostIoOpened(false)
    , m_pWorkBuffer(NULL)
    , m_IsReadStarted(false)
    , m_IsCommandAnalyzing(false)
    , m_PollState(kReady)
    , m_FileLoadState(kBegin)
    , m_pAnalyzeCommandCallback(pAnalyzeCommandCallback)
    , m_pProcessCommandCallback(pProcessCommandCallback)
{
    m_PacketErrorHeader.magic = NW_G3D_EDIT_MAGIC;
    m_PacketErrorHeader.verWord = NW_G3D_EDIT_VERSION;
    m_PacketErrorHeader.dataSize = 0;
    m_PacketErrorHeader.command = SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG;

    m_Socket.Setup(EditSocket::SetupArg());
}

bool
CommandMonitor::Initialize(const InitArg& arg)
{
    m_pAllocator = arg.pAllocator;

    {
        void* pBuffer = m_pAllocator->Alloc(sizeof(EditWorkBuffer), DEFAULT_ALIGNMENT);
        m_pWorkBuffer = new (pBuffer) EditWorkBuffer(m_pAllocator);
    }

    return true;
}

CommandMonitor::~CommandMonitor() throw()
{
    Close();
    ClearWorkBuffer();

    m_pWorkBuffer->~EditWorkBuffer();
    m_pAllocator->Free(m_pWorkBuffer);

    m_pAllocator = NULL;
}

bool
CommandMonitor::IsConnected() const
{
    return m_Socket.IsConnected();
}

bool
CommandMonitor::Open()
{
    m_IsHostIoOpened = m_Socket.Open();
    return m_IsHostIoOpened;
}

void
CommandMonitor::Close()
{
    m_Socket.Close();

    m_IsHostIoOpened = false;
}

void CommandMonitor::PollSocket()
{
    if (!m_Socket.IsWriting())
    {
        m_Socket.ResetWriteFlag();
    }

    if (!m_Socket.IsReading())
    {
        m_Socket.ResetReadFlag();
    }

    m_Socket.Poll();
}

bool CommandMonitor::PollDataCommunication()
{
    PollSocket();

    for (int idx = 0; idx < kEnd; ++idx)
    {
        PollState startState = m_PollState;

        switch (m_PollState)
        {
        case kReady:
            {
                if (m_Socket.ReadASync(&m_PacketHeader, sizeof(PacketHeader)))
                {
                    m_PollState = kPacketAnalyzing;
                }
            }
            break;
        case kPacketAnalyzing:
            {
                if (!m_Socket.IsReading())
                {
                    m_Socket.ResetReadFlag();
                    m_IsReadStarted = false;
                    bool success = AnalyzePacket();
                    if (!success)
                    {
                        ClearState();
                        return false;
                    }

                    m_PollState = m_IsCommandAnalyzing? kCommandAnalyzing : kReady;
                }
            }
            break;
        case kCommandAnalyzing:
            {
                if (!m_Socket.IsReading())
                {
                    AnalyzeCommandArg arg;
                    arg.command = static_cast<CommandFlag>(m_PacketHeader.command);
                    arg.pWorkBuffer = m_pWorkBuffer->GetWorkBufferPtr();
                    arg.workBufferSize = m_pWorkBuffer->Size();
                    bool success = m_pAnalyzeCommandCallback(arg);
                    if (!success)
                    {
                        ClearState();
                        return false;
                    }

                    m_Socket.ResetReadFlag();
                    m_IsCommandAnalyzing = false;
                    m_PollState = kCommandProcessing;
                }
            }
            break;
        default:
            break;
        }

        if (!m_Socket.IsConnected())
        {
            ClearState();
            return false;
        }

        if (startState == m_PollState)
        {
            return true;
        }
    }

    return true;
}

bool CommandMonitor::PollDataEdit()
{
    switch (m_PollState)
    {
    case kCommandProcessing:
        {
            ProcessCommandArg arg;
            arg.command = static_cast<CommandFlag>(m_PacketHeader.command);
            arg.pWorkBuffer = m_pWorkBuffer->GetWorkBufferPtr();
            arg.workBufferSize = m_pWorkBuffer->Size();
            bool success = m_pProcessCommandCallback(arg);
            if (!success)
            {
                ClearState();
                return false;
            }

            m_PollState = kReady;
        }
        break;
    default:
        break;
    }

    return true;
}

void
CommandMonitor::ClearState()
{
    m_Socket.ResetWriteFlag();
    m_Socket.ResetReadFlag();

    m_PollState = kReady;
    m_FileLoadState = kEnd;

    ClearWorkBuffer();
}

void
CommandMonitor::ClearWorkBuffer()
{
    m_pWorkBuffer->Clear();
}

void
CommandMonitor::TryResizeWorkBuffer(int bufferSize)
{
    m_pWorkBuffer->Resize(bufferSize);
}

bool CommandMonitor::AnalyzePacket()
{
    // 正常なパケットではない場合は、処理を抜ける
    bool isAbnormalPacket = IsAbnormalPacket(&m_PacketHeader);
    if (isAbnormalPacket)
    {
        NW_G3D_EDIT_PRINT("Abnormal packet received.\n");
        m_PacketHeader.command = SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG;
        m_PacketHeader.dataSize = 0;
        return false;
    }

    NW_G3D_EDIT_PRINT("EditServer::AnalyzePacket : %s\n", GetEditCommandString(static_cast<CommandFlag>(m_PacketHeader.command)));
    u32 commandCategory = m_PacketHeader.command & COMMAND_CATEGORY_FLAG_MASK;
    switch(commandCategory)
    {
    case FILEDATA_CATEGORY_FLAG:
    case MATERIAL_CATEGORY_FLAG:
    case EDIT_CATEGORY_FLAG:
    case ANIMATION_CATEGORY_FLAG:
    case SHADER_CATEGORY_FLAG:
    case BONE_CATEGORY_FLAG:
    case MODEL_CATEGORY_FLAG:
    case MODEL_ANIMATION_CATEGORY_FLAG:
    case SCENE_ANIMATION_CATEGORY_FLAG:
    case PICK_CATEGORY_FLAG:
    case OTHER_CATEGORY_FLAG:
        TryResizeWorkBuffer(m_PacketHeader.dataSize);
        m_IsCommandAnalyzing = m_Socket.ReadASync(m_pWorkBuffer->GetWorkBufferPtr(), m_PacketHeader.dataSize);
        return true;
    case SYSTEM_CATEGORY_FLAG:
        if (m_PacketHeader.command != SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG)
        {
            TryResizeWorkBuffer(m_PacketHeader.dataSize);
            m_IsCommandAnalyzing = m_Socket.ReadASync(m_pWorkBuffer->GetWorkBufferPtr(), m_PacketHeader.dataSize);
        }
        return true;
    default:
        // 今の所、ここには、パケットエラーの時しか処理がこないはず
        m_Socket.WriteSync(&m_PacketErrorHeader, sizeof(m_PacketErrorHeader));
        return false;
    }
}

bool
CommandMonitor::SendBeginFreeze()
{
    if (!IsConnected())
    {
        return false;
    }
    m_SendPacketHeader.command = SYSTEM_BEGIN_FREEZE_COMMAND_FLAG;
    m_SendPacketHeader.dataSize = 0;
    m_SendPacketHeader.magic = NW_G3D_EDIT_MAGIC;
    m_SendPacketHeader.verWord = NW_G3D_EDIT_VERSION;
    return m_Socket.WriteSync(&m_SendPacketHeader, sizeof(m_SendPacketHeader));
}

bool
CommandMonitor::SendEndFreeze()
{
    if (!IsConnected())
    {
        return false;
    }
    m_SendPacketHeader.command = SYSTEM_END_FREEZE_COMMAND_FLAG;
    m_SendPacketHeader.dataSize = 0;
    m_SendPacketHeader.magic = NW_G3D_EDIT_MAGIC;
    m_SendPacketHeader.verWord = NW_G3D_EDIT_VERSION;
    return m_Socket.WriteSync(&m_SendPacketHeader, sizeof(m_SendPacketHeader));
}

}}}} // namespace evfl::edit::internal

#endif // EVFL_CONFIG_USE_HOSTIO
