﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>

#include <nn/ndd/ndd_Api.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/detail/Service/ndd_Service.sfdl.h>
#include <nn/ndd/detail/Service/ndd_ServiceName.h>

#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator

namespace nn { namespace ndd
{

namespace {
bool isInitialized = false;
nn::sf::SharedPointer<IService> g_Service;
nn::sf::SharedPointer<IServiceCreator> g_ServiceCreator;
struct  NddServiceByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, NddServiceByHipcTag> NddServiceAllocator;
nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
}   // namespace

// NddServiceAllocator を静的コンストラクタで初期化するためのヘルパー
class NddServiceAllocatorInitializer
{
public:

    NddServiceAllocatorInitializer() NN_NOEXCEPT
    {
        NddServiceAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_NddServiceAllocatorInitializer;



void Initialize() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(isInitialized)
    {

    }
    else
    {
        //ServiceCreator
        NN_SDK_ASSERT(!g_ServiceCreator);
        nn::sf::SharedPointer<IServiceCreator> serviceCreator;
        auto result = nn::sf::CreateHipcProxyByName<IServiceCreator, NddServiceAllocator::Policy>(&serviceCreator, ServiceName);
        NN_ABORT_UNLESS(result.IsSuccess());
        g_ServiceCreator = serviceCreator;//[Todo]ポインタを直に渡してしまっても良さそう
        NN_UNUSED(result);

        //Service
        nn::sf::SharedPointer<nn::ndd::IService> service;
        result = g_ServiceCreator->CreateNddService(&service);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_SDK_ASSERT(service);
        g_Service = service;

        //ServiceCreator
        NN_SDK_ASSERT(g_ServiceCreator);
        g_ServiceCreator = nullptr;

        isInitialized = true;
    }
    UnlockMutex(&g_Mutex);
}

void EnableAutoCommunication() NN_NOEXCEPT
{
    auto result = g_Service->EnableAutoCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void DisableAutoCommunication() NN_NOEXCEPT
{
    auto result = g_Service->DisableAutoCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

bool IsAutoCommunicationEnabled() NN_NOEXCEPT
{
    bool isEnabled;
    auto result = g_Service->IsAutoCommunicationEnabled(&isEnabled);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isEnabled;
}

bool IsNetworkActive() NN_NOEXCEPT
{
    bool isActive;
    auto result = g_Service->IsNetworkActive(&isActive);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isActive;
}

void AcquireSendDataUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEvent);
    nn::sf::NativeHandle handle;
    auto result = g_Service->AcquireSendDataUpdateEvent(&handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::AttachReadableHandleToSystemEvent(pEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
}

void AddSendData(const SendDataDescription& data) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS_EQUAL(data.dataSize, SendDataSizeMax);
    auto result = g_Service->AddSendData(data);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void ClearSendData() NN_NOEXCEPT
{
    auto result = g_Service->ClearSendData();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetSendData(SendDataDescription buffer[], int offset, int count) NN_NOEXCEPT
{
    //[todo]複数データ対応
    NN_ABORT_UNLESS_NOT_NULL(&buffer[0]);
    NN_ABORT_UNLESS(count > 0);

    if(offset != 0)
    {
        return 0;
    }
    int sendDataCount;
    auto result = g_Service->GetSendData(&buffer[0], &sendDataCount, offset, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return sendDataCount;
}

void AcquireReceiveDataEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEvent);
    nn::sf::NativeHandle handle;
    auto result = g_Service->AcquireReceiveDataEvent(&handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::AttachReadableHandleToSystemEvent(pEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
}

uint32_t GetCurrentReceiveDataCounter() NN_NOEXCEPT
{
    uint32_t counter;
    auto result = g_Service->GetCurrentReceiveDataCounter(&counter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return counter;
}

uint32_t GetOldestReceiveDataCounter() NN_NOEXCEPT
{
    uint32_t counter;
    auto result = g_Service->GetOldestReceiveDataCounter(&counter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return counter;
}

uint32_t GetNextReceiveDataCounter() NN_NOEXCEPT
{
    uint32_t counter;
    auto result = g_Service->GetNextReceiveDataCounter(&counter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return counter;
}

uint32_t GetRecentReceiveDataCounter(uint32_t counter, int count) NN_NOEXCEPT
{
    uint32_t recentReceiveDataCounter;
    auto result = g_Service->GetRecentReceiveDataCounter(&recentReceiveDataCounter, counter, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return recentReceiveDataCounter;
}

int GetAvailableReceiveDataCount(uint32_t counter) NN_NOEXCEPT
{
    int count;
    auto result = g_Service->GetAvailableReceiveDataCount(&count, counter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return count;
}

int GetReceiveData(ReceiveDataDescription buffer[], uint32_t* pNextCounter, uint32_t counter, int count) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_NOT_NULL(pNextCounter);
    int receiveDataCount;
    nn::sf::OutArray<ReceiveDataDescription> outReceiveData(buffer, count);
    auto result = g_Service->GetReceiveData(outReceiveData, &receiveDataCount, pNextCounter, counter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return receiveDataCount;
}

void AddReceiveData(const ReceiveDataDescription& data) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS_EQUAL(data.dataSize, ReceiveDataSizeMax);
    auto result = g_Service->AddReceiveData(data);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void ClearReceiveData() NN_NOEXCEPT
{
    auto result = g_Service->ClearReceiveData();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void ClearDataIdFilter() NN_NOEXCEPT
{
    auto result = g_Service->ClearDataIdFilter();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void AcquireDeviceScanEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEvent);
    nn::sf::NativeHandle handle;
    auto result = g_Service->AcquireDeviceScanEvent(&handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::AttachReadableHandleToSystemEvent(pEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
}

/*
void LendDeviceScanMemory(nn::os::NativeHandle handle) NN_NOEXCEPT
{

}

void PullbackDeviceScanMemory(nn::os::NativeHandle handle) NN_NOEXCEPT
{

}
*/

void StartDeviceScan() NN_NOEXCEPT
{
    auto result = g_Service->StartDeviceScan();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void CancelDeviceScan() NN_NOEXCEPT
{
    auto result = g_Service->CancelDeviceScan();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetDeviceScanResult(ReceiveDataDescription buffer[], int count) NN_NOEXCEPT
{
    int scanResultCount;
    nn::sf::OutArray<ReceiveDataDescription> outScanResult(buffer, count);
    auto result = g_Service->GetDeviceScanResult(outScanResult, &scanResultCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return scanResultCount;
}

//! @}

}} // namespace nn::ndd
