﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkAssert.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_DetectApi.h>
#include <nn/wlan/wlan_Result.h>

#include <nn/ndd/detail/ndd_Wlan.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/detail/ndd_Packet.h>

namespace nn { namespace ndd { namespace wlan { namespace driver {

namespace {

const size_t SendSizeMax     = 1317;
const size_t ReceiveSizeMax  = 1318;
const size_t SendHeaderSize    = 17;
const size_t ReceiveHeaderSize = 18;
const size_t SendDataSizeMax    = SendSizeMax - SendHeaderSize;//typesと名前被り
const size_t ReceiveDataSizeMax = ReceiveSizeMax - ReceiveHeaderSize;//typesと名前被り

//送信バッファはデータ部をそのまま渡すため、不要
uint8_t g_ReceiveBuffer[ReceiveSizeMax];//受信バッファは ヘッダ部+データ部

//設計上最大値でも溢れないことをチェック（内部実装における境界のため、両レイヤの最大値を比較）
NN_STATIC_ASSERT(SendDataSizeMax    == Packet::PacketSizeMax);
NN_STATIC_ASSERT(ReceiveDataSizeMax == Packet::PacketSizeMax);

NN_STATIC_ASSERT(sizeof(nn::wlan::DetectHash) == DataIdSize);

//[todo]1つ上の階層での管理を検討。上位は1スレッドなので、このままでも問題は無い
uint32_t g_RxEntryId;
}

void Initialize()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    nn::wlan::InitializeDetectManager();
    auto result = nn::wlan::Detect::OpenMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void Finalize()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
    nn::wlan::FinalizeDetectManager();
}

void EnableNetwork()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::StartCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void DisableNetwork()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::StopCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void EnableSend(const void* pData, size_t size, DataId dataId)
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    NN_ABORT_UNLESS_LESS_EQUAL(size, SendDataSizeMax);

    auto subType = nn::wlan::ActionFrameType_Detect;
    auto intervalMs = 100;
    nn::wlan::DetectHash detectHash;
    memcpy(&detectHash.hash[0], &dataId.raw[0], DataIdSize);

    NN_NDD_LOG("dataSize = %d\n",size);
    PrintByteDataAsHex(pData, size);
    auto result = nn::wlan::Detect::StartPeriodicActionFrame(subType, detectHash, static_cast<const uint8_t*>(pData), size, intervalMs);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void DisableSend()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::CancelPeriodicActionFrame();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void Sleep()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::RequestSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void Awake()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::RequestWakeUp();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void DetectiveSleep(const void* pData, size_t size, DataId dataId)
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    //[todo]HashListの登録

    NN_ABORT_UNLESS_LESS_EQUAL(size, SendDataSizeMax);
    auto subType = nn::wlan::ActionFrameType_Detect;
    nn::wlan::DetectHash detectHash;
    memcpy(&detectHash.hash[0], &dataId.raw[0], DataIdSize);
    NN_NDD_LOG("dataSize = %d\n",size);
    PrintByteDataAsHex(pData, size);
    nn::Result result;
    result = nn::wlan::Detect::SetActionFrameForSleep(subType, detectHash, static_cast<const uint8_t*>(pData), size);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視

    result = nn::wlan::Detect::ReserveDetectSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視

    result = nn::wlan::Detect::RequestSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void DetectiveAwake()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::RequestWakeUp();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

void CreateReceiveEntry()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    uint16_t afTypes[] = {   // 受信したいActionFrameType
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect)
    };
    auto result = nn::wlan::Detect::CreateRxEntryForActionFrame(&g_RxEntryId, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
    NN_NDD_LOG("rxEntry = %d\n", g_RxEntryId);
}

void DeleteReceiveEntry()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::DeleteRxEntryForActionFrame(g_RxEntryId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

bool Receive(void* pBuffer, size_t* pSize, size_t size)
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pSize);

    size_t receiveSize;
    nn::wlan::MacAddress macAddress;
    uint16_t channel;
    int16_t rssi;
    nn::os::Tick tick;

    auto result = nn::wlan::Detect::GetActionFrame(&macAddress, &g_ReceiveBuffer[0], ReceiveSizeMax, &receiveSize,
            g_RxEntryId, &channel, &rssi, &tick); // ブロックされる
    if(result.IsSuccess())
    {
        //解析用ダンプ
        /*
        NN_NDD_LOG("Wlan Receive\n");
        NN_NDD_LOG("macAddress = ");
        PrintByteDataAsHex(&macAddress.GetMacAddressData()[0], nn::wlan::MacAddress::MacAddressSize);
        NN_NDD_LOG("headerSize = %d\n", ReceiveHeaderSize);
        NN_NDD_LOG("dataSize = %d\n", receiveSize - ReceiveHeaderSize);
        PrintByteDataAsHex(&g_ReceiveBuffer[0], receiveSize);
        */

        NN_ABORT_UNLESS_LESS_EQUAL(receiveSize, ReceiveSizeMax);//これには引っかからない。仮にオーバーしたらAPIが失敗を返す
        NN_ABORT_UNLESS(size >= receiveSize - ReceiveHeaderSize);//提供されたバッファにコピーできるか確認
        memcpy(pBuffer, &g_ReceiveBuffer[ReceiveHeaderSize], receiveSize - ReceiveHeaderSize);
        *pSize = receiveSize - ReceiveHeaderSize;
        return true;
    }
    else if(result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription())
    {
        NN_NDD_LOG("Wlan Receive cancelled\n");
        return false;
    }
    else if(result.GetDescription() == nn::wlan::ResultBufferTooShort().GetDescription())
    {
        NN_NDD_LOG("buffer too small\n");
        return false;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
        return false;
    }
}

void CancelReceive()
{
    NN_NDD_LOG("Wlan %s\n", __FUNCTION__);
    auto result = nn::wlan::Detect::CancelGetActionFrame(g_RxEntryId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);//[todo]DeviceAbnormalの無視
}

}}}}
