﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/ndd/detail/ndd_Storage.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/ndd_PrivateResult.h>

namespace nn { namespace ndd {

uint16_t File::GetFileIndex(uint16_t dataIndex)
{
    return dataIndex / DataCountMax;
}

uint16_t File::GetDataIndexInFile(uint16_t dataIndex)
{
    return dataIndex % DataCountMax;
}

void DummyFs::Write(const File& file, uint16_t index)
{
    NN_ABORT_UNLESS(index < FileCountMax);
    m_File[index] = file;
}

void DummyFs::Read(File* pFile, uint16_t index) const
{
    NN_ABORT_UNLESS(index < FileCountMax);
    *pFile = m_File[index];
}

Storage::Storage()
{
    //[todo]復旧
    Clear();
}

void Storage::Write(const ReceiveDataDescription& rdd)
{
    if(m_WriteCache.dataCount == File::DataCountMax)
    {
        //WriteCacheが一杯の場合は、使用中のキャッシュをfsに書き込み、新しいキャッシュを生成
        ApplyWriteCache();
    }

    m_WriteCache.file.rdd[m_WriteCache.dataCount] = rdd;
    ++m_WriteCache.dataCount;
}

void Storage::Read(ReceiveDataDescription* pRdd, uint16_t dataIndex)
{
    NN_ABORT_UNLESS(dataIndex < File::DataCountMax * DummyFs::FileCountMax);

    //WriteCacheからの読み出しを試行
    if(ReadFromWriteCache(pRdd, dataIndex))
    {
        return;
    }

    //ReadCacheからの読み出しを試行
    if(ReadFromReadCache(pRdd, dataIndex))
    {
        return;
    }

    //ReadCacheを、所望のdataIndexが含まれる領域に更新し、読み出し
    ChangeReadCache(dataIndex);
    if(ReadFromReadCache(pRdd, dataIndex))
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
}

void Storage::Clear()
{
    m_ReadCache.fileIndex = 0;
    m_ReadCache.dataCount = 0;
    m_WriteCache.fileIndex = 0;
    m_WriteCache.dataCount = 0;
    //[todo]fsのクリア
}

void Storage::ApplyWriteCache()
{
    //dataが一杯（File::DataCountMax）のファイルのみを書く前提
    //[todo]一杯以外への対応
    m_DummyFs.Write(m_WriteCache.file, m_WriteCache.fileIndex);
    if(m_WriteCache.fileIndex < DummyFs::FileCountMax - 1)
    {
        ++m_WriteCache.fileIndex;
    }
    else
    {
        m_WriteCache.fileIndex = 0;
    }
    m_WriteCache.dataCount = 0;
}

void Storage::ChangeReadCache(uint16_t dataIndex)
{
    //detaが一杯（File::DataCountMax）のファイルのみが存在する前提
    //[todo]一杯以外への対応
    auto fileIndex = File::GetFileIndex(dataIndex);
    NN_ABORT_UNLESS(fileIndex < DummyFs::FileCountMax);
    m_ReadCache.fileIndex = fileIndex;
    m_ReadCache.dataCount = File::DataCountMax;
    m_DummyFs.Read(&m_ReadCache.file, fileIndex);
}

bool Storage::ReadFromWriteCache(ReceiveDataDescription* pRdd, uint16_t dataIndex)
{
    auto fileIndex = File::GetFileIndex(dataIndex);
    auto dataIndexInFile = File::GetDataIndexInFile(dataIndex);
    if(m_WriteCache.fileIndex == fileIndex)
    {
        if(m_WriteCache.dataCount > dataIndexInFile)
        {
            *pRdd = m_WriteCache.file.rdd[dataIndexInFile];
            return true;
        }
    }
    return false;
}

bool Storage::ReadFromReadCache(ReceiveDataDescription* pRdd, uint16_t dataIndex)
{
    auto fileIndex = File::GetFileIndex(dataIndex);
    auto dataIndexInFile = File::GetDataIndexInFile(dataIndex);
    if(m_ReadCache.fileIndex == fileIndex)
    {
        if(m_ReadCache.dataCount > dataIndexInFile)
        {
            *pRdd = m_ReadCache.file.rdd[dataIndexInFile];
            return true;
        }
    }
    return false;
}

}}
