﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/os.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>

#include <nn/ndd/detail/ndd_ReceiveSubmodule.h>
#include <nn/ndd/detail/ndd_Receive.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/detail/ndd_Queue.h>
#include <nn/ndd/detail/ndd_Packet.h>
#include <nn/ndd/detail/ndd_Wlan.h>

namespace nn { namespace ndd { namespace wlan { namespace receive {

namespace {

RingBuffer g_RingBuffer;
ScanBuffer g_ScanBuffer;
RejectDataIdContainer g_RejectIdContainer;
Message g_Message;

//スレッド
const size_t ThreadStackSize = 4096;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackForReceive[ThreadStackSize];
nn::os::ThreadType  g_ReceiveThread;

//受信パケットの加工用
const size_t ReceiveSizeMax = 1300;
uint8_t g_ReceiveBuffer[ReceiveSizeMax];
NN_STATIC_ASSERT(ReceiveSizeMax == Packet::PacketSizeMax);
Packet g_Packet;
ReceiveDataDescription g_ReceiveDataDescription;

//停止指示。停止同期は現状スレッド破棄
bool g_IsStopRequest;
}

//下層からの受信
void Add(const void* pData, size_t size)
{
    NN_ABORT_UNLESS_NOT_NULL(pData);
    NN_SDK_ASSERT_LESS_EQUAL(size, ReceiveSizeMax);

    if(g_Packet.SetFromPacket(pData, size))
    {
        //正式なパケットのため続行
    }
    else
    {
        return;
    }

    //データ整形
    //[todo]コピー回数最適化
    memcpy(&g_ReceiveDataDescription.data[0], g_Packet.GetDataPtr(), g_Packet.GetDataSize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&g_ReceiveDataDescription.receiveTimePoint));
    g_ReceiveDataDescription.dataSize = g_Packet.GetDataSize();

    //スキャンバッファへの追加
    g_ScanBuffer.Add(g_ReceiveDataDescription);

    //リングバッファへの追加
    auto dataId = g_Packet.GetDataId();
    if(g_RejectIdContainer.IsReject(dataId))
    {
        //NN_NDD_LOG("Received data is Rejected.\n");
        //PrintByteDataAsHex(&dataId.raw[0], DataIdSize);
        //重複なので何もしない
    }
    else
    {
        g_RejectIdContainer.Add(dataId);
        g_RingBuffer.Add(g_ReceiveDataDescription);

        g_Message.Set(ApiId_System_Received);
        queue::EnQueue(g_Message);
    }
}

void ReceiveThread(void* arg)
{
    while(!g_IsStopRequest)
    {
        size_t receiveSize;
        bool isReceive;
        isReceive = nn::ndd::wlan::driver::Receive(&g_ReceiveBuffer[0], &receiveSize, ReceiveSizeMax);//ブロック
        if(isReceive)
        {
            Add(&g_ReceiveBuffer[0], receiveSize);
        }
        //[todo]キャンセル時はループ回らないようにする
    }
}

void Initialize()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    g_IsStopRequest = true;
    g_RingBuffer.Initialize();
}

void Start()
{
    // ActionFrame受信スレッド作成
    g_IsStopRequest = false;
    auto result = nn::os::CreateThread( &g_ReceiveThread, ReceiveThread, nullptr, g_ThreadStackForReceive, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(ndd, Receiver));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&g_ReceiveThread, NN_SYSTEM_THREAD_NAME(ndd, Receiver));
    nn::os::StartThread(&g_ReceiveThread);
}

void Stop()
{
    //[todo]Start直後のStopで、無効なRxEntryにキャンセルしうる件対処。RxEntry出来るまでStartをブロックするか、Stop側のCancelのエラーハンドリングを柔軟にするか
    g_IsStopRequest = true;
    nn::ndd::wlan::driver::CancelReceive();
    WaitThread(&g_ReceiveThread);
    DestroyThread(&g_ReceiveThread);
}

//上層からの受信（テスト用）
void Add(const ReceiveDataDescription& data)
{
    g_ScanBuffer.Add(data);
    g_RingBuffer.Add(data);
}

void Clear()
{
    g_RingBuffer.Clear();
}

void ClearDataIdFilter()
{
    g_RejectIdContainer.Clear();
}

uint32_t GetCurrentCounter()
{
    return g_RingBuffer.GetCurrentCounter();
}

uint32_t GetOldestCounter()
{
    return g_RingBuffer.GetOldestCounter();
}

uint32_t GetNextCounter()
{
    return g_RingBuffer.GetNextCounter();
}

uint32_t GetRecentCounter(uint32_t counter, int count)
{
    return g_RingBuffer.GetRecentCounter(counter, count);
}

uint16_t GetAvailableCount(uint32_t counter)
{
    return g_RingBuffer.GetAvailableCount(counter);
}

bool Get(ReceiveDataDescription* pBuffer, uint32_t* pNextCounter, uint32_t counter)
{
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pNextCounter);
    return g_RingBuffer.Get(pBuffer, pNextCounter, counter);
}

int Get(ReceiveDataDescription buffer[], uint32_t* pNextCounter, int count, uint32_t counter)
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS(count > 0);
    NN_ABORT_UNLESS_NOT_NULL(pNextCounter);

    *pNextCounter = counter;
    for(int i=0;i<count;++i)
    {
        auto isExist = g_RingBuffer.Get(&buffer[i], pNextCounter, *pNextCounter);
        if(!isExist)
        {
            return i;
        }
    }
    return count;
}

int GetDeviceScanResult(ReceiveDataDescription buffer[], int count)
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS(count > 0);
    auto resultCount = std::min<uint16_t>(count, g_ScanBuffer.GetCount());
    for(int i = 0;i<resultCount;++i)
    {
        buffer[i] = g_ScanBuffer.Get(i);
    }
    return resultCount;
}

void ClearDeviceScanResult()
{
    g_ScanBuffer.Clear();
}

}}}}
