﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/ndd/detail/ndd_Main.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/detail/ndd_Handler.h>
#include <nn/ndd/detail/Service/ndd_Impl.h>
#include <nn/ndd/detail/Service/ndd_ServiceName.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h> // ndd サービスを稼動させるかの fwdbg settings チェック用。常時稼動させるようになったら消す。

namespace nn { namespace ndd { namespace detail
{


// SF 用
namespace {
struct MyServerManagerOption
{
    static const size_t PointerTransferBufferSize = 4096;
};

class NddServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<4, 1, MyServerManagerOption>
{
};

NddServerManager*  g_pNddServerManager;
std::aligned_storage<sizeof(NddServerManager), NN_ALIGNOF(NddServerManager)>::type g_NddServerManagerStorage;

void InitializeNddServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pNddServerManager);
    g_pNddServerManager = new (&g_NddServerManagerStorage) NddServerManager;

    // サービス名の登録とポートの初期化
    auto sessionCountMax = 4;
    typedef nn::sf::ObjectFactory<nn::sf::impl::StaticOneAllocationPolicy>  Factory;
    g_pNddServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::ndd::IServiceCreator, nn::ndd::ServiceCreatorImpl>(), sessionCountMax, nn::ndd::ServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pNddServerManager->Start();
}

void FinalizeNddServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pNddServerManager);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pNddServerManager->~NddServerManager();
    g_pNddServerManager = nullptr;
}

void LoopNddServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pNddServerManager);
    g_pNddServerManager->LoopAuto();
}

void RequestStopNddServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pNddServerManager);
    g_pNddServerManager->RequestStop();
}
} // namespace

void NddServerThread(void *arg)
{
    char* pThreadName = static_cast<char*>(arg);
    NN_UNUSED( pThreadName );
    NN_NDD_LOG("Started ServerThread %s\n",pThreadName);
    LoopNddServer();
}


namespace {
//スレッド共通
const size_t ThreadStackSize = 4096;

//Workerスレッド
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackForWorker[ThreadStackSize];
nn::os::ThreadType  g_WorkerThread;

NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForService[ThreadStackSize];
nn::os::ThreadType  g_ServiceThread;
char g_ThreadNameForService[]="ServiceThread";
}

bool IsNddServiceEnabled() NN_NOEXCEPT
{
    bool enabled;
    auto size = nn::settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "ndd", "service_enabled");
    if( size != sizeof(enabled) )
    {
        return false;
    }
    return enabled;
}

void NddMainThread(void* arg)
{
    NN_DETAIL_NDD_ERROR("----------------------------------------\n");
    NN_DETAIL_NDD_ERROR("!!! NDD works !!!\n");
    NN_DETAIL_NDD_ERROR("----------------------------------------\n");
    //[todo]Mainスレッドはworkerにする

    nn::ndd::handler::InitializeWorker();//[todo]Workの先頭に移行
    InitializeNddServer();

    {
        auto result = nn::os::CreateThread(&g_WorkerThread, nn::ndd::handler::Work, nullptr, g_ThreadStackForWorker, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(ndd, Worker));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SetThreadNamePointer(&g_WorkerThread, NN_SYSTEM_THREAD_NAME(ndd, Worker));
        nn::os::StartThread(&g_WorkerThread);
    }

    {
        auto result = nn::os::CreateThread(&g_ServiceThread, NddServerThread, g_ThreadNameForService, g_ThreadStackForService, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(ndd, IpcServer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SetThreadNamePointer(&g_ServiceThread, NN_SYSTEM_THREAD_NAME(ndd, IpcServer));
        nn::os::StartThread(&g_ServiceThread);
    }

    nn::os::WaitThread(&g_WorkerThread);
    nn::os::WaitThread(&g_ServiceThread);
    NN_UNUSED(RequestStopNddServer);
    NN_UNUSED(FinalizeNddServer);
}

}}} // namespace nn::ndd::detail

