﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/ndd/detail/ndd_Context.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/ndd_PrivateResult.h>

namespace nn { namespace ndd {

void Context::Initialize()
{
    nn::Result result;
    result = nn::os::CreateSystemEvent(&m_SendDataUpdateEvent, nn::os::EventClearMode_AutoClear, true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::os::CreateSystemEvent(&m_ReceiveDataEvent, nn::os::EventClearMode_AutoClear, true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::os::CreateSystemEvent(&m_DeviceScanEvent, nn::os::EventClearMode_AutoClear, true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::InitializeEvent(&m_ReplyEvent, false, nn::os::EventClearMode_AutoClear);
}

void Context::Finalize()
{
    nn::os::DestroySystemEvent(&m_SendDataUpdateEvent);
    nn::os::DestroySystemEvent(&m_ReceiveDataEvent);
    nn::os::DestroySystemEvent(&m_DeviceScanEvent);
    nn::os::FinalizeEvent(&m_ReplyEvent);
}

nn::os::NativeHandle Context::AcquireHandle(EventId eventId)
{
    switch(eventId)
    {
        case EventId_SendDataUpdate:
        {
            return nn::os::GetReadableHandleOfSystemEvent(&m_SendDataUpdateEvent);
            break;
        }
        case EventId_ReceiveData:
        {
            return nn::os::GetReadableHandleOfSystemEvent(&m_ReceiveDataEvent);
            break;
        }
        case EventId_DeviceScan:
        {
            return nn::os::GetReadableHandleOfSystemEvent(&m_DeviceScanEvent);
            break;
        }
        default:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
        }
    }
}

void Context::Signal(EventId eventId)
{
    switch(eventId)
    {
        case EventId_SendDataUpdate:
        {
            nn::os::SignalSystemEvent(&m_SendDataUpdateEvent);
            break;
        }
        case EventId_ReceiveData:
        {
            nn::os::SignalSystemEvent(&m_ReceiveDataEvent);
            break;
        }
        case EventId_DeviceScan:
        {
            nn::os::SignalSystemEvent(&m_DeviceScanEvent);
            break;
        }
        default:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
        }
    }
}

void Context::SignalReply()
{
    nn::os::SignalEvent(&m_ReplyEvent);
}

void Context::WaitReply()
{
    nn::os::WaitEvent(&m_ReplyEvent);
    //[todo]コピーしてreturn
}

void Context::SetHandle(Handle handle)
{
    m_Handle = handle;
}

Handle Context::GetHandle() const
{
    return m_Handle;
}

}}
