﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/account.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/nd/detail/nd_FirmwareDebugSettings.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/nd/service/nd_Service.h>
#include <nn/ndd.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace nd { namespace service {

namespace {

Result VerifyUserExistence(const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUser());
    bool existence;
    NN_RESULT_DO(account::GetUserExistence(&existence, uid));
    NN_RESULT_THROW_UNLESS(existence, ResultNonExistentUser());
    NN_RESULT_SUCCESS;
}

} // ~nn::nd::service::<anonymous>

// ServiceBase

ServiceBase::ServiceBase(
    nn::MemoryResource& allocator,
    ReceiveDataManager& receiveDataManager,
    SendDataManager& sendDataManager) NN_NOEXCEPT
    : m_Allocator(allocator)
    , m_ReceiveDataManager(receiveDataManager)
    , m_SendDataManager(sendDataManager)
{
}

// ServiceForSystem

ServiceForSystem::ServiceForSystem(
    nn::MemoryResource& allocator,
    ReceiveDataManager& receiveDataManager,
    SendDataManager& sendDataManager,
    ScanManager& scanManager) NN_NOEXCEPT
    : ServiceBase(allocator, receiveDataManager, sendDataManager)
    , m_ScanManager(scanManager)
{
}

nn::Result ServiceForSystem::SetSystemData(const nn::account::Uid& uid, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));
    NN_RESULT_THROW_UNLESS(0 < buffer.GetSize() && buffer.GetSize() <= SystemDataSizeMax, ResultInvalidDataSize());

    m_SendDataManager.SetSystemData(uid, buffer.GetPointerUnsafe(), buffer.GetSize());
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::ClearSystemData(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));

    m_SendDataManager.ClearSystemData(uid);
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::GetNeighborInfoUpdateEventForSystem(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    auto pEvent = m_ReceiveDataManager.GetNeighborInfoUpdateEventForSystem();
    *pOutHandle = sf::NativeHandle(pEvent->GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::GetReceivableNeighborInfoCountForSystem(nn::sf::Out<std::int32_t> pOutCount, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));

    *pOutCount = m_ReceiveDataManager.GetReceivableNeighborInfoCountForSystem(uid);
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::ReceiveNeighborInfoForSystem(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::nd::NeighborInfoForSystem>& outInfo, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));

    *pOutCount = m_ReceiveDataManager.ReceiveNeighborInfoForSystem(outInfo.GetData(), static_cast<int>(outInfo.GetLength()), uid);
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::SetSender(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));
    m_SendDataManager.SetSender(uid);
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::GetSender(nn::sf::Out<bool> pOutSenderExists, nn::sf::Out<nn::account::Uid> pOutSender) NN_NOEXCEPT
{
    *pOutSenderExists = m_SendDataManager.GetSender(pOutSender.GetPointer());
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::CreateScannerForSystem(nn::sf::Out<nn::sf::SharedPointer<nn::nd::service::IScannerForSystem>> pOut) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::nd::service::IScannerForSystem> p;
    NN_RESULT_DO(m_ScanManager.CreateScannerForSystem(&p));
    *pOut = p;
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::AddReceivedNeighborInfoForSystemForDebug(const nn::sf::InArray<nn::nd::NeighborInfoForSystem>& info) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(detail::FirmwareDebugSettings::IsDevelopmentFunctionEnabled(), ResultDisabledDevelopmentFunction());

    for( size_t i = 0; i < info.GetLength(); i++ )
    {
        ndd::ReceiveDataDescription data;
        detail::MakeReceiveDataDescription(&data, info[i]);
        ndd::AddReceiveData(data);
    }
    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::GetSendDataForDebug(nn::sf::Out<nn::ndd::SendDataDescription> pOut, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(detail::FirmwareDebugSettings::IsDevelopmentFunctionEnabled(), ResultDisabledDevelopmentFunction());
    NN_RESULT_DO(VerifyUserExistence(uid));

    m_SendDataManager.GetSendDataForDebug(pOut.GetPointer(), uid);

    NN_RESULT_SUCCESS;
}

nn::Result ServiceForSystem::ClearReceiveCounterForDebug(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(detail::FirmwareDebugSettings::IsDevelopmentFunctionEnabled(), ResultDisabledDevelopmentFunction());
    NN_RESULT_DO(VerifyUserExistence(uid));

    m_ReceiveDataManager.ClearReceiveCounterForDebug(uid);

    NN_RESULT_SUCCESS;
}

// ServiceForApplication

ServiceForApplication::ServiceForApplication(
    nn::MemoryResource& allocator,
    ReceiveDataManager& receiveDataManager,
    SendDataManager& sendDataManager) NN_NOEXCEPT
    : ServiceBase(allocator, receiveDataManager, sendDataManager)
{
}

nn::Result ServiceForApplication::SetApplicationData(const nn::account::Uid& uid, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));
    NN_UNUSED(buffer);

    NN_RESULT_THROW(ResultNotImplemented());
}

nn::Result ServiceForApplication::ClearApplicationData(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyUserExistence(uid));

    NN_RESULT_THROW(ResultNotImplemented());
}

}}} // ~namespace nn::nd::service
