﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/service/nd_IService.sfdl.h>
#include <nn/nd/service/nd_Server.h>
#include <nn/nd/service/nd_Service.h>
#include <nn/nd/service/nd_ServiceNames.h>
#include <nn/nd/service/nd_ServiceResource.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_LmemUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace nd { namespace service {

namespace
{

// TODO: パラメータ調整

static const int NeighborDetectionSessionCountMax = 6;
static const int NeighborDetectionPortCountMax = 2;
struct NeighborDetectionServerManagerOption
{
    static const bool CanDeferInvokeRequest = false;
    static const int ObjectInSubDomainCountMax = 128;
    static const size_t PointerTransferBufferSize = 4096;
    static const int SubDomainCountMax = 16;
};

class NeighborDetectionServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<NeighborDetectionSessionCountMax, NeighborDetectionPortCountMax, NeighborDetectionServerManagerOption>
{
};

std::aligned_storage<sizeof(NeighborDetectionServerManager), NN_ALIGNOF(NeighborDetectionServerManager)>::type g_ServerManagerStorage;
NeighborDetectionServerManager* g_pServerManager;

std::aligned_storage<sizeof(ServiceResource), NN_ALIGNOF(ServiceResource)>::type g_ServiceResourceStorage;
ServiceResource* g_pServerResource;

NN_ALIGNAS(4096) char g_ServerThreadStack[16 * 1024];
nn::os::ThreadType g_ServerThread;

NN_ALIGNAS(4096) char g_NetworkThreadStack[4 * 1024];

nn::MemoryResource& GetMemoryResource() NN_NOEXCEPT
{
    static Bit8 s_HeapMemory[8 * 1024];
    NN_FUNCTION_LOCAL_STATIC(lmem::HeapHandle, s_HeapHandle, = lmem::CreateExpHeap(s_HeapMemory, sizeof(s_HeapMemory), lmem::CreationOption_ThreadSafe));
    NN_FUNCTION_LOCAL_STATIC(sf::ExpHeapMemoryResource, s_MemoryResource, { s_HeapHandle });
    return s_MemoryResource;
}

} // ~nn::nd::service::<anonymous>

void StartServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pServerManager);

    g_pServerManager = new (&g_ServerManagerStorage) NeighborDetectionServerManager;
    g_pServerResource = new (&g_ServiceResourceStorage) ServiceResource(GetMemoryResource(), g_NetworkThreadStack, sizeof(g_NetworkThreadStack));
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(g_pServerResource->GetSystemServicePointer(), 4, ServiceNameForSystem));
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(g_pServerResource->GetApplicationServicePointer(), 2, ServiceNameForApplication));

    g_pServerManager->Start();

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&g_ServerThread, [](void*) {
        NN_DETAIL_ND_TRACE("[nd] IpcServer started.\n");
        g_pServerManager->LoopAuto();
    }, nullptr, g_ServerThreadStack, sizeof(g_ServerThreadStack), NN_SYSTEM_THREAD_PRIORITY(nd, IpcServer)));
    os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(nd, IpcServer));
    os::StartThread(&g_ServerThread);
}

}}} // ~nn::nd::service
