﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/nd/service/nd_ReceiveDataManager.h>
#include <nn/ndd.h>
#include <nn/ndd/ndd_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace nd { namespace service {

// ReceiveDataManager::UserData

void ReceiveDataManager::UserData::Reset() NN_NOEXCEPT
{
    *this = UserData{};
}

void ReceiveDataManager::UserData::Initialize(const account::Uid& _uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(_uid);
    NN_SDK_REQUIRES(!m_Uid);

    m_Uid = _uid;
    m_ReceivableCount = DefaultReceivableCount;
    // 初期化のタイミングより後に受信したデータから取得可能。
    m_NextReadCounter = ndd::GetNextReceiveDataCounter();
}

bool ReceiveDataManager::UserData::IsInitialized() const NN_NOEXCEPT
{
    return static_cast<bool>(m_Uid);
}

// ReceiveDatamanager

void ReceiveDataManager::Initialize(const util::Span<const account::Uid>& uids) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(static_cast<int>(uids.size()), account::UserCountMax);

    int count = 0;
    for( auto& u : uids )
    {
        m_UserDatas[count++].Initialize(u);
    }
}

void ReceiveDataManager::Update(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    UpdateUnsafe(newUids);
}

int ReceiveDataManager::GetReceivableNeighborInfoCountForSystem(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    return std::min(pUserData->m_ReceivableCount, ndd::GetAvailableReceiveDataCount(pUserData->m_NextReadCounter));
}

int ReceiveDataManager::ReceiveNeighborInfoForSystem(NeighborInfoForSystem out[], int outCount, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES_GREATER(outCount, 0);
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    auto startCounter = pUserData->m_NextReadCounter; // ログ用。
    NN_UNUSED(startCounter);
    pUserData->m_NextReadCounter = ndd::GetRecentReceiveDataCounter(pUserData->m_NextReadCounter, pUserData->m_ReceivableCount);

    auto networkUserId = m_UserIdManager.GetNetworkUserId(uid);

    // TORIAEZU: ワークバッファとしての ndd::ReceiveDataDescription を static で確保。数は適当。
    NN_FUNCTION_LOCAL_STATIC(ndd::ReceiveDataDescription, s_ReceiveDataDescription[3]);

    int actualOutCount = 0;
    while( actualOutCount < outCount )
    {
        auto count = ndd::GetReceiveData(s_ReceiveDataDescription, &pUserData->m_NextReadCounter, pUserData->m_NextReadCounter,
                                         std::min(static_cast<int>(NN_ARRAY_SIZE(s_ReceiveDataDescription)), (outCount - actualOutCount)));
        if( count == 0 )
        {
            break;
        }
        auto parseFailureCount = 0;
        for( int i = 0; i < count; i++ )
        {
            if( !detail::ParseReceiveDataDescription(&out[actualOutCount + i - parseFailureCount], s_ReceiveDataDescription[i]) )
            {
                parseFailureCount++;
            }
            if( networkUserId && out[actualOutCount + i - parseFailureCount].networkUserId &&
                (networkUserId == out[actualOutCount + i - parseFailureCount].networkUserId) )
            {
                // TORIAEZU: 自分のデータをパース失敗扱いで飛ばす。パース失敗ではないのでもう少しいい感じにしたい。
                parseFailureCount++;
            }
        }
        actualOutCount += (count - parseFailureCount);
    }
    NN_DETAIL_ND_TRACE("[nd] ReceiveDataManager::ReceiveNeighborInfoForSystem : Received %d data. Counter is updated from %u to %u\n",
        actualOutCount, startCounter, pUserData->m_NextReadCounter);
    return actualOutCount;
}

os::SystemEvent* ReceiveDataManager::GetNeighborInfoUpdateEventForSystem() NN_NOEXCEPT
{
    return &m_ReceiveEventForSystem;
}

void ReceiveDataManager::SignalNeighborInfoUpdateEventForSystem() NN_NOEXCEPT
{
    m_ReceiveEventForSystem.Signal();
}

void ReceiveDataManager::ClearReceiveCounterForDebug(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    pUserData->m_NextReadCounter = 0u;
}

void ReceiveDataManager::UpdateUnsafe(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    // 削除されたユーザーを探す
    for( auto& u : m_UserDatas )
    {
        if( !u.IsInitialized() )
        {
            continue;
        }
        if( std::find(newUids.begin(), newUids.end(), u.m_Uid) != newUids.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] ReceiveDataManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is deleted.\n", u.m_Uid._data[0], u.m_Uid._data[1]);
        u.Reset();
    }

    // 追加されたユーザーを探す
    for( auto& newU : newUids )
    {
        if( std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&newU](const auto& u) { return u.m_Uid == newU; }) != m_UserDatas.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] ReceiveDataManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is added.\n", newU._data[0], newU._data[1]);
        auto u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [](const auto& u) { return !u.IsInitialized(); });
        NN_ABORT_UNLESS(!(u == m_UserDatas.end()), "[nd] ReceiveDataManager::UpdateUnsafe : UserDatas is full.");
        u->Initialize(newU);
    }
}

ReceiveDataManager::UserData* ReceiveDataManager::FindOrInitUserDataUnsafe(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    auto u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&uid](const UserData& u) { return u.m_Uid == uid; });
    if( u != m_UserDatas.end() )
    {
        return &(*u);
    }
    // ユーザー追加時に初期化しているので基本的には到達しないはず。EventHandler に処理が回る前に API 呼び出しされた場合に到達する。。
    // ユーザーの削除と追加が連続して行われていた場合には追加されたユーザーの分の領域が空いていない可能性があるので、空き領域を探して初期化するのではなく Update を呼び出す。
    int accountCount;
    account::Uid uids[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uids, NN_ARRAY_SIZE(uids)));
    UpdateUnsafe({ uids, accountCount });
    u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&uid](const UserData& u) { return u.m_Uid == uid; });
    NN_ABORT_UNLESS(u != m_UserDatas.end());
    return &(*u);
}

}}} // ~nn::nd::service
