﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/user/migration_UserMigrationStateManager.h>

#include <nn/fs/fs_Result.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/user/migration_UserMigrationInternalTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_FirmwareVersion.h>

namespace nn { namespace migration { namespace user {

namespace {

static const char UserMigrationServerStatePath[] = "/server.bin";
static const char UserMigrationClientStatePath[] = "/client.bin";

struct StoredState
{
    uint32_t fwVersion;
    MigrationState state;
};

uint32_t GetFirmwareVersion() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    settings::system::FirmwareVersion fwVer;
    settings::system::GetFirmwareVersion(&fwVer);
    return static_cast<uint32_t>(fwVer.GetComparableVersion());
#else
    return 0x00u;
#endif
}

void StoreImpl(const detail::AbstractStorage& storage, const char* path, const void* data, size_t dataSize) NN_NOEXCEPT
{
    auto lock = storage.AcquireWriteLock();
    auto r = storage.Create(path, dataSize);
    if (!(r.IsSuccess() || fs::ResultPathAlreadyExists::Includes(r)))
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Write(path, data, dataSize));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Commit());
}
bool TryLoadImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const detail::AbstractStorage& storage, const char* path) NN_NOEXCEPT
{
    auto r = storage.Read(pOutActualSize, buffer, bufferSize, path);
    if (!(r.IsSuccess() || fs::ResultPathNotFound::Includes(r)))
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    return r.IsSuccess();
}
void DeleteImpl(const detail::AbstractStorage& storage, const char* path) NN_NOEXCEPT
{
    auto lock = storage.AcquireWriteLock();
    auto r = storage.Delete(path);
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Commit());
}

bool IsValidTransition(MigrationState to, MigrationState from) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidState(to));
    NN_SDK_ASSERT(IsValidState(from));
    NN_SDK_ASSERT(to != MigrationState_Started);

    switch (to)
    {
    case MigrationState_InInitialization:
        switch (from)
        {
        case MigrationState_Started:
            return true;
        default:
            return false;
        }
    case MigrationState_InTransfer:
        switch (from)
        {
        case MigrationState_InInitialization:
            return true;
        default:
            return false;
        }
    case MigrationState_InFinalization:
        switch (from)
        {
        case MigrationState_InInitialization:
        case MigrationState_InTransfer:
            return true;
        default:
            return false;
        }
    case MigrationState_Finalized:
        switch (from)
        {
        case MigrationState_InFinalization:
            return true;
        default:
            return false;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

inline bool IsIgnorableServer0400(MigrationState state) NN_NOEXCEPT
{
    switch (state)
    {
    case MigrationState_Started:
    case MigrationState_InInitialization:
    case MigrationState_InTransfer:
        // サーバー側は、終了処理を始めるまでは中断状態を無視しても影響はない
        return true;
    case MigrationState_InFinalization:
    case MigrationState_Finalized:
        return false;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
inline bool IsIgnorableClient0400(MigrationState state) NN_NOEXCEPT
{
    switch (state)
    {
    case MigrationState_Started:
    case MigrationState_InInitialization:
        // クライアント側は、転送処理を始めるまでは中断状態を無視しても影響はない
        return true;
    case MigrationState_InTransfer:
    case MigrationState_InFinalization:
    case MigrationState_Finalized:
        return false;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
} // ~namespace nn::migration::user::<anonymous>

Result StateManager::TryLoadLastMigrationServerState(bool* pOutExists, MigrationState* pOut) const NN_NOEXCEPT
{
    StoredState store;
    size_t readSize;
    if (!TryLoadImpl(&readSize, &store, sizeof(store), m_Storage, UserMigrationServerStatePath))
    {
        // 中断状態がない
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }
    NN_ABORT_UNLESS_EQUAL(readSize, sizeof(store));

    if (store.fwVersion != GetFirmwareVersion())
    {
        NN_MIGRATION_DETAIL_INFO("[StateManager] Old version of server state found: %d (Firmware: %08x)\n", store.state, store.fwVersion);
        // NOTE: SIGLO-72174
        // サーバー側は、終了処理を始めるまでは中断状態を無視しても影響はない
        NN_RESULT_THROW_UNLESS(IsIgnorableServer0400(store.state), detail::ResultUnexpectedCurrentServerStateVersion());
        NN_MIGRATION_DETAIL_INFO("[StateManager] Old version ignored\n");
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(IsValidState(store.state), detail::ResultUnhandlableCurrentServerState());
    if (!IsResumable(store.state))
    {
        // 再開可能ではない状態は存在しないものとする
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }

    *pOutExists = true;
    *pOut = store.state;
    NN_RESULT_SUCCESS;
}
Result StateManager::TryLoadLastMigrationClientState(bool* pOutExists, MigrationState* pOut) const NN_NOEXCEPT
{
    StoredState store;
    size_t readSize;
    if (!TryLoadImpl(&readSize, &store, sizeof(store), m_Storage, UserMigrationClientStatePath))
    {
        // 中断状態がない
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }
    NN_ABORT_UNLESS_EQUAL(readSize, sizeof(store));

    if (store.fwVersion != GetFirmwareVersion())
    {
        NN_MIGRATION_DETAIL_INFO("[StateManager] Old version of client state found: %d (Firmware: %08x)\n", store.state, store.fwVersion);
        // NOTE: SIGLO-72174
        // クライアント側は、転送処理を始めるまでは中断状態を無視しても影響はない
        NN_RESULT_THROW_UNLESS(IsIgnorableClient0400(store.state), detail::ResultUnexpectedCurrentClientStateVersion());
        NN_MIGRATION_DETAIL_INFO("[StateManager] Old version ignored\n");
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(IsValidState(store.state), detail::ResultUnhandlableCurrentClientState());
    if (!IsResumable(store.state))
    {
        // 再開可能ではない状態は存在しないものとする
        *pOutExists = false;
        NN_RESULT_SUCCESS;
    }

    *pOutExists = true;
    *pOut = store.state;
    NN_RESULT_SUCCESS;
}
bool StateManager::TryLoadServerState(MigrationState* pOut) const NN_NOEXCEPT
{
    StoredState store;
    size_t readSize;
    if (!TryLoadImpl(&readSize, &store, sizeof(store), m_Storage, UserMigrationServerStatePath))
    {
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(readSize, sizeof(store));
    if (store.fwVersion != GetFirmwareVersion())
    {
        return false;
    }
    NN_SDK_ASSERT(IsValidState(store.state));
    *pOut = store.state;
    return true;
}
bool StateManager::TryLoadClientState(MigrationState* pOut) const NN_NOEXCEPT
{
    StoredState store;
    size_t readSize;
    if (!TryLoadImpl(&readSize, &store, sizeof(store), m_Storage, UserMigrationClientStatePath))
    {
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(readSize, sizeof(store));
    if (store.fwVersion != GetFirmwareVersion())
    {
        return false;
    }
    NN_SDK_ASSERT(IsValidState(store.state));
    *pOut = store.state;
    return true;
}
void StateManager::StoreServerState(MigrationState state) const NN_NOEXCEPT
{
    StoredState store = {GetFirmwareVersion(), state};
    StoreImpl(m_Storage, UserMigrationServerStatePath, &store, sizeof(store));
}
void StateManager::StoreClientState(MigrationState state) const NN_NOEXCEPT
{
    StoredState store = {GetFirmwareVersion(), state};
    StoreImpl(m_Storage, UserMigrationClientStatePath, &store, sizeof(store));
}
Result StateManager::UpdateServerState(MigrationState state) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidState(state));

    MigrationState currentState;
    NN_RESULT_THROW_UNLESS(TryLoadServerState(&currentState), detail::ResultServerStateDataLoadError());
    NN_RESULT_THROW_UNLESS(IsValidTransition(state, currentState), detail::ResultInvalidServerStateTransition());

    StoreServerState(state);
    NN_RESULT_SUCCESS;
}
Result StateManager::UpdateClientState(MigrationState state) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidState(state));

    MigrationState currentState;
    NN_RESULT_THROW_UNLESS(TryLoadClientState(&currentState), detail::ResultClientStateDataLoadError());
    NN_RESULT_THROW_UNLESS(IsValidTransition(state, currentState), detail::ResultInvalidClientStateTransition());

    StoreClientState(state);
    NN_RESULT_SUCCESS;
}
void StateManager::ResetServer() const NN_NOEXCEPT
{
    MigrationState currentState;
    if (TryLoadServerState(&currentState) && currentState != MigrationState_Finalized)
    {
        NN_MIGRATION_DETAIL_WARN("[Server] Resetting ongoing migration: state=%d\n", currentState);
    }

    DeleteImpl(m_Storage, UserMigrationServerStatePath);
}
void StateManager::ResetClient() const NN_NOEXCEPT
{
    MigrationState currentState;
    if (TryLoadClientState(&currentState) && currentState != MigrationState_Finalized)
    {
        NN_MIGRATION_DETAIL_WARN("[Client] Resetting ongoing migration: state=%d\n", currentState);
    }

    DeleteImpl(m_Storage, UserMigrationClientStatePath);
}
Result StateManager::TryGetLastMigrationInfo(bool* pOut, LastMigrationInfo* pOutInfo) const NN_NOEXCEPT
{
    bool serverResult;
    MigrationState serverState;
    NN_RESULT_DO(TryLoadLastMigrationServerState(&serverResult, &serverState));

    bool clientResult;
    MigrationState clientState;
    NN_RESULT_DO(TryLoadLastMigrationClientState(&clientResult, &clientState));

    NN_ABORT_UNLESS(!serverResult || !clientResult);

    if (serverResult)
    {
        LastMigrationInfo info = {
            MigrationRole::MigrationRole_Server,
            serverState,
        };
        *pOut = true;
        *pOutInfo = info;
        NN_RESULT_SUCCESS;
    }
    else if (clientResult)
    {
        LastMigrationInfo info = {
            MigrationRole::MigrationRole_Client,
            clientState,
        };
        *pOut = true;
        *pOutInfo = info;
        NN_RESULT_SUCCESS;
    }
    *pOut = false;
    NN_RESULT_SUCCESS;
}
Result StateManager::StartServer(StateController** pOut) NN_NOEXCEPT
{
    bool exists;
    MigrationState state;
    NN_RESULT_DO(TryLoadLastMigrationServerState(&exists, &state));
    NN_RESULT_THROW_UNLESS(!exists, detail::ResultInvalidProtocol());

    m_Storage.Cleanup();
    StoreServerState(MigrationState_Started);

    *pOut = &m_ServerController;
    NN_RESULT_SUCCESS;
}
Result StateManager::ResumeServer(StateController** pOut) NN_NOEXCEPT
{
    bool exists;
    MigrationState state;
    NN_RESULT_DO(TryLoadLastMigrationServerState(&exists, &state));
    NN_RESULT_THROW_UNLESS(exists, detail::ResultInvalidProtocol());

    *pOut = &m_ServerController;
    NN_RESULT_SUCCESS;
}
Result StateManager::StartClient(StateController** pOut) NN_NOEXCEPT
{
    bool exists;
    MigrationState state;
    NN_RESULT_DO(TryLoadLastMigrationClientState(&exists, &state));
    NN_RESULT_THROW_UNLESS(!exists, detail::ResultInvalidProtocol());

    m_Storage.Cleanup();
    StoreClientState(MigrationState_Started);

    *pOut = &m_ClientController;
    NN_RESULT_SUCCESS;
}
Result StateManager::ResumeClient(StateController** pOut) NN_NOEXCEPT
{
    bool exists;
    MigrationState state;
    NN_RESULT_DO(TryLoadLastMigrationClientState(&exists, &state));
    NN_RESULT_THROW_UNLESS(exists, detail::ResultInvalidProtocol());

    *pOut = &m_ClientController;
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::user
