﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/migration_UserMigrationApi.h>

#include <mutex>

#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/user/migration_Interface.sfdl.h>
#include <nn/migration/migration_UserMigrationServiceTypes.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_MutexApi.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>

namespace nn { namespace migration {

namespace {
struct Lock
{
    os::MutexType data;

    void lock() NN_NOEXCEPT
    {
        os::LockMutex(&data);
    }
    void unlock() NN_NOEXCEPT
    {
        os::UnlockMutex(&data);
    }
} g_Lock = {NN_OS_MUTEX_INITIALIZER(true)};
sf::SharedPointer<user::IService> (*g_Initializer)() = nullptr;
sf::SharedPointer<user::IService> g_Ptr = nullptr;

sf::SharedPointer<user::IService> GetServicePointerViaHipc() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    std::lock_guard<Lock> lock(g_Lock);
    // セッションマネージャ
    NN_FUNCTION_LOCAL_STATIC(sf::HipcSimpleClientSessionManager, s_HipcDomain);
    // セッションアロケータ
    static const size_t SizeOfHeapToAcquireObject = 1024u;
    struct UserMigrationServiceAllocatorTag {};
    typedef sf::ExpHeapStaticAllocator<SizeOfHeapToAcquireObject, UserMigrationServiceAllocatorTag> Allocator;
    // HIPC 経由で取得したサービスオブジェクトのポインタ
    typedef sf::SharedPointer<user::IService> ServicePointer;
    NN_FUNCTION_LOCAL_STATIC(ServicePointer, s_PtrHipc, = nullptr);

    if (!s_PtrHipc)
    {
        Allocator::Initialize(lmem::CreationOption_NoOption);
        NN_ABORT_UNLESS_RESULT_SUCCESS((s_HipcDomain.InitializeByName<user::IService, Allocator::Policy>(&s_PtrHipc, UserMigrationServiceName)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(s_HipcDomain.SetSessionCount(1));
    }
    return s_PtrHipc;

#elif defined(NN_BUILD_CONFIG_OS_WIN)
    return nullptr;
#else
#error "Unsupported OS specified"
#endif
}

sf::SharedPointer<user::IService> GetServicePointer() NN_NOEXCEPT
{
    std::lock_guard<Lock> lock(g_Lock);
    if (!g_Ptr)
    {
        if (g_Initializer)
        {
            g_Ptr = g_Initializer();
            NN_ABORT_UNLESS_NOT_NULL(g_Ptr);
        }
        else
        {
            g_Ptr = GetServicePointerViaHipc();
        }
    }
    return g_Ptr;
}

} // ~namespace nn::migration::<anonymous>

void DebugSetUserMigrationInitializer(sf::SharedPointer<user::IService> (*initializer)()) NN_NOEXCEPT
{
    std::lock_guard<Lock> lock(g_Lock);
    g_Initializer = initializer;
}
void DebugResetUserMigrationServiceAccessor() NN_NOEXCEPT
{
    std::lock_guard<Lock> lock(g_Lock);
    g_Ptr = nullptr;
}

LastUserMigrationState GetLastUserMigrationState() NN_NOEXCEPT
{
    bool exist;
    user::LastMigrationInfo raw;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->TryGetLastMigrationInfo(&exist, &raw));
    return exist
        ? user::Convert(raw)
        : LastUserMigrationState::Unresumable;
}

bool GetLastUserMigrationStateInDetail(user::LastMigrationInfo* pOut) NN_NOEXCEPT
{
    bool exist;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->TryGetLastMigrationInfo(&exist, pOut));
    return exist;
}

Result CleanupUserMigration(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        switch (GetLastUserMigrationState())
        {
        case LastUserMigrationState::Unresumable:
            NN_RESULT_SUCCESS;

        case LastUserMigrationState::ClientResumable:
        case LastUserMigrationState::ClientResumeSuggested:
            NN_RESULT_DO(ResumeUserMigrationClient(buffer, bufferSize).Abort());
            break;

        case LastUserMigrationState::ServerResumable:
        case LastUserMigrationState::ServerResumeSuggested:
            NN_RESULT_DO(ResumeUserMigrationServer(buffer, bufferSize).Abort());
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

UserMigrationServer CreateUserMigrationServer(const account::Uid& uid, const UserMigrationServerProfile& profile, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, RequiredWorkBufferSizeForUserMigrationOperation);
    NN_SDK_REQUIRES_EQUAL(bufferSize % os::MemoryPageSize, 0u);

    sf::SharedPointer<user::IServer> p;
    os::TransferMemory memory(buffer, bufferSize, os::MemoryPermission_None);
    sf::NativeHandle handle(memory.Detach(), true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->CreateServer(&p, uid, profile, std::move(handle), static_cast<uint32_t>(bufferSize)));

    return UserMigrationServer(p.Detach());
}
UserMigrationServer ResumeUserMigrationServer(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, RequiredWorkBufferSizeForUserMigrationOperation);
    NN_SDK_REQUIRES_EQUAL(bufferSize % os::MemoryPageSize, 0u);

    sf::SharedPointer<user::IServer> p;
    os::TransferMemory memory(buffer, bufferSize, os::MemoryPermission_None);
    sf::NativeHandle handle(memory.Detach(), true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->ResumeServer(&p, std::move(handle), static_cast<uint32_t>(bufferSize)));

    return UserMigrationServer(p.Detach());
}

UserMigrationClient CreateUserMigrationClient(const UserMigrationClientProfile& profile, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, RequiredWorkBufferSizeForUserMigrationOperation);
    NN_SDK_REQUIRES_EQUAL(bufferSize % os::MemoryPageSize, 0u);

    sf::SharedPointer<user::IClient> p;
    os::TransferMemory memory(buffer, bufferSize, os::MemoryPermission_None);
    sf::NativeHandle handle(memory.Detach(), true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->CreateClient(&p, profile, std::move(handle), static_cast<uint32_t>(bufferSize)));

    return UserMigrationClient(p.Detach());
}
UserMigrationClient ResumeUserMigrationClient(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, RequiredWorkBufferSizeForUserMigrationOperation);
    NN_SDK_REQUIRES_EQUAL(bufferSize % os::MemoryPageSize, 0u);

    sf::SharedPointer<user::IClient> p;
    os::TransferMemory memory(buffer, bufferSize, os::MemoryPermission_None);
    sf::NativeHandle handle(memory.Detach(), true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetServicePointer()->ResumeClient(&p, std::move(handle), static_cast<uint32_t>(bufferSize)));

    return UserMigrationClient(p.Detach());
}

}} // ~namespace nn::migration
