﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_SharedBufferConnection.h>
#include <nn/migration/idc/migration_SharedBufferConnectionManager.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

void swap(SharedBufferConnection& o0, SharedBufferConnection& o1) NN_NOEXCEPT
{
    std::swap(o0.m_pManager, o1.m_pManager);
    std::swap(o0.m_Id, o1.m_Id);
    std::swap(o0.m_pSendBuffer, o1.m_pSendBuffer);
    std::swap(o0.m_pRecvBuffer, o1.m_pRecvBuffer);
}

SharedBufferConnection::SharedBufferConnection() NN_NOEXCEPT
    : m_pManager(nullptr)
    , m_Id(-1)
    , m_pSendBuffer(nullptr)
    , m_pRecvBuffer(nullptr)
{
}

SharedBufferConnection::SharedBufferConnection(SharedBufferConnectionManager* pManager, int id, SharedBufferConnectionManager::SharedBuffer* pSendResource, SharedBufferConnectionManager::SharedBuffer* pRecvResource) NN_NOEXCEPT
    : m_pManager(pManager)
    , m_Id(id)
    , m_pSendBuffer(pSendResource)
    , m_pRecvBuffer(pRecvResource)
{
}

SharedBufferConnection::SharedBufferConnection(SharedBufferConnection&& rhs) NN_NOEXCEPT
    : m_pManager(rhs.m_pManager)
    , m_Id(rhs.m_Id)
    , m_pSendBuffer(rhs.m_pSendBuffer)
    , m_pRecvBuffer(rhs.m_pRecvBuffer)
{
    rhs.m_pManager = nullptr;
    rhs.m_Id = -1;
    rhs.m_pSendBuffer = nullptr;
    rhs.m_pRecvBuffer = nullptr;
}

SharedBufferConnection::~SharedBufferConnection() NN_NOEXCEPT
{
    if( m_pManager != nullptr )
    {
        Close();
    }
}

SharedBufferConnection& SharedBufferConnection::operator =(SharedBufferConnection&& rhs) NN_NOEXCEPT
{
    SharedBufferConnection tmp(std::move(rhs));
    swap(*this, tmp);
    return *this;
}

Result SharedBufferConnection::Send(
    size_t* pOutSentSize, const void* stream, size_t size, int timeoutSeconds,
    const migration::detail::Cancellable* pCancellable,
    detail::TransferSpeedMonitor* pSpeedMonitor) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSentSize);
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);

    NN_SDK_REQUIRES_NOT_NULL(m_pSendBuffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pRecvBuffer);

    NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());

    const int PollIntervalMs = 10;

    os::TimerEvent timeoutTimer(os::EventClearMode_AutoClear);
    if( timeoutSeconds > 0 )
    {
        timeoutTimer.StartOneShot(nn::TimeSpan::FromSeconds(timeoutSeconds));
    }

    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Resume();
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Pause();
        }
    };

    while( m_pSendBuffer->IsPopOpen() && !m_pSendBuffer->WaitPushable(PollIntervalMs) )
    {
        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());
        NN_RESULT_THROW_UNLESS(!timeoutTimer.TryWait(), idc::ResultTimeout());
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Update(0);
            NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
        }
    }
    // 相手が切断済みの場合は即座にその旨の Result を返す。Windows の Socket, 送信サイズが小さいときの NX の Socket とは挙動が異なる。
    NN_RESULT_THROW_UNLESS(m_pSendBuffer->IsPopOpen(), idc::ResultPeerClosed());
    *pOutSentSize = m_pSendBuffer->Push(stream, size);
    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Update(*pOutSentSize);
        NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
    }
    NN_RESULT_SUCCESS;
}

Result SharedBufferConnection::Receive(
    size_t *pOutReceivedSize, void* outStream, size_t outStreamSize, int timeoutSeconds,
    const migration::detail::Cancellable* pCancellable,
    detail::TransferSpeedMonitor* pSpeedMonitor) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReceivedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    NN_SDK_REQUIRES_GREATER(outStreamSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);

    NN_SDK_REQUIRES_NOT_NULL(m_pSendBuffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pRecvBuffer);

    NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());

    const int PollIntervalMs = 10;

    os::TimerEvent timeoutTimer(os::EventClearMode_AutoClear);
    if( timeoutSeconds > 0 )
    {
        timeoutTimer.StartOneShot(nn::TimeSpan::FromSeconds(timeoutSeconds));
    }

    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Resume();
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Pause();
        }
    };

    while( !m_pRecvBuffer->WaitPoppable(PollIntervalMs) )
    {
        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());
        NN_RESULT_THROW_UNLESS(!timeoutTimer.TryWait(), idc::ResultTimeout());
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Update(0);
            NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
        }
        // 相手がクローズしていても受信できるデータがある限りは受信するためループ内でチェック。
        NN_RESULT_THROW_UNLESS(m_pRecvBuffer->IsPushOpen(), idc::ResultPeerClosed());
    }
    *pOutReceivedSize = m_pRecvBuffer->Pop(outStream, outStreamSize);
    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Update(*pOutReceivedSize);
        NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
    }
    NN_RESULT_SUCCESS;
}

void SharedBufferConnection::Close() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pManager);
    m_pManager->Close(this);
    m_pManager = nullptr;
}

}}};
