﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_LoginSession.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace migration { namespace detail {

namespace {

bool IsUserRegistered(const account::Uid& uid) NN_NOEXCEPT
{
    bool existence;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserExistence(&existence, uid));
    return existence;
}

bool IsNsaRegistered(const account::Uid& uid) NN_NOEXCEPT
{
    bool registered;
    account::NetworkServiceAccountAdministrator admin;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountAdministrator(&admin, uid));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
    return registered;
}

Result DeleteUserSaveData(const account::Uid& uid, const Cancellable* pCancellable) NN_NOEXCEPT
{
    ns::UserSaveDataStatistics total;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(ns::CalculateUserSaveDataStatistics(&total, uid));

    ns::ProgressMonitorForDeleteUserSaveDataAll prog;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(ns::DeleteUserSaveDataAll(&prog, uid));

    os::TimerEvent timer(os::EventClearMode_ManualClear);
    timer.StartPeriodic(TimeSpan::FromMilliSeconds(500), TimeSpan::FromMilliSeconds(500));
    os::SystemEvent e;
    prog.GetSystemEvent(&e);
    do
    {
        os::WaitAny(e.GetBase(), timer.GetBase());
        NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());
        timer.Clear();

        prog.Update();
        auto current = prog.GetStatistics();
        NN_MIGRATION_DETAIL_INFO(
            " - DeleteUserSaveData: %u of %u (%llu of %llu bytes)\n",
            current.count, total.count,
            current.sizeInBytes, total.sizeInBytes);
    } while (!e.TryWait());

    return prog.GetResult();
}

template <int IntervalMilliSeconds>
Result WaitAccountAsyncContext(account::AsyncContext&& asyncContext, const Cancellable* pCancellable) NN_NOEXCEPT
{
    account::AsyncContext ctx = std::move(asyncContext);
    os::SystemEvent e;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    if (!detail::CancellableWait<IntervalMilliSeconds>(e, pCancellable))
    {
        ctx.Cancel();
        e.Wait();
        NN_RESULT_THROW(ResultCanceled());
    }
    return ctx.GetResult();
}

Result RestoreUserRights(const account::Uid& uid, const Cancellable* pCancellable) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_UNUSED(pCancellable);

    NN_RESULT_DO(npns::Suspend());
    NN_UTIL_SCOPE_EXIT
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(npns::Resume());
    };

    // NT 登録
    auto r = npns::CreateJid();
    if (!r.IsSuccess())
    {
        NN_MIGRATION_DETAIL_WARN("<!> npns::CreateJid failed with %03d-%04d (%08lx)\n", r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
    }
    NN_RESULT_DO(nn::npns::UploadTokenToBaaS(uid));
#else
    NN_UNUSED(uid);
    NN_UNUSED(pCancellable);
#endif
    NN_RESULT_SUCCESS;
}

} // ~namesapce nn::migration::detail::<anonymous>

Result ClearUserData(const account::Uid& uid, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(uid);

    if (IsUserRegistered(uid) && IsNsaRegistered(uid))
    {
        NN_RESULT_DO(ns::UnregisterNetworkServiceAccount(uid));
    }

    NN_RESULT_DO(DeleteUserSaveData(uid, pCancellable));

    if (IsUserRegistered(uid))
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(uid));
    }
    NN_RESULT_SUCCESS;
}

LocalUserLoginSession::LocalUserLoginSession() NN_NOEXCEPT
    : m_Uid(account::InvalidUid)
    , m_Initialized(false)
{
}

void LocalUserLoginSession::Initialize(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(uid);
    m_Uid = uid;
    m_Initialized = true;
}

Result LocalUserLoginSession::GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Initialized);
    account::NetworkServiceAccountManager manager;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&manager, m_Uid));
    return manager.GetNetworkServiceAccountId(pOut);
}

Result LocalUserLoginSession::AcquireNetworkServiceAccountIdToken(
    size_t* pOut, char* buffer, size_t bufferSize,
    const account::SystemProgramIdentification& sysAppId, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Initialized);

    account::NetworkServiceAccountManager manager;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&manager, m_Uid));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(manager.SetSystemProgramIdentification(sysAppId));

    account::AsyncContext ctx;
    NN_RESULT_DO(manager.EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx));
    NN_RESULT_DO(WaitAccountAsyncContext<500>(std::move(ctx), pCancellable));

    if (!buffer)
    {
        NN_SDK_ASSERT(!pOut);
        NN_SDK_ASSERT(bufferSize == 0u);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(manager.LoadNetworkServiceAccountIdTokenCache(pOut, buffer, bufferSize));
    NN_RESULT_SUCCESS;
}

Result LocalUserLoginSession::DeleteUser(const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    return ClearUserData(m_Uid, pCancellable);
}

// ------------------------------------------------------------------------------------------------
// RemoteUserLoginSession

RemoteUserLoginSession::RemoteUserLoginSession(void* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Lock(false)
    , m_Initialized(false)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar);
    NN_UNUSED(bufferSize);
    m_Buffer = reinterpret_cast<WorkMemory*>(buffer);
}
account::SessionId RemoteUserLoginSession::Create() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    m_Registrar = decltype(m_Registrar)();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::CreateExternalNetworkServiceAccountRegistrar(&m_Registrar, m_Buffer, sizeof(*m_Buffer)));
    m_Initialized = true;
    m_Uid = account::InvalidUid;
    m_IsRegistered = false;

    account::SessionId sessionId;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Registrar.GetOAuthProcedureSessionId(&sessionId));
    return sessionId;
}
Result RemoteUserLoginSession::GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    return m_Registrar.GetNetworkServiceAccountId(pOut);
}
Result RemoteUserLoginSession::GetNickname(account::Nickname* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    return m_Registrar.GetNickname(pOut);
}
Result RemoteUserLoginSession::GetProfileImage(size_t *pOut, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, account::ProfileImageBytesMax);

    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    return m_Registrar.GetProfileImage(pOut, buffer, bufferSize);
}

Result RemoteUserLoginSession::AcquireNetworkServiceAccountIdToken(
    size_t* pOut, char* buffer, size_t bufferSize,
    const account::SystemProgramIdentification& sysAppId, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);

    NN_RESULT_DO(m_Registrar.SetSystemProgramIdentification(sysAppId));

    account::AsyncContext ctx;
    NN_RESULT_DO(m_Registrar.EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx));
    NN_RESULT_DO(WaitAccountAsyncContext<500>(std::move(ctx), pCancellable));

    if (!buffer)
    {
        NN_SDK_ASSERT(!pOut);
        NN_SDK_ASSERT(bufferSize == 0u);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(m_Registrar.LoadNetworkServiceAccountIdTokenCache(pOut, buffer, bufferSize));
    NN_RESULT_SUCCESS;
}

void RemoteUserLoginSession::SetUid(const account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    NN_SDK_ASSERT(!m_Uid);
    m_Uid = uid;
}

void RemoteUserLoginSession::RegisterUser() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    NN_SDK_ASSERT(m_Uid);
    auto r = m_Registrar.RegisterUserAs(m_Uid);
    if (!(r.IsSuccess()))
    {
        if (!account::ResultUidAlreadyRegistered::Includes(r))
        {
            // 登録成功 or すでに登録済みの Uid 以外は abort
            NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
        }
        NN_MIGRATION_DETAIL_WARN("<!> Uid is already registered: %016llx_%016llx\n", m_Uid._data[0], m_Uid._data[1]);
        return;
    }
    m_IsRegistered = true;
}

Result RemoteUserLoginSession::RegisterNetworkServiceAccount(const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Initialized);
    NN_SDK_ASSERT(m_Uid);
    if (m_IsRegistered)
    {
        account::AsyncContext ctx;
        NN_RESULT_DO(m_Registrar.RegisterNetworkServiceAccountAsync(&ctx));
        NN_RESULT_DO(WaitAccountAsyncContext<500>(std::move(ctx), pCancellable));
    }
    else
    {
        account::AsyncContext ctx;
        NN_RESULT_DO(m_Registrar.RegisterNetworkServiceAccountAsync(&ctx, m_Uid));
        NN_RESULT_DO(WaitAccountAsyncContext<500>(std::move(ctx), pCancellable));
    }
    m_Registrar = decltype(m_Registrar)();
    return RestoreUserRights(m_Uid, pCancellable);
}

// ------------------------------------------------------------------------------------------------
// ClientLoginSessionHolder

ClientLoginSessionHolder::ClientLoginSessionHolder(void* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Session(buffer, bufferSize)
    , m_Lock(false)
{
}
ClientLoginSessionHolder::~ClientLoginSessionHolder() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Counter.Get() == 0);
}
UniqueResource<RemoteUserLoginSession> ClientLoginSessionHolder::Acquire() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_Counter.Get() == 0);
    return UniqueResource<RemoteUserLoginSession>(m_Session, m_Counter);
}

}}} // ~namespace nn::migration::detail
