﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/manu/manu_UfioRequestHipcServer.h>

#include <nn/nn_Common.h>
#include <nn/sf/sf_HipcServer.h>
#include "manu_CreateUfioRequest.h"
#include "manu_ServiceName.h"
#include <type_traits>
#include <new>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace manu {

namespace {

struct ServerOptions
{
    static const size_t PointerTransferBufferSize = 4096;
};

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1, ServerOptions>
{
};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

}

void InitializeUfioRequestServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // CreateUfioRequestByDfc() で得られるオブジェクトを、サービス名で関連付けて、システムに登録する。
    // これによって、ここで指定したサービス名に対して nn::sf::CreateHipcProxyByName() を呼ぶことで、
    // 指定したオブジェクトへの HIPC プロキシ参照を得ることができる。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけば良い。
    auto sessionCountMax = 30;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMyServerManager->RegisterObjectForPort(GetUfioRequestByDfc(), sessionCountMax, ManuServiceName));

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
}

void LoopUfioRequestServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // サーバの処理ループ実行を実行する。
    // - ポートへの接続要求
    // - セッションへの処理要求
    // RequestStop が呼ばれるまで返らない。
    g_pMyServerManager->LoopAuto();
}

void RequestStopUfioRequestServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
}

void FinalizeUfioRequestServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

}}
