﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/diag/text/diag_SdkTextLdn.h>
#include <nn/err/err_Api.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Service/ldn_CreateService.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn { namespace ldn { namespace
{
    ::nn::sf::SharedPointer<detail::IMonitorService> g_pMonitorService;
    ::nn::sf::SharedPointer<detail::ILocalCommunicationService> g_pLocalCommunicationService;
    ::nn::sf::SharedPointer<detail::IUserLocalCommunicationService>
        g_pUserLocalCommunicationService;
    ::nn::sf::SharedPointer<detail::ISystemLocalCommunicationService>
        g_pSystemLocalCommunicationService;

    const int DummyProcessId = 0;

    // 基本的には NN_RESULT_DO と同等ですが、
    // Debug/Develop ビルドでは ResultBadRequest 発生時にクラッシュさせます。
    #define NN_LDN_RESULT_DO(result) do\
    {\
        ::nn::Result tmpResult = (result);\
        NN_SDK_ASSERT(!nn::ldn::ResultUnauthorizedLocalCommunicationId::Includes(tmpResult),\
            NN_TEXT_LDN("許可されていないローカル通信識別子が指定されました。"));\
        NN_SDK_ASSERT(!::nn::ldn::ResultBadRequest::Includes(tmpResult));\
        NN_RESULT_DO(tmpResult);\
    } while (NN_STATIC_CONDITION(false))

}}} // namespace nn::ldn::<unnamed>

namespace nn { namespace ldn
{
    Result Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!g_pUserLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは既に初期化されています。"));

        // LDN プロセスのサービスをオープンします。
        auto pServiceCreator = detail::CreateUserServiceCreator();

        // このクライアントプロセス用のサービスオブジェクトを取得します。
        nn::sf::SharedPointer<detail::IUserLocalCommunicationService> pLocalCommunicationService;
        pServiceCreator->CreateUserLocalCommunicationService(&pLocalCommunicationService);
        NN_SDK_ASSERT_NOT_NULL(pLocalCommunicationService);
        pServiceCreator = nullptr;

        // LDN ライブラリを初期化します。
        Result result = pLocalCommunicationService->Initialize(DummyProcessId);
        if (result.IsSuccess())
        {
            g_pLocalCommunicationService = pLocalCommunicationService;
            g_pUserLocalCommunicationService = pLocalCommunicationService;
        }
        NN_LDN_RESULT_DO(result);
        NN_RESULT_SUCCESS;
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pUserLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));

        // LDN ライブラリを解放します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pUserLocalCommunicationService->Finalize());
        g_pLocalCommunicationService = g_pSystemLocalCommunicationService;
        g_pUserLocalCommunicationService = nullptr;
    }

    Result InitializeSystem() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!g_pSystemLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは既に初期化されています。"));

        // LDN プロセスのサービスをオープンします。
        auto pServiceCreator = detail::CreateSystemServiceCreator();

        // このクライアントプロセス用のサービスオブジェクトを取得します。
        nn::sf::SharedPointer<detail::ISystemLocalCommunicationService> pLocalCommunicationService;
        pServiceCreator->CreateSystemLocalCommunicationService(&pLocalCommunicationService);
        NN_SDK_ASSERT_NOT_NULL(pLocalCommunicationService);
        pServiceCreator = nullptr;

        // LDN ライブラリを初期化します。
        Result result = pLocalCommunicationService->Initialize(DummyProcessId);
        if (result.IsSuccess())
        {
            g_pLocalCommunicationService = pLocalCommunicationService;
            g_pSystemLocalCommunicationService = pLocalCommunicationService;
        }
        NN_LDN_RESULT_DO(result);
        NN_RESULT_SUCCESS;
    }

    void FinalizeSystem() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pSystemLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));

        // LDN ライブラリを解放します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pSystemLocalCommunicationService->Finalize());
        g_pLocalCommunicationService = g_pUserLocalCommunicationService;
        g_pSystemLocalCommunicationService = nullptr;
    }

    void InitializeMonitor() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!g_pMonitorService,
            NN_TEXT_LDN("LDN ライブラリは既に初期化されています。"));

        // LDN プロセスのサービスをオープンします。
        auto pServiceCreator = detail::CreateMonitorServiceCreator();

        // このクライアントプロセス用のサービスオブジェクトを取得します。
        nn::sf::SharedPointer<detail::IMonitorService> pMonitorService;
        pServiceCreator->CreateMonitorService(&pMonitorService);
        NN_SDK_ASSERT_NOT_NULL(pMonitorService);
        pServiceCreator = nullptr;

        // LDN ライブラリを初期化します。
        Result result = pMonitorService->Initialize();
        if (result.IsSuccess())
        {
            g_pMonitorService = pMonitorService;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    void FinalizeMonitor() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMonitorService, NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMonitorService->Finalize());
        g_pMonitorService = nullptr;
    }

    void AttachStateChangeEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);

        // LDN ライブラリが保持するイベントのハンドルを取得します。
        nn::sf::NativeHandle readableHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            g_pLocalCommunicationService->GetStateChangeEvent(&readableHandle));

        // 引数として受け取ったイベントオブジェクトにハンドルをアタッチします。
        nn::os::AttachReadableHandleToSystemEvent(
            pOutSystemEvent,
            readableHandle.GetOsHandle(),
            readableHandle.IsManaged(),
            nn::os::EventClearMode_AutoClear
        );
        readableHandle.Detach();
    }

    State GetState() NN_NOEXCEPT
    {
        Bit32 state;

        // 解放されないように Shared Pointer を複製しておきます。
        auto pLocalCommunicationService = g_pLocalCommunicationService;

        // LDN ライブラリが初期化されていれば状態を取得します。
        if (pLocalCommunicationService && pLocalCommunicationService->GetState(&state).IsSuccess())
        {
            return static_cast<State>(state);
        }
        else
        {
            return State_None;
        }
    }

    State GetStateForMonitor() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMonitorService, NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        Bit32 state;
        NN_LDN_RESULT_DO(g_pMonitorService->GetState(&state));
        return static_cast<State>(state);
    }

    Result Scan(NetworkInfo* outBuffer, int* pOutCount, int bufferCount,
                const ScanFilter& filter, int channel) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pOutCount);
        NN_SDK_REQUIRES(0 < bufferCount);
        NN_SDK_REQUIRES(channel == AutoChannel || channel == 1 || channel == 6 || channel == 11 ||
                        channel == 36 || channel == 40 || channel == 44 || channel == 48);
        NN_SDK_REQUIRES((filter.flag & ScanFilterFlag_NetworkType) == 0 ||
                        (filter.networkType & ~(NetworkType_General | NetworkType_Ldn)) == 0);
        NN_SDK_REQUIRES((filter.flag & ScanFilterFlag_Ssid) == 0 || IsValidSsid(filter.ssid));
        NN_SDK_REQUIRES((filter.flag & ScanFilterFlag_Bssid) == 0);
        NN_SDK_REQUIRES((filter.flag & ~ScanFilterFlag_All) == 0);

        // ScanFilter の予約領域を 0 埋めします。
        ScanFilter copiedFilter = { };
        copiedFilter.networkId = filter.networkId;
        copiedFilter.networkType = filter.networkType;
        copiedFilter.ssid = filter.ssid;
        copiedFilter.flag = filter.flag & ScanFilterFlag_All & (~ScanFilterFlag_Bssid);

        // Scan を実行します。
        nn::sf::OutArray<NetworkInfo> outScanResultList(outBuffer, bufferCount);
        int16_t count;
        int16_t ch = static_cast<int16_t>(channel);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->Scan(
            outScanResultList, &count, copiedFilter, ch));

        // 成功した場合には取得した NetworkInfo の個数を保存します。
        *pOutCount = count;
        NN_RESULT_SUCCESS;
    }

    Result GetNetworkInfo(
        NetworkInfo* pOutNetwork, NodeLatestUpdate *pOutUpdates, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutNetwork);
        NN_SDK_REQUIRES_NOT_NULL(pOutUpdates);
        NN_SDK_REQUIRES(0 < bufferCount);
        nn::sf::OutArray<NodeLatestUpdate> updates(pOutUpdates, bufferCount);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->GetNetworkInfoAndHistory(
            pOutNetwork, updates));
        NN_RESULT_SUCCESS;
    }

    Result GetNetworkInfo(NetworkInfo* pOutNetwork) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutNetwork);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->GetNetworkInfo(pOutNetwork));
        NN_RESULT_SUCCESS;
    }

    Result GetNetworkInfoForMonitor(NetworkInfo* pOutNetwork) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMonitorService, NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutNetwork);
        NN_LDN_RESULT_DO(g_pMonitorService->GetNetworkInfo(pOutNetwork));
        NN_RESULT_SUCCESS;
    }

    Result GetIpv4Address(Ipv4Address* pOutAddress, SubnetMask* pOutMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
             NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_SDK_REQUIRES_NOT_NULL(pOutMask);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->GetIpv4Address(pOutAddress, pOutMask));
        NN_RESULT_SUCCESS;
    }

    Result GetIpv4AddressForMonitor(Ipv4Address* pOutAddress, SubnetMask* pOutMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMonitorService, NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_SDK_REQUIRES_NOT_NULL(pOutMask);
        NN_LDN_RESULT_DO(g_pMonitorService->GetIpv4Address(pOutAddress, pOutMask));
        NN_RESULT_SUCCESS;
    }

    Result OpenAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->OpenAccessPoint());
        NN_RESULT_SUCCESS;
    }

    Result CloseAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->CloseAccessPoint());
        NN_RESULT_SUCCESS;
    }

    Result CreateNetwork(
        const NetworkConfig& network,
        const SecurityConfig& security,
        const UserConfig& user) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_MINMAX(network.nodeCountMax, 1, NodeCountMax);
        NN_SDK_REQUIRES(network.channel == 1 || network.channel == 6 || network.channel == 11 ||
                        network.channel == 36 || network.channel == 40 || network.channel == 44 ||
                        network.channel == 48 || network.channel == AutoChannel);
        NN_SDK_REQUIRES_MINMAX(network.localCommunicationVersion,
                               LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_SDK_REQUIRES(security.securityMode == SecurityMode_Debug ||
                        security.securityMode == SecurityMode_Product ||
                        security.securityMode == SecurityMode_SystemDebug);
        NN_SDK_REQUIRES_MINMAX(security.passphraseSize, PassphraseSizeMin, PassphraseSizeMax);

        // NetworkConfig の予約領域を 0 埋めします。
        NetworkConfig copiedNetworkConfig = { };
        copiedNetworkConfig.intentId = MakeIntentId(
            network.intentId.localCommunicationId, network.intentId.sceneId);
        copiedNetworkConfig.channel = network.channel;
        copiedNetworkConfig.nodeCountMax = network.nodeCountMax;
        copiedNetworkConfig.localCommunicationVersion = network.localCommunicationVersion;

        // UserConfig の予約領域を 0 埋めします。
        UserConfig copiedUserConfig = { };
        std::memcpy(copiedUserConfig.userName, user.userName, UserNameBytesMax);

        // ネットワークを構築します。
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->CreateNetwork(
            copiedNetworkConfig, security, copiedUserConfig));
        NN_RESULT_SUCCESS;
    }

    Result CreateNetworkPrivate(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int addressEntryCount,
        const AddressEntry* addressEntries) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_MINMAX(network.nodeCountMax, 1, NodeCountMax);
        NN_SDK_REQUIRES(network.channel == 1 || network.channel == 6 || network.channel == 11 ||
                        network.channel == 36 || network.channel == 40 || network.channel == 44 ||
                        network.channel == 48 || network.channel == AutoChannel);
        NN_SDK_REQUIRES_MINMAX(network.localCommunicationVersion,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_SDK_REQUIRES(securityConfig.securityMode == SecurityMode_Debug ||
                        securityConfig.securityMode == SecurityMode_Product ||
                        securityConfig.securityMode == SecurityMode_SystemDebug);
        NN_SDK_REQUIRES_MINMAX(securityConfig.passphraseSize,
                               PassphraseSizeMin, PassphraseSizeMax);
        NN_SDK_REQUIRES_MINMAX(addressEntryCount, 0, AddressEntryCountMax);
        NN_SDK_REQUIRES(addressEntryCount == 0 || addressEntries != nullptr);

        // NetworkConfig の予約領域を 0 埋めします。
        NetworkConfig copiedNetworkConfig = {};
        copiedNetworkConfig.intentId = MakeIntentId(
            network.intentId.localCommunicationId, network.intentId.sceneId);
        copiedNetworkConfig.channel = network.channel;
        copiedNetworkConfig.nodeCountMax = network.nodeCountMax;
        copiedNetworkConfig.localCommunicationVersion = network.localCommunicationVersion;

        // UserConfig の予約領域を 0 埋めします。
        UserConfig copiedUserConfig = {};
        std::memcpy(copiedUserConfig.userName, user.userName, UserNameBytesMax);

        // ネットワークを構築します。
        nn::sf::InArray<AddressEntry> entries(addressEntries, addressEntryCount);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->CreateNetworkPrivate(
            copiedNetworkConfig, securityConfig, securityParam, copiedUserConfig, entries));
        NN_RESULT_SUCCESS;
    }

    Result DestroyNetwork() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->DestroyNetwork());
        NN_RESULT_SUCCESS;
    }

    Result SetAdvertiseData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_MINMAX(dataSize, 0U, AdvertiseDataSizeMax);
        NN_SDK_REQUIRES(data != nullptr || dataSize == 0);
        nn::sf::InBuffer inBuffer(static_cast<const char*>(data), dataSize);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->SetAdvertiseData(inBuffer));
        NN_RESULT_SUCCESS;
    }

    Result SetStationAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(policy == AcceptPolicy_AlwaysAccept ||
                        policy == AcceptPolicy_AlwaysReject ||
                        policy == AcceptPolicy_BlackList ||
                        policy == AcceptPolicy_WhiteList);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->SetStationAcceptPolicy(
            static_cast<Bit8>(policy)));
        NN_RESULT_SUCCESS;
    }

    Result Reject(Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->Reject(ipv4Address));
        NN_RESULT_SUCCESS;
    }

    Result AddAcceptFilterEntry(const NodeInfo& station) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->AddAcceptFilterEntry(station.macAddress));
        NN_RESULT_SUCCESS;
    }

    Result AddAcceptFilterEntry(MacAddress station) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->AddAcceptFilterEntry(station));
        NN_RESULT_SUCCESS;
    }

    Result ClearAcceptFilter() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->ClearAcceptFilter());
        NN_RESULT_SUCCESS;
    }

    Result OpenStation() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->OpenStation());
        NN_RESULT_SUCCESS;
    }

    Result CloseStation() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->CloseStation());
        NN_RESULT_SUCCESS;
    }

    Result Connect(
        const NetworkInfo& network,
        const SecurityConfig& security,
        const UserConfig& user,
        int version,
        ConnectOption option) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(network.common.channel == 1 || network.common.channel == 6 ||
                        network.common.channel == 11 || network.common.channel == 36 ||
                        network.common.channel == 40 || network.common.channel == 44 ||
                        network.common.channel == 48);
        NN_SDK_REQUIRES(security.securityMode == SecurityMode_Debug ||
                        security.securityMode == SecurityMode_Product ||
                        security.securityMode == SecurityMode_SystemDebug ||
                        security.securityMode == SecurityMode_Any);
        NN_SDK_REQUIRES_MINMAX(security.passphraseSize, PassphraseSizeMin, PassphraseSizeMax);
        NN_SDK_REQUIRES_MINMAX(version,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_SDK_REQUIRES_EQUAL(option & ~ConnectOption_All, ConnectOption_None);

        // UserConfig の予約領域を 0 埋めします。
        UserConfig copiedUserConfig = {};
        std::memcpy(copiedUserConfig.userName, user.userName, UserNameBytesMax);

        // ネットワークに接続します。
        Result result = g_pLocalCommunicationService->Connect(
            network, security, copiedUserConfig, version, static_cast<Bit32>(option));
        if ((option & ConnectOption_ShowErrorMessage) != 0 &&
            ResultIncompatibleVersion::Includes(result))
        {
            nn::err::ShowError(result);
        }
        NN_LDN_RESULT_DO(result);
        NN_RESULT_SUCCESS;
    }

    Result Connect(
        const NetworkInfo& network,
        const SecurityConfig& security,
        const UserConfig& user,
        int version) NN_NOEXCEPT
    {
        return Connect(network, security, user, version, ConnectOption_Default);
    }

    Result ConnectPrivate(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int version,
        ConnectOption option) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(network.channel == 1 || network.channel == 6 || network.channel == 11 ||
                        network.channel == 36 || network.channel == 40 || network.channel == 44 ||
                        network.channel == 48);
        NN_SDK_REQUIRES_MINMAX(network.localCommunicationVersion,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_SDK_REQUIRES(securityConfig.securityMode == SecurityMode_Debug ||
                        securityConfig.securityMode == SecurityMode_Product ||
                        securityConfig.securityMode == SecurityMode_SystemDebug);
        NN_SDK_REQUIRES_MINMAX(securityConfig.passphraseSize,
                               PassphraseSizeMin, PassphraseSizeMax);
        NN_SDK_REQUIRES_EQUAL(option & ~ConnectOption_All, ConnectOption_None);

        // NetworkConfig の予約領域を 0 埋めします。
        NetworkConfig copiedNetworkConfig = {};
        copiedNetworkConfig.intentId = MakeIntentId(
            network.intentId.localCommunicationId, network.intentId.sceneId);
        copiedNetworkConfig.channel = network.channel;
        copiedNetworkConfig.nodeCountMax = network.nodeCountMax;
        copiedNetworkConfig.localCommunicationVersion = network.localCommunicationVersion;

        // UserConfig の予約領域を 0 埋めします。
        UserConfig copiedUserConfig = {};
        std::memcpy(copiedUserConfig.userName, user.userName, UserNameBytesMax);

        // ネットワークに接続します。
        Result result = g_pLocalCommunicationService->ConnectPrivate(
            copiedNetworkConfig, securityConfig, securityParam,
            copiedUserConfig, version, static_cast<Bit32>(option));
        if ((option & ConnectOption_ShowErrorMessage) != 0 &&
            ResultIncompatibleVersion::Includes(result))
        {
            nn::err::ShowError(result);
        }
        NN_LDN_RESULT_DO(result);
        NN_RESULT_SUCCESS;
    }

    Result ConnectPrivate(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int version) NN_NOEXCEPT
    {
        return ConnectPrivate(
            network, securityConfig, securityParam, user, version, ConnectOption_Default);
    }

    Result Disconnect() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->Disconnect());
        NN_RESULT_SUCCESS;
    }

    Result GetSecurityParameter(SecurityParameter* pOutSecurityParameter) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutSecurityParameter);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->GetSecurityParameter(
            pOutSecurityParameter));
        NN_RESULT_SUCCESS;
    }

    Result GetNetworkConfig(NetworkConfig* pOutNetworkConfig) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(pOutNetworkConfig);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->GetNetworkConfig(
            pOutNetworkConfig));
        NN_RESULT_SUCCESS;
    }

    Result ScanPrivate(NetworkInfo* outBuffer, int* pOutCount, int bufferCount,
                       const ScanFilter& filter, int channel) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pOutCount);
        NN_SDK_REQUIRES(0 < bufferCount);
        NN_SDK_REQUIRES(channel == AutoChannel || channel == 1 || channel == 6 || channel == 11 ||
                        channel == 36 || channel == 40 || channel == 44 || channel == 48);
        NN_SDK_REQUIRES((filter.flag & ScanFilterFlag_NetworkType) == 0 ||
                        (filter.networkType & ~(NetworkType_General | NetworkType_Ldn)) == 0);
        NN_SDK_REQUIRES((filter.flag & ScanFilterFlag_Ssid) == 0 || IsValidSsid(filter.ssid));
        NN_SDK_REQUIRES((filter.flag & ~ScanFilterFlag_All) == 0);

        // ScanFilter の予約領域を 0 埋めします。
        ScanFilter copiedFilter = {};
        copiedFilter.networkId = filter.networkId;
        copiedFilter.networkType = filter.networkType;
        copiedFilter.bssid = filter.bssid;
        copiedFilter.ssid = filter.ssid;
        copiedFilter.flag = filter.flag & ScanFilterFlag_All;

        // Scan を実行します。
        nn::sf::OutArray<NetworkInfo> outScanResultList(outBuffer, bufferCount);
        int16_t count;
        int16_t ch = static_cast<int16_t>(channel);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->ScanPrivate(
            outScanResultList, &count, copiedFilter, ch));

        // 成功した場合には取得した NetworkInfo の個数を保存します。
        *pOutCount = count;
        NN_RESULT_SUCCESS;
    }

    DisconnectReason GetDisconnectReason() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));

        // 切断理由を取得します。
        int16_t reason;
        Result result = g_pLocalCommunicationService->GetDisconnectReason(&reason);

        // 成功した場合にはその切断理由を、失敗した場合には適当な切断理由を返します。
        if (result.IsSuccess())
        {
            return static_cast<DisconnectReason>(reason);
        }
        else
        {
            return DisconnectReason_Unknown;
        }
    }

    Result SetOperationMode(OperationMode mode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(mode == OperationMode_Stable || mode == OperationMode_HighSpeed);
        NN_LDN_RESULT_DO(g_pSystemLocalCommunicationService->SetOperationMode(
            static_cast<Bit32>(mode)));
        NN_RESULT_SUCCESS;
    }

    Result SetWirelessControllerRestriction(WirelessControllerRestriction restriction) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pLocalCommunicationService || !g_pMonitorService,
            NN_TEXT_LDN("監視モードではこの機能は使用できません。"));
        NN_SDK_REQUIRES(g_pLocalCommunicationService,
            NN_TEXT_LDN("LDN ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(restriction == WirelessControllerRestriction_Disabled ||
                        restriction == WirelessControllerRestriction_Enabled);
        NN_LDN_RESULT_DO(g_pLocalCommunicationService->SetWirelessControllerRestriction(
            static_cast<Bit32>(restriction)));
        NN_RESULT_SUCCESS;
    }

}} // namespace nn::ldn
