﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/ldn/detail/ldn_Context.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // コンテキストの最大数です。
    const int ContextCountMax = 16;

    // コンテキストを保存するテーブルです。インデックスがコンテキストの識別子に相当します。
    Context g_ContextTable[ContextCountMax];

    // コンテキストテーブルを保護する Mutex です。
    nn::os::Mutex g_Mutex(false);

    // コンテキストに割り当てるカウンタ値です。
    int32_t g_Counter;

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    int CreateContext() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(g_Mutex);

        // 未使用のコンテキストを検索します。
        for (int i = 0; i < ContextCountMax; ++i)
        {
            if (g_ContextTable[i].contextId == 0)
            {
                ++g_Counter;
                int contextId = (g_Counter << 4) | i;
                g_ContextTable[i].contextId = contextId;
                return contextId;
            }
        }

        // 未使用のコンテキストが見つかりませんでした。
        NN_SDK_ASSERT(false, "failed to create a context");
        return InvalidContextId;
    }

    void DestroyContext(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(g_Mutex);
        NN_SDK_ASSERT_GREATER(contextId, 0);
        auto& context = g_ContextTable[contextId % ContextCountMax];
        NN_SDK_ASSERT_EQUAL(context.contextId, contextId);
        if (contextId > 0 && context.contextId == contextId)
        {
            std::memset(&context, 0, sizeof(Context));
        }
    }

    Context* GetContext(int contextId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER(contextId, 0);
        auto& context = g_ContextTable[contextId % ContextCountMax];
        NN_SDK_ASSERT_EQUAL(context.contextId, contextId);
        return contextId > 0 && context.contextId == contextId ? &context : nullptr;
    }

}}} // namespace nn::ldn::detail
