﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/TcpIp/ldn_LdnAutoIp.h>
#include <nn/ldn/detail/Utility/ldn_Random.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // IPv4 アドレスの範囲は 169.254.x.y/24 (1 <= x <= 127 && 1 <= y <= 127) とします。
    static const int NetworkAddressCountMax = 127;
    static const int HostAddressCountMax    = 127;

    // サブネットマスクです。
    const auto LdnAutoIpSubnetMask = MakeSubnetMask(24);

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    size_t LdnAutoIpServer::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(AddressInfo) * HostAddressCountMax;
    }

    LdnAutoIpServer::LdnAutoIpServer(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_AddressSpace(static_cast<AddressInfo*>(buffer)),
          m_Counter(0),
          m_NetworkAddress(ZeroIpv4Address),
          m_MyAddress(ZeroIpv4Address)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(AddressInfo));
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_UNUSED(bufferSize);
        std::memset(m_AddressSpace, 0, sizeof(AddressInfo) * HostAddressCountMax);
    }

    LdnAutoIpServer::~LdnAutoIpServer() NN_NOEXCEPT
    {
        if (m_MyAddress != ZeroIpv4Address)
        {
            StopServer();
        }
    }

    void LdnAutoIpServer::AddEntry(
        Ipv4Address ipv4Address, MacAddress macAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_MyAddress, ZeroIpv4Address);

        // IPv4 アドレスの正当性を検証します。
        auto address = ConvertToNetworkByteOrder(ipv4Address.raw);
        uint8_t* raws = reinterpret_cast<uint8_t*>(&address);
        if (raws[0] != 169 || raws[1] != 254 ||
            raws[2] == 0 || NetworkAddressCountMax < raws[2] ||
            raws[3] == 0 || HostAddressCountMax < raws[3])
        {
            NN_LDN_LOG_WARN("invalid address: %u.%u.%u.%u\n",
                raws[0], raws[1], raws[2], raws[3]);
            return;
        }

        // ネットワーク・アドレスの正当性を検証します。
        auto networkAddress = nn::ldn::MakeNetworkAddress(
            ipv4Address, LdnAutoIpSubnetMask);
        if (m_NetworkAddress == ZeroIpv4Address)
        {
            m_NetworkAddress = networkAddress;
        }
        else if (m_NetworkAddress != networkAddress)
        {
            return;
        }

        // テーブルにアドレスを保存します。
        m_AddressSpace[raws[3] - 1].macAddress = macAddress;
    }

    void LdnAutoIpServer::ClearEntries() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_MyAddress, ZeroIpv4Address);

        // 静的割当テーブルを初期化します。
        std::memset(m_AddressSpace, 0, sizeof(AddressInfo) * HostAddressCountMax);
        m_NetworkAddress = ZeroIpv4Address;
    }

    void LdnAutoIpServer::StartServer(MacAddress macAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_MyAddress, ZeroIpv4Address);

        // 静的割当が与えられていない場合は適当なネットワーク・アドレスを生成します。
        if (m_NetworkAddress == ZeroIpv4Address)
        {
            auto x = static_cast<uint8_t>((Random<uint32_t>() % NetworkAddressCountMax) + 1);
            m_NetworkAddress = MakeIpv4Address(169, 254, x, 0);
        }

        // 自身に IPv4 アドレス 169.254.x.y を割り当てます。
        m_MyAddress = Assign(macAddress);
    }

    void LdnAutoIpServer::StopServer() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        for (int i = 0; i < HostAddressCountMax; ++i)
        {
            AddressInfo& info = m_AddressSpace[i];
            if (info.isAssigned)
            {
                info.isAssigned = false;
                info.counter = m_Counter;
            }
        }
        ++m_Counter;
        m_MyAddress = ZeroIpv4Address;
    }

    Ipv4Address LdnAutoIpServer::Assign(MacAddress macAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_NetworkAddress, ZeroIpv4Address);

        // 割り当てる IPv4 アドレスを決定します。
        int      index   = -1;
        uint32_t diffMax = 0;
        for (int i = 0; i < HostAddressCountMax; ++i)
        {
            const AddressInfo& info = m_AddressSpace[i];
            if (macAddress == info.macAddress)
            {
                // 同じ MAC アドレスに対して IPv4 アドレスを割り当てたことがある場合は
                // その IPv4 アドレスを優先的に割り当てます。
                NN_SDK_ASSERT(!info.isAssigned);
                index = i;
                break;
            }
            else if (!info.isAssigned)
            {
                // 利用されていない期間が長いアドレスを優先的に割り当てます。
                uint32_t diff = m_Counter - info.counter;
                if (index < 0 || diffMax < diff)
                {
                    index   = i;
                    diffMax = diff;
                }
            }
        }
        NN_SDK_ASSERT(0 <= index);

        // IPv4 アドレスの割り当てを記録します。
        AddressInfo& info = m_AddressSpace[index];
        info.macAddress = macAddress;
        info.isAssigned = true;

        // IPv4 アドレスを生成して返します.
        auto hostAddress = static_cast<Bit32>(index + 1);
        Ipv4Address ipv4Address = { m_NetworkAddress.raw | hostAddress };
        return ipv4Address;
    }

    void LdnAutoIpServer::Free(Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_NetworkAddress, ZeroIpv4Address);

        // ホストアドレスを取得します。
        Bit32 hostAddress = ipv4Address.raw & 0xFF;
        NN_SDK_ASSERT(1 <= hostAddress && hostAddress <= HostAddressCountMax);

        // ネットワークアドレスを取得します。
        Ipv4Address networkAddress = MakeNetworkAddress(ipv4Address, LdnAutoIpSubnetMask);
        NN_SDK_ASSERT_EQUAL(m_NetworkAddress, networkAddress);

        // ホストアドレス、ネットワークアドレスに問題が無ければ割り当てを解除します。
        if (1 <= hostAddress && hostAddress <= HostAddressCountMax &&
            m_NetworkAddress == networkAddress)
        {
            AddressInfo& info = m_AddressSpace[hostAddress - 1];
            NN_SDK_ASSERT(info.isAssigned);
            if (info.isAssigned)
            {
                info.isAssigned = false;
                info.counter = m_Counter;
                ++m_Counter;
            }
        }
    }

    Ipv4Address LdnAutoIpServer::GetIpv4Address() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return m_MyAddress;
    }

    Ipv4Address LdnAutoIpServer::GetGatewayAddress() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return LocalLoopbackIpv4Address;
    }

    SubnetMask LdnAutoIpServer::GetSubnetMask() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return LdnAutoIpSubnetMask;
    }

    LdnAutoIpClient::LdnAutoIpClient() NN_NOEXCEPT
      : m_MyAddress(ZeroIpv4Address),
        m_NetworkAddress(ZeroIpv4Address),
        m_Gateway(ZeroIpv4Address)
    {
    }

    LdnAutoIpClient::~LdnAutoIpClient() NN_NOEXCEPT
    {
        if (m_MyAddress != ZeroIpv4Address)
        {
            StopClient();
        }
    }

    void LdnAutoIpClient::StartClient(Ipv4Address server, Ipv4Address client) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_MyAddress, ZeroIpv4Address);
        m_MyAddress = client;
        m_NetworkAddress = MakeNetworkAddress(client, LdnAutoIpSubnetMask);
        m_Gateway = server;
        NN_SDK_ASSERT_EQUAL(m_NetworkAddress, MakeNetworkAddress(server, LdnAutoIpSubnetMask));
    }

    void LdnAutoIpClient::StopClient() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        m_MyAddress = ZeroIpv4Address;
        m_NetworkAddress = ZeroIpv4Address;
        m_Gateway = ZeroIpv4Address;
    }

    Ipv4Address LdnAutoIpClient::GetIpv4Address() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return m_MyAddress;
    }

    Ipv4Address LdnAutoIpClient::GetGatewayAddress() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return m_Gateway;
    }

    SubnetMask LdnAutoIpClient::GetSubnetMask() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_MyAddress, ZeroIpv4Address);
        return LdnAutoIpSubnetMask;
    }

}}} // namespace nn::ldn::detail
